import warnings
import numpy as np
import processing

from qgis.core import QgsProject
from qgis.core import QgsVectorLayer, QgsRasterLayer
from qgis.core import QgsField
from qgis.core import QgsFeature
from qgis.core import NULL
from qgis.core import QgsProcessing
from qgis.core import QgsProcessingAlgorithm

from qgis.PyQt.QtWidgets import QAction
from qgis.PyQt.QtWidgets import QFileDialog




class dataBridge:
    def __init__(self):
        self._layerInputs = None
        self._gridResolution_m = None

        self._layers = None

        self._returnvalues = None

    def importDataFromDialog(self, dlg):
        """
        Uses the dialog
        box construct to populate the necessary fields.

        input: the dialog box object

        output: none

        side effects: populates the information ABOUT
        the fields and prepares the
        dataBridge object to provide
        data from the tables/layers/settings
        """

        # import information about the layer settings
        self._layerInputs = dlg.get_layer_input_values()

        # whether to use union or intersection of the layers
        self._extent_type = dlg.get_extent_type()

        # get actual QgsVectorLayer list of all the layers that are of interest
        self._layers = [
            QgsProject.instance().mapLayersByName(layerName)[0]
            for layerName in self._layerInputs
            if self._layerInputs[layerName] is not None
        ]

        # import the information about the size of the grid in meters
        self._gridResolution_m = dlg.get_grid_resolution()

        # get the raster layer that is of interest, if set
        self._rasterLayer = dlg.get_rasterLayerToProcess()

        # this is the user-set threshold value for points to return, if such a value was set.
        self._returnvalues = dlg.get_minimum_return_value_info()
        
        # user-set label for the sector for exporting
        self._sector = dlg.get_sectorText()

        # path and points layer to export for rencat formatting
        self._exportPath = dlg.get_exportPath()
        self._exportLayer = dlg.get_exportLayer()
        
    def checkAtLeastOneLayer(self): 
        """
        Checks that at least one layer was selected for processing in the raster. If not, raises ValueError.
        """
        if len(self.getLayers()) == 0: 
            raise ValueError("Please check at least one 'Include?' box so the plugin has at least one layer to base a raster on!")
        

    def checkLayersCRSandUnits(self):
        """
        Double-checks that all the layers are in the same, meters-based, CRS. If not,
        throws a fatal error. If success, is silent.

        The project CRS is not currently checked because it's not relevant for our purposes.
        """

        firstCRS = None

        for layer in self.getLayers():
            # check that we have a crs to compare to across all layers
            if firstCRS is None:
                firstCRS = layer.crs()
            else:
                if layer.crs() != firstCRS:
                    raise ValueError(
                        f"CRSes are not identical between selected layers. All selected layers should be in the same, meters- or feet-based,  CRS."
                    )

        # check that the shared crs across all layers uses meters
        if firstCRS.mapUnits().name not in ["Meters", "Feet"]:
            errstr = f"All layers have the same CRS, but it needs to use meters or feet. Currently is {firstCRS.mapUnits().name}"
            raise ValueError(errstr)

    def checkLayerTypes(self):
        """
        Double-checks that all layers being processed are either QgsVectorLayer or QgsRasterLayer type. Errors if not.

        Returns: none

        """
        recognized_types = (QgsVectorLayer, QgsRasterLayer)
        for layer in self.getLayers():
            if not isinstance(layer, recognized_types):
                raise ValueError(
                    f"Layer {layer} is of unrecognized type {type(layer)}. Recognized types are {recognized_types}"
                )

    def findSuperExtent(self):
        """
        Determine the overall extent on which to operate.
        By default, this is the union of all extents, so that all selected layers' extents lie
        within this extent - they may completely fill the extent, also (e.g. if you have a single layer).

        However, if the user has selected "intersection" as the option, it will only give the intersection of the two extents.

        Note that if a layer is filtered, then the extent of that layer will encompass only those
        areas/items that fulfill the filter, NOT the entire set. (So if you have some layer that
        covers the entire continental US but filter it to Kansas,
        your layer's extent will only cover those facilities which lie within Kansas, not those
        in Florida.)

        inputs: none

        returns: QgsRectangle object with the dimensions of the combined extent of all selected layers.
        """
        dims = None

        if self._extent_type == "union":
            for layer in self.getLayers():
                ext = layer.extent()
                # layer extent validity check
                if not ext.isFinite():
                    raise ValueError(
                        f"Layer {layer.name()} has a non-finite extent. This sometimes occurs when a layer is projected into a new CRS, but it has features outside the bounds of that CRS's designed extent. (Try doing everything in WGS84; it's global.) "
                    )
                if dims is None:
                    dims = ext
                else:
                    dims.combineExtentWith(ext)
        elif self._extent_type == "intersection":
            for layer in self.getLayers():
                ext = layer.extent()
                # layer extent validity check
                if not ext.isFinite():
                    raise ValueError(
                        f"Layer {layer.name()} has a non-finite extent. This sometimes occurs when a layer is projected into a new CRS, but it has features outside the bounds of that CRS's designed extent. (Try doing everything in WGS84; it's global.) "
                    )
                if dims is None:
                    dims = ext
                else:
                    dims = dims.intersect(ext)
        else:
            raise ValueError(
                f"{self._extent_type} is not a recognized form of extent combination option."
            )

        return dims

    # -----------------getters -------------

    def getLayers(self):
        return self._layers

    def getGridResolution(self):
        return self._gridResolution_m

    def getFieldofLayerByLayerName(self, layername: str):
        try:
            layerdata = self._layerInputs[layername]
        except KeyError:
            raise ValueError(f"No layer by name {layername}.")
        return layerdata["field"]

    def getIfUsingFieldByLayerName(self, layername: str):
        """
        Provided a layer name, returns whether there's a field whose data should be used
            in rasterizing/heatmapping that layer.
        If a layer by that name does not exist, will ValueError.

        Returns:
            bool. True if there is such a field, else false.

        """

        try:
            layerdata = self._layerInputs[layername]
        except KeyError:
            raise ValueError(f"No layer by name {layername}.")
        return layerdata["usefield"]

    def getInterLayerWeightByLayerName(self, layername: str):
        """
        Provided a layer name, returns what weight should be used for that
        layer's values when aggregating across layers.
        If a layer by that name does not exist, will ValueError.

        Returns:
            float

        """
        try:
            layerdata = self._layerInputs[layername]
        except KeyError:
            raise ValueError(f"No layer by name {layername}.")
        return layerdata["weight"]

    def getLayerRadiusByLayerName(self, layername: str):
        """
        Provided a layer name, returns what radius should be used in calculating
        the buffer for that layer's points or lines (if it's a polygon layer, that's
        not really relevant).
        If a layer by that name does not exist, will ValueError.

        Returns:
            float.

        """
        try:
            layerdata = self._layerInputs[layername]
        except KeyError:
            raise ValueError(f"No layer by name {layername}.")
        return layerdata["radius"]

    def getRasterLayerToProcess(self):
        return self._rasterLayer

    def getReturnValueInfo(self):
        return self._returnvalues
        
    def getExportSector(self): 
        return self._sector

    def getExportPath(self):
        return self._exportPath

    def getExportLayer(self):
        return self._exportLayer
