# -*- coding: utf-8 -*-

"""
/***************************************************************************
 RenameBands
                                 A QGIS plugin
 Rename and reorder raster bands
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2021-09-09
        copyright            : (C) 2021 by J. Pierson, UMR 6554 LETG, CNRS
        email                : julie.pierson@univ-brest.fr
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'J. Pierson, UMR 6554 LETG, CNRS'
__date__ = '2021-09-09'
__copyright__ = '(C) 2021 by J. Pierson, UMR 6554 LETG, CNRS'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import (QgsProcessing,
                       QgsProcessingAlgorithm,
                       QgsProcessingParameterRasterLayer,
                       QgsProcessingParameterMatrix,
                       QgsProcessingParameterRasterDestination)
from osgeo import gdal


class RenameBandsAlgorithm(QgsProcessingAlgorithm):
    """
    This is an algorithm that takes a raster layer and
    rename and/or reorder its bands.
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    INPUT = 'INPUT'
    TABLE = 'TABLE'
    OUTPUT = 'OUTPUT'

    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        # input raster layer
        self.addParameter(
            QgsProcessingParameterRasterLayer(
                self.INPUT,
                self.tr('Input raster layer'),
                [QgsProcessing.TypeVectorAnyGeometry]
            )
        )
        
        # table to reaorder and rename bands
        self.addParameter(
            QgsProcessingParameterMatrix(
                self.TABLE, 
                self.tr('Band names and order'), 
                numberRows=0, 
                hasFixedNumberRows = False, 
                headers=['future name', 'order'], 
                defaultValue=[None,None]
            )
        )
            
        # output raster layer
        self.addParameter(
            QgsProcessingParameterRasterDestination(
                'OUTPUT',
                self.tr('Raster output')
            )
        )



    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """

        # RETRIEVE INPUTS AND OUTPUT
        input_raster = self.parameterAsRasterLayer(parameters, self.INPUT, context)
        band_table = self.parameterAsMatrix(parameters, self.TABLE, context)
        output_raster_path = self.parameterAsOutputLayer(parameters, self.OUTPUT, context)
        
        # GETTING USEFUL PAREMETERS
        
        # number of columns in table is used later on, change it here if necessary
        nb_col = 2
        # index of column used for band order
        index_bandorder = 1
        # index of column used for band description
        index_banddesc = 0
        
        # transform band_table in a nested list, with 1 element per band, and name and order of each band
        # [['band 1', 1], ['band 2', 2], ['band 3', 3]]
        band_table = [band_table[n:n+nb_col] for n in range(0, len(band_table), nb_col)]
        # get input raster path
        input_raster_path = input_raster.source()
        # read raster with gdal
        input_raster = gdal.Open(input_raster_path)
        
        
        # REORDERING BANDS
        
        # get order of bands as a list, i.e. [3,2,1]
        band_order = [i[index_bandorder] for i in band_table]
        # reordering bands using gdal_translate
        kwargs = {
                'bandList': band_order
                }
        output_raster = gdal.Translate(output_raster_path, input_raster_path, **kwargs)
        
        
        # RENAMING BANDS
        
        # get number of bands of input raster
        nbands = input_raster.RasterCount
        # get number of rows in table
        nrows = len(band_table)
        # check if band number = row count in band_table
        if nbands != nrows:
            message = 'Number of rows in table is different from number of bands in input raster'
            feedback.reportError(QCoreApplication.translate('Rename bands', message))
            return{}
        
        # reorder band table so that bands are in same order as in output raster
        sorted_band_table = sorted(band_table, key = lambda x: x[index_bandorder])
        
        # iterate over each band to set its description using table input
        for band_number in range(nbands):
            band = output_raster.GetRasterBand(band_number + 1)
            band.SetDescription(sorted_band_table[band_number][index_banddesc])
            
            
        # RETURNING RESULTS

        # Return the results of the algorithm as a dictionary
        return {self.OUTPUT: output_raster_path}

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'Rename and reorder bands'

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr(self.name())

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr(self.groupId())

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return ''

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return RenameBandsAlgorithm()
