# -*- coding: utf-8 -*-

"""
***************************************************************************
    BandTableWidget.py
    ---------------------
    Date                 : 2021-09-14
    Copyright            : (C) 2021 by J. Pierson, UMR 6554 LETG, CNRS
    Email                : julie.pierson@univ-brest.fr
    Based upon           : ReliefColorsWidget.py (C) 2016 by Alexander Bruy
***************************************************************************
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
***************************************************************************
"""

__author__ = 'J. Pierson, UMR 6554 LETG, CNRS'
__date__ = '2021-09-14'
__copyright__ = '(C) 2021 by J. Pierson, UMR 6554 LETG, CNRS'

import os

#from PyQt5 import QtWidgets
from qgis.PyQt import uic
from qgis.PyQt.QtCore import Qt, pyqtSlot
from qgis.PyQt.QtWidgets import (QTreeWidgetItem,
                                 QMessageBox,
                                 QInputDialog,
                                 QListWidget
                                 )
from qgis.core import (QgsApplication, 
                       QgsProject,
                       QgsMapLayer
                       )
from processing.gui.wrappers import WidgetWrapper

pluginPath = os.path.dirname(__file__)
WIDGET, BASE = uic.loadUiType(os.path.join(pluginPath, 'mergebandtablewidgetbase.ui'))

# global variable for selected layers in combobox
global selectedLayers

class MergeBandTableWidget(BASE, WIDGET):

    def __init__(self):
        super(MergeBandTableWidget, self).__init__(None)
        self.setupUi(self)
        
        # allow for multiple selection in QtListWidget for bands to merge
        self.multiLayers.setSelectionMode(self.multiLayers.MultiSelection)
        
        selectedLayers = self.multiLayers.selectedItems()
        #self.multiLayers.checkedItemsChanged.connect(self.layersChanged)
        self.multiLayers.itemSelectionChanged.connect(lambda: self.layersChanged(selectedLayers))
        self.btnAdd.setIcon(QgsApplication.getThemeIcon('/symbologyAdd.svg'))
        self.btnRemove.setIcon(QgsApplication.getThemeIcon('/symbologyRemove.svg'))
        self.btnUp.setIcon(QgsApplication.getThemeIcon('/mActionArrowUp.svg'))
        self.btnDown.setIcon(QgsApplication.getThemeIcon('/mActionArrowDown.svg'))
        self.layer = None
        
        # when alg is launched by user, populate combobox with loaded raster layers
        self.populateMultiLayers()
        
        # this is how to display a message for debugging
        #QMessageBox.information(None, self.tr('window name'), self.tr('message'))
     
    # populate checkable combobox with every loaded single band raster layer
    def populateMultiLayers(self):
        # get a list of loaded raster layers
        layerList = QgsProject.instance().mapLayers().values()
        rasterList = [i for i in list(layerList) if i.type() == QgsMapLayer.RasterLayer]
        # keep only single band rasters in list
        rasterNames = [i.name() for i in list(rasterList) if i.bandCount() == 1]
        # populate combobox
        for r in rasterNames:
            self.multiLayers.addItem(r)
    
    # when user selects or unselects a layer in combobox, changes only this layer in table
    def layersChanged(self, selectedLayers):
        # get selected raster layers
        rasterNames = self.multiLayers.selectedItems()
        rasterNames = [r.text() for r in rasterNames]
        # if there is at least one selected raster
        if len(rasterNames) != 0:
            for r in rasterNames:
                # if it was not selected before
                if r not in selectedLayers:
                    # add it to table
                    self._addBandData(r)
                    # add it to list of selected layers
                    selectedLayers.append(r)
        # if a layer was selected before but not anymore
        for r in selectedLayers:
            if r not in rasterNames:
                # remove it from table
                item = self.bandClassTree.findItems(r, Qt.MatchExactly, 0)[0]
                self.bandClassTree.invisibleRootItem().removeChild(item)
                item = None
                # remove it from the list of selected layers
                selectedLayers.remove(r)
    
    # add selected raster names to table
    def _addBandData(self, rastername):
        item = QTreeWidgetItem()
        item.setText(0, rastername)
        self.bandClassTree.addTopLevelItem(item)
        
    # add line in table when add button is clicked
    @pyqtSlot()
    def on_btnAdd_clicked(self):
        pass
        
    # remove selected line in table when remove button is clicked
    @pyqtSlot()
    def on_btnRemove_clicked(self):
        selectedItems = self.bandClassTree.selectedItems()
        for item in selectedItems:
            self.bandClassTree.invisibleRootItem().removeChild(item)
            item = None
    
    # move down selected line in table when down button is clicked
    @pyqtSlot()
    def on_btnDown_clicked(self):
        selectedItems = self.bandClassTree.selectedItems()
        for item in selectedItems:
            currentIndex = self.bandClassTree.indexOfTopLevelItem(item)
            if currentIndex < self.bandClassTree.topLevelItemCount() - 1:
                self.bandClassTree.takeTopLevelItem(currentIndex)
                self.bandClassTree.insertTopLevelItem(currentIndex + 1, item)
                self.bandClassTree.setCurrentItem(item)
    
    # move up selected line in table when up button is clicked
    @pyqtSlot()
    def on_btnUp_clicked(self):
        selectedItems = self.bandClassTree.selectedItems()
        for item in selectedItems:
            currentIndex = self.bandClassTree.indexOfTopLevelItem(item)
            if currentIndex > 0:
                self.bandClassTree.takeTopLevelItem(currentIndex)
                self.bandClassTree.insertTopLevelItem(currentIndex - 1, item)
                self.bandClassTree.setCurrentItem(item)

    # when a cell in 2nd column is clicked, open dialog box for entering value
    @pyqtSlot(QTreeWidgetItem, int)
    def on_bandClassTree_itemDoubleClicked(self, item, column):
        if not item:
            return

        if column == 1:
            d, ok = QInputDialog.getText(None,
                                           self.tr('Band name'),
                                           self.tr('Enter new band name')
                                           )
            if ok:
                item.setText(1, str(d))

    # return table values in a list, one element for each row
    # [['band 1', 'new name 1'], ['band 2', 'new name 2']]
    def bandNames(self):
        band_names = []
        for i in range(self.bandClassTree.topLevelItemCount()):
            item = self.bandClassTree.topLevelItem(i)
            if item:
                row = [item.text(0), item.text(1)]
                band_names.append(row)
        return band_names

    def setLayers(self, layers):
        self.layers = layers
        self.updateTable(layers)
        
    def updateTable(self, layers):
        item = QTreeWidgetItem()
        item.setText(0, '0.00')
        item.setText(1, '0.00')
        self.bandClassTree.addTopLevelItem(item)

    # when launching alg from history in processing, setting parameters
    def setValue(self, value):
        self.bandClassTree.clear()
        # getting raster names
        param = value.split(';')
        param = [i.split(',') for i in param]
        rasterNames = [i[0] for i in param]
        # keeping only loaded layers
        rasterNames = [i for i in rasterNames if len(QgsProject.instance().mapLayersByName(i)) != 0]
        # checking items in combobox
        self.multiLayers.setSelected(rasterNames)
        # clearing table since rasters may not have been added in right order when checking items
        self.bandClassTree.clear()
        # for each raster
        for r in param:
            # if a raster is loaded
            if r[0] in rasterNames:
                # populate columns in table
                item = QTreeWidgetItem()
                item.setText(0, r[0])
                item.setText(1, r[1])
                self.bandClassTree.addTopLevelItem(item)

    # return a list each element for a table row
    # ['band 1, new name 1', 'band 2, new name 2']
    def value(self):
        allValues = ''
        # adding table content to values
        for b in self.bandNames():
            allValues += '{0},{1};'.format(b[0], b[1])
        return allValues[:-1]


class MergeBandTableWidgetWrapper(WidgetWrapper):

    def createWidget(self):
        return MergeBandTableWidget()

    def postInitialize(self, wrappers):
        for wrapper in wrappers:
            if wrapper.param.name == self.param.parent:
                self.setLayer(wrapper.value())
                wrapper.widgetValueHasChanged.connect(self.parentValueChanged)
                break

    def parentValueChanged(self, wrapper):
        self.setLayer(wrapper.parameterValue())

    def setLayer(self, layer):
        if isinstance(layer, QgsMapLayer):
            layer = layer.source()
        self.widget.setLayer(layer)
        self.widget.updateTable(layer)

    def setValue(self, value):
        self.widget.setValue(value)

    def value(self):
        return self.widget.value()
