# -*- coding: utf-8 -*-
"""
/***************************************************************************
 RemoteDB
                                 A QGIS plugin
 Manage and open SSH connections to remote database servers
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2023-03-16
        git sha              : $Format:%H$
        copyright            : (C) 2023 by Lymperis Efstathios
        email                : geo.elymperis@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
import os.path

from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, Qt, QUrl
from qgis.PyQt.QtGui import QIcon, QDesktopServices
from qgis.PyQt.QtWidgets import QAction, QDialog, QListWidgetItem, QLabel, QMessageBox

from .resources import *
from .remote_db_dockwidget import RemoteDBDockWidget
from .core.ConnectionManager import ConnectionManager
from .core.ConnectionListItem import ConnectionListItem
from .core.AddConnectionDialog import AddConnectionDialog


class RemoteDB:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface

        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)

        # initialize locale
        locale = QSettings().value("locale/userLocale")[0:2]
        locale_path = os.path.join(self.plugin_dir, "i18n", f"RemoteDB_{locale}.qm")

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(" Remote DB Plugin")
        self.toolbar = self.iface.addToolBar("RemoteDB")
        self.toolbar.setObjectName("RemoteDB")

        self.pluginIsActive = False
        self.dockwidget = None

        self.connectionManager = ConnectionManager()

    # noinspection PyMethodMayBeStatic

    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate("RemoteDB", message)

    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None,
    ):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            self.toolbar.addAction(action)

        if add_to_menu:
            self.iface.addPluginToDatabaseMenu(self.menu, action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = os.path.join(
            os.path.dirname(__file__), "img", "icon.png"
        )  # ":/plugins/remote_db/icon.png"
        self.add_action(
            icon_path,
            text=self.tr("Open SSH tunnels to remote databases"),
            callback=self.run,
            parent=self.iface.mainWindow(),
        )

    def onClosePlugin(self):
        """Cleanup necessary items here when plugin dockwidget is closed"""

        # Discnonect all active connections on plugin close
        for conn in self.connectionManager.available_connections:
            if conn.is_connected:
                conn.disconnect()

        self.dockwidget.closingPlugin.disconnect(self.onClosePlugin)
        self.pluginIsActive = False

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginDatabaseMenu(self.tr("& Remote DB Plugin"), action)
            self.iface.removeToolBarIcon(action)
        # remove the toolbar
        del self.toolbar

    # --------------------------------------------------------------------------

    def run(self):
        """Run method that loads and starts the plugin"""

        if not self.pluginIsActive:
            self.pluginIsActive = True

            if self.dockwidget is None:
                # Create the dockwidget (after translation) and keep reference
                self.dockwidget = RemoteDBDockWidget()

            # connect to provide cleanup on closing of dockwidget
            self.dockwidget.closingPlugin.connect(self.onClosePlugin)

            self.dockwidget.add_conn_btn.clicked.connect(self.add_connection_dialog)

            self.dockwidget.refresh_btn.clicked.connect(self.refresh_connections)

            self.dockwidget.open_settings_dir_btn.clicked.connect(
                self.open_settings_folder
            )

            self.populate_connections_list()

            self.iface.addDockWidget(Qt.LeftDockWidgetArea, self.dockwidget)
            self.dockwidget.show()

    def populate_connections_list(self):
        self.dockwidget.conn_list_widget.clear()

        # If no connections exist, display a message
        if len(self.connectionManager.available_connections) == 0:
            custom_widget = QLabel()
            custom_widget.setTextFormat(Qt.RichText)
            custom_widget.setText(
                "   <strong> No connections. </strong> Add a new connection using the button above!"
            )
            custom_widget.setWordWrap(True)
            new_item = QListWidgetItem(self.dockwidget.conn_list_widget)
            new_item.setSizeHint(custom_widget.sizeHint())
            self.dockwidget.conn_list_widget.addItem(new_item)
            self.dockwidget.conn_list_widget.setItemWidget(new_item, custom_widget)

        available_connections = self.connectionManager.available_connections
        available_connections.sort(key=lambda d: d.name)
        for available_connection in self.connectionManager.available_connections:
            # Instantiate a new ConnectionListItem
            custom_widget = ConnectionListItem(
                connection=available_connection,
                connectionManager=self.connectionManager,
                parent=None,
            )

            # Signals
            custom_widget.connectionEdited.connect(self.populate_connections_list)

            new_item = QListWidgetItem(self.dockwidget.conn_list_widget)
            new_item.setSizeHint(custom_widget.sizeHint())
            self.dockwidget.conn_list_widget.addItem(new_item)
            self.dockwidget.conn_list_widget.setItemWidget(new_item, custom_widget)

    def refresh_connections(self):
        self.connectionManager.refresh_connections()
        self.populate_connections_list()

    def add_connection_dialog(self):
        dialog = AddConnectionDialog()
        result = dialog.exec_()
        if result == QDialog.Accepted:
            connection_info = dialog.get_connection_info()

            try:
                self.connectionManager.add_connection(connection_info)

                # Repopulate connections list
                self.populate_connections_list()

            # Duplicate connection Name
            except ReferenceError as e:
                notify_user = QMessageBox(self.dockwidget)
                notify_user.setText(str(e))
                notify_user.exec_()
            # Invalid port
            except ValueError as e:
                print(e)
                notify_user = QMessageBox(self.dockwidget)
                notify_user.setText(str(e))
                notify_user.exec_()

    def open_settings_folder(self):
        QDesktopServices.openUrl(
            QUrl.fromLocalFile(self.connectionManager.SETTINGS_FOLDER)
        )
