# -*- coding: utf-8 -*-
"""
/***************************************************************************
 Reloader
                                 A QGIS plugin
 Reload selected layer(s)
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2023-02-09
        git sha              : $Format:%H$
        copyright            : (C) 2023 by Maarten Pronk
        email                : git@evetion.nl
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
import os.path
from os.path import isfile

from qgis.core import Qgis
from qgis.PyQt.QtCore import (
    QCoreApplication,
    QFileSystemWatcher,
    QSettings,
    QTranslator,
)
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QMessageBox

# Initialize Qt resources from file resources.py
from .resources import *

# Import the code for the dialog
# from .reloader_dialog import ReloaderDialog


class Reloader:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr("&Reloader")

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None
        self.watchers = {}

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate("Reloader", message)

    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None,
    ):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(self.menu, action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ":/plugins/reloader/layer-reload.png"
        self.add_action(
            icon_path,
            text=self.tr("Reload selected layer(s)"),
            callback=self.reload,
            parent=self.iface.mainWindow(),
        )

        icon_path = ":/plugins/reloader/layer-reopen.png"
        self.add_action(
            icon_path,
            text=self.tr("Reopen selected layer(s)"),
            callback=self.reopen,
            parent=self.iface.mainWindow(),
        )

        icon_path = ":/plugins/reloader/layer-watch.png"
        self.add_action(
            icon_path,
            text=self.tr("Start watching layer(s) for changes"),
            callback=self.watch,
            parent=self.iface.mainWindow(),
        )

        icon_path = ":/plugins/reloader/layer-unwatch.png"
        self.add_action(
            icon_path,
            text=self.tr("Stop watching layer(s) for changes"),
            callback=self.unwatch,
            parent=self.iface.mainWindow(),
        )

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(self.tr("&Reloader"), action)
            self.iface.removeToolBarIcon(action)

    def reload(self):
        """Reload selected layer(s)."""
        layers = self.iface.layerTreeView().selectedLayers()

        if len(layers) == 0:
            mw = self.iface.mainWindow()
            QMessageBox.warning(mw, "Reloader", "No selected layer(s).")
            return 1
        else:
            for layer in layers:
                layer.reload()
                layer.triggerRepaint()

    def reopen(self):
        """Reopen selected layer(s), which also updates the extent in contrast to `reload`."""
        layers = self.iface.layerTreeView().selectedLayers()

        if len(layers) == 0:
            mw = self.iface.mainWindow()
            QMessageBox.warning(mw, "Reloader", "No selected layer(s).")
            return 1
        else:
            for layer in layers:
                layer.setDataSource(layer.source(), layer.name(), layer.providerType())
                layer.triggerRepaint()

    def watch(self):
        """Start watching selected layer(s) for changes."""
        layers = self.iface.layerTreeView().selectedLayers()
        print(self.watchers)
        if len(layers) == 0:
            mw = self.iface.mainWindow()
            QMessageBox.warning(mw, "Reloader", "No selected layer(s).")
            return 1
        else:
            for layer in layers:
                layer.reload()
                path = layer.dataProvider().dataSourceUri()
                if not isfile(path):
                    self.iface.messageBar().pushMessage(
                        "Warning",
                        f"Can't watch {layer.name()} for updates because it is not a local path.",
                        level=Qgis.Warning,
                        duration=5,
                    )
                else:
                    watcher = QFileSystemWatcher()
                    watcher.addPath(path)
                    def reload():
                        layer.reload()
                        layer.triggerRepaint()
                    watcher.fileChanged.connect(reload)
                    self.watchers[layer.id()] = watcher

    def unwatch(self):
        """Stop watching selected layer(s) for changes."""
        layers = self.iface.layerTreeView().selectedLayers()

        if len(layers) == 0:
            mw = self.iface.mainWindow()
            QMessageBox.warning(mw, "Reloader", "No selected layer(s).")
            return 1
        else:
            for layer in layers:
                watcher = self.watchers.pop(layer.id(), None)
                if watcher is None:
                    self.iface.messageBar().pushMessage(
                        "Warning",
                        f"Can't stop watching {layer.name()} because we never started watching it.",
                        level=Qgis.Warning,
                        duration=5,
                    )
                else:
                    del(watcher)
