from PyQt5.QtWidgets import QSizePolicy, QWidget, QVBoxLayout, QPushButton, QLabel, QHBoxLayout, QCheckBox, QLineEdit, QStackedWidget
from PyQt5.QtCore import pyqtSignal, Qt
from PyQt5.QtGui import QPixmap
from .HoverLabel import HoverLabel
from ..utils import services
from ..constants import RE_USER_TOOLS
import os
from qgis.core import QgsMessageLog, QgsSettings

class ValidationTab(QWidget):
    loggedIn = pyqtSignal(str, str)
    loggedOut = pyqtSignal()
    invalidInput = pyqtSignal()
    loginIssue = pyqtSignal()
    """
    This class is used to display the search results. basically just to make a label clickable.
    """

    def __init__(self, parent, settings: QgsSettings, *args, **kwargs):
        #
        super().__init__(*args, **kwargs)
        self.parent = parent
        self.settings = settings

        self.username = ""

        self.setObjectName("t1MainWidget")
        self.mainLayout = QVBoxLayout(self)
        self.mainLayout.setContentsMargins(0,0,0,0)
        self.mainLayout.setObjectName("t1MainLayout")
        self.mainLayout.setAlignment(Qt.AlignTop)

        # Setting size policy so it fits the entire tab
        self.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Expanding)
        # i dont think i would need a scroll area here so im skipping it.

        self.stackedWidget = QStackedWidget(self)
        self.mainLayout.addWidget(self.stackedWidget)

        self.loginScreen = self.loginSetup()
        self.userScreen = self.userInfoSetup()

        self.stackedWidget.addWidget(self.loginScreen)
        self.stackedWidget.addWidget(self.userScreen)

        # Set initial screen
        self.stackedWidget.setCurrentWidget(self.loginScreen)

        self.setLayout(self.mainLayout)

    def loginSetup(self):
        """Function to set up the login UI

        Args:
            parent (_type_): Parent layout
        """
        # Login Title
        loginScreen = QWidget()
        layout = QVBoxLayout(loginScreen)
        layout.setAlignment(Qt.AlignTop)

        self.loginTitle = QLabel("<h2>Login</h2>")
        self.loginTitle.setTextFormat(Qt.RichText)
        layout.addWidget(self.loginTitle)

        # Username and password input fields
        self.usernameInput = QLineEdit()
        self.passwordInput = QLineEdit()
        self.usernameInput.setPlaceholderText("Username")
        self.passwordInput.setPlaceholderText("Password")
        self.passwordInput.setEchoMode(QLineEdit.Password)

        # If username was saved previously, then autofill this field
        savedUsername = self.settings.value("realearth/validation/username", "")
        savedPassword = self.settings.value("realearth/validation/password", "")
        if savedUsername and savedPassword:
            self.usernameInput.setText(savedUsername)
            self.passwordInput.setText(savedPassword)

        layout.addWidget(self.usernameInput)
        layout.addWidget(self.passwordInput)

        # hbox for login options
        self.loginOptionsHbox = QHBoxLayout()
        self.loginOptionsHbox.setAlignment(Qt.AlignLeft)
        layout.addLayout(self.loginOptionsHbox)

        # Remember me checkbox
        self.rememberMeCheckbox = QCheckBox("Remember Me")
        self.loginOptionsHbox.addWidget(self.rememberMeCheckbox)
        
        self.loginOptionsHbox.addStretch()

        # Forgot password label
        self.forgotPasswordLabel = QLabel(f"<a href='{RE_USER_TOOLS}'>Forgot Password?/Register</a>")
        self.forgotPasswordLabel.setOpenExternalLinks(True)
        self.forgotPasswordLabel.setTextFormat(Qt.RichText)
        self.loginOptionsHbox.addWidget(self.forgotPasswordLabel)

        layout.addSpacing(10)

        # Login button
        self.loginButton = QPushButton("Login")
        self.loginButton.clicked.connect(self.onLogin)
        layout.addWidget(self.loginButton)

        return loginScreen
    
    def userInfoSetup(self):
        """Sets up the screen for user info

        Returns:
            QWidget: A widget containing the user information
        """
        # Setting up the log in screen
        userInfoScreen = QWidget()
        layout = QVBoxLayout(userInfoScreen)
        layout.setAlignment(Qt.AlignTop)
        
        # Logged in Label
        self.loggedInLabel = QLabel("<h2>Logged In</h2>")
        self.loggedInLabel.setTextFormat(Qt.RichText)
        
        # Label for username
        self.usernameLabel = QLabel(f"<h3>{self.username}</h3>")
        self.usernameLabel.setTextFormat(Qt.RichText)

        # Log out button
        self.logOutButton = QPushButton("Log Out")
        self.logOutButton.clicked.connect(self.onLogOut)

        # Adding widgets
        layout.addWidget(self.loggedInLabel)
        layout.addWidget(self.usernameLabel)

        layout.addWidget(self.logOutButton)

        return userInfoScreen

    def apiValidateSetup(self, parent):
        """Takes care of the setting up of api input

        Args:
            parent (_type_): parent layout
        """
        self.validationTitle = QLabel("<h2>API Validation</h2>")
        self.validationTitle.setTextFormat(Qt.RichText)
        parent.addWidget(self.validationTitle)

        self.apiInput = QLineEdit()
        self.apiInput.setPlaceholderText("API Key")
        parent.addWidget(self.apiInput)

        # Remember me checkbox
        self.rememberAPICheckbox = QCheckBox("Remember Me")
        parent.addWidget(self.rememberAPICheckbox)

        parent.addSpacing(10)

        # validate button
        self.validateButton = QPushButton("Validate API")
        self.validateButton.clicked.connect(self.onValidate)
        parent.addWidget(self.validateButton)

    def toggleVisibility(self):
        """Makes the password visible/invisible.
        """
        # TODO: Figure out where to get the icon
        if self.passwordInput.echoMode() == QLineEdit.Normal:
            self.passwordInput.setEchoMode(QLineEdit.Password)
        else:
            self.passwordInput.setEchoMode(QLineEdit.Normal)

    def onLogin(self):
        """Logins the user
        """
        # TODO: Add the actual logic
        username = self.usernameInput.text()
        password = self.passwordInput.text()

        sessionKey = services.getSessionKey(username, password)
        if sessionKey is None:
            self.invalidInput.emit()
            return
        
        # Tries to get the access Token
        self.accessToken = services.getAccessToken(username, sessionKey)
        if self.accessToken is None:
            self.loginIssue.emit()
            return
        
        # Saves the login information when successfully getting access token.
        if self.rememberMeCheckbox.isChecked():
            # Remember me is checked, so save the user information
            self.settings.setValue('realEarth/validation/username', username)
            self.settings.setValue('realEarth/validation/password', password)
        else:
            # Nothing is checked, so clear the saved settings
            self.settings.setValue('realEarth/validation/username', '')
            self.settings.setValue('realEarth/validation/password', '')

        self.username = username

        # Change the username displayed on user screen first
        self.usernameLabel.setText(self.username)

        self.stackedWidget.setCurrentWidget(self.userScreen)
        
        # Sending successful login
        self.loggedIn.emit(self.username, self.accessToken)

    def onLogOut(self):
        """Logs the user out
        """
        self.stackedWidget.setCurrentWidget(self.loginScreen)
        self.username = ""

        # prefil the username/password input with saved password if there is one
        username = self.settings.value('realEarth/validation/username', '')
        password = self.settings.value('realEarth/validation/password', '')
        self.usernameInput.setText(username)
        self.passwordInput.setText(password)
        # TODO: Might need to reset saved username and password or add a setting to signal logged in or not

        self.loggedOut.emit()

    def onValidate(self):
        """Validates the user, this is automatically called
        """
        username = self.settings.value('realEarth/validation/username', '')
        password = self.settings.value('realEarth/validation/password', '')

        if not len(username) or not len(password):
            return

        sessionKey = services.getSessionKey(username, password)
        if sessionKey is None:
            self.invalidInput.emit()
            return
        
        # Tries to get the access Token
        accessToken = services.getAccessToken(username, sessionKey)
        if accessToken is None:
            self.loginIssue.emit()
            return
        
        self.loggedIn.emit(username, accessToken)
        # Change to user info screen
        self.username = username
        self.stackedWidget.setCurrentWidget(self.userScreen)
        self.usernameLabel.setText(self.username)
    