from PyQt5.QtWidgets import QLabel, QVBoxLayout, QHBoxLayout, QWidget, QGridLayout
from PyQt5.QtCore import Qt, pyqtSignal
from PyQt5.QtGui import QPixmap, QIcon
from qgis.core import QgsMessageLog
from datetime import datetime
from ..constants import PRODUCT_URL, LEGEND_URL, CATALOG_URL
import urllib.request
import requests
import os
class ProductOverviewWindow(QWidget):
    """This is a window that displays the metadata of the plugin.
    Opened when the version is pressed
    """
    windowClosed = pyqtSignal(QWidget)
    def __init__(self, id, theme):
        """Initializes the product overview window

        Args:
            id (str): Id of the product to look for
        """
        super().__init__()

        x = requests.get(PRODUCT_URL.format(id=id))
        self.product = x.json()[0]
        if theme == "Blend of Gray":
            self.setStyleSheet("background-color: #c0c0c0")

        # Setting window icon
        path = os.path.join(os.path.dirname(os.path.realpath(__file__)), "../images/ssec-logo.png")
        self.setWindowIcon(QIcon(path))

        # Main layout of the window
        self.resize(500, 400)
        self.mainLayout = QVBoxLayout()
        self.mainLayout.setAlignment(Qt.AlignTop)
        self.setLayout(self.mainLayout)
        self.setWindowTitle("Product Information")

        # Title Label
        # Title of the plugin
        self.titleLayout = QHBoxLayout()
        self.mainLayout.addLayout(self.titleLayout)
        self.titleLabel = QLabel()
        self.titleLabel.setTextFormat(Qt.RichText)
        self.titleLabel.setText(f"<h3>{self.product['name']}</h3>")
        self.titleLabel.setTextInteractionFlags(Qt.TextSelectableByMouse)
        self.titleLayout.addWidget(self.titleLabel)

        # Grid layout for more info
        self.gridLayout = QGridLayout()
        self.mainLayout.addLayout(self.gridLayout)
        self.gridLayout.setAlignment(Qt.AlignLeft)
        self.gridLayout.setVerticalSpacing(15)
        self.gridLayoutRow = 0

        # ID
        self.addColumns("ID:", self.product["id"])

        # Categories
        self.addColumns("Categories:", self.product["categories"].__str__())

        # Desc
        description = self.product["description"] if self.product["description"] != "" else "None"
        self.addColumns("Description:", description)

        # Legend, if it has one
        if self.product["showlegend"]:
            legendURL = LEGEND_URL.format(id=self.product['id'])
            self.addColumns("Legend:", legendURL, True)
        else:
            self.addColumns("Legend:", "N/A")
        # URL
        url = f"<a href=\"{self.product['url']}\">Visit {self.product['url']} </a>" if self.product["url"] != "" else "None"
        self.addColumns("More Information:", url)

        # Times
        times = len(self.product["times"])
        self.addColumns("Times:", str(times))

        # Dealing with date time
        if times > 0:
            # if the time field is not empty, we can then add earliest and latest time
            latestTime = datetime.strptime(self.product["times"][times - 1], "%Y%m%d.%H%M%S")
            latestTime = latestTime.strftime("%Y-%m-%d %H:%M:%S")
            
            earliestTime = datetime.strptime(self.product["times"][0], "%Y%m%d.%H%M%S")
            earliestTime = earliestTime.strftime("%Y-%m-%d %H:%M:%S")
        else:
            earliestTime = "N/A"
            latestTime = "N/A"

        self.addColumns("Earliest:", earliestTime)
        self.addColumns("Latest:", latestTime)

        # Catalog
        url = CATALOG_URL.format(id=self.product['id'])
        self.addColumns("", f"<a href=\"{url}\">View in Product Catalog</a>")

        # Resize the window so that it fits the content
        self.adjustSize()
        self.setFixedSize(self.width(), self.height())
    
    def addColumns(self, key, value, image=False):
        """Adds a hbox with a label key on the left and a value key on the right

        Args:
            row (int): row number to add to the grid layout
            key (_type_): key to add
            value (_type_): value to add
            image (bool): if type image
        """
        # Main hbox layout

        # Setting up the key on the left
        keyLabel = QLabel()
        keyLabel.setTextFormat(Qt.RichText)
        keyLabel.setAlignment(Qt.AlignRight)
        keyLabel.setText(f"<span style='font-size: 12px;'>{key}&nbsp;&nbsp;</span>")
        keyLabel.setMinimumHeight(18)
        self.gridLayout.addWidget(keyLabel, self.gridLayoutRow, 0)

        self.gridLayout.setSpacing(2)

        # Setting up the value on the right
        valueLabel = QLabel(value)
        valueLabel.setTextFormat(Qt.RichText)
        valueLabel.setAlignment(Qt.AlignLeft)
        if not image:
            valueLabel.setFixedWidth(self.width() - keyLabel.frameWidth() -self.gridLayout.getContentsMargins()[0])
            # ID on realEarth Browser has a monospace font, so i'm hardcoding this.
            if key == "ID:":
                # Not sure if monospace is supported on all OS though
                valueLabel.setText(f"<span style='font-size: 12px; font-family: \"Courier New\", Courier, monospace;'>{value}</span>")
            else:
                valueLabel.setText(f"<span style='font-size: 12px;'>{value}</span>")

            valueLabel.setTextInteractionFlags(Qt.TextSelectableByMouse | Qt.LinksAccessibleByMouse)
            valueLabel.setOpenExternalLinks(True)
            valueLabel.setWordWrap(True)
        else:
            with urllib.request.urlopen(value) as url:
                data = url.read()
                pixmap = QPixmap()
                pixmap.loadFromData(data)
                valueLabel.setPixmap(pixmap)
        valueLabel.setMinimumHeight(15)
        self.gridLayout.addWidget(valueLabel, self.gridLayoutRow, 1)
        self.gridLayoutRow += 1
    
    def getId(self):
        """Returns the id of the product the window is currently displaying

        Returns:
            str: id
        """
        return self.product["id"]
    
    def closeEvent(self, event):
        self.windowClosed.emit(self)
        super().closeEvent(event)

        
