from PyQt5.QtWidgets import QSizePolicy, QWidget, QVBoxLayout, QScrollArea, QPushButton, QFrame, QLabel, QHBoxLayout, QCheckBox, QLineEdit, QSpinBox, QMessageBox
from PyQt5.QtCore import pyqtSignal, Qt
from PyQt5.QtGui import QIntValidator
from qgis.core import QgsMessageLog, QgsSettings

class PreferencesTab(QWidget):
    settingsApplied = pyqtSignal()
    settingsChanged = pyqtSignal(bool)
    """
    This class is used to display the search results. basically just to make a label clickable.
    """

    def __init__(self, parent, settings: QgsSettings, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.parent = parent
        self.settings = settings
        # save the options currently displayed
        self.currOptions = {}

        # Checks unsavedd settings have been changed
        self.changed = False

        self.setObjectName("t3MainWidget")
        self.mainLayout = QVBoxLayout(self)
        self.mainLayout.setContentsMargins(0,5,0,5)
        self.mainLayout.setObjectName("t3MainLayout")

        # Setting size policy so it fits the entire tab
        self.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Expanding)

        # Setting up the scroll area
        self.scrollWidget = QWidget(self)
        self.scrollVBox = QVBoxLayout(self.scrollWidget)
        self.scrollVBox.setAlignment(Qt.AlignTop)
        self.scrollWidget.setLayout(self.scrollVBox)

        self.scrollArea = QScrollArea(self)
        self.scrollArea.setWidgetResizable(True)
        self.scrollArea.setWidget(self.scrollWidget)
        self.scrollArea.setObjectName("t3ScrollArea")
        #self.scrollArea.setStyleSheet("border-radius: 2px;")
        self.scrollArea.setFrameShape(QFrame.NoFrame)
        # Adding scroll area to the main layout
        self.mainLayout.addWidget(self.scrollArea)

        # Adding the apply button at the bottom
        self.applyButton = QPushButton("Apply", self)
        self.applyButton.clicked.connect(self.applyHandler)
        self.applyButton.setEnabled(False)
        self.applyButton.setToolTip("No changes detected")
        self.mainLayout.addWidget(self.applyButton)

        # sets up the perfomrance section
        self.performanceSection(self.scrollVBox)

        self.scrollVBox.addSpacing(10)

        self.debugSection(self.scrollVBox)

        self.setLayout(self.mainLayout)


    def performanceSection(self, parent):
        """Sets up the performance sction of the setting. value would be the widget
        """
        # Frame for performance settings
        self.performanceFrame = QFrame()
        self.performanceFrame.setObjectName("preferenceFrame")
        self.performanceFrame.setFrameShape(QFrame.NoFrame)

        # Title of the performance section
        performanceLabel = QLabel("Performance")
        performanceLabel.setTextFormat(Qt.RichText)
        performanceLabel.setStyleSheet("font-size:15px; font-weight:bold;")
        
        # divider line
        performanceLine = QFrame()
        performanceLine.setFrameShape(QFrame.HLine)
        performanceLine.setFrameShadow(QFrame.Sunken)
        performanceLine.setMaximumHeight(1)
        performanceLine.setStyleSheet("background-color:#dcdcdc;")

        # layout for the performance section
        self.pfLayout = QVBoxLayout(self.performanceFrame)
        self.pfLayout.setContentsMargins(0, 0, 0, 10)
        self.pfLayout.setAlignment(Qt.AlignTop)
        self.pfLayout.addWidget(performanceLabel)
        self.performanceFrame.setLayout(self.pfLayout)
        parent.addWidget(self.performanceFrame)

        self.pfLayout.addWidget(performanceLine)
        #################################### Start to add settings  #########################################################
        # Add settings. each setting being a hbox
        hboxIconToggle = QHBoxLayout()
        hboxIconToggle.setContentsMargins(0, 0, 0, 0)
        hboxIconToggle.setAlignment(Qt.AlignLeft | Qt.AlignVCenter)
        self.pfLayout.addLayout(hboxIconToggle)

        # Setting the text for the checkbox
        iconToggleText = QLabel("Load product icons")
        iconToggleText.setTextFormat(Qt.RichText)
        iconToggleText.setStyleSheet("font-size:12px;")
        hboxIconToggle.addWidget(iconToggleText)

        # Setting the checkbox
        iconToggleCheckBox = QCheckBox()
        #iconToggleCheckBox.setStyleSheet("border:1px solid #b0b0b0;")
        # Set default value from the settings
        iconToggleCheckBox.setChecked(self.settings.value("realEarth/iconToggle", True, type=bool))
        hboxIconToggle.addWidget(iconToggleCheckBox)
        
        self.currOptions["iconToggle"] = iconToggleCheckBox
        ###################################################################################################3
        # Now for the max search results
        hboxMaxResult = QHBoxLayout()
        hboxMaxResult.setContentsMargins(0, 0, 0, 0)
        hboxMaxResult.setAlignment(Qt.AlignLeft | Qt.AlignVCenter)
        self.pfLayout.addLayout(hboxMaxResult)

        # Setting the text for the label
        maxResultText = QLabel("Maximum Search Results Displayed")
        maxResultText.setTextFormat(Qt.RichText)
        maxResultText.setStyleSheet("font-size:12px;")
        hboxMaxResult.addWidget(maxResultText)

        # Setting the QLineEdit
        maxResultInput = QLineEdit()
        maxResultInput.setStyleSheet("border:1px solid #b0b0b0;")
        # Validator so it only accepts integers as input
        onlyInt = QIntValidator()
        onlyInt.setRange(0, 1000)
        maxResultInput.setValidator(onlyInt)

        # Set default value from the settings
        maxResultInput.setText(str(self.settings.value("realEarth/maxResults", 40, type=int)))
        maxResultInput.setMaximumWidth(25)
        maxResultInput.setStyleSheet("border-radius: 3px; border: 1px solid #b0b0b0")
        hboxMaxResult.addWidget(maxResultInput)
        hboxMaxResult.addStretch()
        self.currOptions["maxResults"] = maxResultInput
        #############################################################################################33#####
        # Layer Refetch Interval
        hboxRefetchInterval = QHBoxLayout()
        hboxRefetchInterval.setContentsMargins(0, 0, 0, 0)
        hboxRefetchInterval.setAlignment(Qt.AlignLeft | Qt.AlignVCenter)
        self.pfLayout.addLayout(hboxRefetchInterval)

        # Setting the text for the label
        refetchIntervalText = QLabel("Layer Refetch Interval")
        refetchIntervalText.setTextFormat(Qt.RichText)
        refetchIntervalText.setStyleSheet("font-size:12px;")
        hboxRefetchInterval.addWidget(refetchIntervalText)

        # Setting the QSpinBox, basically a QLineEdit with increment decrement buttons
        refetchIntervalInput = QSpinBox()
        #refetchIntervalInput.setStyleSheet("border:1px solid #b0b0b0;")
        refetchIntervalInput.setAlignment(Qt.AlignRight)
        
        # Set min and max values allowed
        refetchIntervalInput.setMinimum(1)
        refetchIntervalInput.setMaximum(360)  # max 6 hours

        # Set default value from the settings
        refetchIntervalInput.setValue(self.settings.value("realEarth/refetchPeriod", 5, type=int))
        refetchIntervalInput.setMaximumWidth(50)

        #refetchIntervalInput.setStyleSheet("border-radius: 3px; border: 1px solid #b0b0b0")
        hboxRefetchInterval.addWidget(refetchIntervalInput)

        # Setting the minute text
        minuteText = QLabel("minute(s)")
        minuteText.setTextFormat(Qt.RichText)
        minuteText.setStyleSheet("font-size:12px;")
        hboxRefetchInterval.addWidget(minuteText)

        hboxRefetchInterval.addStretch()
        self.currOptions["refetchInterval"] = refetchIntervalInput

        ######################################################################################################
        iconToggleCheckBox.stateChanged.connect(self.checkSync)
        maxResultInput.textChanged.connect(self.checkSync)
        refetchIntervalInput.textChanged.connect(self.checkSync)

    def debugSection(self, parent):
        """Sets up the performance sction of the setting. value would be the widget
        """
        # Frame for debug settings
        self.debugFrame = QFrame()
        self.debugFrame.setObjectName("debugFrame")
        self.debugFrame.setFrameShape(QFrame.NoFrame)

        # Title of the debug section
        debugLabel = QLabel("Debug")
        debugLabel.setTextFormat(Qt.RichText)
        debugLabel.setStyleSheet("font-size:15px; font-weight:bold;")
        
        # divider line
        debugLine = QFrame()
        debugLine.setFrameShape(QFrame.HLine)
        debugLine.setFrameShadow(QFrame.Sunken)
        debugLine.setMaximumHeight(1)
        debugLine.setStyleSheet("background-color:#dcdcdc;")

        # layout for the debug section
        self.debugLayout = QVBoxLayout(self.debugFrame)
        self.debugLayout.setContentsMargins(0, 0, 0, 10)
        self.debugLayout.setAlignment(Qt.AlignTop)
        self.debugLayout.addWidget(debugLabel)
        self.debugFrame.setLayout(self.debugLayout)
        parent.addWidget(self.debugFrame)

        self.debugLayout.addWidget(debugLine)
        #################################### Start to add settings  #########################################################
        # Clear Saved Local Settings
        # Add settings. each setting being a hbox
        hboxClear = QHBoxLayout()
        hboxClear.setContentsMargins(0, 0, 0, 0)
        hboxClear.setAlignment(Qt.AlignLeft | Qt.AlignVCenter)
        self.debugLayout.addLayout(hboxClear)

        clearButton = QPushButton("Reset Saved Local Data", self.debugFrame)
        clearButton.setObjectName("ClearButton")
        clearButton.setToolTip("Reset saved local data such as settings and favorites")
        clearButton.clicked.connect(self.clearSettings)
        hboxClear.addWidget(clearButton)

    def clearSettings(self):
        """The function clears all the saved data related to realEarth in QSettings
        """
        ret = QMessageBox.question(self, '', "Are you sure you want to reset saved local data?\nA reload of the plugin will be required for things to work properly", QMessageBox.Yes | QMessageBox.No)
        if ret == QMessageBox.Yes:
            allKeys = self.settings.allKeys()
            for key in allKeys:
                if key.startswith('realEarth/'):
                    self.settings.remove(key)
            ret = QMessageBox.information(self, '', "Data have been reset")

    def checkSync(self):
        """This checks if the modified settings is the same as before"""
        iconToggle = self.settings.value("realEarth/iconToggle", True, type=bool)
        maxResults = self.settings.value("realEarth/maxResults", 40, type=int)
        refetchPeriod = self.settings.value("realEarth/refetchPeriod", 5, type=int)
        
        # Checks if the settings are changed
        if (iconToggle != self.currOptions["iconToggle"].isChecked() or \
            str(maxResults) != self.currOptions["maxResults"].text() or \
            refetchPeriod != self.currOptions["refetchInterval"].value()):
            if not self.changed:
                # If one of the setting changed and self.changed was not originally true, then enable apply button
                self.applyButton.setEnabled(True)
                self.changed = True
                # emitting the signal only in the if statement since i don't want to emit too many signals
                self.settingsChanged.emit(True)
        elif self.changed:
            # changes are the same as before but self.changed is set to true, change it back
            self.changed = False
            self.applyButton.setEnabled(False)
            self.applyButton.setToolTip("No changes detected")
            self.settingsChanged.emit(False)

        # Check if text fields are empty
        if self.currOptions["maxResults"].text() == "" or self.currOptions["refetchInterval"].value() == "":
            self.applyButton.setEnabled(False)
            self.applyButton.setToolTip("Empty Field detected")
        elif self.changed:
            # Else only set apply button to enabled if changes have been made
            self.applyButton.setEnabled(True)
            self.applyButton.setToolTip("Apply the changes")

    def applyHandler(self, event=None):
        """Handles the button pressed event, which updates the settings"""
        # Update the settings
        self.settings.setValue("realEarth/iconToggle", self.currOptions["iconToggle"].isChecked())
        self.settings.setValue("realEarth/maxResults", int(self.currOptions["maxResults"].text()))
        self.settings.setValue("realEarth/refetchPeriod", int(self.currOptions["refetchInterval"].value()))
        self.applyButton.setEnabled(False)
        self.applyButton.setToolTip("No changes detected")
        self.changed = False
        self.settingsChanged.emit(False)
        self.settingsApplied.emit()
        self.settings.sync()