from PyQt5.QtWidgets import QLabel, QVBoxLayout, QHBoxLayout, QWidget, QGridLayout, QScrollArea
from PyQt5.QtCore import Qt, pyqtSignal
from PyQt5.QtGui import QPixmap, QIcon
from qgis.core import QgsMessageLog
import re
import os
class PluginOverviewWindow(QWidget):
    """This is a window that displays the metadata of the plugin.
    Opened when the version is pressed
    """
    windowClosed = pyqtSignal()

    def __init__(self):
        """Initializes the window, and set up the ui
        """
        super().__init__()
        
        # Main layout of the window
        self.resize(500, 400)
        self.mainLayout = QVBoxLayout()
        self.mainLayout.setAlignment(Qt.AlignTop)
        self.setLayout(self.mainLayout)

        # Setting window icon
        path = os.path.join(os.path.dirname(os.path.realpath(__file__)), "../images/ssec-logo.png")
        self.setWindowIcon(QIcon(path))
        self.setWindowTitle("Plugin Overview")

        # Setting up the scroll Area
        self.scrollArea = QScrollArea(self)
        self.scrollArea.setWidgetResizable(True)
        self.scrollWidget = QWidget()
        self.scrollVBox = QVBoxLayout(self.scrollWidget)
        self.scrollVBox.setAlignment(Qt.AlignTop)
        self.scrollWidget.setLayout(self.scrollVBox)
        self.scrollArea.setWidget(self.scrollWidget)

        self.mainLayout.addWidget(self.scrollArea)

        self.metaData = self.getMetadata()

        # Title of the plugin
        self.titleLayout = QHBoxLayout()
        self.scrollVBox.addLayout(self.titleLayout)
        self.titleLabel = QLabel()
        self.titleLabel.setTextFormat(Qt.RichText)
        self.titleLabel.setText("<h1>RealEarth</h1>")
        self.titleLayout.addWidget(self.titleLabel)

        # Adding icon
        self.titleLayout.addStretch()
        self.iconLabel = QLabel()
        self.titleLayout.addWidget(self.iconLabel)
        self.iconLabel.setPixmap(QPixmap(os.path.join(os.path.dirname(__file__), '../images/ssec-logo.png')))
        self.iconLabel.setScaledContents(True)
        self.iconLabel.setFixedSize(50,35)

        # Adding plugin description
        self.descriptionLabel = QLabel()
        self.descriptionLabel.setText(f"<h3>{self.metaData['description']}</h2>")
        self.descriptionLabel.setAlignment(Qt.AlignLeft)
        self.descriptionLabel.setWordWrap(True)
        self.scrollVBox.addWidget(self.descriptionLabel)

        self.scrollVBox.addSpacing(8)

        # Adding about
        self.aboutLabel = QLabel()
        #self.aboutLabel.setText(self.metaData["about"])
        self.aboutLabel.setTextFormat(Qt.RichText)
        self.aboutLabel.setText(f"<span style='font-size: 12px;'>{self.metaData['about']}</span>")
        self.aboutLabel.setAlignment(Qt.AlignLeft)
        self.aboutLabel.setWordWrap(True)
        self.scrollVBox.addWidget(self.aboutLabel)

        self.scrollVBox.addSpacing(15)

        # Adding a grid layout for the plugin metadata
        self.gridLayout = QGridLayout()
        self.gridLayout.setAlignment(Qt.AlignLeft)
        self.gridLayout.setVerticalSpacing(10)

        # Adding category
        self.addColumns(0, "Category", self.metaData["category"])
        
        # Adding tags
        tags = re.split(r",\s*", self.metaData["tags"])
        toAdd = ""
        for tag in tags:
            toAdd += tag + "&nbsp;&nbsp;&nbsp;"
        self.addColumns(1, "Tags", toAdd)

        # More info
        homepage = f"<a href=\"{self.metaData['homepage']}\">homepage</a>"
        bugTracker = f"<a href=\"{self.metaData['tracker']}\">bug tracker</a>"
        codeRepository = f"<a href=\"{self.metaData['repository']}\">code repository</a>"

        toAdd = homepage + "&nbsp;&nbsp;&nbsp;" + bugTracker + "&nbsp;&nbsp;&nbsp;" + codeRepository
        self.addColumns(2, "More info", toAdd)

        # Author
        self.addColumns(3, "Author", self.metaData['author'])
        
        # Version
        self.addColumns(4, "Installed version", self.metaData["version"])

        # Changelog
        self.metaData['changelog'].replace("\n", "<br>")
        self.addColumns(5, "Changelog", self.metaData["changelog"])


        self.scrollVBox.addLayout(self.gridLayout)
        self.resize(700, 600)

    def addColumns(self, row, key, value):
        """Adds a hbox with a label key on the left and a value key on the right

        Args:
            row (int): row number to add to the grid layout
            key (_type_): key to add
            value (_type_): value to add
        """
        # Main hbox layout

        # Setting up the key on the left
        keyLabel = QLabel()
        keyLabel.setTextFormat(Qt.RichText)
        keyLabel.setAlignment(Qt.AlignRight)
        keyLabel.setText(f"<span style='font-size: 12px;font-weight:bold;'>{key}&nbsp;&nbsp;</span>")
        keyLabel.setMinimumHeight(18)
        self.gridLayout.addWidget(keyLabel, row, 0)

        self.gridLayout.setSpacing(2)

        # Setting up the value on the right
        valueLabel = QLabel(value)
        valueLabel.setTextFormat(Qt.RichText)
        valueLabel.setAlignment(Qt.AlignLeft)
        valueLabel.setFixedWidth(self.width() - keyLabel.frameWidth() -self.gridLayout.getContentsMargins()[0])
        valueLabel.setWordWrap(True)
        valueLabel.setText(f"<span style='font-size: 12px'>{value}</span>")
        valueLabel.setMinimumHeight(15)
        self.gridLayout.addWidget(valueLabel, row, 1)

    def getMetadata(self):
        """Sets up the metadata needed for the window

        Returns:
            dict: _description_
        """
        # Add version
        metaData = {}
        with open(os.path.join(os.path.dirname(__file__), "../metadata.txt")) as fp:
            lines = fp.readlines()
            changelog = ""
            changelogLogging = False
            for line in lines:
                temp = line.split("=")
                # Since chagelog is stored a bit differently, so extract it a bit differently as well
                if line[0] == "\n" and changelogLogging:
                    # We are done with the changelog logging
                    changelogLogging = False
                if temp[0] == "changelog":
                    changelogLogging = True
                    continue
                if changelogLogging:
                    changelog += line + "<br>"
                    continue
                
                if len(temp) > 1:
                    metaData[temp[0]] = temp[1]

            metaData["changelog"] = changelog

        return metaData
    
    def closeEvent(self, event):
        """Emits a signal when the window is closed

        Args:
            event (_type_): event
        """
        self.windowClosed.emit()
        super().closeEvent(event)