# -*- coding: utf-8 -*-
"""
/***************************************************************************
 RealEarth
                                 A QGIS plugin
 description placeholder
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2024-06-05
        git sha              : $Format:%H$
        copyright            : (C) 2024 by ssecPlaceholder
        email                : ssecPlaceholder
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, Qt, QTimer
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QApplication
from qgis.core import QgsProject, QgsRasterLayer, QgsMessageLog, QgsVectorLayer, Qgis, QgsVectorTileLayer
from qgis.gui import QgsMessageBar
from .customWidget.CustomDropDown import CustomDropDown
from .customWidget.SearchResult import SearchResult
from collections import defaultdict
from xml.etree import ElementTree as ET
from .utils.utils import getWfs, checkTimestamp, checkTime
from .constants import MVT_URL, WMTS_URL, WFS_URL, MVT_URL_TIME
from datetime import datetime
import re
import json
# Initialize Qt resources from file resources.py
from .resources import *

# Import the code for the DockWidget
from .realEarth_dockwidget_base import Ui_RealEarthDockWidgetBase
from .realEarth_dockwidget import RealEarthDockWidget
import os.path
from .style.styles import get_style
from time import time
import urllib.parse

class RealEarth:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface

        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)

        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'RealEarth_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&RealEarth')
        # TODO: We are going to let the user set this up in a future iteration. did i add this?
        self.toolbar = self.iface.addToolBar(u'RealEarth')
        self.toolbar.setObjectName(u'RealEarth')
        
        # Connect the project loaded event
        QgsProject.instance().readProject.connect(self.onProjectLoaded)

        #print "** INITIALIZING RealEarth"

        self.pluginIsActive = False
        self.dockwidget = None


    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('RealEarth', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            self.toolbar.addAction(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action


    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = os.path.dirname(__file__) + "/images/ssec-logo.png"
        self.add_action(
            icon_path,
            text=self.tr(u'RealEarth'),
            callback=self.run,
            parent=self.iface.mainWindow())

    #--------------------------------------------------------------------------

    def onClosePlugin(self):
        """Cleanup necessary items here when plugin dockwidget is closed"""

        #print "** CLOSING RealEarth"

        # disconnects
        self.dockwidget.closingPlugin.disconnect(self.onClosePlugin)
        
        # Closes all the windows that are opened
        for window in SearchResult.openedWindow:
            window.close()
        for window in CustomDropDown.openedWindow:
            window.close()

        if hasattr(self, 'timer'):
            self.timer.stop()

        if self.dockwidget.ui.aboutWindow is not None:
            self.dockwidget.ui.aboutWindow.close()

        ############################### Need to disconnect everything################################################
        self.dockwidget.ui.addLayerButton.clicked.disconnect(self.addLayer)
        self.dockwidget.ui.timeList.doubleClicked.disconnect(self.addLayer)
        self.dockwidget.ui.updateButton.clicked.disconnect(self.updateLayer)
        # connecting basemap menu item click event
        for basemap in self.dockwidget.ui.baseMapActionGroup.actions():
            basemap.triggered.disconnect(self.setBaseMap)
        # connecting label menu item click event
        for label in self.dockwidget.ui.labelActionGroup.actions():
            label.triggered.disconnect(self.setLabel)
        self.iface.layerTreeView().selectionModel().selectionChanged.disconnect(self.updateUpdateButton)
        QgsProject.instance().layerWillBeRemoved.disconnect(self.onLayerRemoved)
        self.timer.timeout.disconnect(self.timerEvent)
        self.dockwidget.ui.preferencesWidget.settingsApplied.disconnect(self.checkIntervalChange)

        # Validation tab signal disconnection
        self.dockwidget.ui.validationWidget.loggedIn.disconnect(self.loggedInHandler)
        self.dockwidget.ui.validationWidget.loggedOut.disconnect(self.loggedOutHandler)
        self.dockwidget.ui.validationWidget.invalidInput.disconnect(self.invalidInputHandler)
        self.dockwidget.ui.validationWidget.loginIssue.disconnect(self.loginIssueHandler) 


        # remove this statement if dockwidget is to remain
        # for reuse if plugin is reopened
        # Commented next statement since it causes QGIS crashe
        # when closing the docked window:
        # self.dockwidget = None
        self.savePluginState(0)
        self.pluginIsActive = False


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""

        #print "** UNLOAD RealEarth"

        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&RealEarth'),
                action)
            self.iface.removeToolBarIcon(action)
        # remove the toolbar
        del self.toolbar

    #--------------------------------------------------------------------------

    def extract_features(self, json_data):
        """Since each layer can only have one geometry type, this function extracts the different geometry types into its
        own GeoJson object.

        Args:
            json_data (_type_): dictionary containing the GeoJson data

        Returns:
            _type_: List of GeoJson objects each being its own geometry type
        """
        # each key is a geometry type, each storing a list of featuress
        featureCollection = defaultdict(list)
        for feature in json_data["features"]:
            featureCollection[feature["geometry"]["type"]].append(feature)

        # POESNAV-LSAT9, for example, 2 features. LineString and Point
        #QgsMessageLog.logMessage(f"{featureCollection.keys()}", "RealEarth")

        # now building a list of GeoJson objects readible by QGIS
        output = []
        for i in featureCollection.keys():
            # im hard coding type for now since this is the only one i have seen so far
            output.append({"type": "FeatureCollection", "features": featureCollection[i]})

        return output
    
    def findLabel(self):
        """Finds the label layer and returns the layer"""
        for layer in QgsProject.instance().mapLayers().values():
            if layer.customProperty("mapType") == "Label":
                return layer
        # finds no label layer
        return None

    def addLayer(self):
        """Adds a new layer(s) from the selected timestamps
        """
        items = self.dockwidget.ui.timeList.selectedItems()

        # No items to add
        if len(items) == 0:
            return
        
        data = []
        options = {}

        # first get the layer options
        for radio in self.dockwidget.ui.options:
            options[radio.objectName()] = radio.isChecked()

        # unchecks the auto refetch checkbox
        if self.dockwidget.ui.refetchLayerOption.isChecked():
            self.dockwidget.ui.refetchLayerOption.setChecked(False)

        for i in items:
            data.append(i.data(Qt.UserRole))

        # get the id
        id = data[0]["id"]

        wfs_params = None
        if options["renderVector"] and self.dockwidget.ui.renderVectorRadio.isEnabled():
            # For render normal vecotr
            # Checking if the option is enabled.
            # Get all the wfs params first
            # This is a dictionary. each key being a time and value being all the polygon types at the time
            try:
                wfs_params = getWfs(id)
            except Exception as e:
                self.iface.messageBar().pushMessage("RealEarth", f"Failed to retrieve WFS parameters for {id}.")
                return
            
        QApplication.setOverrideCursor(Qt.BusyCursor)
        try:
            for i, d in enumerate(reversed(data)):
                # Stores all the layers of the timestamp, only here because vectors have multiple layers each for a polygon
                layers = []
                # this is to set a unique identifier for each layer in case the name of layer is the same
                currTime = time()

                if options["renderVector"] and self.dockwidget.ui.renderVectorRadio.isEnabled():
                    if d["Time"] == "Animate":
                        # Animation not supported on vector layer yet
                        self.iface.messageBar().pushMessage("RealEarth", "Animation on vector layer is not yet supported.", level=Qgis.Warning, duration=10)
                        continue
                    # vector requires a different object type
                    # TODO: Make aware of Animate
                    # This should give the time alone
                    selectedTime = d["Time"].split("UTC")[0].strip()
                    selectedTime = re.sub(r'\s*\([^)]*\)', '', selectedTime).strip()

                    for polygonType in wfs_params[selectedTime]:
                        layer_name = polygonType["title"]
                        param = {'srsname': polygonType["srsname"],
                                'typename': polygonType["typename"],
                                'version': polygonType["version"],
                                'request': 'GetFeature',
                                'service': polygonType["serviceType"],
                                'client': 'QGISPlugin',
                                }
                        
                        if len(self.token) != 0:
                            param["http-header:RE-Access-Key"] = self.token

                        # Refetch option is on, but still want to make sure this is of current or latest
                        if (d["Latest Forecast"] or d["Current"]) and options["refetchLayer"]:
                            layer_name = "↻ " + layer_name
                            self.toRefetch.append([layer_name, polygonType["originalTitle"], d["Current"], d["Latest Forecast"], d["id"], currTime, "vector"])
                            
                            # everytime something has been added, we need to modify the saved project data
                            toRefetchStr = json.dumps(self.toRefetch)
                            QgsProject.instance().writeEntry("realEarth", "toRefetch", toRefetchStr)

                            # Unhide the clock and make sure timer starts running if not already
                            if self.dockwidget.ui.clock.isHidden():
                                self.dockwidget.ui.clock.show()
                            if not self.timer.isActive():
                                self.timer.start()

                        # Making of the actual vector layer
                        wfs_uri = polygonType["url"] + "?" + urllib.parse.urlencode(param)
                        layer = QgsVectorLayer(wfs_uri, layer_name, "WFS")
                        layer.setCustomProperty("uniqueId", currTime)
                        layer.setCustomProperty("realEarthProduct", "true")
                        layers.append(layer)
                elif options["renderVectorTile"] and self.dockwidget.ui.renderVectorTileRadio.isEnabled():
                    layer_name = d["Title"]

                    if d["Time"] == "Animate":
                        # Animation not supported on vector layer yet
                        self.iface.messageBar().pushMessage("RealEarth", "Animation on vector layer is not yet supported.", level=Qgis.Warning, duration=10)
                        continue
                    # TODO: Make aware of Animate

                    # This is for rendering vecotr tiles, build the params first
                    # This assumes all the time is of the following format 2015-07-01 12:00:00 UTC
                    timeSplit = d['Time'].split(" ")
                    if len(timeSplit) < 3:
                        url = MVT_URL.format(id=id)
                    else:
                        layerDate = timeSplit[0].split("-")
                        layerDate = ''.join(layerDate)
                        layerTime = timeSplit[1].split(":")
                        layerTime = ''.join(layerTime)
                        # TODO: REPLACE DOMAIN AND ID
                        url = MVT_URL_TIME.format(id=id, layerDate=layerDate, layerTime=layerTime)
                    
                    mvt_params = {"type": "xyz",
                            "url": url,
                            "zmax": 14,
                            "zmin": 0,
                            'client': 'QGISPlugin',}
                    
                    if len(self.token) != 0:
                            mvt_params["http-header:RE-Access-Key"] = self.token
                    
                    # Refetch option is on, but still want to make sure this is of current or latest layer
                    if (d["Latest Forecast"] or d["Current"]) and options["refetchLayer"]:
                        layer_name = "↻ " + layer_name
                        self.toRefetch.append([layer_name, d["originalTitle"], d["Current"], d["Latest Forecast"], d["id"], currTime, "vectorTile"])
                        # everytime something has been added, we need to modify the saved project data
                        toRefetchStr = json.dumps(self.toRefetch)
                        QgsProject.instance().writeEntry("realEarth", "toRefetch", toRefetchStr)

                        # Unhide the clock and make sure timer starts running if not already
                        if self.dockwidget.ui.clock.isHidden():
                            #QgsMessageLog.logMessage("Hidden Clock", "RealEarth")
                            self.dockwidget.ui.clock.show()
                        if not self.timer.isActive():
                            self.timer.start()

                    # Joining the params together and create the layer
                    mvt_uri = '&'.join([f"{key}={value}" for key, value in mvt_params.items()])
                    mvtLayer = QgsVectorTileLayer(mvt_uri, layer_name)
                    mvtLayer.setCustomProperty("uniqueId", currTime)
                    mvtLayer.setCustomProperty("realEarthProduct", "true")
                    # This is completely unnecessary as each raster timestamp has only one layer, but just to make it easier
                    # for implementing vector layers, im adding it to the layers list
                    layers.append(mvtLayer)

                else:
                    # A raster layer   
                    # URL pointing to the WMTS GetCapabilities document
                    # NOTE: dpi mode set to 7 is hard coded. I dont understand where do i get the dpiMode but that is what
                    # qgis ueses natively
                    layer_name = d["Title"]
                    wmts_params = {
                        'crs': d["CRS"],
                        'layers': d["Identifier"],
                        'dpiMode': "7",
                        'format': d["Format"],
                        'styles': d["Style"],
                        'tileMatrixSet': d["TileMatrixSet"], 
                        'type':'tile',
                        'tiled':'true',
                        'url': d["url"],
                        'client': 'QGISPlugin',
                    }

                    if len(self.token) != 0:
                            wmts_params["http-header:RE-Access-Key"] = self.token
                    
                    # Refetch option is on, but still want to make sure this is of current or latest layer
                    if (d["Latest Forecast"] or d["Current"]) and options["refetchLayer"]:
                        layer_name = "↻ " + layer_name
                        self.toRefetch.append([layer_name, d["originalTitle"], d["Current"], d["Latest Forecast"], d["id"], currTime, "raster"])
                        # everytime something has been added, we need to modify the saved project data
                        toRefetchStr = json.dumps(self.toRefetch)
                        QgsProject.instance().writeEntry("realEarth", "toRefetch", toRefetchStr)

                        # Unhide the clock and make sure timer starts running if not already
                        if self.dockwidget.ui.clock.isHidden():
                            self.dockwidget.ui.clock.show()
                        if not self.timer.isActive():
                            self.timer.start()

                    # Joining the params together and create the layer
                    wmts_uri = '&'.join([f"{key}={value}" for key, value in wmts_params.items()])
                    wmts_layer = QgsRasterLayer(wmts_uri, layer_name, "wms")
                    wmts_layer.setCustomProperty("uniqueId", currTime)
                    wmts_layer.setCustomProperty("realEarthProduct", "true")
                    # This is completely unnecessary as each raster timestamp has only one layer, but just to make it easier
                    # for implementing vector layers, im adding it to the layers list
                    layers.append(wmts_layer)

                for layer in layers:
                    if options["opacity"]:
                        # Opacity option is checked, so show the opacity
                        layer.setCustomProperty("embeddedWidgets/count", 1)
                        layer.setCustomProperty("embeddedWidgets/0/id", "transparency")

                    if layer.isValid():
                        # Add the layer if valid
                        root = QgsProject.instance().layerTreeRoot()
                        labelLayer = self.findLabel()
                        # If a label layer exists, then push it under the layer
                        if labelLayer is not None:
                            labelNode = root.findLayer(labelLayer.id())
                            parent = labelNode.parent()
                            index = parent.children().index(labelNode)

                            QgsProject.instance().addMapLayer(layer, False)
                            parent.insertLayer(index+1, layer)
                        else:
                            # Else just add it to the top
                            QgsProject.instance().addMapLayer(layer)

                        # Set the layer to be collapsed other than the first layer added so legend won't take up all the space
                        layer_node = root.findLayer(layer.id())
                        if layer_node:
                            layer_node.setExpanded(i==len(data)-1)
                    else:
                        # TODO: Remove layer from refetch?
                        self.iface.messageBar().pushMessage("RealEarth", f"Unable to create layer", level=Qgis.Critical, duration=7)
                        pass
        except Exception as e:
            raise e
        finally:
            QApplication.restoreOverrideCursor()

    def updateLayer(self):
        """Updates the selected layer. All the check here are not really necessary as there is only one property (so far), but
            just to make things a bit easier in the future these checks are added."""
        
        # first get the layer options
        options = {}
        for checkbox in self.dockwidget.ui.options:
            options[checkbox.objectName()] = checkbox.isChecked()
        
        # getting the selected layers
        selectedLayers = self.iface.layerTreeView().selectedLayers()
        for layer in selectedLayers:
            # finding out if the layer is of raster or vector type?
            if type(layer):
                # Getting the current count of the widgets that are embedded
                widgetsCount = layer.customProperty("embeddedWidgets/count", 0)
                # update the layer based on the options
                if options["opacity"]:
                    if not any(layer.customProperty(f"embeddedWidgets/{i}/id") == "transparency" for i in range(widgetsCount)):
                        # Opacity option is checked, and transparency widget is not present, add the widget
                        layer.setCustomProperty("embeddedWidgets/count", 1)
                        layer.setCustomProperty("embeddedWidgets/0/id", "transparency")

                elif not options["opacity"]:
                    # if opacity is unchecked, remove the transparancy widget
                    for i in range(widgetsCount):
                        if layer.customProperty(f"embeddedWidgets/{i}/id") == "transparency":
                            layer.removeCustomProperty(f"embeddedWidgets/{i}/id")

                            # Shifting all other properties
                            for j in range(i, widgetsCount-1):
                                # j is the index of the new property, so we want to move j+1 to j
                                layer.setCustomProperty(f"embeddedWidgets/{j}/id", layer.customProperty(f"embeddedWidgets/{j+1}/id"))
                                layer.removeCustomProperty(f"embeddedWidgets/{j+1}/id")

                            # Decrementing the count of the widgets that are embedded
                            layer.setCustomProperty("embeddedWidgets/count", widgetsCount-1)
                            break
            # Refreshes the layer
            layer.triggerRepaint()
            self.iface.layerTreeView().refreshLayerSymbology(layer.id())

    def setBaseMap(self):
        """Sets the base map for the project. added to the bottom most level of the layer tree.
        """
        action = self.dockwidget.ui.baseMapOption.sender()
        root = QgsProject.instance().layerTreeRoot()
        data = action.data()

        if data is not None:
            # Get the params ready
            layer_name = data["Title"]
            wmts_params = {
                'crs': data["CRS"],
                'layers': data["Identifier"],
                'format': data["Format"],
                'styles': data["Style"],
                'tileMatrixSet': data["TileMatrixSet"], 
                'type':'tile',
                'tiled':'true',
                'url': data["url"],
                'client': 'QGISPlugin',
            }
            # Joining the params together and create the layer
            wmts_uri = '&'.join([f"{key}={value}" for key, value in wmts_params.items()])

            if wmts_uri == self.baseMapSource:
                # selected the same basemap again, so we're done
                return
        elif self.baseMapSource is None:
            # Previous base map is none, and new is also none, do nothing.
            return

        if self.baseMapSource is not None:
            # Loops over the layer when there is already a basemap
            for layer in QgsProject.instance().mapLayers().values():
                if isinstance(layer, QgsRasterLayer) and layer.source() == self.baseMapSource:
                    if data is None:
                        # data is None, means no basemap selected, so we're done
                        # removing the layer
                        QgsProject.instance().removeMapLayer(layer.id())
                        self.baseMapSource = None
                        self.iface.mapCanvas().refresh()
                        return

                    # Adding the layer to the same position
                    layer.setDataSource(wmts_uri, layer_name, "wms")
                    layer.setCustomProperty("mapType", "Basemap")
                    if not layer.isValid():
                        # TODO: Add error message here.
                        return
                    
                    self.baseMapSource = wmts_uri
                    return
        
        # cannot find a basemap / no basemap origionally just add the layer at the bottom
        wmts_layer = QgsRasterLayer(wmts_uri, layer_name, "wms")
        if not wmts_layer.isValid():
            # TODO: Add error message here.
            return
        
        wmts_layer.setCustomProperty("mapType", "Basemap")
        QgsProject.instance().addMapLayer(wmts_layer, False)
        root.addLayer(wmts_layer)
        self.baseMapSource = wmts_uri
        # adds the layer at the very bottom
        #root.addLayer(wmts_layer)

    def setLabel(self):
        """Sets the label/overlay for the project. added to the top most level of the layer tree.
        """
        action = self.dockwidget.ui.labelOption.sender()
        root = QgsProject.instance().layerTreeRoot()
        data = action.data()

        if data is not None:
            # Get the params ready
            layer_name = data["Title"]
            wmts_params = {
                'crs': data["CRS"],
                'layers': data["Identifier"],
                'format': data["Format"],
                'styles': data["Style"],
                'tileMatrixSet': data["TileMatrixSet"], 
                'type':'tile',
                'tiled':'true',
                'url': data["url"],
                'client': 'QGISPlugin',
            }
            # Joining the params together and create the layer
            wmts_uri = '&'.join([f"{key}={value}" for key, value in wmts_params.items()])

            if wmts_uri == self.labelSource:
                # selected the same label again, so we're done
                return
        elif self.labelSource is None:
            # Previous label was none and new label is none, just return
            return

        if self.labelSource is not None:
            # Loops over the layer when there is already a label
            for layer in QgsProject.instance().mapLayers().values():
                if isinstance(layer, QgsRasterLayer) and layer.source() == self.labelSource:
                    if data is None:
                        # data is None, means no label selected, so we're done
                        # removing the layer
                        QgsProject.instance().removeMapLayer(layer.id())
                        self.labelSource = None
                        self.iface.mapCanvas().refresh()
                        return

                    # Adding the layer to the same position
                    layer.setDataSource(wmts_uri, layer_name, "wms")
                    layer.setCustomProperty("mapType", "Label")
                    if not layer.isValid():
                        # TODO: Add error message here.
                        return
                    
                    self.labelSource = wmts_uri
                    return
        
        # cannot find a label / no label origionally just add the layer at the bottom
        wmts_layer = QgsRasterLayer(wmts_uri, layer_name, "wms")
        if not wmts_layer.isValid():
            # TODO: Add error message here.
            return
        wmts_layer.setCustomProperty("mapType", "Label")
        
        QgsProject.instance().addMapLayer(wmts_layer, False)
        root.insertLayer(0,wmts_layer)
        self.labelSource = wmts_uri

    def updateUpdateButton(self):
        """This updates the update button depending if there are selected plugins
        """
        selectedLayers = self.iface.layerTreeView().selectedLayers()
        # See if there are any seleceted layers, if yes enable the button. else diable the button
        # Assuming this is always running with python3 where false = 0
        if not selectedLayers:
            self.dockwidget.ui.updateButton.setEnabled(False)
            self.dockwidget.ui.updateButton.setToolTip("Select layer(s) from the Layers panel to update options")
        else:
            self.dockwidget.ui.updateButton.setEnabled(True)
            self.dockwidget.ui.updateButton.setToolTip("Update the selected layer with the updated checkboxes")
    
    def getUri(self, layer, id):
        """Gets the uri for the wmts layer

        Args:
            layer (_type_): the layer to get the uri from

        Returns:
            str, str, dict: returns the title of the layer, original title, and the uri dictionary
        """
        namespace = {'ows': 'http://www.opengis.net/ows/1.1', 'gml': 'http://www.opengis.net/gml'}
        location = "http://www.opengis.net/wmts/1.0"
        title = layer.find('ows:Title', namespace).text if layer.find('ows:Title', namespace) is not None else None
        originalTitle = title
        uri = {}
        # Used to identify if the layer is latest forecast or current

        # updates the layer title if it is a latest forecast or a current
        if "(Forecast)" in title:
            title = title.replace("(Forecast)", "(Latest Forecast)")
        else:
            title += " (Current)"

        # getting crs
        boundingBox = layer.find('ows:BoundingBox', namespace)
        if boundingBox is not None:
            uri["crs"] = boundingBox.get("crs")

        # getting Id/Layer
        identifier = layer.find("ows:Identifier", namespace)
        uri["layers"] = identifier.text if identifier is not None else None

        # getting Format
        format = layer.find(f'{{{location}}}Format')
        uri["format"] = format.text if format is not None else None

        # Setting dpiMode to 7. Don't know why, but that is what qgis does natively
        uri["dpiMode"] = "7"

        # getting MatrixTile
        matrixTileLink = layer.find(f'{{{location}}}TileMatrixSetLink')
        if matrixTileLink is not None:
            matrixTileSet = matrixTileLink.find(f'{{{location}}}TileMatrixSet')
            uri["tileMatrixSet"] = matrixTileSet.text if matrixTileSet is not None else None
            
        # getting Style
        style = layer.find(f'{{{location}}}Style')
        if style is not None:
            if style.get("isDefault") == "true":
                uri["styles"] = ""
            else:
                styldid = style.find("ows:Identifier", namespace)
                uri["styles"] = styldid.text if styldid is not None else None
        uri["type"] = "tile"
        uri["tiled"] = "true"
        uri["url"] = WMTS_URL.format(id=id)
        
        return title, originalTitle, uri
    def getWfsUri(self, layer, otherInfo):
        """Gets the uri of the wfs layer

        Args:
            layer (_type_): layer content from xml
            otherInfo (_type_): some other information (version, serviceType, productName, url, id)

        Returns:
            str, str, dict: returns the title of the layer, original title, and the uri dictionary
        """
        namespace = {'wfs': 'http://www.opengis.net/wfs/2.0', 'ows': 'http://www.opengis.net/ows/1.1'}
        title = layer.find('wfs:Title', namespace).text if layer.find('wfs:Title', namespace) is not None else None
        originalTitle = title

        typename = layer.find(".//wfs:Name", namespace).text
        srsname = layer.find(".//wfs:DefaultCRS", namespace).text
        title = layer.find('wfs:Title', namespace).text

        # This is also very hardcoded
        time = title.split(f"{otherInfo['productName']}: ", 1)[1].strip() if otherInfo["productName"] in title else title
        # Removing the stuff like (Point) (Line)...
        time = re.sub(r'\s*\([^)]*\)', '', time).strip()

        uri = {}
        uri["typename"] = typename
        uri["srsname"] = srsname
        uri["version"] = otherInfo["version"]
        uri["serviceType"] = otherInfo["serviceType"]
        uri["productName"] = otherInfo["productName"]
        uri["title"] = title
        uri["originalTitle"] = originalTitle
        uri["time"] = time
        uri["url"] = WFS_URL.format(id=otherInfo['id'])

        return title, originalTitle, uri

    def refetchLayers(self):
        """This function refetches layers that have refetch enabld and refreshes the time list
        """
        # decrement the clock by 1
        # TODO: Check duplicate layers
        for i, item in enumerate(self.toRefetch):
            # This is used to check if this item is found inside the layers panel
            found = False
            # Check which one is required to refetch by seeing if there is a new timestamp
            xmlLayer, otherInfo = None, None
            if item[6] == "vector":
                xmlLayer, otherInfo = checkTimestamp(item)
            elif item[6] == "vectorTile":
                # Get time for original title
                originalTitle = item[1]
                # Checks for a new timestamp, if no new time
                # var name is xmlLayer but it really isn't, it just makes it easier for me to write the code
                newTitle, xmlLayer = checkTime(originalTitle, item[4])

            else:
                xmlLayer = checkTimestamp(item)
            # item: (layer name, originalTitle, current?, forecast?, id, time, raster/vector)

            if xmlLayer is not None:
                # we find it in the tree
                for layer in QgsProject.instance().mapLayers().values():
                    if layer.name() == item[0] and layer.customProperty("uniqueId") == item[5]: #item[0] should be the name of layer. item[5] is the time added acting as a unique id
                        
                        # modify the uri
                        if item[6] == "vector":
                            # Different function for wfs
                            title, originalTitle, uri = self.getWfsUri(xmlLayer, otherInfo)
                            # TODO: Right now it checks for token every time. Keep the same token? idk
                            param = {'srsname': uri["srsname"],
                                    'typename': uri["typename"],
                                    'version': uri["version"],
                                    'request': 'GetFeature',
                                    'service': uri["serviceType"],
                                    'client': 'QGISPlugin',
                                    }
                            
                            # Checks for token
                            if len(self.token) != 0:
                                param["http-header:RE-Access-Key"] = self.token

                            # building new uri
                            newUri = uri["url"] + "?" + urllib.parse.urlencode(param)
                        elif item[6] == "vectorTile":
                            layerDate, layerTime = xmlLayer.split(".")
                            # TODO: REPLACE DOMAIN AND ID
                            url = MVT_URL_TIME.format(id = item[4], layerDate = layerDate, layerTime = layerTime)
                        
                            mvt_params = {"type": "xyz",
                                    "url": url,
                                    "zmax": 14,
                                    "zmin": 0,
                                    'client': 'QGISPlugin',}
                            
                            # Checks for token
                            if len(self.token) != 0:
                                mvt_params["http-header:RE-Access-Key"] = self.token

                            newUri = '&'.join([f"{key}={value}" for key, value in mvt_params.items()])
                            # This is here to make it easier outside of the ifelse statment
                            title = originalTitle = newTitle
                        else:
                            # TODO: Same issue here
                            # i think this is wmts
                            title, originalTitle, uri = self.getUri(xmlLayer, item[4])
                            uri['client'] = 'QGISPlugin'
                            
                            # Checks for token
                            if len(self.token) != 0:
                                uri["http-header:RE-Access-Key"] = self.token

                            # building new uri
                            newUri = '&'.join([f"{key}={value}" for key, value in uri.items()])
                        
                        title = f"↻ {title}"
                        # Adding the layer to the same position
                        if item[6] == "vector":
                            provider = "wfs"
                        elif item[6] == "vectorTile":
                            provider = "xyzvectortiles"
                        else:
                            provider = "wms"

                        layer.setDataSource(newUri, title, provider)
                        layer.triggerRepaint()
                        item[0] = title
                        item[1] = originalTitle
                        
                        # Updating refetch
                        toRefetchStr = json.dumps(self.toRefetch)
                        QgsProject.instance().writeEntry("realEarth", "toRefetch", toRefetchStr)
                        found = True
                        break
                
                if not found:
                    # couldn't find the item in the panel, so there is no reason to keep this. remove it.
                    self.toRefetch.remove(item)
                    # Update the saved state of refetch
                    toRefetchStr = json.dumps(self.toRefetch)
                    QgsProject.instance().writeEntry("realEarth", "toRefetch", toRefetchStr)
                    
                    # stop timer and reset time
                    if not len(self.toRefetch):
                        self.timer.stop()
                        self.dockwidget.ui.clock.hide()
                        return
                        
    def timerEvent(self):
        """Decrements the clock label and handles the calling of refetch
        """
        self.timeLeft -=1
        
        if self.timeLeft <= 0:
            #refetch the layers if time is up and reset all the timer and clock
            self.dockwidget.ui.clock.setText(f"Refreshing...")
            self.refetchLayers()
            self.timeLeft = self.settings.value("realEarth/refetchPeriod", 5, type=int)
        
        self.dockwidget.ui.clock.setText(f"Next refetch: {self.timeLeft} minute(s)")
        

    def onLayerRemoved(self, layerId):
        """Removes the layer in self.toRefetch if it is the layer to be removed

        Args:
            layerId (_type_): layerId
        """
        layer = QgsProject.instance().mapLayer(layerId)
        # For each item in self.toRefetch check if the name is the same, if so remove it.
        for item in self.toRefetch:
            if item[0] == layer.name():
                self.toRefetch.remove(item)
                
                # Update the saved state of refetch
                toRefetchStr = json.dumps(self.toRefetch)
                QgsProject.instance().writeEntry("realEarth", "toRefetch", toRefetchStr)

                # stop timer and reset time
                if not len(self.toRefetch):
                    self.timer.stop()
                    self.dockwidget.ui.clock.hide()
                break

    def checkIntervalChange(self):
        """Checks in the setting if there is a change in the interval, if so
        update the timer
        """
        interval = self.settings.value("realEarth/refetchPeriod", 5, type=int)

        if self.currInterval != interval:
            # New interval, so need to update
            self.currInterval = interval
            self.timeLeft = interval

            # update clock
            self.dockwidget.ui.clock.setText(f"Next refetch: {self.timeLeft} minute(s)")

    def findBasemapLabelAndUpdate(self):
        """This function finds the basemap and label from layers panel and updates the currently selected basemap/label
        """
        for layer in QgsProject.instance().mapLayers().values():
            if layer.customProperty("mapType") is not None and layer.customProperty("mapType") == "Basemap":
                # if the layer is of a basemap, then upate the basemap selection
                self.baseMapSource = layer.source()
                
                # Loop through each action in the qmenu
                for action in self.dockwidget.ui.baseMapActionGroup.actions():
                    if action.data() is not None and action.data()["Title"] == layer.name():
                        # update the selection
                        action.setChecked(True)
                        action.trigger()
                        break
            elif layer.customProperty("mapType") is not None and layer.customProperty("mapType") == "Label":
                # if the layer is of a label, then upate the label selection
                self.labelSource = layer.source()
                
                # Loop through each action in the qmenu
                for action in self.dockwidget.ui.labelActionGroup.actions():
                    if action.data() is not None and action.data()["Title"] == layer.name():
                        # update the selection
                        action.setChecked(True)
                        action.trigger()
                        break
    
    def savePluginState(self, state):
        """Saves the state of the plugin to the instance of the project

        Args:
            state (int): state of the plugin. 1: opened, 2: closed
        """
        QgsProject.instance().writeEntry("realEarth", "opened", state)

    def onProjectLoaded(self):
        """Load the plugin if state is 1
        """
        isOpen = QgsProject.instance().readNumEntry("realEarth", "opened")[0]
        if isOpen:
            self.run()

    def loggedInHandler(self, username, token):
        """Handles loggin event

        Args:
            username (str): username
            token (str): token
        """
        # Convert the token to a header with key value format
        self.token = "".join(token.split())

        # Push Message notifying logged in
        self.iface.messageBar().pushMessage("RealEarth", f"Successfully Logged in, {username}", level=Qgis.Success, duration=7)
        self.updateAllLayers(token)

    def strToDict(self, string):
        """convert a string to a dictionary

        Args:
            string (str): stinrg to convert

        Returns:
            dict: converted dictionary
        """
        params = {}
        for param in string.split('&'):
            keyVal = param.split('=', 1)
            if len(keyVal) == 2:
                key, value = keyVal
                params[key] = value
            else:
                params[keyVal[0]] = ''

        return params

    def updateAllLayers(self, token):
        """Updates all alyers with the new token

        Args:
            token (_type_): access token
        """

        # Go through each layer
        layers = QgsProject.instance().mapLayers().values()

        for layer in layers:
            if layer.customProperty("realEarthProduct"):
                layerSource = layer.source()

                # convert string to dict
                sourceDict = self.strToDict(layerSource)
                
                sourceDict["http-header:RE-Access-Key"] = token

                # joining back into a string
                uri = '&'.join([f"{key}={value}" for key, value in sourceDict.items()])

                # Setting data source
                layerName = layer.name()
                provider = layer.dataProvider().name()
                layer.setDataSource(uri, layerName, provider)
                

    def loggedOutHandler(self):
        """Handles the log out event.
        """
        # Reset token
        self.token = ""
        self.iface.messageBar().pushMessage("RealEarth", f"User Logged out", level=Qgis.Info, duration=7)

        self.removeToken()

    def removeToken(self):
        """Removes token from all layers
        """
        # Go through each layer
        layers = QgsProject.instance().mapLayers().values()

        for layer in layers:
            if layer.customProperty("realEarthProduct"):
                layerSource = layer.source()

                # convert string to dict
                sourceDict = self.strToDict(layerSource)

                if "http-header:RE-Access-Key" in sourceDict:
                    del sourceDict["http-header:RE-Access-Key"]

                # joining back into a string
                uri = '&'.join([f"{key}={value}" for key, value in sourceDict.items()])

                # Setting data source
                layerName = layer.name()
                provider = layer.dataProvider().name()
                layer.setDataSource(uri, layerName, provider)


    def invalidInputHandler(self):
        self.iface.messageBar().pushMessage("RealEarth", f"Failed to login: Invalid username or password", level=Qgis.Critical, duration=7)

    def loginIssueHandler(self):
        self.iface.messageBar().pushMessage("RealEarth", f"Failed to login: Something went wrong", level=Qgis.Critical, duration=7)

    def run(self):
        """Run method that loads and starts the plugin"""
        self.settings = QSettings()
        self.theme = self.settings.value("UI/uitheme")

        # Setting is different for mac
        if self.theme is None:
            self.theme = self.settings.value("UI/UITheme")
        
        if not self.pluginIsActive:
            self.pluginIsActive = True

            #print "** STARTING RealEarth"

            # dockwidget may not exist if:
            #    first run of plugin
            #    removed on close (see self.onClosePlugin method)

            # Change the cursor to loading cursor
            QApplication.setOverrideCursor(Qt.BusyCursor)
            if self.dockwidget == None:
                # Create the dockwidget (after translation) and keep reference
                try:
                    self.dockwidget = RealEarthDockWidget()
                    get_style(self.dockwidget, self.theme)
                except Exception as e:
                    QApplication.restoreOverrideCursor()
                    if hasattr(self, 'timer'):
                        self.timer.stop()
                    raise e
            
            # connect to provide cleanup on closing of dockwidget
            self.dockwidget.closingPlugin.connect(self.onClosePlugin)
            
            # show the dockwidget
            # TODO: fix to allow choice of dock location
            self.iface.addDockWidget(Qt.RightDockWidgetArea, self.dockwidget)
            self.dockwidget.show()

            # Restore cursor
            QApplication.restoreOverrideCursor()

        # change plugin state
        self.savePluginState(1)
        self.dockwidget.ui.addLayerButton.clicked.connect(self.addLayer)
        self.dockwidget.ui.timeList.doubleClicked.connect(self.addLayer)
        self.dockwidget.ui.updateButton.clicked.connect(self.updateLayer)

        # This keeps track of the current basemap/label using uri. There is a possibility of user removing the layer manually.
        self.baseMapSource = None
        self.labelSource = None

        # Detect the current basemap/label before connecting the actions to them
        self.findBasemapLabelAndUpdate()

        # connecting basemap menu item click event
        for basemap in self.dockwidget.ui.baseMapActionGroup.actions():
            basemap.triggered.connect(self.setBaseMap)
        # connecting label menu item click event
        for label in self.dockwidget.ui.labelActionGroup.actions():
            label.triggered.connect(self.setLabel)

        # Connect layer click event to enable/disable update button
        self.iface.layerTreeView().selectionModel().selectionChanged.connect(self.updateUpdateButton)
        # Connect layer removed to remove it from toRefetchList
        QgsProject.instance().layerWillBeRemoved.connect(self.onLayerRemoved)
        
        # Start a timer to refresh the list of times
        self.timeLeft = self.settings.value("realEarth/refetchPeriod", 5, type=int)
        # This variable here to check if the new applied interval is the same as the current one
        self.currInterval = self.timeLeft
        self.timer = QTimer()
        self.timer.timeout.connect(self.timerEvent)
        # set to 1 minute interval
        self.timer.setInterval(60000)

        # Loading the layers to refetch from the saved qgis data
        toRefetchStr = QgsProject.instance().readEntry("realEarth", "toRefetch", "[]")[0]
        # This will contain all the layers to refetch. [(layer name, originalTitle, current?, latest forecast?, id, time, raster/vector), ...]
        self.toRefetch = json.loads(toRefetchStr)
        
        if len(self.toRefetch) != 0:
            # Detected things to refetch upon startup, so start the timer
            if self.dockwidget.ui.clock.isHidden():
                self.dockwidget.ui.clock.show()  
            self.timer.start()
        
        # Settings applied, needs to update refetch interval
        self.dockwidget.ui.preferencesWidget.settingsApplied.connect(self.checkIntervalChange)

        # Header used to put the api key in
        self.token = ""

        # Validation tab signal connection
        self.dockwidget.ui.validationWidget.loggedIn.connect(self.loggedInHandler)
        self.dockwidget.ui.validationWidget.loggedOut.connect(self.loggedOutHandler)
        self.dockwidget.ui.validationWidget.invalidInput.connect(self.invalidInputHandler)
        self.dockwidget.ui.validationWidget.loginIssue.connect(self.loginIssueHandler)

        # Attemps to autologin at startup
        self.dockwidget.ui.validationWidget.onValidate()
