from PyQt5.QtWidgets import QLabel, QHBoxLayout, QSizePolicy, QFrame
from PyQt5.QtCore import pyqtSignal, Qt
from PyQt5.QtGui import QPixmap, QIcon
from concurrent.futures import ThreadPoolExecutor
from .productOverviewWindow import ProductOverviewWindow
from .HoverLabel import HoverLabel
import os
import requests
from qgis.core import QgsMessageLog

class SearchResult(QFrame):
    leftClicked = pyqtSignal()
    updateFav = pyqtSignal(list)
    openedWindow = []
    """
    This class is used to display the search results. basically just to make a label clickable.
    """

    def __init__(self, result: tuple, iconUrl, theme, *args, **kwargs):
        """This constructor sets up the search result ui

        Args:
            result (tuple): (score, name, id, desc, late?, categories, type, favorite)
            iconUrl (_type_): url of Icon
            theme (_type_): theme to use for the search results
        """
        super().__init__(*args, **kwargs)
        self.setObjectName("searchResult")
        self.result = result
        self.theme = theme
        
        self.setProperty("clickableLabel", True)
        
        # This layout is to store the icon and the text of a result entry.
        self.layout = QHBoxLayout()

        self.layout.setContentsMargins(0, 0, 0, 0)
        self.setStyleSheet("""
            QLabel#iconLabel { height: 32px; background-color: transparent;}
            QLabel#textLabel { font-size: 18px; background-color: transparent;}
        """)
        self.setLayout(self.layout)

        # Here is the part where i try to load the icons.
        self.iconUrl = iconUrl

        # Setting the default icon to a blank icon
        QPixmapIcon = QPixmap(27, 27)
        QPixmapIcon.fill(Qt.transparent)
        self.defaultIcon = QIcon(QPixmapIcon)  # Default blank icon

        self.exe = ThreadPoolExecutor(max_workers=min(10, os.cpu_count() * 2))
        self.iconFuture = []

        # Setting the tooltip
        description = result[3].replace("\n", " ")
        tooltip = f"<div style='white-space:pre-wrap; max-width: 600px;'>{description}</div>"
        self.setToolTip(tooltip)
        
        # The icon label for the result
        self.iconLabel = QLabel()
        self.iconLabel.setPixmap(QPixmapIcon)
        self.iconLabel.setFixedHeight(40)
        self.iconLabel.setToolTip(tooltip)
        self.iconLabel.mousePressEvent = lambda event: self.leftClickEvent(event)

        self.layout.addSpacing(4)
        self.layout.addWidget(self.iconLabel)
        self.layout.addSpacing(2)

        # Adding a dot label to show if a product is late or not. Orange: late, green: on time
        self.dotLabel = QLabel()
        #self.dotLabel.setStyleSheet("font-size:8px;")
        self.dotLabel.setAlignment(Qt.AlignLeft | Qt.AlignVCenter)

        # Setting color depending on if its late or not.
        color = "#F75D1E" if result[4] else "#38CF38"

        if result[6] == "shape":
            # If a product is of vector, then add v
            self.dotLabel.setText(f"<span style='color: {color}; font-size:12px; font-weight:bold;'>V</span>")
        else:
            self.dotLabel.setText(f"<span style='color: {color}; font-size:12px; font-weight:bold;'>R</span>")

        # Setting tooltip
        shape = "vector" if result[6] == "shape" else result[6]

        self.dotLabel.setToolTip(f"Late: {result[4]}, Shape: {shape}")

        self.layout.addWidget(self.dotLabel)
        
        # Setting text label for the actual text showing up, setting styling to add categories
        self.text = result[1] + "<br><span style='font-size:9px'>"

        for i, cat in enumerate(result[5]):
            # Showing 3 at max
            if i != len(result[5])-1 and i < 3:
                self.text += cat + ", "
            elif i == 3:
                # End
                self.text += cat + ",..."
                break
            else:
                self.text += cat
        self.text += "</span>"
        
        # Setting the text label
        self.textLabel = QLabel(self.text)
        self.textLabel.setTextFormat(Qt.RichText)
        self.textLabel.setStyleSheet("font-size:14px;")
        self.textLabel.setAlignment(Qt.AlignLeft | Qt.AlignVCenter)
        self.textLabel.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Preferred)
        self.textLabel.setToolTip(tooltip)
        self.textLabel.mousePressEvent = lambda event: self.leftClickEvent(event)
        self.layout.addWidget(self.textLabel)
        self.layout.addStretch()

        # setting up favorite button
        self.favLabel = QLabel()
        self.normalStar = QPixmap(os.path.join(os.path.dirname(os.path.realpath(__file__)), "../images/star_normal.png"))
        self.yellowStar = QPixmap(os.path.join(os.path.dirname(os.path.realpath(__file__)), "../images/star_yellow.png"))
        self.fav = result[7]

        if self.fav:
            # If favorite, then set yellow star
            self.favLabel.setPixmap(self.yellowStar)
            self.favLabel.setToolTip("Remove from favorites")
        else:
            self.favLabel.setPixmap(self.normalStar)
            self.favLabel.setToolTip("Add to favorites")

        # build the item used for emitting the signal
        item = [result[1], result[2], result[4], result[3], result[6], result[7]]
        self.favLabel.mousePressEvent = lambda event: self.changeIcon(item)
        self.layout.addWidget(self.favLabel)

        # Setting up icon
        # Setting the information icon
        self.infoIcon = HoverLabel()
        path = os.path.join(os.path.dirname(os.path.realpath(__file__)), "../images/icons.png")
        icon = QPixmap(path).copy(1*16, 9*16, 16, 16)
        self.infoIcon.setPixmap(icon)
        self.infoIcon.enterEvent
        self.infoIcon.setToolTip("Show Product Info")
        self.infoIcon.mousePressEvent = lambda event, id=result[2]: self.showInfo(id)
        self.layout.addWidget(self.infoIcon)

    def changeIcon(self, item):
        """Changes the icon of the fav label
        """
        if self.fav:
            # self.fav value is pre change, so if original value was true, then set the icon to normal star
            self.favLabel.setPixmap(self.normalStar)
            self.fav = False
        else:
            self.favLabel.setPixmap(self.yellowStar)
            self.fav = True
        self.updateFav.emit(item)

        # build the item to pass to the signal emit

    def showInfo(self, id):
        """This function creates an instance of the plugin overview window. called when the info icon is clicked

        Args:
            id (_type_): id of the product
        """
        # loops through each opened window
        for window in SearchResult.openedWindow:
            # Raise the window if window alreay exists
            if window.getId() == id:
                window.raise_()
                return
        
        # No window of the product exists yet, so create an instance of it
        window = ProductOverviewWindow(id, self.theme)
        window.windowClosed.connect(self.removeWindow)
        window.show()
        SearchResult.openedWindow.append(window)

    def removeWindow(self, window):
        """Removes the window from openedWindow

        Args:
            window (Qwidget): window to be removed
        """
        SearchResult.openedWindow.remove(window)

    def leftClickEvent(self, event):
        """Handles the left click event to make the label clickable.

        Args:
            event (_type_): _description_
        """
        if event.button() == Qt.LeftButton:
            self.leftClicked.emit()

    def loadIcon(self):
        """This method does the same thing as loadIconHandler in titledComboBox, go see that for comments.

        Returns:
            _type_: _description_
        """
        if not self.iconFuture:
            self.iconFuture = self.exe.submit(self.fetchIcon)
        if self.iconFuture.done():
            icon = self.iconFuture.result()
            self.iconLabel.setPixmap(icon.pixmap(27, 27))
            return True
        return False

    def fetchIcon(self):
        """This method does a request to the api to get the icon

        Returns:
            _type_: _description_
        """
        response = requests.get(self.iconUrl)
        if response.status_code == 200:
            pixmap = QPixmap()
            pixmap.loadFromData(response.content)
            if pixmap.isNull():
                return self.defaultIcon

            icon = QIcon(pixmap)
            return icon
        else:
            return self.defaultIcon
        
    def closeEvent(self, event):
        """Overriding the close event, so that icon and text gets removed properly."""
        self.iconFuture = None
        self.iconLabel.clear()
        self.textLabel.clear()
        self.layout.removeWidget(self.textLabel)
        self.layout.removeWidget(self.iconLabel)
        self.removeWidget(self.layout)
        super().closeEvent(event)
