from PyQt5.QtWidgets import QWidget, QVBoxLayout, QLabel, QFrame, QHBoxLayout, QSizePolicy
from PyQt5.QtCore import Qt, pyqtSignal, QTimer
from PyQt5.QtGui import QIcon, QPixmap
from .HoverLabel import HoverLabel
from .productOverviewWindow import ProductOverviewWindow
from ..constants import ICON_URL
from concurrent.futures import ThreadPoolExecutor
from qgis.core import QgsMessageLog
import requests
import os

class CustomDropDown(QWidget):
    # emits name, id, shape
    itemSelected = pyqtSignal(str, str, str)
    updateFav = pyqtSignal(list)
    # Stores the instance of the productOverWindow
    openedWindow = []

    def __init__(self, text, settings, parent=None):
        super().__init__(parent)
        self.isExpanded = False
        self.text = text
        self.settings = settings

        # Setting the main layout to contain the entire widget
        self.mainLayout = QVBoxLayout(self)
        self.mainLayout.setContentsMargins(0,0,0,0)
        self.mainLayout.setSpacing(0)
        
        # Frame for the combobox itself
        self.frame = QFrame(self)
        self.frame.setFrameShape(QFrame.StyledPanel)
        self.frame.setFrameShadow(QFrame.Raised)
        self.frame.setProperty("expanded", "false")

        self.frame.setObjectName("comBoxFrame")
        self.frame.setContentsMargins(0,0,0,0)
        self.frame.mousePressEvent = self.toggleDropdown
        
        self.mainLayout.addWidget(self.frame)

        # set layout inside the frame
        self.hbox = QHBoxLayout(self.frame)
        self.hbox.setContentsMargins(0,0,0,0)
        self.hbox.setSpacing(3)

        # Label displaying the icon
        self.iconLabel = QLabel()
        path = os.path.join(os.path.dirname(os.path.realpath(__file__)), "../images/icons.png")
        self.collapsePixmap = self.extractIcon(path, 2*16, 1*16, 16, 16)
        self.expandPixmap = self.extractIcon(path, 4*16, 1*16, 16, 16)
        self.iconLabel.setPixmap(self.collapsePixmap)
        self.iconLabel.setMaximumHeight(40)
        self.iconLabel.setAlignment(Qt.AlignLeft | Qt.AlignVCenter)
        self.hbox.addWidget(self.iconLabel)

        # Label displaying the title
        self.textLabel = QLabel(text)
        self.textLabel.setStyleSheet("padding: 5px; margin: 0px;")
        self.textLabel.setAlignment(Qt.AlignLeft)
        self.textLabel.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Preferred)
        self.textLabel.setMaximumHeight(40)
        self.hbox.addWidget(self.textLabel)
        
        #self.mainLayout.setSpacing(0)
        # Frame and layout to store the items
        self.itemsFrame = QFrame()
        self.itemsFrame.setFrameShape(QFrame.StyledPanel)
        self.itemsFrame.setFrameShadow(QFrame.Raised)
        self.itemsFrame.setObjectName("itemsFrame")
        self.itemsFrame.setContentsMargins(0,0,0,0)

        self.mainLayout.addWidget(self.itemsFrame)
        self.itemsFrame.hide()

        self.itemsLayout = QVBoxLayout()
        self.itemsLayout.setSpacing(0)
        self.itemsLayout.setContentsMargins(0,0,0,0)

        self.itemsFrame.setLayout(self.itemsLayout)
        self.setLayout(self.mainLayout)

        # Stuff below is starting to set up the icon fetching in background
        # this is to store all the loaded icons
        self.iconCache = {}

        # Setting default icon
        QPixmapIcon = QPixmap(27, 27)
        QPixmapIcon.fill(Qt.transparent)
        self.defaultIcon = QIcon(QPixmapIcon)  # Default blank icon
        self.iconLoaded = False

        # max of 10 workers since it doesn't really get significantly faster after that
        self.exe = ThreadPoolExecutor(max_workers=min(10, os.cpu_count() * 2))
        self.count = 0

        # Setting up the icon
        self.starIcon = QPixmap(os.path.join(os.path.dirname(__file__), '../images/star_normal.png'))
        self.starYellowIcon = QPixmap(os.path.join(os.path.dirname(__file__), '../images/star_yellow.png'))

    def addItem(self, data):
        """This function adds a new item to the dropdown list

        Args:
            data (list of lists): list of lists [text, id, late?, desc, type, favorite?]
        """
        self.itemFrameAndHbox = {}
        self.intToProd = []
        self.items = {}
        # Creating the items in the drop down
        for i, item in enumerate(data):
            # Create a layout to put the frame in
            # Need to first create a frame and layout for each item
            self.intToProd.append((item[0], item[1]))
            self.items[item[1]] = item

            frame = QFrame()
            frame.setObjectName("items")
            frame.setFrameShape(QFrame.StyledPanel)
            frame.setFrameShadow(QFrame.Raised)

            # setting tool tip for the frame
            description = item[3].replace("\n", " ")
            tooltip = f"<div style='white-space:pre-wrap; max-width: 600px;'>{description}</div>"

            # setting up the border radius if it is top most or bottom most item
            if i == 0:
                frame.setStyleSheet("QFrame#items{border-top-left-radius: 4px; border-top-right-radius: 4px;}")
            elif i == len(data) - 1:
                frame.setStyleSheet("QFrame#items{border-bottom-left-radius: 4px; border-bottom-right-radius: 4px;}")
            
            hbox = QHBoxLayout(frame)
            hbox.setContentsMargins(0,0,0,0)

            # Assigning the default icon
            itemIcon = QLabel()
            itemIcon.setPixmap(self.defaultIcon.pixmap(27,27))
            itemIcon.setAlignment(Qt.AlignLeft | Qt.AlignVCenter)
            itemIcon.setToolTip(tooltip)
            hbox.addWidget(itemIcon)

            # Setting the late indicator # whether it is late or not
            lateLabel = QLabel()
            lateLabel.setAlignment(Qt.AlignLeft | Qt.AlignVCenter)
            #lateLabel.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Preferred)

            # Setting color depending on if it is late or not
            color = "#F75D1E" if item[2] else "#38CF38"

            if item[4] == "shape":
                # if a product is vector, then put v
                lateText = f"<span style='color: {color}; font-size:12px; font-weight:bold;'>V</span>"
            else:
                lateText = f"<span style='color: {color}; font-size:12px; font-weight:bold;'>R</span>"

            shape = "vector" if item[4] == "shape" else item[4]

            lateLabel.setToolTip(f"Late: {item[2]}, Shape: {shape}")

            lateLabel.setText(lateText)
            hbox.addWidget(lateLabel)
            
            # Setting the text label
            itemLabel = QLabel(item[0])
            itemLabel.mousePressEvent = lambda event, text=item[0], id = item[1], shape = shape: self.selectItem(text, id, shape)

            # 27 for itemIcon, 16 for infoIcon
            itemLabel.setAlignment(Qt.AlignLeft | Qt.AlignVCenter)
            itemLabel.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Preferred)
            itemLabel.setToolTip(tooltip)
            hbox.addWidget(itemLabel)
            frame.setLayout(hbox)
            hbox.addStretch()
            
            # Setting the fav icon
            favLabel = QLabel()
            
            if item[5]:
                # This is favorite add the yellow star icon
                favLabel.setPixmap(self.starYellowIcon)
                favLabel.setToolTip("Remove from favorites")
            else:
                # This is favorite add the normal star icon
                favLabel.setPixmap(self.starIcon)
                favLabel.setToolTip("Add to favorites")
            favLabel.mousePressEvent = lambda event, itemToUse = self.items[item[1]]: self.updateFav.emit(itemToUse)
            hbox.addWidget(favLabel)

            # Setting the information icon
            infoIcon = HoverLabel()
            path = os.path.join(os.path.dirname(os.path.realpath(__file__)), "../images/icons.png")
            icon = self.extractIcon(path, 1*16, 9*16, 16, 16)
            infoIcon.setPixmap(icon)
            infoIcon.setToolTip("Show Product Info")
            infoIcon.mousePressEvent = lambda event, id=item[1]: self.showInfo(id)
            hbox.addWidget(infoIcon)

            self.itemsLayout.addWidget(frame)
            self.itemFrameAndHbox[i] = [item[1], frame, hbox, itemIcon, favLabel, infoIcon]
            self.count += 1


        # Update the comobox title
        self.textLabel.setText(self.text + f" ({self.count})")

    def updateFavHandler(self, item):
        self.updateFav.emit(item)

    def removeAll(self):
        """This function removes all items from the dropdown list"""
        # This stops the timer from trying to fetch icons
        if hasattr(self, 'timer'):
            self.timer.stop()

        self.itemsLayout.setContentsMargins(0,0,0,0)
        self.itemsFrame.setLayout(self.itemsLayout)
        self.mainLayout.addWidget(self.itemsFrame)
        self.count = 0
        self.textLabel.setText(self.text)
        for i in self.itemFrameAndHbox.keys():
            for j in self.itemFrameAndHbox[i]:
                if not isinstance(j, str):
                    j.deleteLater()
                    j = None
        self.itemFrameAndHbox.clear()

    def changeIcon(self, item, frame):
        """Handles the changing icon and the signal emit

        Args:
            item (_type_): item of the object
            frame (_type_): frame of the object
        """
        # item[5] is post changed value.
        if not item[5]:
            # If it is favorite, then replace the yellow star icon
            frame[4].setPixmap(self.starIcon)
            frame[4].setToolTip("Add to favorites")
            self.items[item[1]][5] = False
        else:
            # If it is not favorite, then replace the star icon
            frame[4].setPixmap(self.starYellowIcon)
            frame[4].setToolTip("Remove from favorites")
            self.items[item[1]][5] = True
            

    def updateStar(self, item):
        """Updates the star icon for the product given fav or not

        Args:
            item(_list_): list of properties of the item
        """
        # find the item in the itemFrameAndHbox dict
        for i in self.itemFrameAndHbox.keys():
            if self.itemFrameAndHbox[i][0] == item[1]:
                # change the star icon
                self.changeIcon(item, self.itemFrameAndHbox[i])
    def showInfo(self, id):
        """This function creates an instance of the plugin overview window. called when the info icon is clicked

        Args:
            id (_type_): id of the product
        """
        # loops through each opened window
        for window in CustomDropDown.openedWindow:
            # Raise the window if window alreay exists
            if window.getId() == id:
                window.raise_()
                return
        
        # No window of the product exists yet, so create an instance of it
        theme = self.settings.value("UI/uitheme")

        if theme is None:
            theme = self.settings.value("UI/UITheme")
        window = ProductOverviewWindow(id, theme)
        window.windowClosed.connect(self.removeWindow)
        window.show()
        CustomDropDown.openedWindow.append(window)

    def removeWindow(self, window):
        """Removes the window from openedWindow

        Args:
            window (QWidget): window to be removed
        """

        CustomDropDown.openedWindow.remove(window)
        
    def expand(self):
        """Expands the dropdown, used when expanding all the dropdown"""
        if not self.isExpanded:
            # if not expanded, then expand
            self.itemsFrame.show()
            self.frame.setProperty("expanded", "true")
            self.frame.style().unpolish(self.frame)
            self.frame.style().polish(self.frame)
            self.iconLabel.setPixmap(self.expandPixmap)
            # Check if the show icon is toggled, if so, start loading the icons
            if self.settings.value("realEarth/iconToggle", True, type=bool):
                if not self.iconLoaded:
                    QTimer.singleShot(0, self.loadIconHandler)
                    self.iconLoaded = True
            QTimer.singleShot(0, self.loadIconHandler)
            self.isExpanded = True

    def collapse(self):
        """Collapse the dropdown, used when collapsing all the dropdown"""
        if self.isExpanded:
            # if expanded, then collapse
            self.itemsFrame.hide()
            self.frame.setProperty("expanded", "false")
            self.frame.style().unpolish(self.frame)
            self.frame.style().polish(self.frame)
            self.iconLabel.setPixmap(self.collapsePixmap)
            
            self.isExpanded = False

    def fetchIcon(self, item):
        """Fetches the icon from the API and returns the icon

        Args:
            i (_type_): the index of the item in the combobox

        Returns:
            i, icon: the index and the icon associated with it. return default if fails
        """
        # check if the icon is already in the cache.
        product = self.intToProd[item][0]

        if(product in self.iconCache):
            return item, self.iconCache[product]
        
        url = ICON_URL.format(id=self.intToProd[item][1])
        x = requests.get(url)
        if(x.status_code == 200):
            # success

            # first stores the response a a pixmap then turn it into a QIcon
            pixmap = QPixmap()
            pixmap.loadFromData(x.content)
            
            if pixmap.isNull():
                # Image is broken
                return item, self.defaultIcon
            
            icon = QIcon(pixmap)
            self.iconCache[product] = icon
            return item, icon
        else:
            return item, self.defaultIcon # return default if fails to get

    def loadIconHandler(self):
        """this function (hopefully) loads the icons in the background

        Args:
            max_workers: max workers. Defaults to 5.
        """

        # this funct
        self.futures = []
        for i in range(self.count):
            self.futures.append(self.exe.submit(self.fetchIcon, i))
        self.timer = QTimer()

        # This repeatedly updates Icon. Icon will update if completedFuture has something stored inside.
        self.timer.timeout.connect(self.updateIcon)
        self.timer.start(100)

    def updateIcon(self):
        """updates the icon and force the combobox to repaint."""
        # Gets a list of all futures that are done
        completedFutures = []
        for f in self.futures:
            if f.done():
                completedFutures.append(f)
        
        # For each completed futures, set the icon.
        for future in completedFutures:
            index, icon = future.result()
            self.setItemIcon(index, icon.pixmap(27, 27))
            self.futures.remove(future)
        if not self.futures:
            self.timer.stop()
    
    def toggleDropdown(self, event):
        """This function expands the dropdown if it is not expanded, collapses if it is expanded."""
        self.isExpanded = not self.isExpanded

        if self.isExpanded:
            # Expands the dropdown
            self.itemsFrame.show()
            self.frame.setProperty("expanded", "true")
            self.frame.style().unpolish(self.frame)
            self.frame.style().polish(self.frame)
            self.iconLabel.setPixmap(self.expandPixmap)
            # checks if show icon is toggled, if so, start loading the icons
            if self.settings.value("realEarth/iconToggle", True, type=bool):
                if not self.iconLoaded:
                    QTimer.singleShot(0, self.loadIconHandler)
                    self.iconLoaded = True
        else:
            # Collapse the dropdown
            self.itemsFrame.hide()
            self.frame.setProperty("expanded", "false")
            self.frame.style().unpolish(self.frame)
            self.frame.style().polish(self.frame)
            self.iconLabel.setPixmap(self.collapsePixmap)

    def setItemIcon(self, index, icon):
        """ Sets the item icon"""
        self.itemFrameAndHbox[index][3].setPixmap(icon)

    def selectItem(self, text, id, shape):
        """Emits the signal with details of the selected itms.

        Args:
            text (_type_): name of product
            id (_type_): id of product
            shape (_type_): raster or vector
        """
        if not self.isExpanded:
            # i think this might get accidentally triggered during startup, so its just here to prevent that.
            return
        self.itemSelected.emit(text, id, shape)
        
    def extractIcon(self, path, x, y, width, height):
        """This function extracts an icon from the given image

        Args:
            path (_type_): path to the image
            x (_type_): x coord
            y (_type_): y coord
            width (_type_): width of the icon
            height (_type_): height of the icon 
        """
        return(QPixmap(path).copy(x,y,width, height))