import requests
from PyQt5.QtWidgets import QListWidget, QListWidgetItem
from PyQt5.QtCore import Qt, QCoreApplication
from xml.etree import ElementTree as ET
from collections import defaultdict
from qgis.core import QgsMessageLog
from datetime import datetime
from ..constants import WMTS_URL, WFS_URL, PRODUCT_TIME
import re

def searchTime(product):
    """This funtion searches for the time associated with the product.

    Args:
        product (tuple): (name, id)

    Returns:
        list of dictionaries: list of dictionaries of all the neccessary information to create a layer for the wmts 
    """
    url = WMTS_URL.format(id=product[1])

    try:
        x = requests.get(url)
    except requests.exceptions.RequestException as e:
        raise e

    root = ET.fromstring(x.content)

    # Setting up XML namespaces.

    namespace = {'ows': 'http://www.opengis.net/ows/1.1', 'gml': 'http://www.opengis.net/gml'}
    location = "http://www.opengis.net/wmts/1.0"
    contents = root.find(f'.//{{{location}}}Contents')
    productName = product[0]

    layers = contents.findall(f'.//{{{location}}}Layer')

    layerInfo = []
    latestForecast = False
    current = False

    for layer in layers:
        # We want to skip anything that says current or future
        title = layer.find('ows:Title', namespace).text if layer.find('ows:Title', namespace) is not None else None
        if title == productName + ': Current' or title == productName + ": Latest Forecast":
            continue
        
        info = {}
        # Used to identify if the layer is latest forecast or current
        info["Latest Forecast"] = False
        info["Current"] = False
        info["originalTitle"] = title

        # updates the layer title if it is a latest forecast or a current
        if not latestForecast and "(Forecast)" in title:
            title = title.replace("(Forecast)", "(Latest Forecast)")
            latestForecast = True
            info["Latest Forecast"] = True
        elif "Animate" not in title and not current and "(Forecast)" not in title:
            title += " (Current)"
            current = True
            info["Current"] = True

        # If it is the first forecast or the first none forecast, we need to change the title
        
        info["Title"] = title

        # getting crs
        boundingBox = layer.find('ows:BoundingBox', namespace)
        if boundingBox is not None:
            crsSplit = boundingBox.get("crs").split("crs:")
            crs = crsSplit[1] if len(crsSplit) > 1 else crsSplit[0]
            crs = ":".join(crs.split('::'))
            info["CRS"] = crs

        # getting Id/Layer
        identifier = layer.find("ows:Identifier", namespace)
        info["Identifier"] = identifier.text if identifier is not None else None

        # getting Format
        format = layer.find(f'{{{location}}}Format')
        info["Format"] = format.text if format is not None else None

        # getting MatrixTile
        matrixTileLink = layer.find(f'{{{location}}}TileMatrixSetLink')
        if matrixTileLink is not None:
            matrixTileSet = matrixTileLink.find(f'{{{location}}}TileMatrixSet')
            info["TileMatrixSet"] = matrixTileSet.text if matrixTileSet is not None else None
            
        # getting Style
        style = layer.find(f'{{{location}}}Style')
        if style is not None:
            if style.get("isDefault") == "true":
                info["Style"] = ""
            else:
                styldid = style.find("ows:Identifier", namespace)
                info["Style"] = styldid.text if styldid is not None else None

        # getting Extract time from title
        info["Time"] = info["Title"].split(f"{productName}: ", 1)[1].strip() if productName in info["Title"] else info["Title"]
    
        # url
        info["url"] = url
        
        #id: The product id
        info["id"] = product[1]

        layerInfo.append(info)

    return layerInfo

def updateList(masterProduct, widget, list: QListWidget, toAdd):
    """This function updates the Time selection list with the given list of stuff to add.

    Args:
        masterProduct (_type_): The selected product
        widget (_type_): the groupbox holding the widget, just here to change the title
        list (QListWidget): the list to modify
        toAdd (_type_): things to add
    """
    # Clearing the list
    list.clear()
    list.verticalScrollBar().setValue(list.verticalScrollBar().minimum())
    
    # Adding things to the new list
    for item in toAdd:
        # for each item set the item name and the entire dictionary associated with it.
                
        item_toAdd = QListWidgetItem(item["Time"])
        item_toAdd.setData(Qt.UserRole, item)
        list.addItem(item_toAdd)

    # updating the title after retrieving the time
    _translate = QCoreApplication.translate
    if masterProduct != "": 
        widget.setTitle(_translate("RealEarthDockWidgetBase", f"{masterProduct}: Time Selection ({len(toAdd)})"))
    else:
        widget.setTitle(_translate("RealEarthDockWidgetBase", "Select a Product From Above"))

def getWmts(responseAndUrl):
    """This function fetches the layers of the wmts link(url)

    Args:
        response (_type_): response of the wmts request

    Returns:
        dictionary: Key: Provider, value: list of layers
    """
    root = ET.fromstring(responseAndUrl[0].content)
    url = responseAndUrl[1]

    # Setting up XML namespaces.

    namespace = {'ows': 'http://www.opengis.net/ows/1.1', 'gml': 'http://www.opengis.net/gml'}
    location = "http://www.opengis.net/wmts/1.0"
    contents = root.find(f'.//{{{location}}}Contents')

    layers = contents.findall(f'.//{{{location}}}Layer')

    # Each key is the provider, and stores a list of dictionaries for a product
    layerInfo = defaultdict(list)

    for layer in layers:
        info = {}
        info["Title"] = layer.find('ows:Title', namespace).text if layer.find('ows:Title', namespace) is not None else None

        # getting abstract
        info["Abstract"] = layer.find('ows:Abstract', namespace).text if layer.find('ows:Abstract', namespace) is not None else None

        # getting crs
        boundingBox = layer.find('ows:BoundingBox', namespace)
        if boundingBox is not None:
            info["CRS"] = boundingBox.get("crs")

        # getting Id/Layer
        identifier = layer.find("ows:Identifier", namespace)
        info["Identifier"] = identifier.text if identifier is not None else None

        # getting Format
        format = layer.find(f'{{{location}}}Format')
        info["Format"] = format.text if format is not None else None

        # getting MatrixTile
        matrixTileLink = layer.find(f'{{{location}}}TileMatrixSetLink')
        if matrixTileLink is not None:
            matrixTileSet = matrixTileLink.find(f'{{{location}}}TileMatrixSet')
            info["TileMatrixSet"] = matrixTileSet.text if matrixTileSet is not None else None
            
        # getting Style. i dont see style, so i'm hardcoding it.
        info["Style"] = ""

        # extracting the provider and the name, hardcoding this because this is the only format i see.
        info["Provider"], info["Name"] = info["Title"].split(" - ", 1)
    
        # url
        info["url"] = url
        layerInfo[info["Provider"]].append(info)

    return layerInfo

def getWfs(id):
    """Gets the wfs parameters for 

    Args:
        id (_type_): id of the product

    Returns:
        dictionary: Key: layer, value: wfs_parameters
    """
    output = defaultdict(list)

    url = WFS_URL.format(id=id)
    try:
        x = requests.get(url)
    except requests.exceptions.RequestException as e:
        raise e
    
    root = ET.fromstring(x.content)

    # Setting up XML namespaces.

    namespace = {'wfs': 'http://www.opengis.net/wfs/2.0', 'ows': 'http://www.opengis.net/ows/1.1'}

    productName = root.find(f'.//ows:Title', namespace).text
    version = root.find('.//ows:ServiceTypeVersion', namespace).text
    serviceType = root.find('.//ows:ServiceType', namespace).text

    for polygonType in root.findall('.//wfs:FeatureType', namespace):
        typename = polygonType.find(".//wfs:Name", namespace).text
        srsname = polygonType.find(".//wfs:DefaultCRS", namespace).text
        title = polygonType.find('wfs:Title', namespace).text

        # This is also very hardcoded
        originalTitle = title
        time = title.split(f"{productName}: ", 1)[1].strip() if productName in title else title
        # Removing the stuff like (Point) (Line)...
        time = re.sub(r'\s*\([^)]*\)', '', time).strip()

        info = {}
        info["typename"] = typename
        info["srsname"] = srsname
        info["version"] = version
        info["serviceType"] = serviceType
        info["productName"] = productName
        info["title"] = title
        info["originalTitle"] = originalTitle
        info["time"] = time
        info["url"] = url

        output[time].append(info)
    
    return output

def checkTime(title, id):
    """Checks for the timestamp, this is for MVT since we only need to compare the time

    Args:
        title (_type_): title that contains the timestamp
        id (_type_): check id

    Returns:
        str, str: new title and time, none if no new time found
    """
    # TODO: CHANGE URL AND ID
    url = PRODUCT_TIME.format(id=id)

    # Getting the time from the title
    match = re.search(r'\d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2}', title)
    if match:
        timeSplit = match.group(0).split(" ")
        layerDate = timeSplit[0].split("-")
        layerDate = ''.join(layerDate)
        layerTime = timeSplit[1].split(":")
        layerTime = ''.join(layerTime)
        
        dateTime = layerDate + '.' + layerTime

        # Get the time list of the layer
        response = requests.get(url)
        if response.status_code == 200:
            responseDict = response.json()
            # doing this since i know there is only one item in the dictionary
            id, time = next(iter(responseDict.items()))
            newTime = time[-1]
            if newTime != dateTime:
                # create the title
                dateTimeObj = datetime.strptime(newTime, "%Y%m%d.%H%M%S")
                timeReplacement = dateTimeObj.strftime("%Y-%m-%d %H:%M:%S")

                # replacement the time into the title
                newTitle = title[:match.start()] + timeReplacement + title[match.end():len(title)]

                return newTitle, newTime

    return None, None

def checkTimestamp(item):
    """Checks if there is a new timestamp

    Args:
        item (list):[layer name, original title, current?, forecast?, id, time, raster/vector]

    Returns:
        _type_: The layer within the xml file
    """
    layerTitle = item[1]

    if item[6] == "vector":
        # if it is vector, there is a different way to check
        url = WFS_URL.format(id=item[4])
        try:
            x = requests.get(url)
        except requests.exceptions.RequestException as e:
            raise e
        root = ET.fromstring(x.content)
        # Setting up XML namespaces.

        namespace = {'wfs': 'http://www.opengis.net/wfs/2.0', 'ows': 'http://www.opengis.net/ows/1.1'}
        location = "http://www.opengis.net/wmts/1.0"

        # retrieves the polygon type
        polygonMatch = re.search(r'\(.*?\)', item[1])
        polygon = polygonMatch.group() if polygonMatch else ""

        productName = root.find(f'.//ows:Title', namespace).text
        version = root.find('.//ows:ServiceTypeVersion', namespace).text
        serviceType = root.find('.//ows:ServiceType', namespace).text

        for layer in root.findall('.//wfs:FeatureType', namespace):
            title = layer.find('wfs:Title', namespace).text if layer.find('wfs:Title', namespace) is not None else None

            # want to skip the ones that are either latest forecast or current
            if "Current" in title or "Latest Forecast" in title:
                continue
            # iirc item[2] is if it's latest forecast.
            # A lot of check here but basically we dont look for Forecast, Current, Animate, and layer where polygon doesn't match
            if ((item[3] and "Forecast" in title) or (item[2] and "Forecast" not in title)) and "Animate" not in title and polygon in title:
                # Finding the first instance of forecast or none forecast depending if original layer is forecast or current
                if layerTitle != title:
                    # Returning otherInfo becuase this is a necessary information for wfs layer
                    otherInfo = {"productName":productName, "version":version, "serviceType":serviceType, "id":item[4]}
                    return layer, otherInfo
                # still the same forecast. returning None and None because the caller expects two things being returned
                return None, None
    else:
        # Raster layer. doing a lot of the same thing but minor difference.
        url = WMTS_URL.format(id=item[4])

        try:
            x = requests.get(url)
        except requests.exceptions.RequestException as e:
            raise e

        root = ET.fromstring(x.content)

        # Setting up XML namespaces.

        namespace = {'ows': 'http://www.opengis.net/ows/1.1', 'gml': 'http://www.opengis.net/gml'}
        location = "http://www.opengis.net/wmts/1.0"
        contents = root.find(f'.//{{{location}}}Contents')
        layerTitle = item[1]
        # need to remove extra stuff in the title

        layers = contents.findall(f'.//{{{location}}}Layer')

        for layer in layers:
            title = layer.find('ows:Title', namespace).text if layer.find('ows:Title', namespace) is not None else None

            # want to skip the ones that are either latest forecast or current
            if "Current" in title or "Latest Forecast" in title:
                continue
            # iirc item[2] is if it's latest forecast
            if ((item[3] and "Forecast" in title) or (item[2] and "Forecast" not in title)) and "Animate" not in title:
                # Finding the first instance of forecast or none forecast depending if original layer is forecast or current
                if layerTitle != title:
                    # It means latest forecast is now different
                    return layer
                # still the same forecast
                return None


