# -*- coding: utf-8 -*-
"""
/***************************************************************************
 ReadOnlySwitcher
                                 A QGIS plugin
Read-Only Switcher enables you to quickly toggle the read-only flag for one 
or more selected vector layers directly from the Layers panel, eliminating 
the need to open the Project Properties dialog.

When multiple layers with mixed read-only statuses are selected, the plugin 
sets all layers to read-only by default to ensure data safety.

Raster layers and non-vector layers are automatically excluded from the 
toggling process, with clear notifications to the user.

This tool streamlines layer management, protects your data from accidental 
edits, and is especially useful in collaborative environments where different 
users may need different editing permissions on the same project.


 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2024-04-12
        git sha              : $Format:%H$
        copyright            : (C) 2024 by Lei Ding
        email                : lleidding@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QMessageBox
from qgis.utils import iface
from qgis.PyQt.QtGui import QIcon
from qgis.core import QgsMapLayer

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .read_only_switcher_dialog import ReadOnlySwitcherDialog
import os.path


class ReadOnlySwitcher:
    """The plugin class."""

    def __init__(self, iface):
        """Constructor."""
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'ReadOnlySwitcher_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Read-Only Switcher')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API."""
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('ReadOnlySwitcher', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar."""
        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            pass
            #self.iface.addPluginToMenu(
             #   self.menu,
              #  action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/read_only_switcher/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'Read-only switcher'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&Read-Only Switcher'),
                action)
            self.iface.removeToolBarIcon(action)
    
    
    def determine_new_state(self, layers):
        """
        Input: list of QgsVectorLayer
        Output: (has_non_vector, new_state, has_mixed_status, previous_state, vector_layers)
        """
        has_non_vector = False
        vector_layers = []
        
        for layer in layers:
            if not layer.type() == QgsMapLayer.VectorLayer:
                has_non_vector = True
            else:
                vector_layers.append(layer)
        
        if len(vector_layers) == 0:
            QMessageBox.warning(None, "Warning", "No vector layers selected!")
            return None  # Return none to avoid unpacking
        
        read_only_statuses = {layer.readOnly() for layer in vector_layers}
        has_mixed_status = len(read_only_statuses) > 1
        
        if has_mixed_status:
            # When mixed status, all change to read-only for data safety
            new_state = True
            previous_state = "mixed"
        else:
            current_state = read_only_statuses.pop()
            new_state = not current_state
            previous_state = "read-only" if current_state else "editable"
        
        return has_non_vector, new_state, has_mixed_status, previous_state, vector_layers
    
    
    #Change layer status
    def toggle_read_only(self,layers, new_state):
        for layer in layers:             
            layer.setReadOnly(new_state)



    def run(self):
        selected_layers = iface.layerTreeView().selectedLayers()
        if not selected_layers:
            QMessageBox.warning(None, "Warning", "No layers selected!")
            return
        
        result = self.determine_new_state(selected_layers)
        if result is None:  # if no vector
            return
        
        has_non_vector, new_state, has_mixed_status, previous_state, vector_layers = result
        self.toggle_read_only(vector_layers, new_state)

        message = (
            f"Successfully changed {len(vector_layers)} layer(s):\n"
            f"- Previous state: {previous_state}\n"
            f"- New state: {'read-only' if new_state else 'editable'}\n"
        )
        if has_non_vector:
            message +="\nSelected non-vector layers excluded in the process.\n"
        if has_mixed_status:
            message += "\n😵‍💫Note: Mixed states detected. All layers changed to read-only for data safety."

        QMessageBox.information(None, "Layer Status Updated", message)