# -*- coding: utf-8 -*-

import os, logging, sys
from functools import partial

from qgis.gui import *
from qgis.core import *
from qgis.utils import iface
from qgis.PyQt import QtGui, QtWidgets, uic, QtXml
from qgis.PyQt.QtCore import *

from .rdi_param import rdiParam
from .rdi_tools import quote, clearLayout
from .rdi_layer import rdiLayer, rdiLayersCollection

from .rdi_meta import rdiMeta


memoRdiLegend = {}



class rdiFilter():
    def __init__(self, dock, layout):
        self.dock = dock
        self.filterMode = None
        self.imageFolder = self.dock.imageFolder
        self.initData()
        self.makeFilters(layout)
        
    def makeFilters(self, layout):
        gbox = layout

        first = None
        w = QtWidgets.QGroupBox("Filtrer sur éléments")
        gbox.addWidget(w)
        vbox = QtWidgets.QVBoxLayout(w)
        vbox.setSpacing(2)
        for k,v in self.filterOptions.items():
            b = QtWidgets.QRadioButton(v, w)
            b.clicked.connect(partial(self.toggleFilter, b))
            vbox.addWidget(b)
            if self.filterMode==v or self.filterMode==None and not first:
                first = b


        self.labelTitle = QtWidgets.QLabel()
        font = QtGui.QFont()
        font.setItalic(True)
        self.labelTitle.setFont(font)
        gbox.addWidget(self.labelTitle)

        w = QtWidgets.QWidget()
        gbox.addWidget(w)
        vbox = QtWidgets.QVBoxLayout(w)
        self.widgetFilterlegend = w
        self.dock.legend = rdiLegend(self.dock, vbox)
        for v in self.dock.dyn.getAll().values():
            if v.rdi.style.isFilterable:
                try:
                    name = v.rdi.style.json['func']
                    legend = v.rdi.style.legend
                    self.dock.legend.add(name, legend, v.rdi)
                except:
                    pass
                
        w = QtWidgets.QWidget()
        gbox.addWidget(w)
        hbox = QtWidgets.QHBoxLayout(w)
        self.widgetFilterSelect = w
        ico = QtGui.QIcon(os.path.join(self.imageFolder,'filter.png'))
        butt = QtWidgets.QPushButton(ico, " Filtrer")
        hbox.addWidget(butt)
        butt.clicked.connect(self.filterSelection)
        self.widgetFilterSelect.butt = butt
        ico = QtGui.QIcon(os.path.join(self.imageFolder,'back.png'))
        butt = QtWidgets.QPushButton(ico, " Annuler")
        hbox.addWidget(butt)
        butt.clicked.connect(self.filterReset)
        self.widgetFilterSelect.buttCancel = butt
        self.widgetFilterSelect.buttCancel.setEnabled(False)
        
        w = QtWidgets.QWidget()
        gbox.addWidget(w)
        hbox = QtWidgets.QHBoxLayout(w)
        self.widgetFilterVisible = w
        ico = QtGui.QIcon(os.path.join(self.imageFolder,'filter.png'))
        butt = QtWidgets.QPushButton(ico, " Filtrer")
        hbox.addWidget(butt)
        butt.clicked.connect(self.filterVisible)
        self.widgetFilterVisible.butt = butt
        ico = QtGui.QIcon(os.path.join(self.imageFolder,'back.png'))
        butt = QtWidgets.QPushButton(ico, " Annuler")
        hbox.addWidget(butt)
        butt.clicked.connect(self.filterReset)
        self.widgetFilterVisible.buttCancel = butt
        self.widgetFilterVisible.buttCancel.setEnabled(False)

        first.click()
        

    
    def toggleFilter(self, b):
        self.filterMode = b.text()
        if b.text()==self.filterOptions['no']:
            self.labelTitle.setText(f"")
        if b.text()==self.filterOptions['legend']:
            self.labelTitle.setText(f"Filtrer depuis les items de la légende")
            self.widgetFilterlegend.show()
            self.dock.legend.makeContext()
        else:
            self.widgetFilterlegend.hide()
        if b.text()==self.filterOptions['select']:
            self.labelTitle.setText(f"Filtrer depuis votre sélection")
            self.widgetFilterSelect.show()
        else:
            self.widgetFilterSelect.hide()
        if b.text()==self.filterOptions['visible']:
            self.labelTitle.setText(f"Filtrer depuis les éléments visibles")
            self.widgetFilterVisible.show()
        else:
            self.widgetFilterVisible.hide()
        self.filterReset()
    
    def filterSelection(self):
        self.widgetFilterSelect.buttCancel.setEnabled(True)
        for v in self.dock.dyn.getAll().values():
            sel = v.selectedFeatures()
            pkey = v.rdi.getIdentFeature()
            try:
                lst = list(map(lambda x: x.attribute(x.fieldNameIndex(pkey)), sel))
            except:
                pkey = v.rdi.getIdentFeature(force=True)
                lst = list(map(lambda x: x.attribute(x.fieldNameIndex(pkey)), sel))
            v.rdi.setContextFromList(pkey, lst)
        self.dock.actuLayersAlea()
        self.dock.actuLayersEnjeu()
        
    def filterVisible(self):
        self.widgetFilterVisible.buttCancel.setEnabled(True)
        for v in self.dock.dyn.getAll().values():
            pkey = v.rdi.getIdentFeature() 
            vis = []
            renderer = v.rdi.vlayer.renderer().clone()
            ctx = QgsRenderContext()
            renderer.startRender(ctx, QgsFields())
            for feature in v.rdi.vlayer.getFeatures():
                ctx.expressionContext().setFeature(feature)
                if renderer.willRenderFeature(feature, ctx):
                    vis.append(feature)
            renderer.stopRender(ctx)
            lst = list(map(lambda x: x.attribute(x.fieldNameIndex(pkey)), vis))
            v.rdi.setContextFromList(pkey, lst)
        self.dock.actuLayersAlea()
        self.dock.actuLayersEnjeu()
        
    def filterReset(self):
        self.dock.actuIcoTabFil()
        self.widgetFilterSelect.buttCancel.setEnabled(False)
        for v in self.dock.dyn.getAll().values():
            v.rdi.context = None
        self.dock.actuLayersAlea()
        self.dock.actuLayersEnjeu()
        
    def initData(self):
        self.filterOptions = {
            'no': "Aucun", 
            'legend': "Légende", 
            'select': "Sélection", 
            'visible': "Visible",
        }
        
        

class rdiLegend():
    def __init__(self, dock, layout):
        self.dock = dock
        self.items = {}
        self.ori = {}
        self.layout = layout
    
    def setlayout(self, layout):
        self.layout = layout
    
    def add(self, name, catalog, rdi):
        self.ori[rdi] = name
        if not name in self.items:
            item = rdiLegendItem(name, catalog, self.layout)
            item.clicked.connect(partial(self.makeContext, item))
            self.items[name] = item
        else:
            item = self.items[name]
            item.show()
        return item
    
    def makeContext(self, item=None):
        for rdi,name in self.ori.items():
            
            if not item:
                item = self.items[name]
            if item.name==name:
                rdi.setContextFromLegend(item)
        self.dock.actuLayersAlea()
        self.dock.actuLayersEnjeu()
    
    def remove(self, rdi):
        if rdi in self.ori:
            del self.ori[rdi]
            self.clean()
        
    def clean(self):
        lst = self.ori.values()
        for name,item in self.items.items():
            if not name in lst:
                item.hide()


class rdiLegendItem(QObject):
    
    clicked = pyqtSignal()
    
    def __init__(self, name, catalog, layout):
        QObject.__init__(self)
        self.name = name
        self.catalog = catalog
        self.layout = layout
        self.elts = []
        self.draw()
        
    def draw(self):
        if 'list' in self.catalog:
            if not self.name in memoRdiLegend:
                memoRdiLegend[self.name] = {}
            # l = QtWidgets.QLabel()
            # self.layout.addWidget(l)
            # self.space = l
            gb = QtWidgets.QGroupBox(self.name)
            self.layout.addWidget(gb)
            lgb = QtWidgets.QVBoxLayout(gb) 
            lgb.setSpacing(0)
            for i,e in enumerate(self.catalog['list']):
                elt = rdiLegendItemElt(i, e, lgb)
                if i in memoRdiLegend[self.name] and memoRdiLegend[self.name][i]==False:
                    elt.setChecked(False)
                elt.clicked.connect(self.click)
                self.elts.append(elt)  
            self.group = gb
    
    def click(self, ind, checked):
        memoRdiLegend[self.name][ind] = checked
        self.clicked.emit()
    
    def hide(self):
        # self.space.hide()
        self.group.hide()
        
    def show(self):
        # self.space.show()
        self.group.show()
    


class rdiLegendItemElt(QObject):
    
    clicked = pyqtSignal(int, bool)
    
    def __init__(self, ind, elt, layout):
        QObject.__init__(self)
        self.layout = layout
        self.elt = elt
        self.ind = ind
        self.draw()
        
    def draw(self):
        hbox = QtWidgets.QHBoxLayout()
        hbox.setSpacing(5)
        self.layout.addLayout(hbox)
        l = QtWidgets.QLabel()
        l.setStyleSheet(f"background-color:{self.elt['color']};")
        l.setFixedSize(17, 17)
        hbox.addWidget(l)
        cb = QtWidgets.QCheckBox(self.elt['label'])
        cb.setChecked(True)
        cb.stateChanged.connect(self.click)
        hbox.addWidget(cb)
        self.cb = cb
        
    def click(self):
        self.clicked.emit(self.ind, self.cb.isChecked())
    
    def setChecked(self, bool):
        self.cb.setChecked(bool)
    
    def isChecked(self):
        return self.cb.isChecked()
                
   


class rdiAlleviate(QObject):
    def __init__(self, parent):
        QObject.__init__(self)
        self.dock = parent
        self.excludes = {}
        self.windows = {}
        self.meta = rdiMeta()
    
    def openWindow(self, code):
        if code in self.windows.keys():
            return
        buttons = QtWidgets.QDialogButtonBox.StandardButton.Ok | QtWidgets.QDialogButtonBox.StandardButton.Cancel
        dialog = QgsDialog(iface.mainWindow(),
                  fl=Qt.WindowFlags(),
                  buttons=buttons,
                  orientation=Qt.Orientation.Horizontal)
        dialog.setWindowTitle(f"Sélectionner les {self.dock.param.get(code)}") 
        self.windows[code] = dialog
        coll = rdiLayersCollection(self.dock, code)
        coll.expand = True
        coll.excludes = {}
        w = QtWidgets.QWidget(dialog)       
        w.setMinimumWidth(280)
        dialog.layout().addWidget(w)
        gboxt = QtWidgets.QVBoxLayout(w)
        cb = self.makeLayerSelectTitle(gboxt, coll)
        sc = QtWidgets.QScrollArea(dialog)
        sc.setMinimumHeight(400)
        sc.setWidgetResizable(True)
        dialog.layout().addWidget(sc)
        w = QtWidgets.QWidget()
        gbox = QtWidgets.QVBoxLayout()
        gbox.setSpacing(0)
        gbox.setAlignment(Qt.AlignmentFlag.AlignTop)
        w.setLayout(gbox)
        sc.setWidget(w)
        cb.currentIndexChanged.connect(partial(self.rebuiltSelectList, cb, w))
        self.rebuiltSelectList(cb, w)
        dialog.accepted.connect(partial(self.rebuiltList, coll))
        self.setInfos(coll)
        dialog.show()
        dialog.finished.connect(partial(self.closeWindow, code))

    def closeWindow(self, code):
        del self.windows[code]

    def makeLayerSelectTitle(self, layout, coll):
        hbox = QtWidgets.QHBoxLayout()
        layout.addLayout(hbox)
        l = QtWidgets.QLabel()
        img = os.path.join(self.dock.imageFolder,f"{coll.code}.png")
        ico = QtGui.QIcon(img)
        l.setPixmap(ico.pixmap(15,15))
        l.setMaximumWidth(20)
        hbox.addWidget(l)
        l = QtWidgets.QLabel(self.dock.param.get(coll.code))
        font = QtGui.QFont()
        font.setBold(True)
        l.setFont(font)
        hbox.addWidget(l)
        l = QtWidgets.QLabel()
        font = QtGui.QFont()
        font.setBold(True)
        l.setFont(font)
        hbox.addWidget(l)
        coll.infos = l
        hbox2 = QtWidgets.QHBoxLayout()
        layout.addLayout(hbox2)
        cb = QtWidgets.QCheckBox("TOUS")
        cb.coll = coll
        coll.cbAll = cb
        cb.stateChanged.connect(partial(self.setAll, cb))
        hbox2.addWidget(cb)
        l = QtWidgets.QLabel("Regroupement par")
        hbox2.addWidget(l)
        hbox2.setAlignment(l, Qt.AlignmentFlag.AlignRight)
        cb = QtWidgets.QComboBox()
        for k,v in {'bloc':"Thèmes", 'bloczone':"Thèmes+Zones", 'zone':"Zones", 'zonebloc':"Zones+Thèmes"}.items():
            cb.addItem(v,k)
        cb.coll = coll
        hbox2.addWidget(cb)
        hbox2.setAlignment(cb, Qt.AlignmentFlag.AlignRight)
        return cb
    
    def rebuiltSelectList(self, cb, widget):
        gbox = widget.layout()
        clearLayout(gbox)
        self.makeLayerSelectList(gbox, cb.coll, regroup=cb.currentData())
        
    def makeLayerSelectList(self, layout, coll, regroup=None): 
        coll.blocs = {}
        if regroup.find('bloc')==0: order = 'bloc'
        else: order = 'parent_name'
        for elt in self.dock.psql.getTableAdmin(coll.code, order=order):
            if elt['id'] not in coll.dict:
                rdi = rdiLayer(elt)
                coll.addAbstract(rdi)
                first = True
            else:
                rdi = coll.dict[elt['id']]
                first = False
            if not rdi.tableName():
                continue
            name = rdi.name
            if rdi.parent(): name = f"{elt.get('parent_name')}  ({name})"
            if rdi.bloc() not in (None,""): name = f"[{rdi.bloc()}] {name}"
            key = None
            lay = layout
            
            cb = QtWidgets.QCheckBox(name)
            cb.rdi = rdi
            rdi.cb = cb
            if (first and self.isExclude(rdi)) or rdi.id() in coll.excludes.keys():
                coll.excludes[rdi.id()] = True
            else:
                cb.setChecked(True)
                
            if regroup.find('bloc')==0:
                if rdi.bloc() not in (None,""):
                    name = f"Thème [{rdi.bloc()}]"
                    key = f"bloc_{rdi.bloc()}"
                    lay = self.addGroup(coll, key, cb, name=name, layout=lay, bold=True)
                if regroup.find('zone')>0 and rdi.parent()!=None:
                        name = f"Zone {coll.find(rdi.parent()).name}"
                        key = f"zone_{rdi.bloc()}_{rdi.parent()}"
                        lay = self.addGroup(coll, key, cb, name=name, layout=lay, italic=True)
           
            if regroup.find('zone')==0:
                if rdi.parent()!=None:
                    name = f"Zone {coll.find(rdi.parent()).name}"
                    key = f"zone_{rdi.parent()}"
                    lay = self.addGroup(coll, key, cb, name=name, layout=lay, bold=True)
                if regroup.find('bloc')>0 and rdi.bloc() not in (None,""):
                    name = f"Thème [{rdi.bloc()}]"
                    key = f"bloc_{rdi.parent()}_{rdi.bloc()}"
                    lay = self.addGroup(coll, key, cb, name=name, layout=lay, italic=True)

            lay.addWidget(cb)
            cb.stateChanged.connect(partial(self.setExclude, cb))
        l = QtWidgets.QLabel("")
        layout.addWidget(l)
        for k,g in coll.blocs.items():
            g.detect()
   
    def setEnabled(self, bool):
        try:
            self.send.setEnabled(bool)
        except:
            pass
   
    def setAll(self, cb):
        bool = cb.isChecked()
        coll = cb.coll
        for rdi in coll.getElts():
            if not rdi.cb:
                continue
            rdi.cb.setChecked(bool)
    
    def setInfos(self, coll):
        tot = 0
        nb = 0
        for rdi in coll.getElts():
            if not rdi.cb:
                continue
            tot += 1
            if rdi.cb.isChecked():
                nb += 1
        try:
            l = coll.infos
            l.setText(f" ({nb}/{tot})")
            cb = coll.cbAll
            bool = (nb==tot)
            cb.blockSignals(True)
            cb.setChecked(bool)
            cb.blockSignals(False)
        except:
            pass
   
    def setExclude(self, cb):
        self.setEnabled(True)
        rdi = cb.rdi
        coll = rdi.coll
        if not cb.isChecked():
            coll.excludes[rdi.id()] = True
        else:
            if rdi.id() in coll.excludes.keys():
                del coll.excludes[rdi.id()]
        self.setInfos(coll)
    
    def isExclude(self, rdi):
        return self.isExcludeId(rdi.id())
            
    def isExcludeId(self, id):
        if id in self.excludes.keys():
            return True

    def rebuiltList(self, coll):
        for rdi in coll.getElts():
            if not rdi.cb:
                continue
            cb = rdi.cb
            if not cb.isChecked():
                self.excludes[rdi.id()] = True
            else:
                try:
                    del self.excludes[rdi.id()]
                except:
                    pass
        self.save()
        self.setEnabled(False)
        self.dock.actuTabs(['dynamic', 'statistic'])
    
    def addGroup(self, coll, key, cb=None, name=None, layout=None, bold=False, italic=False):
        if key not in coll.blocs:
            group = rdiAlleviateGroup(self.dock)
            coll.blocs[key] = group
            lay = None
            if layout is not None:
                lay = group.setTitle(layout, name, bold, italic)
        else:
            lay = coll.blocs[key].widget.layout()
        if cb is not None:
            coll.blocs[key].add(cb)
        return lay    
        
    def save(self):
        d = {}
        d['base'] = self.getBase()
        d['table'] = self.getTable()
        d['list'] = list(self.excludes.keys())
        if self.dock.param.get('filterMemo'):
            self.meta.writeIni('filter', d)
        else:
            self.meta.delIni('filter')
                
    def recupIni(self):
        if not self.dock.param.get('filterMemo'): return
        d = self.meta.readIni('filter')
        if not isinstance(d, dict): return
        if d.get('base')!=self.getBase(): return
        if d.get('table')!=self.getTable(): return
        self.excludes = {}
        for id in d.get('list',[]):
            self.excludes[int(id)] = True
            
    def getBase(self):
        return f"{self.dock.psql.PGdb}({self.dock.psql.PGhost})"
        
    def getTable(self):
        return f"{self.dock.psql.admin('tableAdmin')})"
        
class rdiAlleviateGroup(QObject):
    def __init__(self, dock):
        QObject.__init__(self)
        self.dock = dock
        self.cb = None
        self.list = []
        
    def setTitle(self, layout, name, bold, italic):
        hbox = QtWidgets.QHBoxLayout()
        layout.addLayout(hbox)
        self.title = rdiAlleviateGroupTitle(self.dock, name, bold, italic)
        hbox.addWidget(self.title)
        cbg = QtWidgets.QCheckBox()
        hbox.addWidget(cbg)
        cbg.stateChanged.connect(self.affect)
        self.cb = cbg
        self.widget = QtWidgets.QWidget()
        layout.addWidget(self.widget)
        self.widget.hide()
        vbox = QtWidgets.QVBoxLayout(self.widget)
        vbox.setSpacing(0)
        vbox.setContentsMargins(10,0,0,0)
        self.title.clicked.connect(self.clicAndDrop)
        return vbox
    
    def add(self, item):
        self.list.append(item)
        item.stateChanged.connect(self.detect)
        
    def affect(self):
        if not self.cb:
            return
        state = self.cb.isChecked()
        for item in self.list:
            item.setChecked(state)
        
    def detect(self):
        if not self.cb:
            return
        state = True
        for item in self.list:
            state = state and item.isChecked()
        self.cb.blockSignals(True)
        self.cb.setChecked(state)
        self.cb.blockSignals(False)
        
    def clicAndDrop(self, state):
        if state=='plus':
            self.widget.hide()
        if state=='minus':
            self.widget.show()



class rdiAlleviateGroupTitle(QtWidgets.QWidget):
    clicked = pyqtSignal(str)
    def __init__(self, dock, title, bold, italic):
        QtWidgets.QHBoxLayout.__init__(self)
        self.path = {'doss':dock.imageFolder}
        self.layout = QtWidgets.QHBoxLayout(self)
        self.layout.setSpacing(0)
        self.layout.setContentsMargins(0,0,0,0)
        self.initState()
        self.makeCross()
        self.makeLabel(title, bold, italic)
    
    def makeCross(self):
        l = QtWidgets.QLabel()
        l.setMaximumWidth(20)
        self.layout.addWidget(l)
        self.cross = l
        self.affectImg()
        
    def affectImg(self):
        ico = QtGui.QIcon(os.path.join(self.path['doss'], self.path[self.state]))
        self.cross.setPixmap(ico.pixmap(15,15))
    
    def makeLabel(self, txt, bold, italic):
        l = QtWidgets.QLabel(txt)
        if bold:
            font = QtGui.QFont()
            font.setBold(True)
            l.setFont(font)
        if italic:
            font = QtGui.QFont()
            font.setItalic(True)
            l.setFont(font)    
        self.layout.addWidget(l)
    
    def initState(self):
        self.state = 'plus'
        self.cd = {
            'plus': 'minus',
            'minus': 'plus',
        }
        self.path['plus'] = 'plus.png'
        self.path['minus'] = 'minus.png'
        
    def changeState(self):
        self.state = self.cd[self.state]
        self.affectImg()
    
    def mousePressEvent(self, event):
        self.changeState()
        self.clicked.emit(self.state)

 
