# -*- coding: utf-8 -*-

import os, logging, sys
from functools import partial

from qgis.gui import *
from qgis.core import *
from qgis.utils import iface
from PyQt5 import QtGui, QtWidgets, uic, QtXml
from PyQt5.QtCore import *

from .rdi_param import rdiParam, rdiParamBloc
from .rdi_tools import quote, makeUniqueIdent, clearLayout
from .rdi_trade import rdiTrade
from .rdi_postgis import tmpQuery
from .rdi_async import mGlobalTask

from .rdi_meta import rdiMeta

import math

class rdiMask():
    def __init__(self, parent):
        self.dock = parent
        self.meta = rdiMeta()
        self.initData()
        self.table = None
        self.layer = None
        self.trade = rdiTrade(self)
        self.selected = False
        self.postgres = False
        self.canvas = self.dock.parent.iface.mapCanvas()
        self.tool = rdiMaskTool(self)
        self.prevTool = None
         
    def recupIni(self):
        mask = self.meta.readIni("mask")
        if mask is not None and self.dock.psql.isTable(mask):
            self.table = mask
            self.affLayer()
        else:
            self.table = None
    
    def paint(self, layout):
        
        l = QtWidgets.QLabel("")
        layout.addWidget(l)
        gb = rdiParamBloc("filter_mask", "Masque de travail", self.dock, memo=False)
        layout.addWidget(gb)
        elts = [
            {
                'label': u"Activer",
                'help': 'p5-option-mask',
                'description': u"Permet de rajouter un masque pour ne travailler que sur une sous-partie de l'aléa",
                'ini': 'maskWorking',
                'tab': ['layer'],
                'func': ['self.actuIcoTabFil()'],
                'title': True,
            },
        ]
        gb.add(elts, self.dock.changeParameter)
        self.layout = gb.getLayout()

        if self.table!=None:
            self.makeChoosen()
        else:
            self.makeToChoose()

    def toggleChoice(self, b):
        self.choice = b.key   
    
    def makeToChoose(self):
        layout = self.layout
        clearLayout(layout)
        italic = QtGui.QFont()
        italic.setItalic(True)
        normal = QtGui.QFont()
        normal.setItalic(False)
        w = QtWidgets.QComboBox()
        w.setFont(italic)
        layout.addWidget(w)
        for i,(k,v) in enumerate(self.options.items()):
            if v.get('ico'):  w.addItem(QtGui.QIcon(os.path.join(self.dock.imageFolder, v.get('ico'))), v.get('label',k),k)
            else: w.addItem(v.get('label',k),k)
            w.setItemData(i, normal, Qt.FontRole)
        w.setItemData(0, italic, Qt.FontRole)
        self.chooseOption = w
        w = QtWidgets.QWidget()
        layout.addWidget(w)
        form = QtWidgets.QFormLayout(w)
        form.setContentsMargins(0,10,0,0) 
        self.chooseOption.currentIndexChanged.connect(partial(self.adaptToChoose, form))
        
        self.adaptToChoose(form)

        butt = QtWidgets.QPushButton("Charger")
        butt.setFixedSize(130, 30)
        layout.addWidget(butt)
        layout.setAlignment(butt, Qt.AlignCenter)
        butt.clicked.connect(self.set)
        self.butt = butt
        self.butt.hide()
    
    def adaptToChoose(self, form):
        self.setTool(False)
        clearLayout(form)
        if self.chooseOption.currentIndex()>0:
            self.chooseOption.setFont(QtGui.QFont())
            self.chooseOption.setItemData(0, '', Qt.UserRole-1)
            self.butt.show()
        if self.chooseOption.currentData()=='qgis':
            combo = QgsMapLayerComboBox()
            combo.setFilters(QgsMapLayerProxyModel.PolygonLayer)
            form.addRow("Couche", combo)
            cb = QtWidgets.QCheckBox("Entités sélectionnées uniquement")
            if self.selected:
                cb.setChecked(True)
            cb.stateChanged.connect(self.actuSelected)
            form.addRow("", cb)
            self.selectedCB = cb
            self.comboQgis = combo
        if self.chooseOption.currentData()=='file':
            hbox = QtWidgets.QHBoxLayout()
            form.addRow("Couche", hbox)
            ico = QtGui.QIcon(os.path.join(self.dock.imageFolder, "folder.png"))
            b = QtWidgets.QPushButton(ico, "")
            b.setMaximumWidth(30)
            b.clicked.connect(self.openFile)
            hbox.addWidget(b)
            l = QtWidgets.QLabel()
            hbox.addWidget(l)
            self.labelFile = l
            self.labelFile.file = None
            self.openFile()
        if self.chooseOption.currentData()=='draw':
            w = QtWidgets.QComboBox()
            italic = QtGui.QFont()
            italic.setItalic(True)
            w.setFont(italic)
            form.addRow(w)
            for i,(k,v) in enumerate(self.drawForms.items()):
                w.addItem(v.get('label',k),k)
                w.setItemData(i, QtGui.QFont(), Qt.FontRole)
            w.setItemData(0, italic, Qt.FontRole)
            self.drawOption = w
            w.currentIndexChanged.connect(self.adaptDrawOption)
    
    def adaptDrawOption(self):
        self.drawForm = self.drawOption.currentData()
        if self.drawOption.currentIndex()>0:
            self.drawOption.setFont(QtGui.QFont())
            # self.drawOption.setItemData(0, '', Qt.UserRole-1)
            self.setTool()
        else:
            italic = QtGui.QFont()
            italic.setItalic(True)
            self.drawOption.setFont(italic)
            self.setTool(False)
        txt = ""
        if self.drawForm=='polygon': txt = "Clic gauche pour chaque sommet\nClic droit pour terminer"
        if self.drawForm=='circle': txt = "Clic pour le centre\nDéplacer pour agrandir\nClic pour terminer"
        if self.drawForm=='rect': txt = "Clic pour le premier sommet\nDéplacer pour agrandir\nClic pour terminer"
        if self.drawForm=='custom': txt = "Clic pour commencer\nDéplacer pour dessiner\nClic pour terminer"
        self.drawOption.setToolTip(txt)
    
    def openFile(self):
        dialog = QtWidgets.QFileDialog()
        fileName = dialog.getOpenFileName(None, "Ouvrir une couche vecteur", "", "Vector Files (*.shp *.gpkg)")
        file = fileName[0]
        self.labelFile.file = file
        self.labelFile.setText(os.path.basename(file))
        QtWidgets.QApplication.processEvents()
    
    def setTool(self, bool=True):
        if self.canvas.mapTool()!=self.tool:
            self.prevTool = self.canvas.mapTool()
        if bool: self.canvas.setMapTool(self.tool)
        else: self.canvas.setMapTool(self.prevTool)
    
    def makeChoosen(self):
        layout = self.layout
        clearLayout(layout)
        hbox = QtWidgets.QHBoxLayout()
        layout.addLayout(hbox)
        l = QtWidgets.QLabel("Masque choisi")
        font = QtGui.QFont()
        font.setItalic(True)
        l.setFont(font)
        hbox.addWidget(l)
        ico = QtGui.QIcon(os.path.join(self.dock.imageFolder,'zoom.png'))
        butt = QtWidgets.QPushButton(ico, '')
        butt.setToolTip("Zoomer sur le masque")
        butt.setMaximumWidth(20)
        hbox.addWidget(butt)
        butt.clicked.connect(self.zoom)
        ico = QtGui.QIcon(os.path.join(self.dock.imageFolder,'refresh.png'))
        butt = QtWidgets.QPushButton(ico, '')
        butt.setToolTip("Actualiser")
        butt.setMaximumWidth(20)
        hbox.addWidget(butt)
        butt.clicked.connect(self.affLayer)
        l = QtWidgets.QLabel()
        l.setMaximumWidth(100)
        ico = QtGui.QIcon(self.getSVG())
        l.setPixmap(ico.pixmap(100, 100))
        layout.addWidget(l)
        layout.setAlignment(l, Qt.AlignCenter)
        butt = QtWidgets.QPushButton("Supprimer")
        butt.setFixedSize(130, 30)
        layout.addWidget(butt)
        layout.setAlignment(butt, Qt.AlignCenter)
        butt.clicked.connect(self.delete)
        self.butt = butt

    def makeTmp(self):
        tmp = os.path.join(os.path.dirname(__file__), 'svg', 'tmp')
        if not os.path.isdir(tmp):
            os.mkdir(tmp)
    
    def getSVG(self):
        res = self.dock.psql.getSVG(self.table, maxDecimalDigits=0)
        try:
            info = res[0]
            path = info.get('poly')
            box = list(map(float, info.get('box').replace("(","").replace(")","").split(",")))
            w = max((box[0]-box[2]),(box[1]-box[3]))/50
            vb = f"{int(box[2])-2*w} {-int(box[1])-2*w} {int(box[0]-box[2])+4*w} {int(box[1]-box[3])+4*w}"  
        except:
            path = ""
            vb = "0 0 0 0"
            w = 1
        svg = f''
        svg += f'<svg version="1.1" viewBox="{vb}" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">'
        svg += f'<g fill="#bbbbbb" fill-opacity="0.2" stroke-width="{w}" stroke="#888888">'
        svg += f'<path d="{path}" />'
        svg += f'</g></svg>' 
        self.makeTmp()
        file = os.path.join(os.path.dirname(__file__), 'svg', 'tmp', '_tmp_mask.svg')
        f = QFile(file)
        if not f.open(QIODevice.WriteOnly | QIODevice.Text):
            return
        out = QTextStream(f)
        out << svg 
        f.close()
        return file
    
    def actuComboLayer(self, combo, cb):
        list = ['postgres']
        self.postgres = False
        if cb.isChecked():
            list = []
            self.postgres = True
        combo.setExcludedProviders(list)
        
    def actuSelected(self, cb):
        self.selected = self.selectedCB.isChecked()
  
    def delete(self):
        self.butt.setEnabled(False)
        self.butt.setText("Suppression...")
        QtWidgets.QApplication.processEvents()
        sql = f"DROP TABLE IF EXISTS {self.table}; DELETE FROM {self.dock.psql.admin('tableTmp')} WHERE tablename='{self.table}';"   
        self.dock.psql.sqlSend(sql)
        self.table = None
        self.meta.delIni("mask")
        self.dock.psql.actuMask(self.table)
        self.makeToChoose()
        self.dock.actuIcoTabFil()
        self.dock.actuTabs(['layer'])

    def set(self):
        ok = False
        if self.chooseOption.currentData()=='qgis':
            layer = self.comboQgis.currentLayer()
            if layer is None: return
            ok = partial(self.trade.createTableFromFeatures, layer, self.dock.psql.tmpPrefixe+makeUniqueIdent(), self.dock.psql.PGschema, self.fusionPolygon, selected=self.selectedCB.isChecked())
        if self.chooseOption.currentData()=='file':
            layer = self.labelFile.file
            if layer in (None, ""): return
            ok = partial(self.trade.createTableFromFile, layer, self.dock.psql.tmpPrefixe+makeUniqueIdent(), self.dock.psql.PGschema, self.fusionPolygon)
        if self.chooseOption.currentData()=='draw':
            if len(self.tool.memo.get('pts',[]))<3: return
            self.tool.memo['pts'].append(self.tool.memo['pts'][0])
            ok = partial(self.trade.createTableFromGeometry, self.tool.memo, self.dock.psql.tmpPrefixe+makeUniqueIdent(), self.dock.psql.PGschema, self.fusionPolygon)
            self.tool.reset()
            self.setTool(False)
        if ok:
            self.butt.setEnabled(False)
            self.butt.setText("Chargement...")
            self.insert = ok()


    def fusionPolygon(self, full):
        if not self.dock.psql.isTable(self.insert):
            self.butt.setEnabled(True)
            return
        geom = self.dock.psql.getGeometryField(self.insert)
        sql = f"SELECT 1 as id, ST_Union({quote(geom)}) as geom FROM {self.insert}"
        q = tmpQuery(self.dock.psql, sql)
        q.setPkey('id')
        q.addBD(enjeu="", alea="", requete=f"mask{self.insert}")
        task = mGlobalTask("Fusion masque")
        q.description = f"union entity"
        task.addSubTask(q)
        task.setCallback(self.endInsertion, q.table)
        self.dock.psql.manager.add(task)
  
    def endInsertion(self, table):
        sql = f"DROP TABLE IF EXISTS {self.insert};"   
        self.dock.psql.sqlSend(sql)
        if not self.dock.psql.isTable(table):
            self.butt.setEnabled(True)
            return
        self.table = table
        self.meta.writeIni("mask", self.table)
        self.dock.psql.actuMask(self.table)
        self.makeChoosen()
        self.dock.actuIcoTabFil()
        self.dock.actuTabs(['layer'])
    
    def affLayer(self):
        if self.dock.param.get('maskWorking') and self.table is not None:
            self.dock.psql.getLayerAsync(self.table, f"{self.table}_mask", callback2=self.affLayerEnd)
        else:
            self.layer = self.dock.dyn.removeMapLayer(self)
            
    def affLayerEnd(self, result):
        vlayer = result[0]
        geomType = result[1]
        if not vlayer:
            return
        self.vlayer = vlayer
        self.layer = self.dock.dyn.addInGroup(self, 'mask')
        self.layerId = self.layer.layerId()
        p = self.dock.param

        symbol = QgsFillSymbol().createSimple({'color':p.get('maskFill'), 'stroke':p.get('maskStroke'), 'stroke-width':str(p.get('maskWidth'))})
        c = p.get('maskBurst')
        if not isinstance(c, list) or len(c)<2:
            c = ["#aa000000","#88ffffff"]
        sl = QgsShapeburstFillSymbolLayer(color=QtGui.QColor(c[0]), color2=QtGui.QColor(c[1]), blurRadius=15, useWholeShape=False, maxDistance=10)
        symbol.changeSymbolLayer(0, sl)
        renderer = QgsSingleSymbolRenderer(symbol)
        
        r = QgsInvertedPolygonRenderer(renderer)
        # r.setPreprocessingEnabled(True)
        self.vlayer.setRenderer(r) 
    
    def zoom(self):
        try:self.dock.zoomExtent(self.vlayer.extent())
        except: pass
    
    def initData(self):
        self.options = {
            '': {'label':"Choisir un masque"},
            'draw': {'label':"Dessiner une emprise", 'ico':'paint.png'},
            'qgis': {'label':"Choisir dans Qgis", 'ico':'qgis.png'},
            'file': {'label':"Choisir un fichier", 'ico':'folder.png'},
        }
        self.drawForms = {
            '': {'label':"Choisir une forme"},
            'polygon': {'label':"Polygone"},
            'circle': {'label':"Cercle"},
            'rect':{'label':"Rectangle"},
            'custom': {'label':"Tracé libre"},
        }
           
        
class rdiMaskTool(QgsMapTool):
    def __init__(self, mask):
        self.mask = mask
        self.dock = mask.dock
        self.iface = self.dock.parent.iface
        self.canvas = self.iface.mapCanvas()
        super().__init__(self.canvas)
        self.memo = {}
        self.rubberBand = QgsRubberBand(self.canvas, QgsWkbTypes.GeometryType.PolygonGeometry)
        self.rubberBand.setStrokeColor(QtCore.Qt.darkGray)
        self.rubberBand.setLineStyle(QtCore.Qt.DashLine)
        self.rubberBand.setWidth(2)
        self.started = False
        
    def activate(self):
        super().activate()
        self.canvas.setCursor(QtGui.QCursor(QtCore.Qt.CrossCursor))

    def deactivate(self):
        super().deactivate()
        self.rubberBand.reset(QgsWkbTypes.GeometryType.PolygonGeometry)
        self.canvas.setCursor(QtGui.QCursor(QtCore.Qt.ArrowCursor))
        self.deactivated.emit()

    def canvasMoveEvent(self, e):
        if not self.started: return
        if self.mask.drawForm=='polygon':
            try: lst = self.memo['pts'].copy()
            except: lst = []
            lst.append(e.mapPoint())
            self.drawPoly(lst)    
        if self.mask.drawForm=='custom':
            self.memo['pts'].append(e.mapPoint())
            self.drawPoly(self.memo['pts'])
        if self.mask.drawForm=='circle':
            self.memo['pts'] = []
            nb = 100
            pt = e.mapPoint()
            d = self.memo['center'].distance(pt)
            for i in range(nb):
                a = 2*i*math.pi/nb
                x = math.cos(a)*d+self.memo['center'].x()
                y = math.sin(a)*d+self.memo['center'].y()
                self.memo['pts'].append(QgsPointXY(x,y))
            self.drawPoly(self.memo['pts'])
        if self.mask.drawForm=='rect':
            self.memo['pts'] = []
            pt1 = self.memo['pt1']
            pt2 = e.mapPoint()
            self.memo['pts'].append(QgsPointXY(pt1.x(), pt1.y()))
            self.memo['pts'].append(QgsPointXY(pt1.x(), pt2.y()))
            self.memo['pts'].append(QgsPointXY(pt2.x(), pt2.y()))
            self.memo['pts'].append(QgsPointXY(pt2.x(), pt1.y()))
            self.drawPoly(self.memo['pts'])
    
    def canvasPressEvent(self, e):
        if e.button()==Qt.RightButton:
            if self.mask.drawForm=='polygon': self.end()
            return
        if self.started:
            if self.mask.drawForm!='polygon': 
                self.end()
                return
        else:
            self.start()
        if self.mask.drawForm=='polygon':
            self.memo['pts'].append(e.mapPoint())
        if self.mask.drawForm=='custom':
            self.memo['pts'].append(e.mapPoint())
        if self.mask.drawForm=='circle':
            self.memo['center'] = e.mapPoint()
        if self.mask.drawForm=='rect':
            self.memo['pt1'] = e.mapPoint()

    def canvasReleaseEvent(self, e):
        pass
    
    def start(self):
        self.memo = {'pts':[], 'crs':self.canvas.mapSettings().destinationCrs().authid(), 'type':'polygon'}
        self.started = True
        self.drawPoly()
    
    def end(self):
        self.started = False
        self.drawPoly(self.memo['pts'])    
    
    def reset(self):
        self.started = False
        self.memo = {}
        self.drawPoly()
        
    def drawPoly(self, lst=[]):
        self.rubberBand.reset(QgsWkbTypes.GeometryType.PolygonGeometry)
        for i,p in enumerate(lst):
            b = i==len(lst)-1
            # self.rubberBand.addPoint(QgsPointXY(p.x(), p.y()), b)
            self.rubberBand.addPoint(p, b)
        self.rubberBand.show()

    


        