# -*- coding: utf-8 -*-
import os, sys, configparser, time
from functools import partial

from qgis.core import *
from qgis.gui import *
from qgis.utils import iface
from PyQt5 import QtGui, QtWidgets, uic
from PyQt5.QtCore import *
from PyQt5.QtSql import *

from .rdi_meta import rdiMeta


class rdiParam():

    def __init__(self, first=False):
        self.meta = rdiMeta()
        self.plugin = self.meta.getPlugin()
        self.pluginName = self.meta.getName()
        self.file = os.path.join(os.path.dirname(__file__), "parameter.ini")
        self.config = configparser.ConfigParser(allow_no_value=True)
        self.config.optionxform = str
        self.lastReading = None
        self.initData()
        if first:
            self.getParams()
            self.firstOpenning()
    
    def readParameters(self):
        defParams = self.makeDefaultparameters()
        self.needWriting = False
        if os.path.exists(self.file):
            try:
                self.config.read(self.file)
            except:
                s = open(self.file, mode='r', encoding='utf-8-sig').read()
                open(self.file, mode='w', encoding='utf-8').write(s)
                self.config.read(self.file)
            for s in self.config.sections():
                self.params[s] = dict((x,y) for x,y in self.config.items(s))  
        else:
            self.needWriting = True
        self.fusionParameters(self.meta.readIni("param"))
        self.fusionParameters(defParams)
        if self.needWriting:
            self.writeParameters()
        
    def fusionParameters(self, new):
        if not new:
            self.needWriting = True
            return
        for s, elt in new.items():
            if not s in self.params:
                self.params[s] = {}
                self.needWriting = True
            for k,v in elt.items():
                if not k in self.params[s]:
                    if k!="comment" and k[:1]!="#":
                        self.needWriting = True
                    self.params[s][k] = v

    def formatParameters(self):     
        for section, dic in self.params.items():
            for k, v in dic.items():    
                nv = v
                if v=='True' or v==True:
                    nv = True
                    self.params[section][k] = nv
                    continue
                if v=='False' or v==False:
                    nv = False
                    self.params[section][k] = nv
                    continue
                if v=='None' or v==None:
                    nv = None
                    self.params[section][k] = nv
                    continue
                try:
                    if str(v).find('.')>=0:
                        nv = float(v)
                    else:
                        nv = int(v)
                except:
                    # print(sys.exc_info(), k, v)
                    pass
                self.params[section][k] = nv

    def writeParameters(self):     
        config = configparser.ConfigParser(allow_no_value=True)
        config.optionxform = str         
        config.set(None, "# Fichier de configuration du plugin rdi", None)
        config.set(None, "# A manipuler avec precautions, reserve aux utilisateurs avertis", None)
        config.set(None, "# Pour eviter les erreurs, il est preferable d'utiliser l'interface de parametrage du plugin", None)
        for section, dic in self.params.items():
            config.add_section(section)
            if 'comment' in dic:
                config.set(section, dic['comment'], None)
            for k, v in dic.items():
                if k!='comment':
                    config.set(section, k, str(v))
        f = open(self.file,'w')
        config.write(f)
        f.close()
        self.lastReading = None   
        self.meta.writeIni("param", self.params)
     
    def needReading(self):
        if not self.lastReading:
            return True
        else:
            delay = time.time() - self.lastReading
            if delay>10:
                return True
    
    def getParams(self):
        if self.needReading():
            self.params = {}
            self.readParameters()
            self.formatParameters()
            self.lastReading = time.time()
    
    def getSection(self, section):
        self.getParams()
        return self.params[section.upper()]
    
    def getValue(self, section, key):
        self.getParams()
        try:
            v = self.params[section.upper()][key]
        except:
            v = None
        return v
        
    def changeValue(self, section, key, value):
        self.getParams()
        if (section in self.params) and (key in self.params[section]):
            self.params[section][key] = value
        self.writeParameters()
        
    
    def find(self, key):
        self.getParams()
        error = rdiParamError(key)
        for section, elt in self.params.items():
            v = self.getValue(section, key)
            if v!=None:
                error.addValue(v)
                error.addSection(section, elt['comment'])
        value = error.getValue()
        return value, error
        
    def get(self, key, section=None):        
        if section:
            return self.getValue(section, key)
        else:
            v, err = self.find(key)
            if err.status:
                return v
            else:
                self.raiseError(err)

    def getErr(self, key):
        v, err = self.find(key)
        return v, err
        
    def set(self, key, value, section=None):
        if section:
            self.changeValue(section, key, value)
        else:
            v, err = self.find(key)
            if err.status:
                self.changeValue(err.getSection(), key, value)
            else:
                self.raiseError(err)

    def raiseError(self, err):
        print(err.dump())
        # iface.messageBar().pushMessage(f"plugin RDI", f"Erreur dans la lecture des paramètres, le plugin peut ne pas fonctionner correctement", level=Qgis.Warning, duration=5)
        # raise ValueError(err.dump())

  
    def makeDefaultparameters(self):
        params = {
            'CONN': {
                'comment': u"# Parametres de connexion a la base postgres/postgis",
                'db': "",
                'host': "",
                'password': "",
                'port': "",
                'savePassword': True,
                'schema': "",
                'user': "", 
            },
            'ADMIN': {
                'comment': u"# Modele de donnees des tables d'administration",
                'manualInsertion': False,
                'schema': "",
                'tableAdmin': '_rdi_gestion_liste',
                'tablePar': '_rdi_gestion_parametrage',
                'tableTmp': '_rdi_gestion_stockage',
                'tmpPrefixe': '_tmp_',
            },
            'EXE': {
                'comment': u"# Parametres divers pour l'execution",
                'admin': False,
                'alea': u"Aléas",
                'allowBeta': False,
                'animation': True,
                'asyncGetLayer': False,
                'asyncGetSql': True,
                'autoAssoc': True,
                'autoCheckAlert': 30,
                'autoCheckConnection': 60000,
                'autoLinkLoad': True,
                'autoGenAlert': 100,
                'background': u"Fond",
                'backMap': True,
                'baseAdmin': False,
                'blocDrop': True,
                'blocDropExpand': False,
                'blocDropTargets': [],
                'buff': u"Tampons",
                'buffer': 0,
                'bufferMax': 100,
                'bufferMin': -20,
                'cbCompact': True,
                'cbAssoc': False,
                'cluster': True,
                'clusterBack': False,
                'clusterField': False,
                'clusterPrecision': 10,
                'clusterRound': True,
                'correction': False,
                'cutting': False,
                'enjeu': u"Enjeux",
                'enjeuAlone': False,
                'enjeuLight': u"Enjeux associés",
                'entityCount': False,
                'filterMemo': False,
                'help': True,
                'info': True,
                # 'inversion': True,
                'keepActive': False,
                'link': u"Hyperliens",
                'linkCallDyn': False,
                'linkDist': 0,
                'linkOrder': False,
                'linkOrderChoice': 'parent',
                'linkToolTabs': ['dyn','par','fil','stat'],
                'logs': False,
                'mask': u"Masques",
                'maskWorking': False,
                'queryTimeout': 9000,
                'root': u"rdi",
                'simplify': 0,
                'simplifyMax': 20,
                'snapPrecision': 0.01,
                'statPrecision': 1,
                'statRound': True,
                'stats': u"Statistiques",
                'stockTimeout': 864000,
                'zipLoader': False,
            },
            'PAINT': {
                'comment': u"# Parametres globaux de dessin",
                'aleaOpacity': "0.6",
                'bufferFill': "#20000000",
                'bufferPen': "dot",
                'bufferStroke': "#000000",
                'bufferWidth': 1.3,
                'lightOpacity': 0.3,
                'maskBurst': ["#aa000000","#88ffffff"],
                'maskFill': "#00000000",
                'maskStroke': "#000000",
                'maskWidth': 2,
                'rendererTag': "renderer-v2",
                'rendererTolerance': 25,
                'rendererUnit': "mm",
                'statFill': "#10ff7400",
                'statPen': "dash",
                'statStroke': "#ff7400",
                'statWidth': 1,
                'strokeUnit': "mm",
                'strokeWidth': "0.5",
            },
        }
        
        return params
        

    def initData(self):
        self.translator = {
            'geom' : {
                'label': u"Géométrie",
                'info': u"Ces options impactent le croisement spatial (et donc la construction dynamique de la requête SQL) et le rendu de la couche extraite",
                'list': [
                    {
                        'label': u"Rendu cluster",
                        'help': 'p5-option-cluster',
                        'description': u"Regroupe l'affichage des points à proximité",
                        'ini': 'cluster',
                        'tab': ['parameter', 'dynamic', 'layer'],
                    },
                    {
                        'label': u"Points en fond",
                        'help': 'p5-option-cluster',
                        'description': u"Garde un affichage de l'ensemble des points en léger arrière plan",
                        'ini': 'clusterBack',
                        'tab': ['layer'],
                        'dep': 'cluster'
                    },
                    {
                        'label': u"Tampon (m)",
                        'help': 'p5-option-buffer',
                        'description': u"Ajoute une distance tampon lors des requêtes de croisement spatial",
                        'ini': 'buffer',
                        'tab': ['layer'],
                        'next': {
                            'type': 'range',
                            'min': self.get('bufferMin'),
                            'max': self.get('bufferMax'),
                        },
                    },
                    {
                        'label': u"Simplifier (m)",
                        'help': 'p5-option-simplify',
                        'description': u"Simplifie la géométrie pour accélérer le croisement spatial",
                        'ini': 'simplify',
                        'tab': ['layer'],
                        'next': {
                            'type': 'range',
                            'min': 0,
                            'max': self.get('simplifyMax'),
                        },
                    },
                    {
                        'label': u"Découper résultat",
                        'help': 'p5-option-cut',
                        'description': u"Découpe les enjeux surfaciques ou linéaires pour être strictement inclus dans l'emprise du croisement",
                        'ini': 'cutting',
                        'tab': ['layer'],
                    },
                    {
                        'label': u"Correction à la volée",
                        'help': 'p5-option-corr',
                        'description': u"Rajoute des corrections à la volée sur la géométrie" + \
                                        "\n(permet de contourner quelques erreurs de topologie, mais rajoute du temps de calcul)",
                        'ini': 'correction',
                        'tab': ['layer'],
                    },
                    {
                        'label': u"Décompte des entités",
                        'help': 'p5-option-count',
                        'description': u"Indique le nombre d'entités renvoyées à côté du nom de la couche",
                        'ini': 'entityCount',
                        'func': ['self.majEntityCount()'],
                    },
                ]
            },
            'plugin' : {
                'label': u"Affichage",
                'info': u"Ces options impactent l'affichage du plugin lui-même, comme notamment la présentation des listes de couches enjeux/aléas dans l'onglet de croisement dynamique (le premier)",
                'list': [
                    {
                        'label': u"Liste compacte",
                        'help': 'p5-option-compact',
                        'description': u"Regroupe les couches associées à une même zone en liste déroulante",
                        'ini': 'cbCompact',
                        'tab': ['parameter','dynamic'],
                    },
                    {
                        'label': u"Eviter la superposition",
                        'help': 'p5-option-compact',
                        'description': u"Si des zones sont liées, alors l'affichage de l'une désactivera celle de l'autre",
                        'ini': 'cbAssoc',
                        'tab': ['dynamic'],
                        'dep': 'cbCompact'
                    },
                    {
                        'label': u"Arborescence en thèmes",
                        'help': 'p5-option-block',
                        'description': u"Arborescence sous forme de blocs dépliables selon les thèmes affectés",
                        'ini': 'blocDrop',
                        'tab': ['parameter', 'dynamic'],
                    },
                    {
                        'label': u"Mémoriser les filtres",
                        'help': 'p5-option-filter',
                        'description': u"Mémorise les sélections appliquées via le filtre statique après la déconnexion",
                        'ini': 'filterMemo',
                        'func': ['self.alleviate.save()'],
                    },
                    {
                        'label': f"{self.get('enjeu')} seuls",
                        'help': 'p5-option-alone',
                        'description': f"Montrer la totalité de la couche des {self.get('enjeu')} lorsqu'aucun {self.get('alea')} n'est sélectionné",
                        'ini': 'enjeuAlone',
                        'tab': ['layer'],
                    },
                    {
                        'label': u"Animation",
                        'help': 'p5-option-anim',
                        'description': u"Défilement pointillé pour patienter pendant les traitements longs",
                        'ini': 'animation',
                        'tab': ['journal'],
                    },
                ]
            },
            'alter' : {
                'label': u"Divers",
                'info': u"D'autres options moins utiles avec le temps (ou en version bêta)",
                'list': [
                    {
                        'label': u"Explications",
                        'help': 'p5-option-explain',
                        'description': u"Affichage plus détaillé des fonctionnalités" +\
                                        "\nUtile au début... mais consomme beaucoup d'espace",
                        'ini': 'info',
                        'tab': ['all'],
                    },
                    {
                        'label': u"Aide en ligne",
                        'help': 'p5-option-help',
                        'description': u"Affichage d'un point d'interrogation cliquable" +\
                                        "\npour ouvrir l'aide en ligne sur l'objet ciblé",
                        'ini': 'help',
                        'tab': ['all'],
                    },
                    {
                        'label': u"Administration",
                        'help': 'p5-option-admin',
                        'description': u"Affichage des onglets d'administration",
                        'ini': 'admin',
                        'func': ['self.actuTabAdminVisibility()'],
                    },
                    {
                        'label': u"Rester actif",
                        'description': u"Permet de continuer à interagir avec les couches d'hyperliens même si le plugin est fermé.",
                        'ini': 'keepActive',
                    },
                    {
                        'label': u"Chargement asynchrone",
                        'description': u"Chargement asynchrone des couches. \nCette option améliore la fluidité du plugin, mais repose sur une utilisation dépréciée des librairies QGis. \nPeut générer des plantages de Qgis.",
                        'ini': 'asyncGetLayer',
                        'beta': True,
                    },
                    {
                        'label': u"SQL asynchrone",
                        'description': u"Préparation asynchrone des requêtes SQL. Encore non stabilisée complètement.",
                        'ini': 'asyncGetSql',
                        'beta': True,
                    },
                ]
            },
        }
        self.serverParam = {
            'queryTimeout': 'delay',
            'stockTimeout': 'delay',
            'tmpPrefixe': 'string',
        }
        self.forceResetParam = [
            ('snapPrecision', 'EXE'),
            ('asyncGetSql', 'EXE')
        ]
        
        
        




    def firstOpenning(self):
        file = os.path.join(os.path.dirname(__file__), "firstOpenning.txt")
        params = self.makeDefaultparameters()
        if os.path.exists(file):
            for p,s in self.forceResetParam:
                v = params.get(s).get(p)
                self.set(p, v, s)
            os.remove(file)       

    def forceDefault(self, callback):
        list = ['EXE', 'PAINT']
        qm = QtWidgets.QMessageBox
        rep = qm.question(None, "", f"Tous les paramètres des blocs EXE et PAINT du fichiers parameter.ini seront réinitialisés à la valeur par défaut, continuer ?", qm.Yes | qm.No)
        if not rep==qm.Yes:
            return
        params = self.makeDefaultparameters()
        for s, dic in params.items():
            if s not in list:
                continue
            for k, v in dic.items():    
                v = params.get(s).get(k)
                self.set(k, v, s)     
        callback()







class rdiParamError():
    def __init__(self, key):
        self.key = key
        self.values = []
        self.message = ""
        self.codes = []
        self.sections = []
        self.status = None

    def addValue(self, value):
        self.values.append(value)
        
    def addSection(self, section, text=""):
        self.codes.append(section)
        self.sections.append(text)

    def getValue(self):
        if len(self.values)==1:
            self.message = f"Valeur trouvée pour \"{self.key}\""
            self.status = True
            return self.values[0]
        else:
            self.status = False
            if len(self.values)==0:
                self.message = f"Aucune valeur correpondant à \"{self.key}\""
            else:
                self.message = f"Plusieurs correspondent à \"{self.key}\""
            return None
            
    def getSection(self):    
        if len(self.codes)==1:
            return self.codes[0]
            
    def dump(self):
        txt = self.message
        txt += f"\nStatus: {self.status}"
        txt += "\nValues: " + ", ".join(map(str, self.values))
        txt += "\nCodes: " + ", ".join(map(str, self.codes))
        txt += "\nSections: " + ", ".join(map(str, self.sections))
        return txt
        
        

class rdiParamBloc(QtWidgets.QWidget):
    def __init__(self, code, title, dock, memo=True, parent=None):
        self.dock = dock
        self.ini = None
        dock.blocExists[code] = True
        super().__init__(parent)
        v = QtWidgets.QVBoxLayout(self) 
        v.setContentsMargins(0,0,0,0)
        v.setSpacing(0)
        l = rdiBlocTitle(code, title, memo)
        v.addWidget(l)
        self.gb = QtWidgets.QGroupBox(self)
        QtWidgets.QVBoxLayout(self.gb) 
        v.addWidget(self.gb)
        # gb.setCheckable(True)
        # self.gb.toggled.connect(self.show)
        l.clicked.connect(self.show)
        self.title = l
        if memo and not self.title.isMemoDrop(): 
            self.title.setExpanded(False)
        if not memo and self.title.isMemoDrop(): 
            self.title.setExpanded(False)
        
    def add(self, v, changed=None):
        if isinstance(v, list):
            for e in v: self.add(e, changed)
            return
        if isinstance(v, dict):
            if v.get('title'): 
                layout = self.title.layout
                l = v.get('callback',[])
                l.append(self.activate)
                v['callback'] = l
                v['label'] = "Activer"
                self.ini = v['ini']
            else: 
                layout = self.gb.layout()
            p = rdiParamElt(v, self.dock)
            if changed: p.changed.connect(changed)
            p.draw(layout)
            if v.get('title'): 
                self.activateCB = p
                if not self.title.isExpanded(): p.widget.hide()
                self.activate()
            return
    
    def show(self, b):
        self.gb.setVisible(b)
        try: self.activateCB.widget.setVisible(b)
        except: pass
        
    def getLayout(self):
        return self.gb.layout()
        
    def activate(self):
        self.gb.setEnabled(self.dock.param.get(self.ini))
        

class rdiParamElt(QObject):
    
    changed = pyqtSignal(dict)
    
    def __init__(self, elt, parent):
        QObject.__init__(self)
        # self.param = rdiParam()
        self.dock = parent
        self.param = self.dock.param
        self.elt = elt
        self.type = None
        self.widgets = []
        self.bounds = None
        self.layout = None
        
    def newLine(self, layout):
        w, l = self.newBloc(layout)
        if 'description' in self.elt:
            w.setToolTip(self.elt['description'])
        self.layout = l
        self.widget = w
    
    def newBloc(self, layout):
        # widget = QtWidgets.QWidget()   
        widget = rightClickWidget()   
        layout.addWidget(widget)
        hbox = QtWidgets.QHBoxLayout(widget)
        hbox.setSpacing(0)
        hbox.setAlignment(Qt.AlignLeft)
        if self.elt.get('dep'):
            hbox.setContentsMargins(20,0,0,0)
        else:
            hbox.setContentsMargins(0,0,0,0)
        return widget, hbox
    
    def draw(self, layout):
        if 'dep' in self.elt and not self.param.get(self.elt['dep']):
            return
        if 'beta' in self.elt and not self.param.get('allowBeta'):
            return
        self.newLine(layout)
        self.paint()
        help = self.elt.get('help')
        if help:
            self.dock.makeHelpButton(self.layout, help)
    
    def paint(self, elt=None, layout=None):
        w = None
        if elt==None:
            elt = self.elt
        if layout==None:
            layout = self.layout
        
        type = elt.get('type') or 'checkbox'
        if type=='checkbox':
            w = self.addCb(elt, layout)
            if not self.type:
                self.type = 'boolean'
        elif type=='range':
            w = self.addRange(elt, layout)
            self.type = 'numeric'

        elif type=='combo':
            w = self.addCombo(elt, layout)
            self.type = 'string'
        
        if w:
            self.widgets.append([w, type])

        if 'next' in elt:
            if 'ini' in elt['next']:
                p = rdiParamElt(elt['next'], self.dock)
                p.draw(layout)
                nw = p.widget
            else:
                nw, nl = self.newBloc(layout)
                self.paint(elt['next'], nl)
            if w and type=='checkbox':
                w.stateChanged.connect(partial(self.show, w, nw))
                self.show(w, nw)
                pass   
     
    def getLabel(self, elt):
        txt = elt.get('label',elt.get('ini'))
        if 'beta' in elt: txt += " (bêta)"
        return txt
    
    def addCb(self, elt, layout):
        # try:
        
        w = QtWidgets.QCheckBox(self.getLabel(elt))
        # w = rightClickCb(elt['label']) 
        
        w.setChecked(self.value())
        layout.addWidget(w)
        w.stateChanged.connect(partial(self.changeValue))
        
        return w
        # except:
            # print(sys.exc_info())
            # pass
    
    def addRange(self, elt, layout):
        try:
            if elt.get('label'):
                l = QtWidgets.QLabel(self.getLabel(elt))
                layout.addWidget(l)
            w = rightClickSlider(Qt.Horizontal)
            w.elt = elt
            w.setRange(elt['min'], elt['max'])
            w.setValue(self.value())
            layout.addWidget(w)
            w.sliderReleased.connect(self.changeValue)
            l = QtWidgets.QSpinBox()
            l.setRange(elt['min'], elt['max'])
            l.setValue(self.value())
            layout.addWidget(l)
            l.editingFinished.connect(self.changeValue)
            l.valueChanged.connect(partial(self.setValueOnly, w, l))
            w.valueChanged.connect(partial(self.setValueOnly, l, w))
            
            w.rightClick.connect(partial(self.affMenu, w))
            self.widget.rightClick.connect(partial(self.affMenu, w))
            
            self.rangeWidget = QObject()
            self.rangeWidget.slider = w
            self.rangeWidget.spin = l
            return w
        except:
            # print(sys.exc_info())
            pass 

    def addCombo(self, elt, layout):
        if elt.get('label'):
            l = QtWidgets.QLabel(self.getLabel(elt))
            layout.addWidget(l)
        w = QtWidgets.QComboBox()
        w.elt = elt
        if elt.get('maxWidth'):
            w.setMaximumWidth(elt.get('maxWidth'))
        layout.addWidget(w)
        for k,v in elt.get('elements',{}).items():
            w.addItem(v,k)
        w.setCurrentIndex(w.findData(self.value()))
        w.currentIndexChanged.connect(partial(self.changeValue))
        return w
    
    def setValueOnly(self, toObj, fromObj):
        toObj.blockSignals(True)
        toObj.setValue(fromObj.value())
        toObj.blockSignals(False)

    def show(self, cb, widget):
        if cb.isChecked():
            widget.show()
        else:
            widget.hide()

    def changeValue(self):
        value = self.getValue()
        if value!=None:   
            self.param.set(self.elt['ini'], value)
        else:
            print(f"Error compilation value for {self.elt['ini']}")
        self.changed.emit(self.elt)
    
    
    def value(self):
        return self.param.get(self.elt['ini'])
        
    def getValue(self):
        value = None
        values = []
        for w, type in self.widgets:
            if type=='checkbox':
                if self.type=='numeric':
                    v = 1 if w.isChecked() else 0
                else:
                    v = w.isChecked()
                values.append(v)
            elif type=='range':
                values.append(w.value())
            elif type=='combo':
                values.append(w.currentData())

        if self.type=='numeric':
            value = 1
            for v in values:
                value = value * v

        if self.type=='boolean':
            value = True
            for v in values:
                value = value and v

        
        if self.type=='string':
            value = ""
            for v in values:
                value += v
        
        return value
        
    def affMenu(self, w):
        has = False
        for e in ['Min', 'Max']:
            if self.param.get(f"{self.elt['ini']}{e}"): has = True
        if not has: return
        self.popMenu = QtWidgets.QMenu()
        self.popMenu.setToolTipsVisible(True)
        # self.popMenu.addSection(w.elt['label'])
        self.popMenu.addAction("Modifier les bornes", self.openBounds)
        self.popMenu.popup(QtGui.QCursor.pos())

    def openBounds(self):
        if self.bounds is not None:
            return
        self.bounds = boundsDialog(self.elt)
        self.bounds.finished.connect(self.closeBounds)
        self.bounds.accepted.connect(partial(self.majBounds, self.bounds))
        
    def closeBounds(self):
        self.bounds = None
        
    def majBounds(self, dialog):
        ini = self.elt.get('ini')
        b = dialog.bounds()
        self.rangeWidget.slider.setRange(*b)
        self.rangeWidget.spin.setRange(*b)
        self.param.set(f"{ini}Min", b[0])
        self.param.set(f"{ini}Max", b[1])
        self.param.set(f"{ini}", self.rangeWidget.slider.value())
        


 
class rightClickSlider(QtWidgets.QSlider):
    rightClick = pyqtSignal()
    def __init__(self, orientation=None):
        QtWidgets.QSlider.__init__(self, orientation)
    
    def mousePressEvent(self, event):
        if event.button()==Qt.RightButton:
            self.rightClick.emit()
        else:
            super().mousePressEvent(event)
            
class rightClickWidget(QtWidgets.QWidget):
    rightClick = pyqtSignal()
    def __init__(self, parent=None):
        QtWidgets.QWidget.__init__(self, parent)
    
    def mousePressEvent(self, event):
        if event.button()==Qt.RightButton:
            self.rightClick.emit()
        else:
            super().mousePressEvent(event)
 


class boundsDialog(QgsDialog):
    def __init__(self, context):
        buttons = QtWidgets.QDialogButtonBox.Ok | QtWidgets.QDialogButtonBox.Cancel
        QgsDialog.__init__(self, iface.mainWindow(),
                            fl=Qt.WindowFlags(),
                            buttons=buttons,
                            orientation=Qt.Horizontal)
        # self.setWindowModality(Qt.ApplicationModal)
        self.context = context
        self.param = rdiParam()
        self.draw()
        self.show()
        
    def draw(self):
        self.edits = {}
        ini = self.context['ini']
        self.layout().setAlignment(Qt.AlignTop)
        w = QtWidgets.QWidget(self)
        hbox = QtWidgets.QHBoxLayout(w)
        hbox.setSpacing(5)
        hbox.setContentsMargins(0,5,0,0)
        self.layout().addWidget(w)
        l = QtWidgets.QLabel(f"Modifier les bornes pour : {self.context['label']}", self)
        hbox.addWidget(l)
        w = QtWidgets.QWidget(self)
        self.layout().addWidget(w)
        self.widgetForm = w
        form = QtWidgets.QFormLayout(w)
        form.setSpacing(5)
        for e in ['Min', 'Max']:
            wl = QtWidgets.QLabel(e, self)
            val = self.param.get(f"{ini}{e}")
            wf = QtWidgets.QLineEdit(str(val), self)
            self.edits[e] = wf
            if val is None:
                wf.setText(str(self.context.get('next').get(e.lower())))
                wf.setEnabled(False)
            else:
                wf.label = wl
                wf.textChanged.connect(partial(self.valid, wf))
            self.value(wf)
            form.addRow(wl, wf)
        l = QtWidgets.QLabel("", self)
        l.setStyleSheet("color:red;")
        form.addRow(l)
        self.errorLabel = l
    
    def value(self, w):
        try:
            w.value = int(w.text().strip())
        except:
            pass
    
    def valid(self, w):
        style = ""
        txt = "" 
        bool = True
        try:
            w.value = int(w.text().strip())
        except:
            style = "color:red;"
            txt = f"Entrez un nombre entier"
            bool = False
        if style=="" and self.edits['Min'].value>self.edits['Max'].value:
            style = "color:red;"
            txt = f"Le Max doit être supérieur au Min"
            bool = False
        w.setStyleSheet(style)
        w.label.setStyleSheet(style)
        self.errorLabel.setText(txt)
        self.buttonBox().button(QtWidgets.QDialogButtonBox.Ok).setEnabled(bool)

    def bounds(self):
        return (self.edits['Min'].value, self.edits['Max'].value)
 
 
class rdiBaseParam():
    def __init__(self, psql):
        self.psql = psql
        self.full = psql.admin('tablePar')
        self.resetData()
        self.ext = ['.svg', '.json']
        
    def getCol(self, col, key, code=None):
        sql = f"SELECT {col} from {self.full} WHERE key='{key}'"
        if code is not None:
            sql += f" AND code='{code}'"
        query = self.psql.sqlSend(sql)
        if query and query.next():
            return query.value(0)
 
    def fileDetect(self, key, code, file):
        if file==False:
            return False
        if file==True:
            return True
        if code is not None:
            deb = code.split('_')[0]
            if deb=='file':
                return True
            return
        name, ext = os.path.splitext(key)
        if ext.lower() in self.ext:
            return True
 
    def get(self, key, code=None, file=None):
        if self.fileDetect(key, code, file):
            return self.getCol('blob', key, code)
        else:
            return self.getCol('value', key, code)
            
    def getFile(self, key, code=None):
        return self.getCol('blob', key, code)
 
    def set(self, key, code=None, value=None, blob=None):
        if code is not None:
            codeS = f"'{code}'"
        else:
            codeS = 'NULL'
        id = self.getCol('id', key, code)
        if value is not None:
            data = 'value'
            # value = value.replace("'", "''")
            data_value = value
        if blob is not None:
            data = 'blob'
            data_value = blob
        if id:
            sql = f"UPDATE {self.full} SET {data}=:data_value WHERE id={id}"
        else:
            sql = f"INSERT INTO {self.full} (key, code, {data}) VALUES ('{key}', {codeS}, :data_value)"
        query = QSqlQuery(self.psql.db)
        query.prepare(sql)
        query.bindValue(":data_value", data_value)
        query.exec()
        # query = self.psql.sqlSend(sql)

    def gets(self, key=None, code=None):
        sql = f"SELECT * from {self.full}"
        wl = []
        if key is not None:
            wl.append(f"key='{key}'")
        if code is not None:
            wl.append(f"code='{code}'")
        if len(wl)>0:
            where = " AND ".join(wl)
            sql += f" WHERE {where}"
        query = self.psql.sqlSend(sql)
        return self.psql.makeDictResult(query)
        
    def resetData(self):
        self.dataList = []
        
    def appendData(self, *args, **kwargs):
        self.dataList.append((args, kwargs))
        
    def sendData(self):
        for data in self.dataList:
            args = data[0]
            kwargs = data[1]
            self.set(*args, **kwargs)
        self.resetData()

    def readFile(self, file):
        f = QFile(file)
        if not f.open(QIODevice.ReadOnly):
            return
        blob = QByteArray(f.readAll())
        f.close()
        return blob
    
    def writeFile(self, file, blob):
        f = QFile(file)
        if not f.open(QIODevice.WriteOnly):
            return
        f.write(blob)
        f.close() 
        
    def setFile(self, path, key, code=None):
        blob = self.get(key=key, code=code, file=True)
        if blob:
            # self.makeFolder(path)
            file = os.path.join(path, key)
            tt = key.split('/')
            for i in range(len(tt)-1):
                self.makeFolder(path, tt[i])
            self.writeFile(file, blob)
            return file
            
    def makeFolder(self, path, folder=None):
        if folder:
            new = os.path.join(path, folder)
        else:
            new = path
        if not os.path.exists(new):
            os.mkdir(new)
            

class rdiBlocTitle(QtWidgets.QWidget):
    clicked = pyqtSignal(bool)
    rightClicked = pyqtSignal()
    def __init__(self, code, title, memo=True):
        QtWidgets.QHBoxLayout.__init__(self)
        self.imageFolder = os.path.join(os.path.dirname(__file__),'images')
        self.param = rdiParam()
        self.code = code
        self.memo = memo
        self.path = {}
        self.layout = QtWidgets.QHBoxLayout(self)
        self.layout.setSpacing(0)
        self.layout.setContentsMargins(0,5,0,0)
        self.initState()
        self.makeCross()
        self.makeLabel(title)
    
    def makeCross(self):
        l = QtWidgets.QLabel()
        l.setMaximumWidth(15)
        self.layout.addWidget(l)
        self.cross = l
        self.affectImg()
        
    def affectImg(self):
        ico = QtGui.QIcon(os.path.join(self.imageFolder, self.path[self.state]))
        self.cross.setPixmap(ico.pixmap(10,10))
    
    def makeLabel(self, txt):
        l = QtWidgets.QLabel(txt)
        self.label = l
        font = QtGui.QFont()
        font.setBold(True)
        l.setFont(font)
        self.layout.addWidget(l)
    
    def initState(self):
        self.state = 'minus'
        self.cd = {
            'plus': 'minus',
            'minus': 'plus',
        }
        self.path['plus'] = 'plus.png'
        self.path['minus'] = 'minus.png'
        
    def changeState(self):
        self.state = self.cd[self.state]
        self.affectImg()
        
    def setExpanded(self, b):
        if b and self.state=='plus' or not b and self.state=='minus':
            self.changeState()
            self.clicked.emit(self.isExpanded())
    
    def isExpanded(self):
        return (self.state=='minus')
    
    def mousePressEvent(self, event):
        if event.button()==Qt.RightButton:
            self.rightClicked.emit()
        else:
            self.changeState()
            self.setMemoDrop()
            self.clicked.emit(self.isExpanded()) 

    def isMemoDrop(self):
        l = eval(self.param.get('blocDropTargets'))
        b = False
        if self.code in l: b = True
        return b
    
    def setMemoDrop(self):
        b = self.isExpanded() and self.memo or not self.isExpanded() and not self.memo
        l = eval(self.param.get('blocDropTargets'))
        if b and self.code not in l: l.append(self.code)
        if not b and self.code in l: l.remove(self.code)
        self.param.set('blocDropTargets', l)


