# -*- coding: utf-8 -*-

import os, socket

from qgis.core import *
from qgis.gui import *
from PyQt5.QtCore import *
from PyQt5.QtSql import *
from PyQt5 import QtWidgets, QtGui
from qgis.utils import iface

from functools import partial

from .rdi_param import rdiParam, rdiBaseParam
from .rdi_async import mGlobalTask, manageTasks, abstractQuery
from .rdi_tools import makeUniqueIdent, quote, rdiTimer, flashMsg, compress, clearLayout, extractIP, niceDuration, rdiEditDuration


class rdiPostgis(QObject):
    
    connected = pyqtSignal()
    connecting = pyqtSignal()
    connectionLost = pyqtSignal()
    
    def __init__(self, callback=None, label=None, password=None, psql=None):
        QObject.__init__(self)
        self.actuParams()
        self.pdb = None
        
        if psql is not None:
            self.psql = psql
            self.actuMask(psql.context['maskTable'])
            self.PGdb = psql.PGdb
            self.PGschema = psql.PGschema
            self.PGhost = psql.PGhost
            self.PGport = psql.PGport
            self.PGuser = psql.PGuser
            self.PGpassword = psql.PGpassword
            self.manager = psql.manager
            self.connName = f"Clone:{makeUniqueIdent()}"
            self.isClone = True
            self.db = QSqlDatabase.cloneDatabase(psql.db, self.connName)
            if not self.db.open():
                self.error = "Database Error: %s" % self.db.lastError().text()
            else:
                # print('connect status', self.db.isOpen())
                self.connected.emit()
        else:
            self.PGdb = self.param.get('db')
            self.PGschema = self.param.get('schema', 'CONN')
            if self.PGschema=="":
                self.PGschema = "public"
            self.PGhost = self.param.get('host')
            self.PGport = self.param.get('port')
            self.PGuser = self.param.get('user')
            self.PGpassword = self.param.get('password')
            if password:
                self.PGpassword = password
            self.isClone = False
            logs = self.param.get('logs')
            self.manager = manageTasks(logs)
            self.checkLoginPassword(callback, label)
            self.myIP = extractIP()
        
        self.dialogs = {}
        self.autoTesting = False
        self.paramServer = {}
        self.connected.connect(self.actuContextServer)
    
    def clone(self):
        return rdiPostgis(self, psql=self)
    
    def destroy(self):
        try:
            self.db.close()
            self.delConn(self.connName)
        except:
            pass
 
    def actuParams(self):
        self.param = rdiParam()
        self.initContext()

    def checkLoginPassword(self, callback=None, label=None):
        if self.PGpassword=="" and not self.param.get('savePassword'):
            self.connDialog = connDialog(self)
            self.connDialog.callback = callback
            self.connDialog.label = label
            self.connDialog.accepted.connect(self.setLoginPassword)
            self.connDialog.rejected.connect(self.setLoginPassword)
        else:
            self.switchConnect(callback, label)
    
    def setLoginPassword(self):
        self.PGuser = self.connDialog.login()
        self.PGpassword = self.connDialog.password()
        self.switchConnect(self.connDialog.callback, self.connDialog.label)
        
    def switchConnect(self, callback, label):
        if callback:
            self.asyncConnect(callback, label)
        else:
            self.connect()

    def connect(self):
        self.error = None
        try:
            self.db = self.newConn()
            if not self.db.open():
                self.error = "Database Error: %s" % self.db.lastError().text()
            else:
                # print('connect status', self.db.isOpen())
                self.connected.emit()
        except:
            self.error = "Aucun paramétrage de la connexion Postgis"
        
    def asyncConnect(self, callback, label=None):
        m = mGlobalTask("Tentative de connexion")
        m.task.setCallin(self.connect)
        m.setCallback(callback)
        if label:
            self.startTimerConn(m, label)
        self.manager.add(m)
        
    def startTimerConn(self, m, label):
        timer = rdiTimer(self.manager)
        timer.timeout.connect(self.connecting.emit)
        timer.setLabel(label, deb=label.text()+"Tentative de connexion en cours")
        timer.setMaxPts(10)
        timer.start(200)
        m.setCallback(self.endTimerConn, timer)
        
    def endTimerConn(self, timer):
        try:
            timer.stop()
            del timer
        except:
            pass
        
    def newConn(self, name=None, init=True):
        if not init:
            return QSqlDatabase.cloneDatabase(self.db, name)
        
        if name:
            new = QSqlDatabase.addDatabase("QPSQL", name)
        else:
            new = QSqlDatabase.addDatabase("QPSQL")
        new.setHostName(self.PGhost)
        new.setPort(int(self.PGport))
        new.setDatabaseName(self.PGdb)
        new.setUserName(self.PGuser)
        new.setPassword(self.PGpassword)
        new.setConnectOptions(f"application_name='plugin RDI'")
        return new
    
    def setConn(self, db):
        db.setHostName(self.PGhost)
        db.setPort(int(self.PGport))
        db.setDatabaseName(self.PGdb)
        db.setUserName(self.PGuser)
        db.setPassword(self.PGpassword) 
        return db
        
    def delConn(self, name):
        QSqlDatabase.removeDatabase(name)
    
    def isDbConnected(self):
        if not self.db.isOpen():
            psql = self
            if self.isClone:
                psql = self.psql
            psql.connectionLost.emit()
        else:
            return True

    def simpleRequest(self):
        sql = "select 1"
        try:
            self.db.exec_(sql)
        except:
            pass    

    def endCheckDbAsync(self):
        self.autoTesting = False
        self.isDbConnected()
    
    def checkDbAsync(self):
        if self.autoTesting:
            return
        self.autoTesting = True
        m = mGlobalTask(f"Test de connexion à la base de données")
        m.task.setCallin(self.simpleRequest)
        m.setCallback(self.endCheckDbAsync)
        self.manager.add(m)
        
    def showConnexion(self):
        print(extractIP())
        print(socket.gethostbyname(socket.gethostname()))
        self.cleanQgisConnexion()

    def serverState(self):
        self.openDialog(serverDialog)
        # if not self.serverDialog:
            # self.serverDialog = serverDialog(self)
            # self.serverDialog.finished.connect(self.closeServerState)

    # def closeServerState(self):
        # self.serverDialog = None

    def cleanQgisConnexion(self):
        sql = f"SELECT pg_terminate_backend(pid) from pg_stat_activity" + \
            f" WHERE client_addr='{self.myIP}'" + \
            f" AND datname='{self.PGdb}' " + \
            f" AND xact_start is null " + \
            f" AND application_name IN ('QGIS', '', null)" + \
            f" AND (localtimestamp-state_change)>'00:00:20'"
        description = f"Kill process"
        task = mGlobalTask(description)
        task.addSubTaskSql(self, sql)
        self.manager.add(task)




    def openDialog(self, key):
        classes = {
            'serverState': serverDialog,
            'paramShare': paramDialog,
            'gestionList': gestionDialog,
            'stockTemp': stockageDialog,
            'shareTable': shareDialog,
        }
        if key not in classes:
            return
        dialogClass = classes[key]
        if key not in self.dialogs:
            self.dialogs[key] = dialogClass(self)
            self.dialogs[key].finished.connect(partial(self.closeDialog, key))

    def closeDialog(self, key):
        if key in self.dialogs:
            del self.dialogs[key]






    def initContext(self):
        self.tmpPrefixe = self.param.get('tmpPrefixe')
        self.context = {}
        self.context['buffer'] = max(min(self.param.get('buffer'), self.param.get('bufferMax')), self.param.get('bufferMin'))
        self.context['simplify'] = max(min(self.param.get('simplify'), self.param.get('simplifyMax')), 0)
        self.context['cutting'] = self.param.get('cutting')
        self.context['correction'] = self.param.get('correction')
        self.context['snapPrecision'] = self.param.get('snapPrecision')
        self.context['cluster'] = self.param.get('cluster')
        self.context['clusterField'] = self.param.get('clusterField')
        self.context['clusterPrecision'] = self.param.get('clusterPrecision')
        self.context['enjeuAlone'] = self.param.get('enjeuAlone')
        self.context['maskWorking'] = self.param.get('maskWorking')
        self.context['maskTable'] = None
        
    def actuContextServer(self):    
        if not self.pdb:
            self.pdb = rdiBaseParam(self)
        self.paramServer = {}
        for e in self.pdb.gets(code="param"):
            key = e['key']
            val = e['value']
            try:
                type = self.param.serverParam.get(key)
                if type in ['delay', 'integer']:
                    val = int(val)
                if type in ['float']:
                    val = float(val)
            except:
                pass
            self.paramServer[e['key']] = val
        self.tmpPrefixe = self.getParamServer('tmpPrefixe')
        
    def getParamServer(self, key):
        if self.paramServer.get('shareParam'):
            v = self.paramServer.get(key)
            if v is not None:
                return v
        return self.param.get(key)
    
    def actuMask(self, table):
        self.context['maskTable'] = table
    
    def getTable(self, table, schema=None):
        schema = self.getSchema(schema)
        sql = f"select * from " + self.full(table, schema)
        query = self.sqlSend(sql)
        return self.makeDictResult(query)
    
    def getTableCount(self, table, schema=None):
        schema = self.getSchema(schema)
        sql = f"select count(*) from " + self.full(table, schema)
        query = self.sqlSend(sql)
        return self.makeDictResult(query)
        
    def getSchemas(self):
        sql = f"select nspname from pg_catalog.pg_namespace " + \
            f" where nspname<>'information_schema' and nspname !~ E'^pg_'" + \
            f" order by nspname"
        return self.getList(sql)
        
    def getTables(self, schema, geomOnly=True):
        if geomOnly:
            sql = f"select c.tablename from pg_catalog.pg_tables c, information_schema.columns i " + \
                f" where c.schemaname='{schema}' " + \
                f" and i.table_name = c.tablename " + \
                f" and i.table_schema=c.schemaname " + \
                f" and i.udt_name='geometry' " + \
                f" and c.tablename not like '_tmp_%'" + \
                f" order by c.tablename "
        else:
            sql = f"select tablename from pg_catalog.pg_tables " + \
                f" where schemaname='{schema}'" + \
                f" and tablename not like '_tmp_%'" + \
                f" order by tablename"
        return self.getList(sql)

    def getList(self, sql):
        query = self.sqlSend(sql)
        list = []
        while query and query.next():
            list.append(query.value(0))
        return list  
      
    def isTable(self, table, schema=None, test=None):
        bool = False
        schema = self.getSchema(schema)
        sql = f"select * from pg_catalog.pg_tables where tablename='{table}' and schemaname='{schema}'"
        query = self.sqlSend(sql)
        if query and query.next():
            bool = True
        return bool     
        
    def getColumnsTableAll(self, table, schema=None):
        schema = self.getSchema(schema)
        sql = f"select * from information_schema.columns where table_name='{table}' and table_schema='{schema}'"
        query = self.sqlSend(sql)
        return self.makeDictResult(query) 
        
    def getColumnsTable(self, table, schema=None):
        schema = self.getSchema(schema)
        sql = f"select column_name from information_schema.columns where table_name='{table}' and table_schema='{schema}'"
        return self.getList(sql)
     
    def hasColumn(self, col, table, schema=None):
        schema = self.getSchema(schema)
        sql = f"select column_name from information_schema.columns where table_name='{table}' and table_schema='{schema}' and column_name='{col}'"
        query = self.sqlSend(sql)
        if query and query.next():
            return True

    def sqlSend(self, sql, flash=False):
        if not self.isDbConnected():
            return
        try:
            query = self.db.exec_(sql)
        except:
            msg = f"Aucune connexion à la base de données"
            if not self.isClone:
                print(msg)
            if flash:
                flashMsg(msg)
            return
        if query.lastError().type()!=QSqlError.NoError:
            if not self.isDbConnected():
                return
            if not self.isClone:
                print(sql, query.lastError().text())
            if flash:
                flashMsg(f"{query.lastError().text()}", category='error')
        return query

    def sqlInsert(self, sql, flash=False):
        query = self.sqlSend(sql, flash)
        if query and query.next():
            return query.value(0)

    def makeDictResult(self, query):
        list = []
        if not query:
            return list
        rec = query.record()    
        while (query.next()):
            row = {}
            for i in range(rec.count()):
                row[rec.fieldName(i)] = query.value(i)
            list.append(row)
        return list  
        
    def getTableAdmin2(self, code=None, compact=False):
        sql = f"select * from {self.admin('tableAdmin')}"
        if code:
            sql += f" where code='{code}'"
        sql += f" order by (case when tablename is null or tablename='' then 0 else 1 end), bloc,"
        if self.param.get('cbCompact') or compact:    
            sql += f" (case when parent is null then 0 else 1 end),"
        sql += f" \"order\", (case when name is null or name='' then tablename else name end), id"
        query = self.sqlSend(sql)
        return self.makeDictResult(query) 
    
    def getTableAdmin(self, code=None, compact=False, order='bloc'):
        sql = f"select a.*, b.name as parent_name from {self.admin('tableAdmin')} as a"
        sql += f" left join {self.admin('tableAdmin')} as b on a.parent=b.id"
        if code:
            sql += f" where a.code='{code}'"
        sql += f" order by (case when a.tablename is null or a.tablename='' then 0 else 1 end),"
        if order=='bloc':
            sql += f" (case when a.bloc is null or a.bloc='' then '' else a.bloc end), b.name,"
        else:
            sql += f" b.name, (case when a.bloc is null or a.bloc='' then '' else a.bloc end),"
        if self.param.get('cbCompact') or compact:    
            sql += f" (case when a.parent is null then 0 else 1 end),"
        sql += f" a.\"order\", (case when a.name is null or a.name='' then a.tablename else a.name end), a.id"
        query = self.sqlSend(sql)
        return self.makeDictResult(query) 
        
    def getTableAdminElt(self, id):
        sql = f"select * from {self.admin('tableAdmin')} where id={id}"
        query = self.sqlSend(sql)
        return self.makeDictResult(query) 
        
    def getTableAdminEltFull(self, *ids):
        if len(ids)==0: return []
        sql = f"select a1.*, a2.name as pname, a2.tablename as ptablename, a2.bloc as pbloc"
        sql += f" from {self.admin('tableAdmin')} a1"
        sql += f" left join {self.admin('tableAdmin')} a2 on a2.id=a1.parent"
        sql += f" where a1.id in ({','.join(map(str,ids))})"
        sql += f" order by {self.getOrderFollow('a1.id', ids)}"
        query = self.sqlSend(sql)
        return self.makeDictResult(query) 
    
    def getTableAdminEltChilds(self, *ids, ordered=False):
        if len(ids)==0: return []
        sql = f"select a1.*, a2.name as pname, a2.tablename as ptablename, a2.bloc as pbloc"
        sql += f" from {self.admin('tableAdmin')} a1"
        sql += f" left join {self.admin('tableAdmin')} a2 on a2.id=a1.parent"
        sql += f" where a2.id in ({','.join(map(str,ids))})"
        
        if ordered in (False, True):
            print('make this an error')
            # a = int('f')
        
        o3 = self.getOrderNoNull('a1.bloc', '')
        o2 = self.getOrderFollow('a2.id', ids)
        o1 = self.getOrderNoNull('a1.order')
        if ordered=='parent': sql += f" order by {o2}, {o1}, a1.name, {o3}"
        elif ordered=='bloc': sql += f" order by {o3}, {o1}, a1.name, {o2}"
        elif ordered=='layer': sql += f" order by {o1}, a1.name, {o2}, {o3}"
        query = self.sqlSend(sql)
        return self.makeDictResult(query) 
    
    def formatValue(self, v):
        if isinstance(v, str): return f"'{v}'"
        return v
    
    def getOrderNoNull(self, a, v=0):
        return f"(case when {a} is null then {self.formatValue(v)} else {a} end)"
    def getOrderFollow(self, a, l):
        return f'(case {" ".join([f"when {a}={id} then {i}" for i,id in enumerate(l)])} end)'
    
    def getTableAdminBlocs(self, mixed=False, code=None):
        if mixed:
            field = f"bloc"
        else:
            field = f"concat(code,'_',bloc)"
        where = f"where bloc is not NULL and bloc<>''"
        if code is not None:
            where += f" and code='{code}'"
        sql = f"select {field} from {self.admin('tableAdmin')} {where} group by 1"
        return self.getList(sql)
    
    def getSVG(self, table, schema=None, geom=None, maxDecimalDigits=None):
        schema = self.getSchema(schema)
        if not geom:
            geom = self.getGeometryField(table, schema)
        if maxDecimalDigits is not None:
            st_assvg = f"st_asSVG({quote(geom)}, 1, {maxDecimalDigits})"
        else:
            st_assvg = f"st_asSVG({quote(geom)})"
        sql = f"select {st_assvg} as poly, box({quote(geom)}) from {self.full(table, schema)}"
        query = self.sqlSend(sql)
        return self.makeDictResult(query)
    
    def boxToList(self, box):
        tt = list(map(float, box.replace('BOX(', '') .replace(')', '').replace(' ', ',').split(',')))
        return tt
    
    def getExtent(self, table=None, schema=None, tableList=None):
        sql = self.getExtentSql(table, schema, tableList)
        query = self.sqlSend(sql)
        if query and query.next():
            box = query.value(0)
            return self.boxToList(box)
            
    def getExtentAsync(self, table=None, schema=None, tableList=None, callback=None, *args, **kwargs):
        sql = self.getExtentSql(table, schema, tableList)
        if table is None:
            table = "all"
        description = f"Get Extent for {table}"
        task = mGlobalTask(description)
        task.addSubTaskSql(self, sql, callback)
        self.manager.add(task)
            
    def getExtentSql(self, table=None, schema=None, tableList=None):
        sql = ""
        if table is None:
            col = f"extent"
            tables = "alles"
            sql_list = []  
            if tableList is None:
                tableList = []
                for elt in self.getTableAdmin('alea'):
                    tableList.append((elt.get('tablename'), self.getSchema(elt.get('schemaname')), elt.get('geom')))
            for c in tableList:
                table = c[0]
                schema = c[1]  
                if not table:
                    continue
                geom = c[2]
                sql_list.append(f" SELECT st_extent({quote(geom)}) as extent FROM {self.full(table, schema)} ")
            if len(sql_list)==0:
                return
            union = " UNION ALL ".join(sql_list)
            sql += f"WITH alles AS ({union})"
        else:
            schema = self.getSchema(schema)
            geom = self.getGeometryField(table, schema)
            col = f"{quote(geom)}"
            tables = f"{self.full(table, schema)}"
        sql += f" SELECT st_extent({col}) FROM {tables}"
        return sql  
     
    
    def full(self, table, schema=None):
        schema = self.getSchema(schema)
        s = quote(table)
        if schema and schema!="":
            s = quote(schema) + "." + s
        return s
    
    def getGeomType(self, table, schema=None, geom=None):
        if not table:
            return []
        if not geom:
            geom = self.getGeometryField(table, schema)
        geomType = []
        if geom:
            sql = f"select geometryType({quote(geom)}) from {self.full(table, schema)} group by geometryType({quote(geom)})"
            query = self.sqlSend(sql)
            while query and query.next():
                geomType.append(query.value(0))
        if len(geomType)==0:
            sql = f"SELECT type FROM geometry_columns " + \
                f" WHERE f_table_schema = '{schema}' " + \
                f" AND f_table_name = '{table}' "
            if geom:  
                sql += f" and f_geometry_column = '{geom}'"
            query = self.sqlSend(sql)
            while query and query.next():
                geomType.append(query.value(0))
        return geomType    
        
    def getGeomTypeUnique(self, table, schema=None, geom=None, geomType=None):
        if not geomType:
            geomType = self.getGeomType(table, schema, geom)
        if len(geomType)==1:
            return geomType[0]
        
    def isGeomPoint(self, table, schema=None, geom=None, geomType=None):
        if not geomType:
            geomType = self.getGeomType(table, schema, geom)
        bool = False
        for t in geomType:
            if t and t.lower().find('point')>=0:
                bool = True
        return bool
    
    def isGeomPolygon(self, table, schema=None, geom=None, geomType=None):
        if not geomType:
            geomType = self.getGeomType(table, schema, geom)
        bool = False
        for t in geomType:
            if t and t.lower().find('polygon')>=0:
                bool = True
        return bool
        
    def isGeomLinestring(self, table, schema=None, geom=None, geomType=None):
        if not geomType:
            geomType = self.getGeomType(table, schema, geom)
        bool = False
        for t in geomType:
            if t and t.lower().find('linestring')>=0:
                bool = True
        return bool
    
    def isMultiGeom(self, table, schema=None, geom=None, geomType=None):
        if not geomType:
            geomType = self.getGeomType(table, schema, geom)
        bool = False
        for t in geomType:
            if t and t.lower().find('multi')>=0:
                bool = True
        return bool
    
    def needDump(self, table, schema=None, geom=None):
        bool = self.isGeomPoint(table, schema, geom)
        if bool:
            bool = self.isMultiGeom(table, schema, geom)
        return bool
    
    def getLayer(self, tableName, layerName, schema=None, cond=''):
        schema = self.getSchema(schema)
        if not self.isTable(tableName, schema, test=True):
            if not self.isClone:
                print('no table', tableName, schema)
            return None, None
        uri = QgsDataSourceUri()
        uri.setConnection(self.PGhost, str(self.PGport), self.PGdb, self.PGuser, self.PGpassword)
        geom = self.getGeometryField(tableName, schema)
        pkey = self.getPrimaryKey(tableName, schema)
        geomType = self.getGeomType(tableName, schema, geom)
        uri.setDataSource(schema, tableName, geom, cond, pkey)
        vlayer = QgsVectorLayer(uri.uri(), layerName, "postgres")
        if vlayer.isValid():
            return vlayer, geomType
        else:
            if not self.isClone:
                print('invalid', tableName, schema, geom, pkey)
            return None, None

    def getLayerAsync(self, tableName, layerName, schema=None, cond='', callback2=None, *args, **kwargs):
        
        if not self.param.get('asyncGetLayer'):
            result = self.getLayer(tableName, layerName, schema, cond)
            if callback2:
                callback2(result, *args, **kwargs)
            return
            
        psql = self.clone()
        m = mGlobalTask(f"Recuperation des donnees pour {layerName}")
        m.task.setCallin(psql.getLayer, tableName, layerName, schema, cond)
        m.setAutoKillConnexion(psql)
        m.addCallback(self.cleanQgisConnexion)
        if callback2:
            m.setCallbackWithResult(callback2, *args, **kwargs)
        self.manager.add(m)

    
    
    def getSchema(self, schema=None):
        if not schema or schema=="":
            schema = self.PGschema
        return schema
    
    def getPrimaryKey(self, table, schema=None):
        schema = self.getSchema(schema)
        sql = f"select " + \
			f"		a.attname as column_name " + \
			f"	from " + \
			f"		pg_class t, " + \
			f"		pg_class i, " + \
			f"		pg_index ix, " + \
			f"		pg_attribute a, " + \
			f"		pg_catalog.pg_tables c " + \
			f"	where " + \
			f"		t.oid = ix.indrelid " + \
			f"		and i.oid = ix.indexrelid " + \
			f"		and a.attrelid = t.oid " + \
			f"		and a.attnum = ANY(ix.indkey) " + \
			f"		and t.relkind = 'r' " + \
			f"		and ix.indisprimary = true " + \
			f"		and t.relname = '{table}' " + \
			f"		and c.tablename = t.relname " + \
			f"		and c.schemaname = '{schema}' " + \
			f"	order by " + \
			f"		t.relname, " + \
			f"		i.relname;"
        query = self.sqlSend(sql)
        if query and query.next():
            return str(query.value(0))
     
    def getSequence(self, table, schema=None):
        schema = self.getSchema(schema)
        sql = f"select " + \
            f"		s.sequence_name " + \
            f"	from " + \
            f"		information_schema.columns c, " + \
            f"		information_schema.sequences s " + \
            f"	where " + \
            f"		c.table_catalog = '{self.PGdb}' " + \
            f"		and c.table_name = '{table}' " + \
            f"		and c.table_schema = '{schema}' " + \
            f"		and s.sequence_catalog = table_catalog " + \
            f"		and s.sequence_schema = table_schema " + \
            f"		and c.column_default like '%'||sequence_name||'%' ;"
        query = self.sqlSend(sql)
        if query and query.next():
            return str(query.value(0))
        
    def hasIndex(self, col, table, schema=None):
        schema = self.getSchema(schema)
        sql = f"select " + \
            f"		t.relname as table_name, " + \
            f"		i.relname as index_name, " + \
            f"		a.attname as column_name " + \
            f"  from " + \
            f"      pg_class t, " + \
            f"      pg_class i, " + \
            f"      pg_index ix, " + \
            f"      pg_attribute a, " + \
            f"      pg_catalog.pg_tables c " + \
            f"  where " + \
            f"      t.oid = ix.indrelid " + \
            f"      and i.oid = ix.indexrelid " + \
            f"      and a.attrelid = t.oid " + \
            f"      and a.attnum = ANY(ix.indkey) " + \
            f"      and t.relkind = 'r' " + \
            f"      and c.tablename = t.relname " + \
            f"      and c.tablename = '{table}' " + \
            f"      and c.schemaname = '{schema}' " + \
            f"      and a.attname = '{col}' " + \
            f"  order by " + \
            f"      t.relname, " + \
            f"      i.relname; "
        query = self.sqlSend(sql)
        if query and query.next():
            return True
            
    def makeSpatialIndex(self, table, schema=None, geom=None):
        schema = self.getSchema(schema)
        if not geom:
            geom = self.getGeometryField(table, schema)
        col = f"{table}_geom"
        sql = f"DROP INDEX IF EXISTS {quote(col)} CASCADE; CREATE INDEX {quote(col)} ON {self.full(table, schema)} USING gist({quote(geom)});"
        self.db.exec_(sql)

    def isValidGeos(self, table, schema=None, geom=None):
        schema = self.getSchema(schema)
        if not geom:
            geom = self.getGeometryField(table, schema)
        sql = f"SELECT st_isValid({quote(geom)}) FROM {self.full(table, schema)} WHERE st_isValid({quote(geom)})='f'"
        query = self.sqlSend(sql)
        if query and query.next():
            return False
        else:
            return True
            
    def makeValidGeos(self, table, schema=None, geom=None, type=None):
        schema = self.getSchema(schema)
        if not geom:
            geom = self.getGeometryField(table, schema)
        geomST = f"st_makeValid({quote(geom)})"
        if not type:
            geomType = self.getGeomType(table, schema, geom)
            if self.isGeomPoint(table, schema, geomType=geomType):
                type = "point"
            if self.isGeomLinestring(table, schema, geomType=geomType):
                type = "line"
            if self.isGeomPolygon(table, schema, geomType=geomType):
                type = "polygon"
        if type:
            ctype = {'point':1, 'line':2, 'polygon':3}
            ntype = ctype[type]
            geomST = f"st_collectionExtract({geomST}, {ntype})"
        sql = f"UPDATE {self.full(table, schema)} SET {quote(geom)}={geomST}"
        query = self.sqlSend(sql)    

    def getGeometryField(self, table, schema=None):
        schema = self.getSchema(schema)
        sql = f"select column_name from information_schema.columns " + \
            f"	where table_name = '{table}' " + \
            f"	and udt_name='geometry' " + \
            f"	and table_schema='{schema}' "
        query = self.sqlSend(sql)
        if query and query.next():
            return str(query.value(0))
        
    def getNonGeomFields(self, table, schema=None):
        schema = self.getSchema(schema)
        sql = f"select column_name from information_schema.columns " + \
            f"	where table_name = '{table}' " + \
            f"	and udt_name<>'geometry' " + \
            f"	and table_schema='{schema}' "
        query = self.sqlSend(sql)
        list = []
        while query and query.next():
            list.append(query.value(0))
        return list
    
    def getSrid(self, table, schema=None, geom=None):
        if not geom:
            geom = self.getGeometryField(table, schema)
        sql = f"SELECT st_srid({quote(geom)}) FROM {self.full(table, schema)} limit 1"
        query = self.sqlSend(sql)
        if query and query.next():
            return str(query.value(0))
        
    def proj(self, table, srid, schema=None, geom=None):
        if not geom:
            geom = self.getGeometryField(table, schema)
        st_geom = self.full(table, schema) + "." + quote(geom)
        srid_ori = self.getSrid(table, schema)
        if srid!=srid_ori:
            st_geom = f"st_transform({st_geom}, {srid})"
        return st_geom


    def admin(self, code):
        if code=='tableAdmin':
            return self.full(self.param.get(code), self.param.get('schema', 'ADMIN'))
        else:            
            return self.full(self.param.get(code), self.param.get('schema', 'CONN'))
 
    
    def checkColumnName(self, table, schema, name):
        fields = self.getColumnsTable(table, schema)
        trouve = False
        for f in fields:
            if f==name:
                trouve = True
                break
        return trouve
        
    def getNewColumnName(self, table, schema, name):
        while self.checkColumnName(table, schema, name):
            name += "_"
        return name
    

    def getHull(self, rdi):
        requete = f"hull"
        rdiAttr = rdi.getAttr() 
        rdiArgs = (rdiAttr['table'], rdiAttr['schema'])
        geom = self.getGeometryField(*rdiArgs)
        geomST = f"{quote(geom)}"
        if self.isGeomPolygon(*rdiArgs) and self.context['correction']:
            geomST = f"st_buffer(st_collectionExtract(st_makeValid({geomST}), 3), {self.context['snapPrecision']})"
            requete += "_corr"
        rdiTotal = self.full(*rdiArgs)
        sqlExtract = f"SELECT ST_ConvexHull(ST_Union({geomST})) as geom FROM {rdiTotal}"
        q = tmpQuery(self, sqlExtract)
        q.addBD(enjeu=rdiTotal, alea=rdiTotal, requete=requete)
        return q
 
    def getSimplify(self, rdi):
        rdiAttr = rdi.getAttr() 
        rdiArgs = (rdiAttr['table'], rdiAttr['schema'])
        fields = ", ".join(map(lambda x: f"a.{quote(x)}", self.getNonGeomFields(*rdiArgs)))
        geom = self.getGeometryField(*rdiArgs)
        rdiTotal = self.full(*rdiArgs)
        typeGeomInt = 1
        if self.isGeomPolygon(*rdiArgs):
            typeGeomInt = 3
        if self.isGeomLinestring(*rdiArgs):
            typeGeomInt = 2
        requete = f"simplify{self.context['simplify']}"
        
        tables = f"{rdiTotal} a"
        geomST = f"ST_CollectionExtract(ST_MakeValid(ST_SimplifyPreserveTopology(a.{quote(geom)}, {self.context['simplify']})), {typeGeomInt})"
        groupby = ""
        if self.context['maskWorking'] and self.context['maskTable'] is not None:
            maskTable = self.context['maskTable']
            requete += f"_mask{maskTable}"
            geomMask = f"b.{quote(self.getGeometryField(maskTable))}"
            if self.context['correction']:
                geomMask = f"st_buffer(st_collectionExtract(st_makeValid({geomMask}), 3), {self.context['snapPrecision']})"
            geomST = f"ST_CollectionExtract(ST_Intersection({geomST}, {geomMask}), {typeGeomInt})"
            tables += f", {maskTable} b"
            groupby = f"group by a.{quote(self.getPrimaryKey(*rdiArgs))}"
        sqlExtract = f"SELECT {geomST} as geom, {fields} FROM {tables}"
        q = tmpQuery(self, sqlExtract)
        q.addBD(enjeu=rdiTotal, alea=rdiTotal, requete=requete)
        return q
        
    def getTampon(self, rdi, force=True):
        rdiAttr = rdi.getAttr() 
        rdiArgs = (rdiAttr['table'], rdiAttr['schema'])
        rdiTotal = self.full(*rdiArgs)
        sqlExtract, requete = self.getTamponSQL(rdi, force)
        q = tmpQuery(self, sqlExtract)
        q.addBD(enjeu=rdiTotal, alea=rdiTotal, requete=f"buffer{requete}")
        return q

    def getTamponSQL(self, rdi, buff=False):
        rdiAttr = rdi.getAttr() 
        rdiArgs = (rdiAttr['table'], rdiAttr['schema'])
        geomST = f"a.{quote(self.getGeometryField(*rdiArgs))}"
        requete = ""
        buffer = self.context['buffer'] or 0
        if self.context['buffer'] and self.context['buffer']!=0 or buff:
            requete += f"{self.context['buffer']}"
        if self.isGeomPolygon(*rdiArgs) and self.context['correction']:
            geomST = f"st_collectionExtract(st_makeValid(st_snapToGrid({geomST}, {self.context['snapPrecision']})), 3)"
            requete += "_corr"
        if self.context['simplify']>0:
            geomST = f"ST_MakeValid(ST_SimplifyPreserveTopology({geomST}, {self.context['simplify']}))"
            requete += f"_simpl{self.context['simplify']}"
        geomST = f"ST_UNION({geomST})"
        if self.context['buffer'] and self.context['buffer']!=0 or buff:
            requete += f"_buff{self.context['buffer']}"
            geomST = f"ST_Buffer({geomST}, {buffer})"

        if rdiAttr['context']:
            if 'field' in rdiAttr['context']:
                where = f" WHERE {rdiAttr['context']['field']} NOT IN ({rdiAttr['context']['values']})"
                requete += f":{rdiAttr['context']['field']}:{rdiAttr['context']['values']}"
            if 'condition' in rdiAttr['context']:
                where = f" WHERE {rdiAttr['context']['condition']}"
                if 'excl' in rdiAttr['context']:
                    requete += f":{rdiAttr['context']['excl']}"
                else:
                    requete += f":{compress(rdiAttr['context']['condition'])}"
        else:
            where = ""   

        tables = f"{self.full(*rdiArgs)} a"
        if self.context['maskWorking'] and self.context['maskTable'] is not None:
            maskTable = self.context['maskTable']
            requete += f"_mask{maskTable}"
            geomMask = f"b.{quote(self.getGeometryField(maskTable))}"
            if self.context['correction']:
                geomMask = f"st_buffer(st_collectionExtract(st_makeValid({geomMask}), 3), {self.context['snapPrecision']})"
            geomST = f"ST_CollectionExtract(ST_Intersection({geomST}, ST_UNION({geomMask})), 3)"
            tables += f", {maskTable} b"
        sql = f"SELECT {geomST} as geom FROM {tables} {where}"
        return sql, requete  

    def getInMask(self, rdi):
        rdiAttr = rdi.getAttr() 
        rdiArgs = (rdiAttr['table'], rdiAttr['schema'])
        fields = ", ".join(map(lambda x: f"a.{quote(x)}", self.getNonGeomFields(*rdiArgs)))
        geom = self.getGeometryField(*rdiArgs)
        rdiTotal = self.full(*rdiArgs)
        tables = f"{rdiTotal} a"
        geomST = f"a.{quote(geom)}"
        typeGeomInt = 1
        if self.isGeomPolygon(*rdiArgs):
            typeGeomInt = 3
        if self.isGeomLinestring(*rdiArgs):
            typeGeomInt = 2
        maskTable = self.context['maskTable']
        requete = f"mask{maskTable}"
        geomMask = f"b.{quote(self.getGeometryField(maskTable))}"
        if self.context['correction']:
                geomMask = f"st_buffer(st_collectionExtract(st_makeValid({geomMask}), 3), {self.context['snapPrecision']})"
        geomST = f"ST_CollectionExtract(ST_Intersection({geomST}, {geomMask}), {typeGeomInt})"
        tables += f", {maskTable} b"
        groupby = f"group by a.{quote(self.getPrimaryKey(*rdiArgs))}"
        sqlExtract = f"SELECT {geomST} as geom, {fields} FROM {tables}"
        q = tmpQuery(self, sqlExtract)
        q.addBD(enjeu=rdiTotal, alea=rdiTotal, requete=requete)
        return q

    def getSingle(self, rdi):
        rdiAttr = rdi.getAttr()
        rdiArgs = (rdiAttr['table'], rdiAttr['schema'])
        fields = ", ".join(map(quote, self.getNonGeomFields(*rdiArgs)))
        geom = self.getGeometryField(*rdiArgs)
        rdiTotal = self.full(*rdiArgs)
        sqlExtract = f"SELECT (ST_DUMP({quote(geom)})).geom as geom, {fields} FROM {rdiTotal}"
        q = tmpQuery(self, sqlExtract)
        # q.setOrigin(*rdiArgs, method='dump')
        # q.setPkey(self.getPrimaryKey(*rdiArgs))
        q.addBD(enjeu=rdiTotal, alea=rdiTotal, requete="dump")
        return q
        
    def getSumAttr(self, rdi):
        rdiAttr = rdi.getAttr()
        rdiArgs = (rdiAttr['table'], rdiAttr['schema'])
        fields = ", ".join(map(quote, self.getNonGeomFields(*rdiArgs)))
        geom = self.getGeometryField(*rdiArgs)
        rdiTotal = self.full(*rdiArgs)
        all = f"{rdiTotal}.*"
        requete = ""
        multiple = ""
        if self.needDump(*rdiArgs):
            all = f"(ST_DUMP({quote(geom)})).geom as geom, {fields}"
            requete += "_dump"
            method = 'dump'

        if rdiAttr['field']!="":
            field = rdiAttr['field']
            precision = self.context['clusterPrecision']
            part = self.getNewColumnName(*rdiArgs, "part")
            rdi.part = part
            all += f", {part}"
            multiple =  f" CROSS JOIN generate_series(1, round({rdiTotal}.{quote(field)}*{precision})::integer, 1) {part}"  
            requete += f"_mult{precision}_{field}"
        sqlExtract = f"SELECT {all} FROM {rdiTotal} {multiple}"
        
        q = tmpQuery(self, sqlExtract)
        q.addBD(enjeu=rdiTotal, alea=rdiTotal, requete=requete)
        return q
  
    def getExtractEnjeuU(self, alea, enjeu):
        aleaAttr = alea.getAttr() 
        aleaArgs = (aleaAttr['table'], aleaAttr['schema'])    
        selectBuffer, requete = self.getTamponSQL(alea) 
        q = self.getTampon(alea)
        if q.dispo:
            selectBuffer = f"SELECT geom FROM {self.full(q.table)}"
        return self.getExtractEnjeuUsql(enjeu, selectBuffer, requete, aleaAttr['table'], aleaAttr['schema'])
            
    def getExtractEnjeuUsql(self, enjeu, selectBuffer, requete, table, schema, context={}):
        enjeuAttr = enjeu.getAttr()
        enjeuArgs = (enjeuAttr['table'], enjeuAttr['schema'])
        enjeuTotal = self.full(*enjeuArgs)
        aleaTotal = self.full(table, schema)
        requete = f"intersect{requete}"
        geomEnjeu = self.getGeometryField(*enjeuArgs)    
        all = f"{enjeuTotal}.*"
        multiple = ""
        if context.get('srid'):
            srid = context.get('srid')
        else:
            srid = self.getSrid(table, schema)
        pkey = self.getPrimaryKey(*enjeuArgs)
        enjeuProj = f"{self.proj(enjeuAttr['table'], srid, enjeuAttr['schema'], geomEnjeu)}"
        if self.isGeomPolygon(*enjeuArgs) and self.context['correction']:
            enjeuProj = f"st_buffer(st_collectionExtract(st_makeValid({enjeuProj}), 3), {self.context['snapPrecision']})"
            requete += "_corrE"
        method = None
        
        n = ""
        maskTable = ""
        inMask = ""
        
        if enjeu.isStat==None and enjeu.cluster and self.needDump(*enjeuArgs):
            list = self.getNonGeomFields(*enjeuArgs)
            fields = ", ".join(map(lambda x: f"{enjeuTotal}.{quote(x)}", list))
            all = f"(ST_DUMP({enjeuTotal}.{quote(geomEnjeu)})).geom as geom, {fields}"
            geomEnjeu = "geom"
            requete += "_dump"
            method = 'dump'
            
        if enjeuAttr['field']!="" and enjeu.isStat==None and enjeu.cluster and self.isGeomPoint(*enjeuArgs):
            field = enjeuAttr['field']
            precision = self.context['clusterPrecision']
            part = self.getNewColumnName(*enjeuArgs, "part")
            enjeu.part = part
            all += f", {part}"
            multiple =  f" CROSS JOIN generate_series(1, round({enjeuTotal}.{quote(field)}*{precision})::integer, 1) {part}"  
            requete += f"_mult{precision}_{field}"
       
        if enjeu.isStat==None and self.context['cutting'] and not self.isGeomPoint(*enjeuArgs):
            list = self.getNonGeomFields(*enjeuArgs)
            fields = ", ".join(map(lambda x: f"{enjeuTotal}.{quote(x)}", list))
            if enjeu.inverted:
                if self.context['maskWorking'] and self.context['maskTable'] is not None:
                    maskTable = f", {self.context['maskTable']}"
                    geomMask = f"{self.context['maskTable']}.{quote(self.getGeometryField(self.context['maskTable']))}"
                    enjeuProj = f"ST_Intersection({enjeuProj}, {geomMask})"
                all = f"ST_Multi({enjeuProj}) as geom, {fields}"
            else:
                all = f"ST_Multi(ST_Intersection({enjeuProj}, buffer.geom)) as geom, {fields}"
            geomEnjeu = "geom"
            requete += "_cut"
            method = 'cut'
        
        if enjeu.isStat==None and enjeu.inverted and not context.get('mask'):
            n = "NOT"
            requete += "_not"
            if self.context['maskWorking'] and self.context['maskTable'] is not None:
                maskTable = f", {self.context['maskTable']}"
                geomMask = f"{self.context['maskTable']}.{quote(self.getGeometryField(self.context['maskTable']))}"
                inMask = f" AND ST_Intersects({geomMask},{enjeuProj})"
        
        sqlExtract = f"WITH buffer AS ({selectBuffer})" +\
			f" SELECT {all} FROM {enjeuTotal} , buffer {maskTable} {multiple}" +\
			f" WHERE {n} ST_Intersects(buffer.geom, {enjeuProj}) {inMask}"
            
             
        if enjeu.isStat==None and self.context['cutting'] and not self.isGeomPoint(*enjeuArgs) and enjeu.inverted and not context.get('mask'):
            typeGeomInt = 1
            if self.isGeomPolygon(*enjeuArgs):
                typeGeomInt = 3
            if self.isGeomLinestring(*enjeuArgs):
                typeGeomInt = 2 
            all = f"ST_Multi(ST_CollectionExtract(ST_Difference({enjeuProj}, buffer.geom), {typeGeomInt})) as geom, {fields}"
            sqlExtract += f" UNION SELECT {all} FROM {enjeuTotal} , buffer {maskTable}" +\
                        f" WHERE ST_Intersects(buffer.geom, {self.proj(enjeuAttr['table'], srid, enjeuAttr['schema'], geomEnjeu)})"
        q = tmpQuery(self, sqlExtract)
        if multiple!="":
            q.setPkey(pkey, enjeu.part)
        else:
            q.setPkey(pkey)
        q.setOrigin(*enjeuArgs, method=method)
        q.addBD(enjeu=enjeuTotal, alea=aleaTotal, requete=requete)
        return q
                    
    def getAssembEnjeu(self, enjeu, tList):
        enjeuAttr = enjeu.getAttr()
        enjeuArgs = (enjeuAttr['table'], enjeuAttr['schema'])
        if len(tList)>1:
            method = None
            join = "union"
            if enjeu.cluster and self.needDump(*enjeuArgs):
                method = 'dump'
            if enjeu.inverted:
                join = "intersect" 
            sqlExtract = f" {join} ".join(map(lambda x: f"SELECT * FROM {self.full(x)}", tList))
            q = tmpQuery(self, sqlExtract)
            q.setOrigin(*enjeuArgs, method=method)
            
            # q.setPkey(self.getPrimaryKey(*enjeuArgs))
            q.addBD(enjeu=self.full(*enjeuArgs), alea="+".join(tList), requete=join)
        elif len(tList)>0:
            q = tmpQuery(self, '')
            q.table = tList[0]
            q.dispo = True
        else:
            if self.context['enjeuAlone']:
                if self.context['maskWorking'] and self.context['maskTable'] is not None:
                    requete = f"mask{self.context['maskTable']}"
                    selectBuffer = f"SELECT geom FROM {self.context['maskTable']}"
                    q = self.getExtractEnjeuUsql(enjeu, selectBuffer, requete, self.context['maskTable'], None, {'mask':True})
                else:
                    if enjeu.cluster:
                        q = self.getSumAttr(rdi=enjeu)
                    else:
                        q = tmpQuery(self, '')
                        q.table = enjeuAttr['table']
                        q.dispo = True
            else:
                if self.isClone:
                    q = self.getEmptyTask2(rdi=enjeu)
                    q.noWait = True
                else:
                    task, table = self.getEmptyTask(enjeu)
                    task.setCallback(enjeu.affLayer, table)
                    self.manager.add(task)  
                    q = tmpQuery(self, '')
                    q.table = table
                    q.dispo = True
        return q
        
    def getExtractEnjeu(self, enjeu, aleaList, callback=None, args=(), kwargs={}):
        enjeuAttr = enjeu.getAttr()
        enjeuArgs = (enjeuAttr['table'], enjeuAttr['schema'])
        if enjeu.cluster and self.isGeomPoint(*enjeuArgs):
            enjeu.needBack = True
        else:
            enjeu.needBack = False
        nb = len(aleaList)
        s = "s" if nb>1 else ""
        description = f"Extraction enjeu {quote(enjeuAttr['name'])} sur {nb} emprise{s}"
        task = mGlobalTask(description)
        if enjeu.isStat==None and self.context['cutting'] and not self.isGeomPoint(*enjeuArgs) and len(aleaList)>1:
            self.getFusionAlea(enjeu, aleaList, self.getExtractFusion, task, callback, args, kwargs)
        else:
            self.getFusionAlea(None, aleaList)
            tList = [] 
            for alea in aleaList:
                aleaAttr = alea.getAttr() 
                q = self.getExtractEnjeuU(alea=alea, enjeu=enjeu)
                if not q.dispo:
                    q.description = f"{quote(aleaAttr['name'])}/{quote(enjeuAttr['name'])}"
                    task.addSubTask(q)
                tList.append(q.table)
            q = self.getAssembEnjeu(enjeu=enjeu, tList=tList)
            q.description = f"Assemblage {len(tList)}/{quote(enjeuAttr['name'])}"
            self.endExtractEnjeu(enjeu, task, q, callback, args, kwargs)



    def getExtractEnjeu2(self, enjeu, aleaList, callback=None, args=(), kwargs={}):
        data = {'extract':[], 'assemb':None, 'dispo':True}
        data['enjeu'] = enjeu
        data['aleaList'] = aleaList
        data['callback'] = callback
        data['args'] = args
        data['kwargs'] = kwargs
        enjeuAttr = enjeu.getAttr()
        enjeuArgs = (enjeuAttr['table'], enjeuAttr['schema'])
        dispo = True
        if enjeu.cluster and self.isGeomPoint(*enjeuArgs):
            enjeu.needBack = True
        else:
            enjeu.needBack = False
        nb = len(aleaList)
        s = "s" if nb>1 else ""
        data['description'] = f"Extraction enjeu {quote(enjeuAttr['name'])} sur {nb} emprise{s}"
        
        q = self.getEmptyTask2(rdi=enjeu)
        data['empty'] = q
        
        if enjeu.isStat==None and self.context['cutting'] and not self.isGeomPoint(*enjeuArgs) and len(aleaList)>1:
            data['fusionFirst'] = True
            data2 = self.getFusionTask(aleaList, 'alea')
            data['qEnveloppes'] = data2['qEnveloppes']
            data['qExtract'] = data2['qExtract']
            selectBuffer = f"SELECT geom FROM {self.full(data['qExtract'].table)}"
            srid = self.getSrid(*enjeuArgs)
            q = self.getExtractEnjeuUsql(enjeu, selectBuffer, "", data['qExtract'].table, None, context={'srid':srid})
            q.description = f"Extraction sur alea fusion"
            data['assemb'] = q
        else:
            tList = [] 
            for alea in aleaList:
                aleaAttr = alea.getAttr() 
                q = self.getExtractEnjeuU(alea=alea, enjeu=enjeu)
                q.description = f"{quote(aleaAttr['name'])}/{quote(enjeuAttr['name'])}"
                data['extract'].append(q)
                tList.append(q.table)
            q = self.getAssembEnjeu(enjeu=enjeu, tList=tList)
            q.description = f"Assemblage {len(tList)}/{quote(enjeuAttr['name'])}"
            data['assemb'] = q
        return data


   
    def getExtractEnjeuAsync(self, enjeu, aleaList, callback=None, args=(), kwargs={}):
        psql = self.clone()
        m = mGlobalTask(f"Preparation SQL")
        m.task.setCallin(psql.getExtractEnjeu2, enjeu=enjeu, aleaList=aleaList, callback=callback, args=args, kwargs=kwargs)
        m.setAutoKillConnexion(psql)
        m.setCallbackWithResult(self.endExtractEnjeuAsync)
        self.manager.add(m)
        
    def endExtractEnjeuAsync(self, result):
        if result is None:
            return
        task = mGlobalTask(result['description'])
        if result.get('fusionFirst'):
            for q in result['qEnveloppes']:
                if not q.dispo:
                    task.addSubTask(q)
            q = result['qExtract']
            if q is not None:
                if not q.dispo:    
                    task.addSubTask(q, final=True)
        else:
            self.getFusionAleaAsync(None, result['aleaList'])
            for q in result['extract']:
                if not q.dispo:
                    task.addSubTask(q)
        self.endExtractEnjeu(result['enjeu'], task, result['assemb'], result['callback'], result['args'], result['kwargs'])
    
    def getExtractFusion(self, table, enjeu, task, callback, args, kwargs):
        selectBuffer = f"SELECT geom FROM {self.full(table)}"
        q = self.getExtractEnjeuUsql(enjeu, selectBuffer, "", table, None)
        q.description = f"Extraction sur alea fusion"
        self.endExtractEnjeu(enjeu, task, q, callback, args, kwargs)

    def endExtractEnjeu(self, enjeu, task, q, callback=None, args=(), kwargs={}):
        if callback!=None:
            task.setCallback(callback, q.table, enjeu, *args, **kwargs)
        else:
            task.setCallback(enjeu.affLayerAsync, q.table)
        if not q.dispo:
            task.addSubTask(q, final=True)
        if task.count()==0 or callback!=None or q.noWait:
            enjeu.tableWaiting = q.table
            self.manager.add(task)
        else:
            self.getWaiting(enjeu, self.launchCallback, task, q.table)

    
    
    def launchCallback(self, rdi, task, table):
        rdi.tableWaiting = table
        self.manager.add(task)


    def getEmptyTask2(self, rdi):
        rdiAttr = rdi.getAttr()
        rdiArgs = (rdiAttr['table'], rdiAttr['schema'])
        fields = ", ".join(map(quote, self.getNonGeomFields(*rdiArgs)))
        geom = self.getGeometryField(*rdiArgs)
        rdiTotal = self.full(*rdiArgs)
        sqlExtract = "select * from "+rdiTotal+" limit 0"
        requete = "empty"
        method = None
        if rdi.cluster and self.needDump(*rdiArgs):
            sqlExtract = f"SELECT (ST_DUMP({quote(geom)})).geom as geom, {fields} FROM {rdiTotal} limit 0"
            requete += "_dump"
            method = 'dump'
        q = tmpQuery(self, sqlExtract)
        q.setOrigin(*rdiArgs, method=method)
        q.setPkey(self.getPrimaryKey(*rdiArgs))
        q.addBD(enjeu=rdiTotal, alea=rdiTotal, requete=requete)
        q.description = "empty " + quote(rdiAttr['name'])
        return q

    def getEmptyTask(self, rdi):
        rdiAttr = rdi.getAttr()
        rdiArgs = (rdiAttr['table'], rdiAttr['schema'])
        fields = ", ".join(map(quote, self.getNonGeomFields(*rdiArgs)))
        geom = self.getGeometryField(*rdiArgs)
        task = mGlobalTask("Extraction enjeu vide "+quote(rdiAttr['name']))
        rdiTotal = self.full(*rdiArgs)
        sqlExtract = "select * from "+rdiTotal+" limit 0"
        requete = "empty"
        method = None
        if rdi.cluster and self.needDump(*rdiArgs):
            sqlExtract = f"SELECT (ST_DUMP({quote(geom)})).geom as geom, {fields} FROM {rdiTotal} limit 0"
            requete += "_dump"
            method = 'dump'
        q = tmpQuery(self, sqlExtract)
        q.setOrigin(*rdiArgs, method=method)
        q.setPkey(self.getPrimaryKey(*rdiArgs))
        q.addBD(enjeu=rdiTotal, alea=rdiTotal, requete=requete)
        rdi.tableWaiting = q.table
        if not q.dispo:
            q.description = "empty " + quote(rdiAttr['name'])
            task.addSubTask(q, final=True)
        return task, q.table
        
        
    def getError(self, rdi):
        task, table = self.getEmptyTask(rdi)
        task.setCallback(rdi.affLayer, table, error=True)
        self.manager.add(task)         
    
    def getWaiting(self, rdi, *args, **kwargs):
        task, table = self.getEmptyTask(rdi)
        task.setCallback(rdi.affLayerWaiting, table, *args, **kwargs)
        self.manager.add(task)         

    def getExtractAlea(self, alea, callback=None, args=(), kwargs={}):
        aleaAttr = alea.getAttr()
        if self.context['simplify']>0:
            description = f"Simplification alea {quote(aleaAttr['name'])}"
            task = mGlobalTask(description)
            q = self.getSimplify(rdi=alea)
            alea.tableWaiting = q.table
            task.setCallback(alea.affLayer, q.table, callback=callback, args=args, kwargs=kwargs)
            if not q.dispo:
                q.description = f"Simplification/{quote(aleaAttr['name'])}"
                task.addSubTask(q, final=True)
            self.manager.add(task)
        elif self.context['maskWorking'] and self.context['maskTable'] is not None:
            description = f"Decoupage alea {quote(aleaAttr['name'])}"
            task = mGlobalTask(description)
            q = self.getInMask(rdi=alea)
            alea.tableWaiting = q.table
            task.setCallback(alea.affLayer, q.table, callback=callback, args=args, kwargs=kwargs)
            if not q.dispo:
                q.description = f"Decoupage/{quote(aleaAttr['name'])}"
                task.addSubTask(q, final=True)
            self.manager.add(task)
        else:
            alea.tableWaiting = aleaAttr['table']
            alea.affLayer(callback=callback, args=args, kwargs=kwargs) 
        alea.tamponWaiting = None
        alea.affTampon()
        if self.context['buffer']!=0:
            description = f"Extraction tampon {self.context['buffer']}m alea {quote(aleaAttr['name'])}"
            task = mGlobalTask(description)
            q = self.getTampon(rdi=alea)
            alea.tamponWaiting = q.table
            kwargs['buffer'] = True
            task.setCallback(alea.affTampon, q.table, callback=callback, args=args, kwargs=kwargs)
            if not q.dispo:
                q.description = f"Tampon/{quote(aleaAttr['name'])}"
                task.addSubTask(q, final=True)
            self.manager.add(task)

    def getExtractAleaAsync(self, alea, callback=None, args=(), kwargs={}):
        aleaAttr = alea.getAttr()
        if self.context['simplify']>0:
            tDescription = {'global':f"Simplification alea {quote(aleaAttr['name'])}", 'local':f"Simplification/{quote(aleaAttr['name'])}"}
            psql = self.clone()
            m = mGlobalTask(f"Preparation SQL")
            m.task.setCallin(psql.getSimplify, rdi=alea)
            m.setAutoKillConnexion(psql)
            m.setCallbackWithResult(self.endExtractAleaAsync, alea=alea, tDescription=tDescription, callback=callback, args=args, kwargs=kwargs)
            self.manager.add(m)
        elif self.context['maskWorking'] and self.context['maskTable'] is not None:
            tDescription = {'global':f"Decoupage alea {quote(aleaAttr['name'])}", 'local':f"Decoupage/{quote(aleaAttr['name'])}"}
            psql = self.clone()
            m = mGlobalTask(f"Preparation SQL")
            m.task.setCallin(psql.getInMask, rdi=alea)
            m.setAutoKillConnexion(psql)
            m.setCallbackWithResult(self.endExtractAleaAsync, alea=alea, tDescription=tDescription, callback=callback, args=args, kwargs=kwargs)
            self.manager.add(m)
        else:
            alea.tableWaiting = aleaAttr['table']
            alea.affLayerAsync(callback=callback, args=args, kwargs=kwargs) 
        alea.tamponWaiting = None
        alea.affTampon()
        if self.context['buffer']!=0:
            nkwargs = kwargs.copy()
            nkwargs['buffer'] = True
            tDescription = {'global':f"Extraction tampon {self.context['buffer']}m alea {quote(aleaAttr['name'])}", 'local':f"Tampon/{quote(aleaAttr['name'])}"}
            psql = self.clone()
            m = mGlobalTask(f"Preparation SQL")
            m.task.setCallin(psql.getTampon, rdi=alea)
            m.setAutoKillConnexion(psql)
            m.setCallbackWithResult(self.endExtractAleaAsync, alea=alea, tDescription=tDescription, callback=callback, args=args, kwargs=nkwargs)
            self.manager.add(m)

    def endExtractAleaAsync(self, q, alea, tDescription, callback=None, args=(), kwargs={}):
        if kwargs.get('buffer'):
            action = alea.affTamponAsync
            alea.tamponWaiting = q.table
        else:
            action = alea.affLayerAsync
            alea.tableWaiting = q.table
        aleaAttr = alea.getAttr()
        task = mGlobalTask(tDescription['global'])
        task.setCallback(action, q.table, callback=callback, args=args, kwargs=kwargs)
        if not q.dispo:
            q.description = tDescription['local']
            task.addSubTask(q, final=True)
        self.manager.add(task)


 
    def getFusion(self, rdi, list, mode, callback, args):
        description = f"Extraction emprise totale"
        task = mGlobalTask(description)
        tList = []
        for l in list:
            lAttr = l.getAttr() 
            q = None
            if mode=='alea':
                q = self.getTampon(rdi=l)
            if mode=='enjeu':
                q = self.getHull(rdi=l)
            if q!=None and not q.dispo:
                q.description = f"{quote(lAttr['name'])}"
                task.addSubTask(q)
            tList.append(q.table)
        sqlUnion = " UNION ".join(map(lambda x: f"SELECT * FROM {self.full(x)}", tList))
        if sqlUnion!="":
            sqlExtract = f"SELECT 1 as id, ST_UNION(geom) as geom FROM ({sqlUnion}) as foo"
            q = tmpQuery(self, sqlExtract)
            q.addBD(enjeu="", alea="+".join(tList), requete="fusion")  
            if not q.dispo:    
                task.addSubTask(q, final=True)
            if callback!=None:
                task.setCallback(callback, q.table, rdi, *args)
            self.manager.add(task)    
    
    def getFusionAlea(self, rdi, aleaList, callback=None, *args):
        self.getFusion(rdi, aleaList, 'alea', callback, args)
        
    def getFusionEnjeu(self, rdi, enjeuList, callback=None, *args):
        self.getFusion(rdi, enjeuList, 'enjeu', callback, args)

    
    
    def getFusionTask(self, list, mode):
        data = {'qEnveloppes':[], 'qExtract':None}
        tList = []
        for l in list:
            lAttr = l.getAttr() 
            q = None
            if mode=='alea':
                q = self.getTampon(rdi=l, force=False)
                q.description = f"Tampon sur {quote(lAttr['name'])}"
            if mode=='enjeu':
                q = self.getHull(rdi=l)
                q.description = f"Enveloppe sur {quote(lAttr['name'])}"
            if q!=None:
                data['qEnveloppes'].append(q)
                tList.append(q.table)
        sqlUnion = " UNION ".join(map(lambda x: f"SELECT * FROM {self.full(x)}", tList))
        if sqlUnion!="":
            sqlExtract = f"SELECT 1 as id, ST_UNION(geom) as geom FROM ({sqlUnion}) as foo"
            q = tmpQuery(self, sqlExtract)
            q.addBD(enjeu="", alea="+".join(tList), requete="fusion")  
            q.description = f"fusion sur {len(list)} {self.param.get(mode)}"
            data['qExtract'] = q
        return data

    def endFusionTask(self, result, rdi, callback=None, args=()):
        description = f"Extraction emprise totale"
        task = mGlobalTask(description)
        for q in result['qEnveloppes']:
            if not q.dispo:
                task.addSubTask(q)
        q = result['qExtract']
        if q is not None:
            if not q.dispo:    
                task.addSubTask(q, final=True)
            if callback!=None:
                task.setCallback(callback, q.table, rdi, *args)
            self.manager.add(task) 

    def getFusionAsync(self, rdi, aleaList, mode, callback=None, args=()):
        psql = self.clone()
        m = mGlobalTask(f"Preparation SQL")
        m.task.setCallin(psql.getFusionTask, list=aleaList, mode=mode)
        m.setAutoKillConnexion(psql)
        m.setCallbackWithResult(self.endFusionTask, rdi=rdi, callback=callback, args=args)
        self.manager.add(m)
    
    def getFusionAleaAsync(self, rdi, aleaList, callback=None, *args):
        self.getFusionAsync(rdi, aleaList, 'alea', callback, args)
        
    def getFusionEnjeuAsync(self, rdi, enjeuList, callback=None, *args):
        self.getFusionAsync(rdi, enjeuList, 'enjeu', callback, args)
    
    
    
    
    
    def deleteTable(self, table, schema=None):
        sql = f"DROP TABLE IF EXISTS {self.full(table, schema)}"
        query = self.sqlSend(sql)
    

    def nbTmp(self, callback, *args, **kwargs):
        c = 0
        schema = self.getSchema()
        sql = f"SELECT count(*) from {self.admin('tableTmp')}"
        query = self.sqlSend(sql)
        if query and query.next():
            c = query.value(0)
        else:
            c = 0
        s = "s" if c>1 else ""
        txt = f"{c} requête{s} stockée{s}"
        if callback:
            callback(txt, *args, **kwargs)
    
    def purgeAsync(self, callback, *args, **kwargs):
        c = 0
        prefixe = self.tmpPrefixe.replace("_","\_")
        schema = self.getSchema()
        sql = f"SELECT tablename FROM pg_catalog.pg_tables" + \
            f" WHERE schemaname='{schema}'" + \
            f" AND tablename LIKE ('{prefixe}%')" + \
            f" UNION " + \
            f" SELECT tablename FROM {self.admin('tableTmp')}"
        query = self.sqlSend(sql)
        sql = ""
        while query and query.next():
            table = query.value(0)
            sql += f"DROP TABLE IF EXISTS {self.full(table)} CASCADE;"
            c += 1
        sql += f"TRUNCATE {self.admin('tableTmp')} RESTART IDENTITY;"
        description = f"Purge des tables"
        task = mGlobalTask(description)
        s = "s" if c>1 else ""
        txt = f"{c} table{s} purgée{s}"
        task.setCallback(callback, txt, *args, **kwargs)
        task.addSubTaskSql(self, sql)
        self.manager.add(task)
        

    def purge(self, callback, *args, **kwargs):
        c = 0
        prefixe = self.tmpPrefixe.replace("_","\_")
        schema = self.getSchema()
        sql = f"SELECT tablename FROM pg_catalog.pg_tables" + \
            f" WHERE schemaname='{schema}'" + \
            f" AND tablename LIKE ('{prefixe}%')" + \
            f" UNION " + \
            f" SELECT tablename FROM {self.admin('tableTmp')}"
        query = self.sqlSend(sql)
        while query and query.next():
            table = query.value(0)
            sql = f"DROP TABLE IF EXISTS {self.full(table)} CASCADE;"
            self.db.exec_(sql)
            c += 1
        sql = f"TRUNCATE {self.admin('tableTmp')} RESTART IDENTITY;"
        query = self.sqlSend(sql)
        sql = f"TRUNCATE {self.admin('tableTmp')};"
        query = self.sqlSend(sql)
        s = "s" if c>1 else ""
        txt = f"{c} table{s} purgée{s}"
        if callback:
            callback(txt, *args, **kwargs)





class tmpQuery(abstractQuery):
    def __init__(self, psql, sqlExec):
        abstractQuery.__init__(self, psql)
        self.queryTimeout = self.psql.getParamServer('queryTimeout')
        self.stockTimeout = self.psql.getParamServer('stockTimeout')
        self.sqlExec = sqlExec
        self.adminTable = self.psql.admin('tableTmp')
        self.tmpPrefixe = self.psql.tmpPrefixe
        self.noWait = False
      
    def delete(self, table=None):
        self.psql.deleteTable(table)
        sql = f"DELETE FROM {self.adminTable} WHERE tablename='{table}'"
        query = self.psql.db.exec_(sql)
        
    def checkBD(self, enjeu, alea, requete):
        sql = f"SELECT tablename, dispo, " + \
            f" (DATE_PART('hour',localtimestamp-cree)*3600+DATE_PART('minute',localtimestamp-cree)*60+DATE_PART('second',localtimestamp-cree))" + \
            f" as delai FROM {self.adminTable}" + \
            f" WHERE alea='{alea}' AND enjeu='{enjeu}' AND requete='{requete}'" +\
            f" ORDER BY (case when dispo is null then 1 else 0 end), dispo desc, cree desc"
        query = self.psql.db.exec_(sql)
        while query.next():
            table = query.value(0)
            dispo = query.value(1)
            delai = query.value(2)
            if self.dispo:
                self.delete(table)
            else:
                if dispo and self.isTable(table):
                    if delai>self.stockTimeout:
                        self.delete(table)
                    else:
                        self.dispo = True
                        self.table = table
                else:
                    try:
                        m = self.psql.manager.waiting[self.full(table)]
                    except:
                        m = None
                    if delai>self.queryTimeout:
                        self.delete(table)
                        if m:
                            m.cancel()
                    else:
                        self.table = table
                        self.dependency = m
        
    def addBD(self, enjeu, alea, requete):
        self.checkBD(enjeu, alea, requete)
        self.tmp = self.tmpPrefixe + makeUniqueIdent()
        self.sqlPrepare = f" INSERT INTO {self.adminTable} (tablename, alea, enjeu, cree, requete, sql)" + \
            f" values ('{self.tmp}', '{alea}', '{enjeu}', localtimestamp, '{requete}', '{self.sqlExec}');"
        if not self.table:
            
            self.table = self.tmp

    def makeCreate(self):
        sql = f" CREATE TABLE {self.full()} AS ({self.sqlExec});"
        return sql
    
    def prepare(self):
        self.psql.db.exec_(self.sqlPrepare)
        
    def endCreate(self):
        sql = f" UPDATE {self.adminTable} SET dispo=timeofday()::timestamp WHERE tablename='{self.table}';"
        return sql
        
    def kill(self):
        sql = f" DELETE FROM {self.adminTable} WHERE tablename='{self.table}';"
        query = self.psql.db.exec_(sql)






class connDialog(QgsDialog):
    
    accepted = pyqtSignal(str, str)
    
    def __init__(self, psql):
        buttons = QtWidgets.QDialogButtonBox.Ok | QtWidgets.QDialogButtonBox.Cancel
        QgsDialog.__init__(self, iface.mainWindow(),
                            fl=Qt.WindowFlags(),
                            buttons=buttons,
                            orientation=Qt.Horizontal)
        self.setWindowTitle("Connexion à Postgres") 
        self.setWindowModality(Qt.ApplicationModal)
        self.psql = psql
        self.drawTitle()
        self.drawForm()
        self.show()
        
    def drawTitle(self):
        hbox = QtWidgets.QHBoxLayout(self)
        hbox.setSpacing(5)
        hbox.setContentsMargins(0,5,0,5)
        self.layout().addLayout(hbox)
        l = QtWidgets.QLabel(f"Identifiants de connexion pour ", self)
        hbox.addWidget(l)
        l = QtWidgets.QLabel(f"{self.psql.PGdb}", self)
        font = QtGui.QFont()
        font.setBold(True)
        l.setFont(font)
        hbox.addWidget(l)
        l = QtWidgets.QLabel(f" sur ", self)
        hbox.addWidget(l)
        l = QtWidgets.QLabel(f"{self.psql.PGhost}", self)
        font = QtGui.QFont()
        font.setBold(True)
        l.setFont(font)
        hbox.addWidget(l)
        
    def drawForm(self):
        form = QtWidgets.QFormLayout(self)
        form.setSpacing(5)
        self.layout().addLayout(form)
        wl = QtWidgets.QLabel("Login", self)
        wf = QtWidgets.QLineEdit(self)
        self.loginEdit = wf
        wf.setText(self.psql.PGuser)
        form.addRow(wl, wf)
        wl = QtWidgets.QLabel("Password", self)
        wf = QgsPasswordLineEdit(self)
        self.passwordEdit = wf
        form.addRow(wl, wf)
        
        l = QtWidgets.QLabel("", self)
        self.layout().addWidget(l)
        
    def login(self):
        return self.loginEdit.text()
        
    def password(self):
        return self.passwordEdit.text()
   
   

class serverDialog(QgsDialog):
    def __init__(self, psql):
        buttons = QtWidgets.QDialogButtonBox.Close
        QgsDialog.__init__(self, iface.mainWindow(),
                            fl=Qt.WindowFlags(),
                            buttons=buttons,
                            orientation=Qt.Horizontal)
        # self.setWindowModality(Qt.ApplicationModal)
        self.psql = psql
        self.setWindowTitle(f"Etat du serveur ({self.psql.PGhost})") 
        self.draw()
        self.show()
        self.getState()
        self.timer = QTimer(self)
        self.timer.timeout.connect(self.getState)
        self.timer.start(10000)
        self.finished.connect(self.close)
        
    def getState(self):
        dual = {'me':"Mes connexions", 'other':"Autres connexions"}
        self.dual = dual
        sql = f"SELECT (case when client_addr='{self.psql.myIP}' then '{dual['me']}' else '{dual['other']}' end) as me," + \
            f" application_name as app, " + \
            f" (case when xact_start is null then 'inactif' else 'actif' end) as status, " + \
            f" (DATE_PART('day', min(localtimestamp-state_change))*86400+DATE_PART('hour', min(localtimestamp-state_change))*3600+DATE_PART('minute', min(localtimestamp-state_change))*60+DATE_PART('second', min(localtimestamp-state_change))) as delay," + \
            f" count(*), array_agg(pid) as pids" + \
            f" FROM pg_stat_activity " + \
            f" WHERE query not like '%pg_stat_activity%' " + \
            f" AND datname='{self.psql.PGdb}' " + \
            f" GROUP BY 1, 2, 3" + \
            f" ORDER BY 1 desc, 3, 2"  
        description = f"Etat du serveur"
        task = mGlobalTask(description)
        task.addSubTaskSql(self.psql, sql, self.affState, getAll=True)
        self.psql.manager.add(task)       
    
    def affState(self, result):
        clearLayout(self.vbox)
        self.lines = {}  
        txt = ""
        if result is None:
            txt = f"Pas de connexion réseau"
        elif len(result)==0:
            txt = f"Aucun processus en cours"
        if txt!="":
            l = QtWidgets.QLabel(txt, self)
            font = QtGui.QFont()
            font.setBold(True)
            l.setFont(font)
            self.vbox.addWidget(l)
            return
        
        for l in result:
            self.drawLine(self.vbox, l)

    
    def draw(self):
        self.layout().setAlignment(Qt.AlignTop)
        w = QtWidgets.QWidget(self)
        hbox = QtWidgets.QHBoxLayout(w)
        hbox.setSpacing(5)
        hbox.setContentsMargins(0,5,0,0)
        self.layout().addWidget(w)
        l = QtWidgets.QLabel(f"Processus ouverts sur la base ", self)
        hbox.addWidget(l)
        l = QtWidgets.QLabel(f"{self.psql.PGdb}", self)
        font = QtGui.QFont()
        font.setBold(True)
        l.setFont(font)
        hbox.addWidget(l)
        ico = QtGui.QIcon(os.path.join(os.path.dirname(__file__),'images', "refresh.png"))
        butt = QtWidgets.QPushButton(ico, "", self)
        butt.setToolTip("Actualiser")
        butt.setMaximumWidth(30)
        hbox.addWidget(butt)
        butt.clicked.connect(self.getState)
        w = QtWidgets.QWidget(self)
        hbox = QtWidgets.QHBoxLayout(w)
        hbox.setContentsMargins(20,0,0,0)
        self.layout().addWidget(w)
        cb = QtWidgets.QCheckBox("Hard kill")
        hbox.addWidget(cb)
        # hbox.setAlignment(Qt.AlignRight)
        w = QtWidgets.QWidget(self)
        hbox = QtWidgets.QHBoxLayout(w)
        hbox.setContentsMargins(20,0,0,5)
        self.layout().addWidget(w)
        l = QtWidgets.QLabel("")
        font = QtGui.QFont()
        font.setItalic(True)
        l.setFont(font)
        hbox.addWidget(l)
        self.result = l
        hbox.setAlignment(Qt.AlignRight)
        self.hardkill = cb
        w = QtWidgets.QWidget(self)
        self.layout().addWidget(w)
        vbox = QtWidgets.QVBoxLayout(w)
        vbox.setAlignment(Qt.AlignTop)
        self.vbox = vbox
        
    def drawLine(self, layout, dico):
        app = dico.get("app")
        me = dico.get("me")
        c = dico.get("count")
        status = dico.get("status")
        delay = dico.get("delay")
        
        if me not in self.lines:
            if me!=self.dual['me']:
                l = QtWidgets.QLabel("", self)
                layout.addWidget(l)
            l = QtWidgets.QLabel("", self)
            font = QtGui.QFont()
            font.setBold(True)
            l.setFont(font)
            layout.addWidget(l)
            self.lines[me] = l
            self.lines[me].c = 0
        self.lines[me].c += c
        self.lines[me].setText(f"{me} ({self.lines[me].c}):")
        
        key = f"{me}_{status}"
        if key not in self.lines:
            l = QtWidgets.QLabel("", self)
            font = QtGui.QFont()
            font.setItalic(True)
            l.setFont(font)
            layout.addWidget(l)
            self.lines[key] = l
            self.lines[key].c = 0
        self.lines[key].c += c
        self.lines[key].setText(f"{status} ({self.lines[key].c}):")
        
        hbox = QtWidgets.QHBoxLayout(self)
        hbox.setContentsMargins(10,0,0,0)
        layout.addLayout(hbox)
        l = QtWidgets.QLabel(f'{app} ({niceDuration(delay, precision=1)}) : {c}', self)
        hbox.addWidget(l)
        ico = QtGui.QIcon(os.path.join(os.path.dirname(__file__),'images', "stop.png"))
        butt = QtWidgets.QPushButton(ico, "", self)
        butt.setToolTip("Demande d'interruption pour ces transactions")
        butt.setMaximumWidth(30)
        hbox.addWidget(butt)
        butt.dico = dico
        butt.clicked.connect(partial(self.kill, butt))
        if me!=self.dual['me']:
            butt.setEnabled(False)

    def kill(self, butt, pids):
        butt.setEnabled(False)
        self.result.setText("")
        pids = butt.dico.get('pids').replace("{","(").replace("}",")")
        action = 'cancel'
        if self.hardkill.isChecked():
            action = 'terminate'
        sql = f"SELECT pg_{action}_backend(pid) from pg_stat_activity WHERE pid IN {pids}"
        description = f"Kill process"
        task = mGlobalTask(description)
        task.addSubTaskSql(self.psql, sql)
        task.setCallback(self.endKill)
        self.psql.manager.add(task)
        
    def endKill(self):
        self.result.setText("Demande envoyée")
        self.getState()
        
    def close(self, result):
        self.timer.stop()
        
        
        

class paramDialog(QgsDialog):
    def __init__(self, psql):
        buttons = QtWidgets.QDialogButtonBox.Close
        QgsDialog.__init__(self, iface.mainWindow(),
                            fl=Qt.WindowFlags(),
                            buttons=buttons,
                            orientation=Qt.Horizontal)
        # self.setWindowModality(Qt.ApplicationModal)
        self.psql = psql
        self.param = rdiParam()
        self.pdb = rdiBaseParam(psql)
        self.setWindowTitle(f"Paramètres communs")
        self.code = "param"
        self.listWidget = []
        self.draw()
        self.show()
        
    def draw(self):
        self.layout().setAlignment(Qt.AlignTop)
        w = QtWidgets.QWidget(self)
        hbox = QtWidgets.QHBoxLayout(w)
        hbox.setSpacing(5)
        hbox.setContentsMargins(0,5,0,0)
        self.layout().addWidget(w)
        cb = QtWidgets.QCheckBox("Utiliser les paramètres du serveur", self)
        hbox.addWidget(cb)
        cb.toggled.connect(partial(self.setParam, cb))
        cb.key = 'shareParam'
        self.listWidget.append(cb)
        w = QtWidgets.QWidget(self)
        self.layout().addWidget(w)
        self.widgetForm = w
        form = QtWidgets.QFormLayout(w)
        form.setSpacing(5)
        
        for key, type in self.param.serverParam.items():
            val = str(self.param.get(key))
            wl = QtWidgets.QLabel(key, self)
            wf = None
            if type=='delay':
                wf = rdiEditDuration(val, self)
                wf.setLabel(wl)
            if type=='string':
                wf = QtWidgets.QLineEdit(val, self)
            if wf is not None:
                wf.editingFinished.connect(partial(self.setParam, wf))
                wf.key = key
                self.listWidget.append(wf)
            form.addRow(wl, wf)

        for w in self.listWidget:
            self.majParam(w)
        
    def setParam(self, w):
        val = None
        if isinstance(w, QtWidgets.QCheckBox):
            val = 1 if w.isChecked() else 0
            self.widgetForm.setEnabled(w.isChecked())
        if isinstance(w, rdiEditDuration):
            val = w.get()
        elif isinstance(w, QtWidgets.QLineEdit):
            val = w.text()
        if isinstance(w, QtWidgets.QSpinBox):
            val = w.value()
        if val is not None:
            self.pdb.set(w.key, code=self.code, value=str(val))

    def majParam(self, w):
        val = self.pdb.get(w.key, code=self.code, file=False)
        if val is None:
            self.setParam(w)
            return
        w.blockSignals(True)
        if isinstance(w, QtWidgets.QCheckBox):
            v = bool(int(val))
            w.setChecked(v)
            self.widgetForm.setEnabled(w.isChecked())
        if isinstance(w, rdiEditDuration):
            w.set(val)
        elif isinstance(w, QtWidgets.QLineEdit):
            if val is not None:
                w.setText(str(val))
        if isinstance(w, QtWidgets.QSpinBox):
            if val is not None:
                w.setValue(int(val))                    
        w.blockSignals(False)
        

            
            
class tableDialog(QgsDialog):
    def __init__(self, psql, title, table):      
        buttons = QtWidgets.QDialogButtonBox.Close
        QgsDialog.__init__(self, iface.mainWindow(),
                            fl=Qt.WindowFlags(),
                            buttons=buttons,
                            orientation=Qt.Horizontal)
        # self.setWindowModality(Qt.ApplicationModal)
        self.setSizeGripEnabled(True)
        self.psql = psql
        self.table = table
        self.setWindowTitle(title)
        self.draw()
        self.show()
        
    def draw(self):
        self.layout().setAlignment(Qt.AlignTop)
        # self.model = QSqlTableModel(self, self.psql.db)
        # self.model.setTable(self.psql.admin(self.table))
        # self.model.setEditStrategy(QSqlTableModel.OnManualSubmit)
        self.model = QSqlQueryModel(self)
        self.refresh()
        
        view = QtWidgets.QTableView(self)
        view.setSortingEnabled(True)
        view.setMinimumWidth(600)
        view.setMinimumHeight(400)
        view.setModel(self.model)
        self.layout().addWidget(view)
        
        view.horizontalHeader().sortIndicatorChanged.connect(self.sort)
        
        refreshButton = QtWidgets.QPushButton("Actualiser")
        refreshButton.clicked.connect(self.refresh)
        self.buttonBox().addButton(refreshButton, QtWidgets.QDialogButtonBox.ActionRole)

    def menu(self):
        self.popMenu = QtWidgets.QMenu()
        self.popMenu.addAction(QtGui.QIcon(os.path.join(self.imageFolder, "refresh.png")), "Actualiser", self.refresh)
        self.popMenu.popup(QtGui.QCursor.pos())
    
    def refresh(self, col=None, s=None):
        # self.model.select()
        # print(self.model.lastError().text())
        sql = f"SELECT * FROM {self.psql.admin(self.table)}"
        if col is not None:
            sql += f" ORDER BY {self.model.headerData(col, Qt.Horizontal)}"
        if s==1:
            sql += " DESC"
        self.model.setQuery(sql, self.psql.db)
        
    def sort(self, *v):
        self.refresh(*v)


class gestionDialog(tableDialog):
    def __init__(self, psql):
        tableDialog.__init__(self, psql, "Gestion des données", "tableAdmin")
        
        
class stockageDialog(tableDialog):
    def __init__(self, psql):
        tableDialog.__init__(self, psql, "Requêtes stockées", "tableTmp")
        
class shareDialog(tableDialog):
    def __init__(self, psql):
        tableDialog.__init__(self, psql, "Données partagées", "tablePar")