# -*- coding: utf-8 -*-

import os, logging, json, time, sys
from functools import partial

from qgis.gui import *
from qgis.core import *
from PyQt5 import QtGui, QtWidgets, uic
from PyQt5.QtCore import *
from qgis.utils import iface

from .rdi_tools import clearLayout
from .rdi_meta import rdiMeta
from .rdi_param import rdiParamBloc

class rdiMap():
    def __init__(self, parent):
        self.dock = parent
        self.meta = rdiMeta()
        self.layers = {}
        self.sliders = {}
        QgsProject.instance().layerWasAdded.connect(self.actuComboChoice)
        # QgsProject.instance().layersRemoved.connect(self.actuButtChoice)
        
    def unload(self):
        QgsProject.instance().layerWasAdded.disconnect(self.actuComboChoice)
        # QgsProject.instance().layersRemoved.disconnect(self.actuButtChoice)
    
    def paint(self, layout):
        l = QtWidgets.QLabel()
        layout.addWidget(l)
        gb = rdiParamBloc("par_Map", "Fond de plan", self.dock, memo=False)
        layout.addWidget(gb)
        elts = [
            {
                'help': 'p5-option-back',
                'description': f"Permet de mémoriser un fond de plan qui sera automatiquement chargé au démarrage du plugin",
                'ini': 'backMap',
                'func': ['self.map.affMap()'],
                'tab': ['parameter'],
                'title': True,
            },
        ]
        gb.add(elts, self.dock.changeParameter)
        self.layout = gb.getLayout()
        
        backMap = self.meta.readIni("backMap")
        if isinstance(backMap, list) and len(backMap)>2:
            backMap = self.meta.readIni("backMapList")
            if not isinstance(backMap, list): backMap = []
            backMap.append(self.meta.readIni("backMap"))
            self.meta.writeIni("backMapList", backMap)
            self.meta.delIni("backMap")
            
        w = QtWidgets.QWidget()
        self.layout.addWidget(w)
        v = QtWidgets.QVBoxLayout(w)
        v.setContentsMargins(0,0,0,0)
        self.liste = w
        
        w = QtWidgets.QFrame()
        w.setFrameShape(QtWidgets.QFrame.Shape.HLine)
        w.setFrameShadow(QtWidgets.QFrame.Shadow.Sunken)
        self.layout.addWidget(w)

        h = QtWidgets.QHBoxLayout()
        h.setContentsMargins(0,0,0,0)
        h.setSpacing(0)
        form = QtWidgets.QFormLayout()
        self.layout.addLayout(form)
        form.addRow("Ajouter ",h)
        h.setAlignment(l, Qt.AlignLeft)
        self.combo = QgsMapLayerComboBox()
        self.combo.currentIndexChanged.connect(self.actuButtChoice)
        h.addWidget(self.combo)
        self.combo.line = h
        
        ico = QtGui.QIcon(os.path.join(self.dock.imageFolder, "add.png"))
        butt = QtWidgets.QPushButton(ico, "")
        butt.setToolTip("Ajouter")
        butt.setMaximumWidth(30)
        h.addWidget(butt)
        butt.clicked.connect(self.addBackMap)
        self.combo.add = butt
        
        self.combo.setFilters(QgsMapLayerProxyModel.RasterLayer)
        
        self.actuComboChoice()
        self.actuButtChoice()
        self.listChoice()
        
    def getBackMap(self):
        backMap = self.meta.readIni("backMapList")
        if not isinstance(backMap, list): backMap = []
        return backMap
        
    def setBackMap(self, add=None, supp=None, op=None):
        backMap = self.getBackMap()
        if add is not None: backMap.append(add)
        if supp is not None: backMap.pop(supp)
        if op is not None: backMap[op[0]][3] = op[1]
        self.meta.writeIni("backMapList", backMap)
    
    def actuComboChoice(self, *v):    
        isRaster = len(v)==0
        for l in v:
            if isinstance(l, QgsRasterLayer): isRaster = True
        if not isRaster: return
        lst = []
        doublon = []
        uris = [bm[2] for bm in self.getBackMap()]
        for n,l in QgsProject.instance().mapLayers().items():
            if not isinstance(l, QgsRasterLayer): continue
            uri = l.dataProvider().dataSourceUri()
            if uri in doublon: lst.append(l)
            if uri not in doublon: doublon.append(uri)     
            if uri in uris: lst.append(l)
        self.combo.setExceptedLayerList(lst)
    
    def actuButtChoice(self):
        self.combo.add.setEnabled(self.combo.count()>0)
    
    def setLabel(self, wOri, wDest):
        op = wOri.value()
        wDest.setText(f"{op}%")
        try: 
            l = self.layers[wOri.uri]
            self.setOpacity(l, op/100)
        except: pass
     
    def listChoice(self):
        lay = self.liste.layout()
        clearLayout(lay)
        backMap = self.meta.readIni("backMapList")
        hbox = QtWidgets.QHBoxLayout()
        lay.addLayout(hbox)
        l = QtWidgets.QLabel()
        font = QtGui.QFont()
        font.setItalic(True)
        l.setFont(font)
        hbox.addWidget(l)
        if not isinstance(backMap, list) or len(backMap)==0:
            l.setText("Aucun fond mémorisé")
            return
        s = "s" if len(backMap)>1 else ""
        l.setText(f"{len(backMap)} fond{s} mémorisé{s}")
        ico = QtGui.QIcon(os.path.join(self.dock.imageFolder,'refresh.png'))
        butt = QtWidgets.QPushButton(ico, '')
        butt.setToolTip("Actualiser")
        butt.setMaximumWidth(20)
        hbox.addWidget(butt)
        butt.clicked.connect(self.affMap)
        self.sliders = {}
        for i,bm in enumerate(backMap):
            h = QtWidgets.QHBoxLayout()
            lay.addLayout(h)
            ico = QtGui.QIcon(os.path.join(self.dock.imageFolder, "delete.png"))
            b = QtWidgets.QPushButton(ico, "")
            b.setMaximumWidth(20)
            b.clicked.connect(partial(self.delBackMap, i))
            h.addWidget(b)
            l = QtWidgets.QLabel(f"{bm[0]}")
            h.addWidget(l)
            w = QtWidgets.QSlider(Qt.Horizontal)
            w.ind = i
            w.uri = bm[2]
            self.sliders[bm[2]] = w
            w.setRange(0, 100)
            h.addWidget(w)
            l = QtWidgets.QLabel()
            h.addWidget(l)
            w.valueChanged.connect(partial(self.setLabel, w, l))  
            op = self.getOpacity(bm)
            w.setValue(op)
            w.sliderReleased.connect(partial(self.saveOpacity, w))
            self.slider = w     
    
    def delBackMap(self, i):
        self.setBackMap(supp=i)
        self.actuComboChoice()
        self.listChoice()
        self.affMap()
        
    def addBackMap(self):
        layer = self.combo.currentLayer()
        provider = layer.dataProvider()
        uri = provider.dataSourceUri()
        providerType = provider.name()
        layername = layer.name()
        self.setBackMap(add=[layername, providerType, uri, 100])
        self.actuComboChoice()
        self.listChoice()
        self.affMap()
    
    def getOpacity(self, backMap):
        try: op = int(backMap[3])
        except: op = 100
        return op
        
    
    def setOpacityBM(self, bm):
        op = self.getOpacity(bm)/100
        l = self.layers[bm[2]]
        self.setOpacity(l, op)
    def setOpacity(self, l, op):
        try: l.layer.layer().setOpacity(op)
        except:
            try: l.layer.layer().renderer().setOpacity(op)
            except: pass
        l.layer.layer().triggerRepaint()
     
    def actuSlider(self, l):
        try:
            try: opacity = l.layer.layer().opacity()
            except: opacity = l.layer.layer().renderer().opacity()
            op = int(opacity*100)
            self.sliders[l.uri].setValue(op)
        except:
            pass
    
    def saveOpacity(self, w):
        self.setBackMap(op=(w.ind,w.value()))
        bm = self.getBackMap()[w.ind]
        self.setOpacityBM(bm)
    
    def affMap(self):
        if not self.dock.param.get('backMap'):
            for uri,l in self.layers.items():
                self.dock.dyn.removeMapLayer(l)
            self.layers = {}
            return
        for uri,l in list(self.layers.items()):
            if l.bm not in self.getBackMap():
                self.dock.dyn.removeMapLayer(l)
                del self.layers[uri]
        for bm in self.getBackMap():
            uri = bm[2]
            if uri not in self.layers:
                l = QObject()
                l.uri = uri
                self.layers[uri] = l
            else: l = self.layers[uri]
            l.vlayer = QgsRasterLayer(uri, bm[0], bm[1])
            l.layer = self.dock.dyn.addInGroup(l, 'background')
            self.setOpacityBM(bm)
            l.layerId = l.layer.layerId()
            l.vlayer.repaintRequested.connect(partial(self.actuSlider,l))
            l.bm = bm
        self.dock.zoomIni.go()


class rdiZoom():
    def __init__(self, parent):
        self.dock = parent
        self.meta = rdiMeta()
        self.layer = None
        self.slider = None
        self.timer = QTimer()
        self.timer.setSingleShot(True)
        self.timer.timeout.connect(self.zoom)
        
    def paint(self, layout):
        l = QtWidgets.QLabel()
        layout.addWidget(l)
        # gb = QtWidgets.QGroupBox("Zoom initial")
        gb = rdiParamBloc("par_Zoom", "Zoom initial", self.dock, memo=False)
        layout.addWidget(gb)
        self.layout = gb.getLayout()
        zoomIni = self.meta.readIni("zoomIni")
        if isinstance(zoomIni, list) and len(zoomIni)>3:
            self.showChoice(zoomIni)
        else:
            self.askChoice()
            
    def showChoice(self, zoomIni):
        lgb = self.layout
        clearLayout(lgb)
        
        w = QtWidgets.QWidget()
        lgb.addWidget(w)
        hbox = QtWidgets.QHBoxLayout(w)
        hbox.setSpacing(0)
        hbox.setContentsMargins(0,0,0,0)
        
        l = QtWidgets.QLabel(f"[{zoomIni[0]}   {zoomIni[1]}   {zoomIni[2]}   {zoomIni[3]}]")
        hbox.addWidget(l)
        
        ico = QtGui.QIcon(os.path.join(self.dock.imageFolder,'zoom.png'))
        butt = QtWidgets.QPushButton(ico, '')
        butt.setToolTip("Zoomer sur l'emprise")
        butt.setMaximumWidth(20)
        hbox.addWidget(butt)
        butt.clicked.connect(partial(self.zoom, force=True))
        
        butt = QtWidgets.QPushButton("Supprimer")
        butt.setFixedSize(130, 30)
        lgb.addWidget(butt)
        lgb.setAlignment(butt, Qt.AlignCenter)
        butt.clicked.connect(self.deleteZoomIni)
        
    def go(self):
        self.timer.start(1000)
    
    def zoom(self, force=False):
        if len(self.dock.alea.getEltsSelected())==0 and self.dock.mask.table is None or force:
            box = self.meta.readIni("zoomIni")
            if isinstance(box, list) and len(box)>3:
                rect = QgsRectangle(QRectF(QPointF(float(box[0]), float(box[1])), QPointF(float(box[2]), float(box[3]))))
                self.dock.zoomExtent(rect)
            else:
                self.dock.askZoom()
    
    def askChoice(self):
        lgb = self.layout
        clearLayout(lgb)
        l = QtWidgets.QLabel("Zoom automatique sur l'emprise des couches")
        lgb.addWidget(l)
        font = QtGui.QFont()
        font.setItalic(True)
        l.setFont(font)
        butt = QtWidgets.QPushButton("Emprise actuelle")
        butt.setFixedSize(130, 30)
        lgb.addWidget(butt)
        lgb.setAlignment(butt, Qt.AlignCenter)
        butt.clicked.connect(self.setZoomIni)
        
    def setZoomIni(self):
        rect = iface.mapCanvas().extent()
        tl = rect.toRectF().topLeft()
        br = rect.toRectF().bottomRight()
        coords = [round(tl.x()), round(tl.y()), round(br.x()), round(br.y())]
        # coords = [tl.x(), tl.y(), br.x(), br.y()]
        self.meta.writeIni("zoomIni", coords)
        self.showChoice(coords)
        
    def deleteZoomIni(self):
        self.meta.writeIni("zoomIni", "")
        self.askChoice()