# -*- coding: utf-8 -*-

import os, json, time, sys, traceback
from functools import partial

from qgis.core import *
from PyQt5 import QtGui, QtWidgets, uic
from PyQt5.QtCore import *

from .rdi_tools import rdiTimer, flashMsg, quote, clickLabel, dblClickLabel, makeUniqueIdent
from .rdi_async import mGlobalTask
from .rdi_param import rdiBlocTitle

class rdiLayer(QObject):
    
    clicked = pyqtSignal()
    changed = pyqtSignal()
    
    def __init__(self, meta=None, code=None):
        QObject.__init__(self)
        self.vlayer = None
        self.layer = None
        self.style = None
        self.stat = None
        self.cb = None
        self.combo = None
        self.code = None
        self.checked = None
        self.freeze = False
        self.cluster = None
        self.needBack = None
        self.isStat = None
        self.inverted = False
        self.loaded = False
        self.widget = None
        self.showed = True
        self.icoPaint = None
        self.field = None
        self.icoList = {}
        self.activeTable =None
        self.isRenamed = False
        self.isPainted = False
        self.dbEnv = {}
        self.params = {}
        self.tampon = rdiTampon(self)
        self.light = rdiLight(self)
        self.search = []
        self.name = "no name"
        self.targetBloc = None
        self.setMeta(meta)
     
    def setMeta(self, meta):
        self.meta = meta
        self.parent()
        self.id()
        self.setName()
        self.findKeySearch()
        self.setTimer()
        self.context = None
        self.tableWaiting = self.tableName()
        
    def cbClicked(self):
        pass
        
    def blockSignals(self, bool):
        if self.cb!=None:
            self.cb.blockSignals(bool)
        if self.combo!=None:
            self.combo.blockSignals(bool)
    
    def recup(self, layer):
        self.blockSignals(True)    
        self.cb.setChecked(True)
        self.checked = True
        dock = self.coll.dock
        self.cluster = dock.param.get('cluster')
        if self.combo:
            for i in range(self.combo.count()):
                if self.combo.itemData(i).id()==layer.rdiId:
                    self.combo.setCurrentIndex(i)
                    break
        self.layer = layer.rdi.layer
        self.vlayer = layer
        self.tampon = layer.rdi.tampon
        self.light = layer.rdi.light
        self.layerId = layer.rdi.layerId
        self.style = layer.rdi.style
        self.inverted = layer.rdi.inverted
        try:
            self.field.memo = layer.rdi.field.memo
        except:
            pass
        self.style.rdi = self
        if layer.rdi.icoPaint:
            self.manualPainting()
        
        layer.rdi = self
        self.invertAff()
        
        self.blockSignals(False)    
    
    def getValueFromMeta(self, key, fallback=None):
        val = fallback
        try:
            v = self.meta[key]
            if v:
                val = v
        except:
            val = fallback
        return val
    
    def getJsonFromMeta(self, key):
        try: data = json.loads(self.getValueFromMeta(key))
        except: data = {}
        return data
    
    def setName(self):
        name = str(self.getValueFromMeta('tablename', 'error'))
        name = str(self.getValueFromMeta('name', name))
        try:
            name = self.statName
        except:
            pass
        self.name = name
    
    def deleteLayer(self):
        self.coll.remove(self)
        
    def renameLayer(self):
        dock = self.coll.dock
        self.cb.hide()
        for k,w in self.icoList.items():
            w.hide()
        w = QtWidgets.QLineEdit(self.cb.text())
        self.layout.addWidget(w)
        self.renameEdit = w
        
        if self.combo:
            w = QtWidgets.QLineEdit(self.combo.currentText())
            self.layout.addWidget(w)
            w.returnPressed.connect(partial(self.renameLayerCommit, True))
            self.renameEdit.combo = w
            self.renameEdit.returnPressed.connect(w.setFocus)
            self.renameEdit.combo.returnPressed.connect(partial(self.renameLayerCommit, True))
        else:
            self.renameEdit.returnPressed.connect(partial(self.renameLayerCommit, True))
        
        ico = QtGui.QIcon(os.path.join(dock.imageFolder, "ok.png"))
        b = QtWidgets.QPushButton(ico, "")
        b.setToolTip("Valider")
        b.setFixedSize(22, 22)
        b.clicked.connect(partial(self.renameLayerCommit, True))
        self.layout.addWidget(b)
        self.renameEdit.ok = b
        ico = QtGui.QIcon(os.path.join(dock.imageFolder, "back.png"))
        b = QtWidgets.QPushButton(ico, "")
        b.setToolTip("Annuler")
        b.setFixedSize(22, 22)
        b.clicked.connect(partial(self.renameLayerCommit, False))
        self.layout.addWidget(b)
        self.renameEdit.ko = b
        self.renameEdit.setFocus()

    def renameLayerCommit(self, bool=False):
        if bool:
            self.isRenamed = True
            self.setCustom()
            name = self.renameEdit.text()
            self.name = name
            self.cb.setText(name)
            if self.combo:
                name2 = self.renameEdit.combo.text()
                self.combo.setItemText(self.combo.currentIndex(), name2)
                name = f"{name} ({name2})"
            if self.layer:
                self.layer.setName(name)
        try:
            self.layout.removeWidget(self.renameEdit)
            self.renameEdit.setParent(None)
            if self.combo:
                self.layout.removeWidget(self.renameEdit.combo)
                self.renameEdit.combo.setParent(None)
            self.layout.removeWidget(self.renameEdit.ok)
            self.renameEdit.ok.setParent(None)
            self.layout.removeWidget(self.renameEdit.ko)
            self.renameEdit.ko.setParent(None)
        except:
            # print(sys.exc_info())
            pass
        self.cb.show()
        for k,w in self.icoList.items():
            w.show()

    def renameLayerDefault(self):
        try:
            self.isRenamed = False
            self.setCustom()
            self.setName()
            self.cb.setText(self.name)
            if self.combo:
                # rdi = self.combo.currentData()
                # rdi.setName()
                # self.combo.setItemText(self.combo.currentIndex(), rdi.name)
                for i in range(self.combo.count()):
                    rdi = self.combo.itemData(i)
                    rdi.setName()
                    self.combo.setItemText(i, rdi.name) 
            if self.layer:
                self.layer.setName(self.layerName())
        except:
            print(sys.exc_info())
            pass
        
    def showTab(self):
        if self.vlayer:
            self.coll.dock.showTab(self.vlayer)        
            
    def showProp(self):
        if self.vlayer:
            self.coll.dock.showProp(self.vlayer)    
        
    def newLine(self, layout):
        self.bloc = layout
        wl = QtWidgets.QWidget()
        layout.addWidget(wl)
        hbox = QtWidgets.QHBoxLayout()
        hbox.setSpacing(0)
        hbox.setContentsMargins(0,0,0,0)
        try:
            bp = self.coll.dock.param.get('blocDrop')
            b = self.getValueFromMeta('bloc')
            if bp and self.targetBloc:
                hbox.setContentsMargins(10,0,0,0)
        except:
            pass
        wl.setLayout(hbox)
        self.layout = hbox
        self.widget = wl
        self.widgetName = f"layer_{makeUniqueIdent()}"
        self.widget.setObjectName(self.widgetName)
    
    def addIco(self, context):
        l = rdiLayerLabel(os.path.join(self.coll.dock.imageFolder, context.get('ico')))
        l.setToolTip(context.get('tip'))
        self.layout.addWidget(l)
        l.leftClick.connect(context.get('action'))
        self.icoList[context.get('key')] = l
    
    def addCb(self, layout):
        self.newLine(layout)
        l = rdiLayerLabel(os.path.join(self.coll.dock.imageFolder, "invert.png"))
        self.layout.addWidget(l)
        self.icoInverted = l
        self.invertAff()
        name = self.name
        if self.parent()!=None:
            try: name = f"{self.coll.findParent(self).name}  ({name})"
            except: pass
        if not self.coll.dock.param.get('blocDrop'):
            bloc = self.getValueFromMeta('bloc')
            if bloc and bloc!="" and self.getValueFromMeta('tablename'):
                name = f"[{bloc}] {name}"
        self.cb = rdiLayerCb(name)
        self.layout.addWidget(self.cb)
        # self.cb.stateChanged.connect(self.setLayer)
        self.cb.toggled.connect(self.setLayer)
        self.cb.rightClick.connect(self.affMenuLayer)
        l.leftClick.connect(self.cb.click)
        l.rightClick.connect(self.affMenuLayer)
        self.icoPaint = None
        self.field = rdiLayerField(self) 
        if hasattr(self, 'coll') and not self.coll.dock.param.get('cbCompact') and self.parent()!=None:
            try: self.addKeySearch(self.coll.findParent(self).name.lower())
            except: pass

    def makeIco(self, name):
        path = os.path.join(self.coll.dock.imageFolder, name)
        ico = QtGui.QIcon(path)
        return ico
    
    def affMenuLayer(self):
        self.popMenu = QtWidgets.QMenu()
        self.popMenu.setToolTipsVisible(True)
        self.popMenu.addSection(self.layerName())
        self.popMenu.addAction(self.makeIco("modify.png"), "Renommer", self.renameLayer)
        
        a = self.popMenu.addAction(self.makeIco("paint.png"), "Propriétés", self.showProp)
        b = self.popMenu.addAction(self.makeIco("table.png"), "Afficher les données", self.showTab)
        c = self.popMenu.addAction(self.makeIco("zoom.png"), "Zoomer sur", self.zoomOn)
        if not self.checked or not self.vlayer or self.wait:
            a.setEnabled(False)
            b.setEnabled(False)
            c.setEnabled(False)
        
        if self.coll.matchCode('enjeu'):
            self.popMenu.addSeparator()
            lib = "Inverser"
            ico = "invert.png"
            if self.inverted:
                lib = "Rétablir"
                ico = "invert-ko.png"
            a = self.popMenu.addAction(self.makeIco(ico), lib, self.invert)
            a.setToolTip(f"Permet d'inverser le résultat issu du croisement spatial\n   (enjeux dans l'emprise OU enjeux hors emprise)")
            # if not self.coll.dock.param.get('inversion'):
                # a.setEnabled(False)
            if self.field.active and self.field.isPoint():
                lib = "Somme sur attribut"
                ico = "sum.png"
                if self.field.on:
                    lib = "Compter les entités"
                    ico = "sum-ko.png"
                a = self.popMenu.addAction(self.makeIco(ico), lib, self.field.toggle)
                a.setToolTip(f"Modifie l'affichage du rendu cluster")
        
        if self.coll.matchCode('stats'):
            self.popMenu.addAction(self.makeIco("print.png"), "Imprimer", partial(self.stat.printStat, self))
            
        if self.coll.matchCode('link') and self.coll.dock.param.get("admin"):
            self.popMenu.addSeparator()
            data = self.getJsonFromMeta('link')
            if data.get('start'): ico = self.makeIco("check.png")
            else: ico = QtGui.QIcon()
            self.popMenu.addAction(ico, "Démarrage", partial(self.changeLinkAttr, 'start', not data.get('start')))
            m = self.popMenu.addMenu("Nom affiché")
            for v in self.coll.dock.psql.getColumnsTable(self.tableName(), self.schemaName()):
                if data.get('col')==v: ico = self.makeIco("check.png")
                else: ico = QtGui.QIcon()
                m.addAction(ico, v, partial(self.changeLinkAttr, 'col', v))
                
            m = self.popMenu.addMenu("trié par")
            for k,v in self.coll.dock.link.linkOrderOptions.items():
                if data.get('order')==k: ico = self.makeIco("check.png")
                else: ico = QtGui.QIcon()
                m.addAction(ico, v, partial(self.changeLinkAttr, 'order', k))    
                
        
        if self.isCustom():
            self.popMenu.addSeparator()
            if self.isRenamed:
                self.popMenu.addAction(self.makeIco("modify-ko.png"), "Rétablir nom d'origine", self.renameLayerDefault)
            if self.isPainted:
                self.popMenu.addAction(self.makeIco("paint-ko.png"), "Supprimer style manuel", self.cancelManualPainting)
        
        if self.coll.matchCode('stats'):
            self.popMenu.addSeparator()
            self.popMenu.addAction(self.makeIco("delete.png"), "Supprimer", self.deleteLayer)
        elif self.coll.dock.param.get("admin"):
            self.popMenu.addSeparator()
            self.popMenu.addAction(self.makeIco("database.png"), "Administrer", self.goToAdmin)    
        
        self.highLight(state=True)
        self.popMenu.aboutToHide.connect(self.highLight)
        self.popMenu.popup(QtGui.QCursor.pos())
    
    def changeLinkAttr(self, attr, val):
        data = self.getJsonFromMeta('link')
        data[attr] = val
        dump = json.dumps(data)
        sql = f'update {self.coll.dock.psql.admin("tableAdmin")} set "link"=\'{dump}\' where "id"={self.id()}'
        self.coll.dock.psql.sqlSend(sql)
        self.meta['link'] = dump
    
    def highLight(self, state=False):
        if state:
            self.widget.setStyleSheet(f"QWidget#{self.widgetName} {{border: 1px dotted #000000; background-color: #e8e8e8;}} ");
        else:
            self.widget.setStyleSheet(f"");
    
    def isCustom(self):
        bool = False
        bool = bool or self.isPainted
        bool = bool or self.isRenamed
        return bool
        
    def setCustom(self):  
        bool = self.isCustom()
        font = QtGui.QFont()
        font.setItalic(bool)
        # font.setUnderline(bool)
        self.cb.setFont(font)
    
    def zoomOn(self):
        if self.vlayer:
            self.coll.dock.zoomExtent(self.vlayer.extent())
    
    def invert(self, bool=None):
        if self.coll.matchCode('enjeu'):
            if bool is not None:
                self.inverted = bool
            else:
                self.inverted = not self.inverted
            self.invertAff()
            self.setLayer()
    
    def invertAff(self):
        if self.inverted:
            self.icoInverted.show()
        else:
            self.icoInverted.hide()
    
    def addCombo(self, rdi):
        if not self.combo:
            self.combo = QtWidgets.QComboBox()
            self.layout.addWidget(self.combo)
            self.comboList = []
            self.combo.currentIndexChanged.connect(self.setLayer)    
            self.combo.currentIndexChanged.connect(self.changed.emit)
            self.icoList['combo'] = self.combo
        self.combo.blockSignals(True)
        self.comboList.append(rdi)    
        name = rdi.name
        if not self.coll.dock.param.get('blocDrop'):
            bloc = rdi.getValueFromMeta('bloc')
            if bloc and bloc!="":
                name = f"[{bloc}] {name}"
        self.combo.addItem(name, rdi)
        self.combo.blockSignals(False)
        for key in rdi.search:
            self.addKeySearch(key)
            
    def adaptBlocPosition(self):
        if not self.combo: return
        blocs = []
        for rdi in self.comboList:
            bloc = rdi.getValueFromMeta('bloc')
            if bloc and bloc!="" and bloc not in blocs: blocs.append(bloc)
        if len(blocs)!=1: return
        for i in range(self.combo.count()):    
            rdi = self.combo.itemData(i)
            self.combo.setItemText(i, rdi.name)
        self.cb.setText(f"[{blocs[0]}] {self.cb.text()}")
    
    def addField(self):
        try:
            self.field.setNumeric()
            self.field.makeCombo(self.layout)
        except:
            pass
    
    def resetRenderer(self):
        if self.style:
            self.style.reset()
            
    def setVisible(self, bool):
        if self.layer is not None:
            self.layer.setItemVisibilityChecked(bool)
        if self.light.layer is not None:
            self.light.layer.setItemVisibilityChecked(bool)
        if self.tampon.layer is not None:
            self.tampon.layer.setItemVisibilityChecked(bool)
    
    def setLayer(self):
        self.tableWaiting = self.tableName()
        self.tamponWaiting = None
        self.activeTable = None
        self.error = False
        dock = self.coll.dock
        self.cluster = dock.param.get('cluster')
        self.checked = self.cb.isChecked()
        self.clicked.emit()
        if self.cb.isChecked():
            if self.coll.matchCode('enjeu'):
                dock.actuLayersEnjeu(self) 
            if self.coll.matchCode('alea'):
                dock.actuLayersAlea(self)
                dock.actuLayersEnjeu()
            if self.coll.matchCode('stats'):
                self.tableWaiting = self.tableResult
                self.affLayerAsync(self.tableWaiting)
            if self.coll.matchCode('link'):    
                self.affLayerAsync(self.tableWaiting)
            if self.coll.dock.param.get('cbAssoc'):
                for id in [*self.assoc(),self.id()]:
                    for rdi in self.coll.dock.alea.finds(id):
                        if rdi==self: continue
                        if rdi and rdi.checked:
                            rdi.cb.setChecked(False)
                            rdi.checked = False    
        else:
            self.loaded = False
            self.coll.dock.legend.remove(self)
            self.tableWaiting = None
            self.layer = dock.dyn.removeLayer(self)
            if self.coll.matchCode('alea'):
                dock.actuLayersEnjeu()
            if dock.param.get('asyncGetLayer'):
                dock.psql.cleanQgisConnexion()
        
    
    def affLayerWaiting(self, table=None, callback=None, *args, **kwargs):
        self.affLayerAsync(table, True)
        if callback:
            callback(self, *args, **kwargs)
    
    def affLayerAsync(self, table=None, wait=None, error=None, callback=None, args=(), kwargs={}):
        if self.activeTable is not None and table==self.activeTable[0]:
            # print(f"No need to reload for {self.layerName()}")
            return
        attr = self.getAttr()
        dock = self.coll.dock
        schema = None
        
        if not table:
            table = attr['table']
        if table==attr['table']:
            schema = attr['schema'] 
        if table and self.tableWaiting==table:
            self.activeTable = (table, schema)
            cond = ''
            if table==attr['table']:
                if attr['context'] and 'condition' in attr['context']:
                    cond = attr['context']['condition']
            self.cond = cond
            context = {
                'tableName': table,
                'layerName': attr['name'],
                'schema': schema,
                'cond': cond,
            }
            dock.psql.getLayerAsync(self.activeTable[0], attr['name'], self.activeTable[1], cond, self.affLayerEnd, wait=wait, error=error, callback=callback, args=args, kwargs=kwargs)
            
    def affLayerEnd(self, result, wait=None, error=None, callback=None, args=(), kwargs={}):        
        vlayer = result[0]
        geomType = result[1]
        self.wait = wait
        self.stopWaiting()
        attr = self.getAttr()
        dock = self.coll.dock
        if not vlayer:
            self.activeTable = None
            msg = ""
            if self.coll.matchCode('enjeu'):
                if not wait and not error:
                    dock.psql.getError(self)
                msg = f"L'extraction {quote(attr['name'])} n'a pas fonctionné"
                nb = len(dock.alea.getEltsSelected())
                if nb>0:
                    s = "s" if nb>1 else ""
                    art = f"les {nb}" if nb>1 else "la"
                    msg += f" pour {art} zone{s} sélectionée{s}"
                
            if self.coll.matchCode('alea'):
                self.layer = dock.dyn.removeLayer(self)
                self.cb.setChecked(False)
                msg = f"La zone {quote(attr['name'])} n'a pas une géométrie correcte"
            if msg!="" and dock.connectionStatus:
                flashMsg(msg, category='warning')  
        else:
            
            self.geomType = geomType
            self.affVLayer(vlayer)
            if self.layer:
                if wait:
                    self.startWaiting()
                else:
                    layername = attr['name']
                    precision = 1
                    if attr['field']!="":
                        layername = f"{layername} Ʃ({attr['field']})"
                        precision = dock.param.get('clusterPrecision')
                        self.layer.setName(f"{layername}")
                    if dock.param.get('entityCount') and self.coll.matchCode('enjeu'):
                        self.getEntityCount()
                if error:
                    name = f"[Echec] {attr['name']}"
                    self.layer.setName(name)        
            if self.needBack and dock.param.get('clusterBack'):
                self.vlayer.dataChanged.connect(self.filterPropagation)
                dock.psql.getLayerAsync(self.activeTable[0], attr['name'], self.activeTable[1], self.cond, self.affLightAsync)
                
        if callback is not None:
            callback(*args, **kwargs)        
                
    def getEntityCount(self):
        dock = self.coll.dock
        if dock.param.get('entityCount'):
            psql = dock.psql.clone()
            m = mGlobalTask(f"Preparation SQL")
            m.task.setCallin(psql.getTableCount, *self.activeTable)
            m.setAutoKillConnexion(psql)
            m.setCallbackWithResult(self.setEntityCount)
            dock.psql.manager.add(m)
        else:
            self.setEntityCount()
    
    def setEntityCount(self, result=None):
        attr = self.getAttr()
        dock = self.coll.dock
        layername = attr['name']
        precision = 1
        if attr['field']!="":
            layername = f"{layername} Ʃ({attr['field']})"
            precision = dock.param.get('clusterPrecision') 
        if result is None:    
            self.layer.setName(f"{layername}")
            return
        res = result[0]
        if 'count' in res:
            count = round(res['count']/precision)
            self.layer.setName(f"{layername} [{count}]")
    

    def affLayer(self, table=None, wait=None, error=None, callback=None, args=(), kwargs={}):
        self.wait = wait
        self.stopWaiting()
        attr = self.getAttr()
        dock = self.coll.dock
        schema = None
        if not table:
            table = attr['table']
        if table==attr['table']:
            schema = attr['schema'] 
        if table and self.tableWaiting==table:
            cond = ''
            if table==attr['table']:
                if attr['context'] and 'condition' in attr['context']:
                    cond = attr['context']['condition']
            vlayer, geomType = dock.psql.getLayer(table, attr['name'], schema, cond)
            if not vlayer:
                msg = ""
                if self.coll.matchCode('enjeu'):
                    if not wait and not error:
                        dock.psql.getError(self)
                    msg = f"L'extraction {quote(attr['name'])} n'a pas fonctionné"
                    nb = len(dock.alea.getEltsSelected())
                    if nb>0:
                        s = "s" if nb>1 else ""
                        art = f"les {nb}" if nb>1 else "la"
                        msg += f" pour {art} zone{s} sélectionée{s}"
                    
                if self.coll.matchCode('alea'):
                    self.layer = dock.dyn.removeLayer(self)
                    self.cb.setChecked(False)
                    msg = f"La zone {quote(attr['name'])} n'a pas une géométrie correcte"
                if msg!="" and dock.connectionStatus:
                    flashMsg(msg, category='warning')  
            else:
                self.activeTable = (table, schema)
                self.geomType = geomType
                self.affVLayer(vlayer)
                if self.layer:
                    if wait:
                        self.startWaiting()
                    else:
                        layername = attr['name']
                        precision = 1
                        if attr['field']!="":
                            layername = f"{layername} Ʃ({attr['field']})"
                            precision = dock.param.get('clusterPrecision')
                            self.layer.setName(f"{layername}")
                        if dock.param.get('entityCount') and self.coll.matchCode('enjeu'):
                            res = dock.psql.getTableCount(table, schema)[0]
                            if 'count' in res:
                                count = round(res['count']/precision)
                                self.layer.setName(f"{layername} [{count}]")
                    if error:
                        name = f"[Echec] {attr['name']}"
                        self.layer.setName(name)        
                if self.needBack and dock.param.get('clusterBack'):
                    self.vlayer.dataChanged.connect(self.filterPropagation)
                    vlayer, geomType = dock.psql.getLayer(table, attr['name'], schema, cond)
                    self.affLight(vlayer, geomType)
        if callback is not None:
            callback(*args, **kwargs)


                               
    def affVLayer(self, vlayer):
        dock = self.coll.dock
        self.vlayer = vlayer
        if not self.layer:
            self.layer = dock.dyn.addLayer(self)
        else:
            self.freeze = True
            self.layer = dock.dyn.majLayer(self)
            self.freeze = False
        if self.layer:
            self.layerId = self.layer.layerId()
            self.vlayer.rdi = self
            self.vlayer.rdiId = self.actif()
            self.vlayer.destroyed.connect(self.timer.clear)
            dock.dyn.makeStyle(self)
            self.vlayer.styleChanged.connect(self.manualPainting)
        
        if self.coll.matchCode('link') and not dock.link.cbMod.isChecked():
            a = 'rdi_link'
            if a in self.vlayer.fields().names():
                self.vlayer.setSubsetString(f'"{a}" is not NULL')
            else:
                self.vlayer.setSubsetString(f'False')

                
    def manualPainting(self):
        self.isPainted = True
        self.setCustom()
        if not self.icoPaint:
            l = rdiLayerLabel(os.path.join(self.coll.dock.imageFolder, "paint.png"), align=(Qt.AlignRight | Qt.AlignVCenter))
            l.setToolTip("Supprimer le style défini manuellement")
            self.layout.addWidget(l)
            l.leftClick.connect(self.cancelManualPainting)
            l.hide()
            self.icoPaint = l
            self.icoList['paint'] = l
        if self.style:
            self.style.saveStyle()
        self.symbologyPropagation()
            
    def cancelManualPainting(self):
        self.resetRenderer()
        self.coll.dock.dyn.makeStyle(self)
        self.isPainted = False
        self.setCustom()
        if self.icoPaint:
            w = self.icoPaint
            self.layout.removeWidget(w)
            w.setParent(None)
            del w
            self.icoPaint = None
            del self.icoList['paint']
        
    def actuVisibleElement(self):     
        lst = []
        for feat in self.vlayer.getFeatures():
            pkey = 'id'
            ind = feat.fieldNameIndex(pkey)
            val = feat.attribute(ind)
            lst.append(val)
        # print(lst)

    def setContextFromLegend(self, item):
        if self.coll.dock.filter.filterMode=="Légende":
            lst = []
            l = []
            for i,elt in enumerate(item.elts):
                if i>=len(self.style.filterList):
                    continue
                filter = self.style.filterList[i]
                if not elt.isChecked() and filter!="":
                    lst.append(filter)
                    l.append(str(i))
            if len(lst)>0:
                # self.context = {'condition':" and ".join(map(lambda x: f"NOT ({x})", lst))}       
                self.context = {'condition':" and ".join(map(lambda x: f"NOT ({x})", lst)), 'excl':"legend:"+",".join(l)}       
            else:
                self.context = None
            
    def setContextFromList(self, col, lst):
        if self.coll.dock.filter.filterMode=="Sélection" or self.coll.dock.filter.filterMode=="Visible":
            if len(lst)>0: 
                liste = ",".join(map(lambda x: f"'{x}'", lst))
                self.context = {'condition':f"{col} IN ({liste})", 'excl':f"sel:{col}:"+",".join(map(str,lst))}       
            else:
                self.context = None
    
    def affTampon(self, table=None, callback=None, args=(), kwargs={}):
        attr = self.getAttr()
        dock = self.coll.dock
        if table:
            if table==self.tamponWaiting:
                buff = dock.psql.context.get('buffer')
                vlayer, geomType = dock.psql.getLayer(table, f"[Tampon_{buff}m]: {attr['name']}")
                if not vlayer:
                    msg = f"Le tampon pour {quote(attr['name'])} n'a pas fonctionné"
                    flashMsg(msg, category='warning')  
                    self.tampon.layer = dock.dyn.removeTampon(self)    
                else:
                    self.tampon.vlayer = vlayer
                    self.tampon.geomType = geomType
                    self.tampon.layer = dock.dyn.addTampon(self)
                    self.tampon.layerId = self.tampon.layer.layerId()
                    if self.tampon.layer:
                        dock.dyn.makeTampon(self)
        else:
            dock.dyn.removeTampon(self)
        if callback is not None:
            callback(*args, **kwargs)

    def affTamponAsync(self, table=None, callback=None, args=(), kwargs={}):
        attr = self.getAttr()
        dock = self.coll.dock
        if table:
            if table==self.tamponWaiting:
                buff = dock.psql.context.get('buffer')
                dock.psql.getLayerAsync(table, f"[Tampon_{buff}m]: {attr['name']}", callback2=self.affTamponEnd, callback=callback, args=args, kwargs=kwargs)
        else:
            dock.dyn.removeTampon(self)

    def affTamponEnd(self, result, callback=None, args=(), kwargs={}):
        attr = self.getAttr()
        dock = self.coll.dock
        vlayer = result[0]
        geomType = result[1]
        if not vlayer:
            msg = f"Le tampon pour {quote(attr['name'])} n'a pas fonctionné"
            flashMsg(msg, category='warning')  
            self.tampon.layer = dock.dyn.removeTampon(self)    
        else:
            self.tampon.vlayer = vlayer
            self.tampon.geomType = geomType
            self.tampon.layer = dock.dyn.addTampon(self)
            self.tampon.layerId = self.tampon.layer.layerId()
            if self.tampon.layer:
                dock.dyn.makeTampon(self)
        if callback is not None:
            callback(*args, **kwargs)
        
    
    def affLightAsync(self, result):
        vlayer = result[0]
        geomType = result[1]
        self.affLight(vlayer, geomType)

    def affLight(self, vlayer, geomType):        
        dock = self.coll.dock
        self.light.setOpacity(dock.param.get('lightOpacity'))
        self.light.vlayer = vlayer
        self.light.geomType = geomType
        self.light.layer = dock.dyn.addLight(self)
        self.light.layerId = self.light.layer.layerId()
        if self.light.layer:
            self.light.layer.setName(f"[Fond]: {self.layerName()}")
            dock.dyn.makeLight(self)
            
    def filterPropagation(self):
        if self.light:
            vlayer = QgsVectorLayer(self.vlayer.dataProvider().dataSourceUri(), self.layerName(), "postgres")
            self.affLight(vlayer, self.geomType)
            
    def symbologyPropagation(self):
        if self.light:
            self.coll.dock.dyn.makeLight(self)
    
    def startWaiting(self):
        self.baseName = self.layer.name()
        if self.timer:
            self.timer.start(500)
        self.affWaiting()
        
    def stopWaiting(self):
        try:
            self.timer.stop()
        except:
            pass
        
    def affWaiting(self):
        try:
            name = f"[Extraction{self.timer.pts()}] {self.baseName}"
            self.layer.setName(name)
        except:
            self.stopWaiting()
            pass
    
    def topAttr(self, item):
        attr = self.getValueFromMeta(item)
        if self.combo:
            attr = self.combo.currentData().getValueFromMeta(item)
        return attr  
    
    def tableName(self): 
        return self.topAttr('tablename')
            
    def schemaName(self): 
        return self.topAttr('schemaname')
        
    def layerName(self):
        layerName = self.name
        if self.combo:
            layerName = f"{self.name} ({self.combo.currentText()})"
        if hasattr(self, 'coll') and (not self.coll.dock.param.get('cbCompact') or self.coll.expand) and self.parent()!=None:
            try: layerName = f"{self.coll.findParent(self).name}  ({self.name})"
            except: pass
        return layerName 
    
    def bloc(self):
        bloc = self.getValueFromMeta('bloc')
        return self.getValueFromMeta('bloc')
    
    def styleContent(self):
        style = self.getValueFromMeta('style')
        if self.combo:
            style = self.combo.currentData().getValueFromMeta('style', style)
        return style
        
    def jsStyle(self):
        style = self.styleContent()
        try:
            jsStyle = json.loads(style)
        except:
            if style!=None and style[:9].lower()=="<!doctype":
                jsStyle = {"xml":"1"}
            else:
                jsStyle = {}
        return jsStyle
        
    def getAttr(self):
        try:
            field = self.field.get()
        except:
            field = ""
        dict = {
            'table': self.tableName(),
            'schema': self.schemaName(),
            'context': self.context,
            'name': self.layerName(),
            'field': field,
                }
        return dict
        
    def setDbEnv(self, psql):
        tableName = self.getValueFromMeta('tablename')
        schema = self.getValueFromMeta('schemaname')
        geom = psql.getGeometryField(tableName, schema)
        pkey = psql.getPrimaryKey(tableName, schema)
        geomType = psql.getGeomType(tableName, schema, geom)
        
        self.dbEnv = {
            'geom': geom,
            'pkey': pkey,
            'geomType': geomType,
        }
 
    def getDbEnv(self):
        dbEnv = self.dbEnv
        if self.combo:
            dbEnv = self.combo.currentData().dbEnv
        return dbEnv 
 
    def findKeySearch(self):
        for k, v in self.meta.items():
            if k in ['tablename', 'name', 'tags', 'bloc']:
                self.addKeySearch(str(v).lower())
                
    def addKeySearch(self, key):
        self.search.append(key)
        
    def keyMatch(self, key):
        bool = False
        needle = key.lower()
        for haystack in self.search:
            if haystack.find(needle)>=0:
                bool = True
        return bool
        
    def parent(self):
        try:
            p = self.propParent
        except:
            self.propParent = self.getValueFromMeta('parent')
        return self.propParent
        
    def id(self):
        try:
            p = self.propId
        except:
            self.propId = self.getValueFromMeta('id')
        return self.propId
        
    def actif(self):
        return self.topAttr('id')
    
    def assoc(self):
        try:
            l = self.getValueFromMeta('assoc').split(',')
            l = list(map(int,l))
            return l
        except:
            return []
    
    def setTimer(self):
        self.timer = rdiTimer(self, self.affWaiting)
      
    def getIdentFeature(self):
        pkey = self.topAttr('pkey')
        if pkey==None:
            pkey = self.coll.dock.psql.getPrimaryKey(self.tableName(), self.schemaName())   
        if pkey==None:
            list = self.coll.dock.psql.getNonGeomFields(self.tableName(), self.schemaName())
            if 'id' in list:
                pkey = 'id'
            else:
                pkey = list[0]
        return pkey
        
    def goToAdmin(self):
        self.coll.dock.goToAdmin(rdi=self)

 
 
 
 
class rdiTampon():
    def __init__(self, rdi):
        self.rdi = rdi
        self.layer = None
        self.layerId = None
        self.vlayer = None
        self.geomType = None
        self.style = None
        self.params = {}

    def jsStyle(self):
        return {"tampon":"1"}
    
    def layerName(self):
        return self.rdi.layerName()
        
    def tableName(self):
        return self.rdi.tableName()
        
class rdiLight():
    def __init__(self, rdi):
        self.rdi = rdi
        self.coll = None
        self.layer = None
        self.layerId = None
        self.vlayer = None
        self.geomType = None
        self.style = None
        self.cluster = False
        self.params = {'opacity':0.2}

    def setOpacity(self, opacity):
        self.params['opacity'] = opacity
    
    def jsStyle(self):
        return self.rdi.jsStyle()
    
    def styleContent(self):
        return self.rdi.styleContent()
    
    def layerName(self):
        return self.rdi.layerName()
        
    def tableName(self):
        return self.rdi.tableName()
 
 

class rdiLayerCb(QtWidgets.QCheckBox):
    rightClick = pyqtSignal()
    def __init__(self, txt):
        QtWidgets.QCheckBox.__init__(self, txt)
        self.ctrl = False
    
    def mousePressEvent(self, event):
        if event.button()==Qt.RightButton:
            self.rightClick.emit()
        else:
            super().mousePressEvent(event)
            
    def keyPressEvent(self, event):
        if event.key()==Qt.Key_Control:
            self.ctrl = True
        super(rdiLayerCb, self).keyPressEvent(event)
    
    def keyReleaseEvent(self, event):
        if event.key()==Qt.Key_Control:
            self.ctrl = False
        super(rdiLayerCb, self).keyReleaseEvent(event)
            
class rdiLayerLabel(QtWidgets.QLabel):
    rightClick = pyqtSignal()
    leftClick = pyqtSignal()
    def __init__(self, path, **kwargs):
        QtWidgets.QLabel.__init__(self)
        self.kwargs = kwargs
        self.setIco(path)
    
    def mousePressEvent(self, event):
        if event.button()==Qt.RightButton:
            self.rightClick.emit()
        else:
            self.leftClick.emit()
            
    def setIco(self, path):
        ico = QtGui.QIcon(path)
        self.setMaximumWidth(20)
        self.setPixmap(ico.pixmap(15,15))
        if 'align' in self.kwargs:
            self.setAlignment(self.kwargs.get('align'))
        if 'height' in self.kwargs:
            w = self.kwargs.get('height')
            self.setMaximumWidth(w+5)
            self.setPixmap(ico.pixmap(w,w))

class rdiLayerField(QObject):
    def __init__(self, rdi, psql=None):
        QObject.__init__(self)
        self.rdi = rdi
        self.memo = rdi.topAttr('cluster')
        self.numeric = False
        self.labels = []
        self.on = True
        self.active = False
        try:
            self.psql = self.rdi.coll.dock.psql
            self.inCollection = True
        except:
            self.psql = psql
            self.inCollection = False
    
    def makeContainer(self, layout):
        self.container = QtWidgets.QWidget()
        if self.rdi.icoPaint:
            layout.insertWidget(layout.indexOf(self.rdi.icoPaint), self.container)
        else:
            layout.addWidget(self.container)
        lay = QtWidgets.QHBoxLayout(self.container)
        lay.setSpacing(2)
        lay.setContentsMargins(0,0,0,0)
        self.makeLabel("Ʃ(", 10)

    def makeLabel(self, txt, size):
        l = QtWidgets.QLabel(txt)
        self.labels.append(l)
        l.setMaximumWidth(size)
        self.container.layout().addWidget(l)
    
    def makeCombo(self, layout):
        self.active = True
        self.makeContainer(layout)
        self.combo = QtWidgets.QComboBox()
        self.combo.setToolTip(f"Choix du champ à sommer pour la couche \"{self.rdi.layerName()}\"\nComptera seulement les entités si laissé vide")
        self.container.layout().addWidget(self.combo)
        self.labels.append(self.combo)
        self.makeLabel(")", 5)
        self.majCombo()
        if self.inCollection:
            self.rdi.changed.connect(self.majCombo)
            self.combo.currentIndexChanged.connect(self.change)
            l = QtWidgets.QLabel("")
            self.container.layout().addWidget(l)
            l.hide()
            self.label = l
            self.icoON = QtGui.QIcon(os.path.join(self.rdi.coll.dock.imageFolder,'close.png'))
            self.icoOFF = QtGui.QIcon(os.path.join(self.rdi.coll.dock.imageFolder,'sum.png'))
            butt = QtWidgets.QPushButton(self.icoON, '')
            butt.setToolTip(f"Compter seulement les entités")
            butt.setMaximumWidth(20)
            self.container.layout().addWidget(butt)
            butt.clicked.connect(self.toggle)
            self.toggleButton = butt
            if not self.memo or self.memo=="":
                self.toggle(block=True)
    
    def toggle(self, block=False):
        self.on = not self.on
        if self.on:
            self.label.hide()
            for l in self.labels:
                l.show()
            self.toggleButton.setIcon(self.icoON)
            self.toggleButton.setToolTip(f"Compter seulement les entités")
            self.toggleButton.show()
        else:
            self.label.show()
            for l in self.labels:
                l.hide()
            self.toggleButton.setIcon(self.icoOFF)
            self.toggleButton.setToolTip(f"Faire une somme sur attribut")
            self.toggleButton.hide()
        if not block:
            self.change()
    
    def setNumeric(self):
        self.numeric = True
        try:
            self.majCombo()
        except:
            pass
            
    def isPoint(self):
        geomtype = self.rdi.topAttr('type')
        if geomtype:
            isPoint = (geomtype=='point')
        else:
            isPoint = self.psql.isGeomPoint(self.rdi.tableName(), self.rdi.schemaName())
        return isPoint
    
    def majCombo(self):
        self.combo.blockSignals(True)
        if self.isPoint():
            self.container.show()
            self.rdi.icoList['field'] = self.container
        else:
            self.container.hide()
            if 'field' in self.rdi.icoList:
                del self.rdi.icoList['field']
        while self.combo.count():
            self.combo.removeItem(0)
        self.combo.addItem("", "")
        fields = self.psql.getColumnsTableAll(self.rdi.tableName(), self.rdi.schemaName())
        for e in fields:
            if not self.numeric or e.get('numeric_precision')!=None:
                f = e.get('column_name')
                self.combo.addItem(f, f)
        if self.memo!="":
            self.combo.setCurrentText(self.memo)
        self.combo.blockSignals(False)
        
    def get(self):
        try:
            if self.on:
                return self.combo.currentText()
            else:
                return ""
        except:
            return ""
            
    def change(self):
        self.memo = self.get()
        self.rdi.setLayer()
 
 
 
 
class rdiLayersCollection(QObject):
    
    changed = pyqtSignal()
    
    def __init__(self, dock, code=None):
        QObject.__init__(self)
        self.dock = dock
        self.code = code
        self.dict = {}
        self.dicts = {}
        self.list = []
        self.blocs = {}
        self.expand = False
        self.actu = None

    def title(self):
        try:
            return self.dock.param.get(self.code)
        except:
            return "Titre "+str(self.code)

    def setFilter(self, filter):
        self.filter = filter
    
    def setActu(self, actu):
        self.actu = actu

    def construct(self, layout):
        wg = QtWidgets.QWidget()
        self.wg = wg
        layout.addWidget(wg)
        lg = QtWidgets.QVBoxLayout()
        lg.setSpacing(0)
        lg.setContentsMargins(2,0,2,0)
        wg.setLayout(lg)
        self.makeEntete(lg)
        self.makeList(lg)
    
    def makeEntete(self, lg):
        w = QtWidgets.QWidget()
        lg.addWidget(w)
        hbox = QtWidgets.QHBoxLayout()
        hbox.setContentsMargins(5,8,5,2)
        w.setLayout(hbox)
        l = rdiLayerLabel(os.path.join(self.dock.imageFolder,f"{self.code}.png"))
        hbox.addWidget(l)
        l.rightClick.connect(self.affMenu)        
        # l = QtWidgets.QLabel(self.title())
        l = dblClickLabel(self.title())
        l.rightClick.connect(self.affMenu)
        font = QtGui.QFont()
        font.setBold(True)
        l.setFont(font)
        hbox.addWidget(l)
        filter = QtWidgets.QLineEdit()
        filter.setClearButtonEnabled(True)
        filter.setPlaceholderText("Mots-clefs du filtre ici")
        self.filter = filter
        filter.textChanged.connect(self.affFromFilter)
        hbox.addWidget(filter)
        l = rdiLayerLabel(os.path.join(self.dock.imageFolder, "filter.png"), height=20)
        l.setToolTip(f"Alléger les {self.dock.param.get(self.code)} présentés")
        hbox.addWidget(l)
        l.leftClick.connect(partial(self.dock.alleviate.openWindow, self.code))
        self.filterIco = l
        if self.actu:
            ico = QtGui.QIcon(os.path.join(self.dock.imageFolder,'refresh.png'))
            butt = QtWidgets.QPushButton(ico, '')
            butt.setToolTip("Actualiser")
            butt.setMaximumWidth(20)
            hbox.addWidget(butt)
            butt.clicked.connect(self.actu)
    
    def setFiltered(self, bool=True):
        if bool:
            path = os.path.join(self.dock.imageFolder, "filterOn.png")
        else:
            path = os.path.join(self.dock.imageFolder, "filter.png")
        self.filterIco.setIco(path)
    
    def makeList(self, lg):
        sc = QtWidgets.QScrollArea()
        sc.setWidgetResizable(True)
        lg.addWidget(sc)
        w = QtWidgets.QWidget()
        gbox = QtWidgets.QVBoxLayout()
        gbox.setSpacing(0)
        gbox.setAlignment(Qt.AlignTop)
        w.setLayout(gbox)
        sc.setWidget(w)
        self.layerBloc = gbox
        self.sc = sc
        if self.dock.param.get('blocDrop'):
            self.bloc = rdiLayersBloc(self)

    def add(self, rdi, force=False, bloc=None):
        if rdi.tableName()!=None or bloc:
            self.addAbstract(rdi)
            rdi.clicked.connect(self.changed.emit)
            if self.dock.param.get('blocDrop'):
                if not bloc: bloc = rdi.getValueFromMeta('bloc')
                rdi.targetBloc = bloc
                layout = self.bloc.layout
                if bloc and bloc!="":
                    if bloc not in self.blocs:
                        self.blocs[bloc] = {'list':[], 'bloc':rdiLayersBloc(self, bloc)}
                    layout = self.blocs[bloc]['bloc'].layout
                    self.blocs[bloc]['list'].append(rdi)
            else:
                layout = self.layerBloc
            rdi.addCb(layout)
            
        elif not self.dock.param.get('cbCompact'):
            self.addAbstract(rdi)
    
    def addAbstract(self, rdi):
        self.addToDict(rdi)
        self.list.append(rdi)
        rdi.coll = self
        
    def addToDict(self, rdi):
        id = rdi.id()
        self.dict[id] = rdi
        if id not in self.dicts: self.dicts[id] = []
        if rdi not in self.dicts[id]: self.dicts[id].append(rdi)
    
    def remove(self, rdi):
        try:
            rdi.cb.setChecked(False)
        except:
            pass
        rdi.bloc.removeWidget(rdi.widget)
        rdi.widget.setParent(None)
    
    def find(self, id, deep=False, bloc=None):
        if deep:
            for rdi in self.list:
                if rdi.id()==id: return (rdi, None)
                try:
                    for rdi2 in rdi.comboList:
                        if rdi2.id()==id: return (rdi, rdi2)
                except: pass
            return (None, None)
        else:
            if bloc:
                for rdi in self.dicts.get(id,[]):
                    if rdi.targetBloc==bloc: return rdi
            else:
                try: return self.dict[id]
                except: return None
    
    def finds(self, id):
        return self.dicts.get(id,[])
        
    def findParent(self, rdi):
        id = rdi.parent()
        if id is None: return
        bloc = rdi.getValueFromMeta('bloc')
        if not self.dock.param.get('blocDrop') or bloc in (None, ""): 
            try: return self.finds(id)[0]
            except: return
        for rdi in self.finds(id):
            if not self.dock.param.get('cbCompact'): return rdi
            if rdi.targetBloc==bloc: return rdi
    
    def affFromFilter(self):
        for rdi in self.list:
            if rdi.widget!=None:
                if rdi.keyMatch(self.filter.text()):
                    rdi.widget.show()
                    rdi.showed = True
                else:
                    rdi.widget.hide()
                    rdi.showed = False
        for c,b in self.blocs.items():
            has = False
            for rdi in b['list']:
                has = has or rdi.showed
            if has:
                b['bloc'].widgetTitle.show()
            else:
                b['bloc'].widgetTitle.hide()
            
    def matchCode(self, code):
        return (self.code==code)

    def getEltsSelected(self):
        list = []
        for rdi in self.list:
            if rdi.checked:
                list.append(rdi)
        return list
        
    def getEltsInverted(self):
        list = []
        for rdi in self.list:
            if rdi.inverted:
                list.append(rdi)
        return list
        
    def getElts(self):
        return self.list
        
    def addLayerField(self):
        for rdi in self.list:
            rdi.addField()
            
    def affMenu(self):
        self.popMenu = QtWidgets.QMenu()
        self.popMenu.addSection(self.title())
        ico = QtGui.QIcon(os.path.join(self.dock.imageFolder, "zoom.png"))
        a = self.popMenu.addAction(ico, "Zoom sélection", self.actionZoom)
        if len(self.getEltsSelected())==0:
            a.setEnabled(False)
        self.popMenu.addSeparator()
        ico = QtGui.QIcon(os.path.join(self.dock.imageFolder, "check.png"))
        a = self.popMenu.addAction(ico, "Tout cocher", partial(self.actionCheck, True))
        if len(self.getEltsSelected())==len(self.getElts()):
            a.setEnabled(False)
        ico = QtGui.QIcon(os.path.join(self.dock.imageFolder, "uncheck.png"))
        a = self.popMenu.addAction(ico, "Tout décocher", self.actionCheck)
        if len(self.getEltsSelected())==0:
            a.setEnabled(False)
            
        if self.matchCode('enjeu'):
            self.popMenu.addSeparator()
            ico = QtGui.QIcon(os.path.join(self.dock.imageFolder, "invert.png"))
            a = self.popMenu.addAction(ico, "Tout inverser", partial(self.actionInvert, True))
            if len(self.getEltsInverted())==len(self.getElts()):
                a.setEnabled(False)
            ico = QtGui.QIcon(os.path.join(self.dock.imageFolder, "invert-ko.png"))
            a = self.popMenu.addAction(ico, "Tout rétablir", self.actionInvert)
            if len(self.getEltsInverted())==0:
                a.setEnabled(False)            

        
        self.popMenu.popup(QtGui.QCursor.pos())
    
    def actionZoom(self):
        tableList = []
        for rdi in self.getEltsSelected():
            if rdi.activeTable is not None:
                tableList.append((*rdi.activeTable, rdi.topAttr('geom')))
            else:
                tableList.append((rdi.tableName(), rdi.schemaName(), rdi.topAttr('geom')))
        self.dock.askZoom(tableList)
    
    def actionCheck(self, bool=False):
        if bool:
            nba = (len(self.dock.alea.getElts())-len(self.dock.alea.getEltsSelected()))
            other = None
            if self.code=='alea':
                other = self.dock.enjeu
            if self.code=='enjeu':
                other = self.dock.alea
            if other is not None:
                nb = (len(self.getElts())-len(self.getEltsSelected())) * (len(other.getEltsSelected()))
                if nb>self.dock.param.get('autoCheckAlert'):
                    qm = QtWidgets.QMessageBox
                    rep = qm.question(None, "", f"Cette action peut engendrer un grand nombre de croisements spatiaux : {nb}\nVoulez-vous continuer ?", qm.Yes | qm.No)
                    if rep!=qm.Yes:
                        return
        for rdi in self.list:
            try:
                rdi.cb.setChecked(bool)
            except:
                pass
                
                
    def actionInvert(self, bool=False):
        nb = (len(self.dock.enjeu.getElts())) * (len(self.dock.alea.getEltsSelected()))
        if nb>self.dock.param.get('autoCheckAlert'):
            qm = QtWidgets.QMessageBox
            rep = qm.question(None, "", f"Cette action peut engendrer un grand nombre de croisements spatiaux : {nb}\nVoulez-vous continuer ?", qm.Yes | qm.No)
            if rep!=qm.Yes:
                return
        for rdi in self.list:
            try:
                rdi.invert(bool)
            except:
                pass
                
    def getLoadedLayers(self):
        list = []
        for rdi in self.list:
            if rdi.combo is not None:
                for rdi2 in rdi.comboList:
                    list.append(rdi2)
            else:
                list.append(rdi)
        return list
     
class rdiLayersBloc(QObject):
    changed = pyqtSignal()
    def __init__(self, coll, title=None):
        QObject.__init__(self)
        self.coll = coll
        self.makeTitle(title)
        self.makeBloc()
        
    def makeTitle(self, title):
        self.title = title
        self.code = f"{self.coll.code}_{title}"
        self.widgetTitle = None
        if title:
            w = rdiBlocTitle(self.code, title)
            self.coll.layerBloc.addWidget(w)
            w.clicked.connect(self.clickAndDrop)
            # w.rightClicked.connect(self.menu)
            self.widgetTitle = w
    
    def makeBloc(self):
        w = QtWidgets.QWidget()
        gbox = QtWidgets.QVBoxLayout(w)
        gbox.setSpacing(0)
        gbox.setContentsMargins(0,0,0,0)
        gbox.setAlignment(Qt.AlignTop)
        self.coll.layerBloc.addWidget(w)
        self.layout = gbox
        self.widget = w
        if self.widgetTitle and not self.widgetTitle.isMemoDrop(): 
            self.widgetTitle.setExpanded(False)
            
    def clickAndDrop(self, b):
        if b:
            self.widget.show()
            # self.coll.dock.blocState[self.code] = True
        else:
            self.widget.hide()
            # del self.coll.dock.blocState[self.code]
            
    # def menu(self):
        # self.popMenu = QtWidgets.QMenu()
        # self.popMenu.addSection(self.title)
        # if self.isOpenDrop(): ico = QtGui.QIcon(os.path.join(self.coll.dock.imageFolder, "check.png"))
        # else: ico = QtGui.QIcon()
        # self.popMenu.addAction(ico, "Déplier à l'ouverture", self.setOpenDrop)
        # self.popMenu.popup(QtGui.QCursor.pos())
        
    # def isOpenDrop(self):
        # l = eval(self.coll.dock.param.get('blocDropTargets'))
        # b = False
        # if self.code in l: b = True
        # return b
    
    # def setOpenDrop(self):
        # b = not self.isOpenDrop()
        # l = eval(self.coll.dock.param.get('blocDropTargets'))
        # if b and self.code not in l: l.append(self.code)
        # if not b and self.code in l: l.remove(self.code)
        # self.coll.dock.param.set('blocDropTargets', l)
        
    




        