# -*- coding: utf-8 -*-

import os, sys, time
from functools import partial

from qgis.gui import *
from qgis.core import *
from PyQt5 import QtGui, QtWidgets, uic, QtXml
from PyQt5.QtCore import *
from qgis.utils import iface
try:
    from qgis import processing
except:
    import processing

from .rdi_layer import rdiLayer, rdiLayersCollection
from .rdi_meta import rdiMeta
from .rdi_tools import makeUniqueIdent, clearLayout

class rdiAtlas(QObject):

    def __init__(self, parent):
        QObject.__init__(self)
        self.dock = parent
        self.working = False
        self.label = None
        self.meta = rdiMeta()
        self.folder = self.meta.readIni("autoGenFolder")
        self.initData()
        QgsProject.instance().layoutManager().layoutAdded.connect(self.actuPrinter)
        QgsProject.instance().layoutManager().layoutRemoved.connect(self.actuPrinter)
        QgsProject.instance().layoutManager().layoutRenamed.connect(self.actuPrinter)
        
    def unload(self):
        QgsProject.instance().layoutManager().layoutAdded.disconnect(self.actuPrinter)
        QgsProject.instance().layoutManager().layoutRemoved.disconnect(self.actuPrinter)
        QgsProject.instance().layoutManager().layoutRenamed.disconnect(self.actuPrinter)
    
    def make(self, layout):
        wl = QtWidgets.QWidget()
        layout.addWidget(wl)
        hbox = QtWidgets.QHBoxLayout(wl)
        hbox.setContentsMargins(10,10,0,0)
        l = QtWidgets.QLabel("Dans ")
        l.setFixedSize(35,15)
        hbox.addWidget(l)
        ico = QtGui.QIcon(os.path.join(self.dock.imageFolder, "folder.png"))
        butt = QtWidgets.QPushButton(ico, "")
        butt.setToolTip("Choisir le dossier cible")
        butt.setMaximumWidth(30)
        hbox.addWidget(butt)
        l = QtWidgets.QLabel("")
        l.setMaximumWidth(250)
        hbox.addWidget(l)
        self.label = l
        butt.clicked.connect(self.setFolder)
        
        wl = QtWidgets.QWidget()
        layout.addWidget(wl)
        hbox = QtWidgets.QHBoxLayout(wl)
        hbox.setContentsMargins(30,0,0,0)
        cb = QtWidgets.QCheckBox(f"{self.dock.param.get('alea')} sélectionnés uniquement")
        cb.setChecked(True)
        hbox.addWidget(cb)
        self.aleaCB = cb
        wl = QtWidgets.QWidget()
        layout.addWidget(wl)
        hbox = QtWidgets.QHBoxLayout(wl)
        hbox.setContentsMargins(30,0,0,0)
        cb = QtWidgets.QCheckBox(f"{self.dock.param.get('enjeu')} sélectionnés uniquement")
        cb.setChecked(True)
        hbox.addWidget(cb)
        self.enjeuCB = cb
        
        wl = QtWidgets.QWidget()
        layout.addWidget(wl)
        hbox = QtWidgets.QHBoxLayout(wl)
        hbox.setContentsMargins(10,10,0,5)
        l = QtWidgets.QLabel("Mode de création")
        l.setFixedSize(85,15)
        hbox.addWidget(l)
        hbox.setAlignment(l, Qt.AlignLeft)
        combo = QtWidgets.QComboBox()
        hbox.addWidget(combo)
        hbox.setAlignment(combo, Qt.AlignLeft)
        for k, v in self.options.items():
            combo.addItem(v, k)
        hbox.addWidget(combo)
        combo.currentIndexChanged.connect(self.actuMode)
        self.mode = combo
        l = QtWidgets.QLabel(" ")
        hbox.addWidget(l)
        
        
        # GPKG
        wg = QtWidgets.QWidget()
        layout.addWidget(wg)
        form = QtWidgets.QFormLayout(wg)
        form.setContentsMargins(30,10,10,10)
        self.gpkgOptions = wg
        wg.hide()
        l = QtWidgets.QLabel("Regroupement")
        combo = QtWidgets.QComboBox()
        form.addRow(l,combo)
        for k, v in self.group.items():
            combo.addItem(v, k)
        self.regroup = combo
        cb = QtWidgets.QCheckBox(f"Intégrer le style")
        self.style = cb
        form.addRow("", cb)
        
        # PDF
        wg = QtWidgets.QWidget()
        layout.addWidget(wg)
        form = QtWidgets.QFormLayout(wg)
        form.setContentsMargins(30,10,10,10)
        self.pdfOptions = wg
        wg.hide()
        l = QtWidgets.QLabel("Mise en page")
        combo = QtWidgets.QComboBox()
        form.addRow(l,combo)
        self.printer = combo
        self.actuPrinter()
        l = QtWidgets.QLabel("Atlas")
        combo = QtWidgets.QComboBox()
        combo.addItem("Aucun", None)
        ico = QtGui.QIcon(os.path.join(self.dock.imageFolder, 'stats.png'))
        for elt in self.dock.psql.getTableAdmin('stats', True):
            rdi = rdiLayer(elt)
            table = rdi.tableName()
            if table:
                name = rdi.layerName()
                combo.addItem(ico, name, elt)
        self.atlasChoice = combo
        form.addRow(l, combo)
        l = QtWidgets.QLabel("Nom")
        combo = QtWidgets.QComboBox()
        combo.label = l
        self.atlasName = combo
        form.addRow(l, combo)
        self.atlasChoice.currentIndexChanged.connect(self.actuAtlas)
        l = QtWidgets.QLabel("")
        cb = QtWidgets.QCheckBox(f"Effet pochoir")
        cb.setToolTip(f"Mise en évidence de la zone de l'atlas en grisant autour")
        self.mask = cb
        self.mask.label = l
        form.addRow(l, cb)
        cb = QtWidgets.QCheckBox(f"Cartes thématiques")
        cb.setToolTip(f"Les {self.dock.param.get('enjeu')} sont regroupés selon les thèmes définis ou séparés")
        self.split = cb
        form.addRow("", cb)
        cb = QtWidgets.QCheckBox(f"PDF multi-couches")
        cb.setToolTip(f"Les enjeux seront cliquables sur le PDF")
        self.multi = cb
        form.addRow("", cb) 
        if Qgis.QGIS_VERSION_INT<31600:
            self.multi.hide()


        wl = QtWidgets.QWidget()
        layout.addWidget(wl)
        hbox = QtWidgets.QHBoxLayout(wl)
        ico = QtGui.QIcon(os.path.join(self.dock.imageFolder, "execute.png"))
        butt = QtWidgets.QPushButton(ico, " Générer")
        butt.setToolTip("Lancer la génération de cartes")
        butt.setMaximumWidth(100)
        hbox.addWidget(butt)
        butt.clicked.connect(self.launch)
        self.butt = butt
        
        if self.dock.param.get('allowBeta'):
            wl = QtWidgets.QWidget()
            layout.addWidget(wl)
            hbox = QtWidgets.QHBoxLayout(wl)
            butt = QtWidgets.QPushButton("Aléas Synapse")
            butt.setToolTip("Lancer la génération de cartes")
            butt.setMaximumWidth(100)
            hbox.addWidget(butt)
            butt.clicked.connect(self.saveSynapse)

        self.actuMode()
        self.actuAtlas()
        self.setReady()
    
    def actuMode(self):
        mode = self.mode.currentData()
        if mode=='gpkg':
            self.pdfOptions.hide()
            self.gpkgOptions.show()
        if mode=='pdf':
            self.pdfOptions.show()
            self.gpkgOptions.hide()
    
    def actuPrinter(self):
        while self.printer.count():
            self.printer.removeItem(0)
        self.printer.addItem("Auto", None)
        list = sorted([cView.name() for cView in QgsProject.instance().layoutManager().printLayouts()], key=str.lower)
        for s in list:
            self.printer.addItem(s, s)
    
    def actuAtlas(self):
        while self.atlasName.count():
            self.atlasName.removeItem(0)
        elt = self.atlasChoice.currentData()
        if elt is None:
            self.atlasName.hide()
            self.atlasName.label.hide()
            self.mask.hide()
            self.mask.label.hide()
        else:
            args = (elt.get('tablename'), elt.get('schemaname'))
            fields = self.dock.psql.getColumnsTable(*args)
            for f in fields:
                self.atlasName.addItem(f, f)
            self.atlasName.show()
            self.atlasName.label.show()
            self.mask.show()
            self.mask.label.show()
        
    
    def setReady(self):
        bool = (self.folder is not None and self.folder!="" and not self.working and os.path.exists(self.folder))
        try:
            self.butt.setEnabled(bool)
            name = self.folder
            if len(name)>40:
                name = name[:10] +"[...]"+ name[-25:]
            self.label.setText(name)
            self.label.setToolTip(self.folder)
            self.meta.writeIni("autoGenFolder", self.folder)
            if os.path.exists(self.folder):
                self.label.setStyleSheet("")
            else:
                self.label.setStyleSheet("color:red;")
        except:
            pass
        
    def setFolder(self):
        dialog = QtGui.QFileDialog()
        dialog.setFileMode(QtGui.QFileDialog.Directory)
        dialog.setOption(QtGui.QFileDialog.ShowDirsOnly, True)
        target = dialog.getExistingDirectory(None, 'Choisir un emplacement')
        self.folder = target
        self.setReady()
    
    def initProgress(self):
        self.tot = self.nb
        if self.tot>0:
            self.part = 100/self.tot
        else:
            self.part = 100
        self.setProgress(False)
    
    def setProgress(self, inc=True):
        if inc:
            self.nb -= 1
            if self.nb<0:
                self.nb = 0
        p = int(self.part*(self.tot-self.nb))
        self.progressBar.setValue(p)
        
    def setProgressInfo(self, txt):
        self.progressInfo.setText(txt)
        QtWidgets.QApplication.processEvents()

    def startWorking(self):
        self.working = True
        self.stopMarker = False
        self.butt.setEnabled(False)
        self.dock.tab.setEnabled(False)

        layout = self.dock.outTabInfos.layout()
        clearLayout(layout)
        self.dock.outTabInfos.show()
        l = QtWidgets.QLabel("Génération en cours\nEvitez toute interaction avec Qgis")
        font = QtGui.QFont()
        font.setBold(True)
        l.setFont(font)
        l.setWordWrap(True)
        layout.addWidget(l)
        
        hbox = QtWidgets.QHBoxLayout()
        layout.addLayout(hbox)
        p = QtWidgets.QProgressBar()
        hbox.addWidget(p)
        self.progressBar = p
        ico = QtGui.QIcon(os.path.join(self.dock.imageFolder, "stop.png"))
        butt = QtWidgets.QPushButton(ico, "")
        butt.setToolTip("Arrêter la génération de cartes")
        butt.setMaximumWidth(20)
        hbox.addWidget(butt)
        butt.clicked.connect(self.stop)
        self.stopButton = butt
        
        l = QtWidgets.QLabel("")
        font = QtGui.QFont()
        font.setItalic(True)
        l.setFont(font)
        layout.addWidget(l)
        self.progressInfo = l

    def stopWorking(self):
        self.working = False
        self.stopMarker = False
        self.butt.setEnabled(True)
        self.dock.tab.setEnabled(True)
        self.dock.outTabInfos.hide()
        # clearLayout(self.dock.outTabInfos.layout())
        self.land()
    
    def needClear(self):
        mode = self.mode.currentData()
        if mode=='gpkg' and self.style.isChecked():
            return True
        if mode=='pdf':
            return True
 
    def aleaParallax(self):
        mode = self.mode.currentData()
        if self.mode.currentData()=='gpkg':
            return True
        if self.mode.currentData()=='pdf':
            return False
    
    def enjeuParallax(self):
        mode = self.mode.currentData()
        if self.mode.currentData()=='gpkg':
            return True
        if self.mode.currentData()=='pdf':
            if self.split.isChecked() and not self.multi.isChecked():
                return False
            return True
            
    def confirm(self):
        if self.nb<self.dock.param.get('autoGenAlert'):
            return True
        qm = QtWidgets.QMessageBox
        rep = qm.question(None, "", f"La génération nécessite un grand nombre de croisements spatiaux : {self.nb}\nVoulez-vous continuer ?", qm.Yes | qm.No)
        if rep==qm.Yes:
            return True
        else:
            self.stopWorking()
    
    def getThemeName(self, name):
        i = 1
        while name in self.coll.mapThemes():
            i += 1
            name = f"{base} ({i})"
        return name   
    
    def getModel(self):
        try:
            model = iface.layerTreeView().layerTreeModel()
        except:
            model = iface.layerTreeView().model()
        return model
        
    def getRecord(self):
        return QgsMapThemeCollection.createThemeFromCurrentState(QgsProject.instance().layerTreeRoot(), self.getModel())
        
    def listRdi(self, code):
        lst = []
        root = QgsProject.instance().layerTreeRoot()
        group = root.findGroup(self.dock.param.get(code))
        for l in group.findLayers():
            try:
                rdi = l.layer().rdi
                lst.append(rdi)
                rdi.setVisible(False)
            except:
                pass
        return lst
    
    def matchThema(self, rdi, theme):
        bloc = rdi.topAttr('bloc')
        if theme is None or theme=="":
            if bloc is None or bloc=="":
                return True
        else:
            if bloc==theme:
                return True
    
    def makeThema(self, theme, lst_enjeu):
        lst = []
        for rdi in lst_enjeu:
            if self.matchThema(rdi, theme):
                rdi.setVisible(True)
                lst.append(rdi)
        if len(lst)>0:
            name = f"{self.dock.param.get('enjeu')}_{self.getThemeName(theme)}"
            self.coll.insert(f"1_{name}", self.getRecord())        
        for rdi in lst_enjeu:
            if self.matchThema(rdi, theme):         
                rdi.setVisible(False)
    
    def setThemes(self):
        project = QgsProject.instance()
        self.coll = project.mapThemeCollection()
        self.coll.clear()
        all = f"0_Tout"
        self.coll.insert(all, self.getRecord())
        lst_enjeu = self.listRdi('enjeu')
        lst_theme = self.dock.psql.getTableAdminBlocs(code='enjeu', mixed=True)
        lst_alea = self.listRdi('alea')
        for rdi in lst_alea:
            rdi.setVisible(True)
        for theme in lst_theme:
            self.makeThema(theme, lst_enjeu)
        self.coll.applyTheme(all, QgsProject.instance().layerTreeRoot(), self.getModel())
        # self.coll.removeMapTheme(all)
    
    def getList(self, code):
        list = []
        coll = rdiLayersCollection(self.dock, code)
        for elt in self.dock.psql.getTableAdmin(code, True):
            rdi = rdiLayer(elt)
            
            if self.dock.alleviate.isExclude(rdi):
                continue
            
            coll.addAbstract(rdi)
            if rdi.tableName():
                list.append(rdi)
        return list
    
    def stop(self):
        self.stopMarker = True
        self.setProgressInfo("Arrêt en cours...")
        self.stopButton.setEnabled(False)
    
    def attemptStop(self):
        if self.stopMarker:
            self.stopWorking()
        
    def launch(self):
        if not os.path.exists(self.folder):
            
            return
        self.startWorking()
        self.aleaSelected = self.dock.alea.getEltsSelected()
        self.enjeuSelected = self.dock.enjeu.getEltsSelected()
        self.aleaMemo = self.getList('alea')
        if self.aleaCB.isChecked(): 
            self.aleaMemo = self.dock.alea.getEltsSelected()
        self.enjeuMemo = self.getList('enjeu')
        if self.enjeuCB.isChecked():
            self.enjeuMemo = self.dock.enjeu.getEltsSelected()
        self.nb = len(self.aleaMemo)*len(self.enjeuMemo)
        self.themeMemo = self.dock.psql.getTableAdminBlocs(code='enjeu', mixed=True)
        if not self.confirm():
            return
        self.initProgress()
        if self.needClear():
            for rdi in self.enjeuSelected:
                rdi.cb.setChecked(False)
                rdi.layer = None
            for rdi in self.aleaSelected:
                rdi.cb.setChecked(False)
                rdi.layer = None
        self.dock.closeAllLayer()
        self.dock.mask.affLayer()
        self.dock.map.affMap()
        self.waitings = {}
        self.aleaData = {'list':self.aleaMemo.copy(), 'wait':len(self.aleaMemo), 'done':0, 'target':None}
        self.enjeuData = {}
        self.getAleas()
        
    def land(self):
        self.dock.closeAllLayer()
        return
        if self.needClear():
            for rdi in self.aleaSelected:
                rdi.cb.setChecked(True)
            for rdi in self.enjeuSelected:
                rdi.cb.setChecked(True)
    
    def getAleas(self):
        self.attemptStop()
        if self.aleaData['target'] is not None and self.needClear():
            self.remove(self.aleaData['target'])
        if len(self.aleaData['list'])==0:
            self.stopWorking()
            return
        if self.aleaParallax():
            list = self.aleaData['list'].copy()
            self.aleaData['list'] = []
            for alea in list:
                self.enjeuData[alea] = {'list':self.enjeuMemo.copy(), 'wait':len(self.enjeuMemo), 'done':0, 'target':None}
                self.getAlea(alea)
            return
        self.aleaData['target'] = self.aleaData['list'].pop()
        self.aleaData['target'].layer = None
        self.enjeuData[self.aleaData['target']] = {'list':self.enjeuMemo.copy(), 'wait':len(self.enjeuMemo), 'done':0, 'target':None}
        self.getAlea(self.aleaData['target'])
        
    def getAlea(self, alea):
        self.setProgressInfo(f"{alea.layerName()}: extraction enjeux")
        self.waitings[alea.actif()] = {'done':False, 'addicts':{}, 'enjeu':False}
        if self.dock.param.get('buffer'):
            self.waitings[alea.actif()]['buffer'] = False
        alea.cluster = self.dock.param.get('cluster')
        if self.mode.currentData()=='pdf':
            self.dock.psql.getExtractAlea(alea=alea, callback=self.waitingRoom, args=(alea,))
            elt = self.atlasChoice.currentData()
            if elt is not None:
                atlas = rdiLayer(elt)
                atlas.isStat = True
                self.dock.psql.getExtractEnjeu(enjeu=atlas, aleaList=[alea], callback=self.setAtlas, args=(alea,))
            else:
                self.getEnjeux(alea)
        if self.mode.currentData()=='gpkg':
            self.waitings[alea.actif()]['buffer'] = True
            vlayer, geomType = self.dock.psql.getLayer(alea.tableName(), alea.layerName(), alea.schemaName())
            if self.regroup.currentData()=="all":
                file = self.getFile(ext="gpkg")
            else:            
                file = self.getFile(alea=alea, ext="gpkg")
            name = alea.layerName()
            if self.regroup.currentData()=='all':
                name += ";"
            if self.regroup.currentData()=='alea':  
                name = "_"+name
            self.saveGPKG(alea, vlayer, file, name)
            self.waitingRoom(alea)
            self.getEnjeux(alea)
    
    def setAtlas(self, table, atlas, alea):
        self.atlasTable = table
        self.atlasRdi = atlas
        self.atlasAlea = alea
        self.getEnjeux(alea)
 
    def getEnjeux(self, alea):
        self.attemptStop()
        if self.enjeuData[alea]['target'] is not None and self.needClear():
            self.remove(self.enjeuData[alea]['target'])
        try:
            for enjeu in self.enjeuData[alea]['clear']:
                self.remove(enjeu)
        except:
            pass
        if len(self.enjeuData[alea]['list'])==0:   
            self.getAleas()
            return
        self.waitings[alea.actif()]['enjeu'] = True
        if self.enjeuParallax():
            for enjeu in self.enjeuMemo:
                self.waitings[alea.actif()]['addicts'][enjeu.actif()] = {'done':False}
            list = self.enjeuData[alea]['list'].copy()
            self.enjeuData[alea]['list'] = []
            for enjeu in list:
                self.getEnjeu(alea, enjeu)
            return 
        if self.split.isChecked():    
            self.enjeuData[alea]['theme'] = None
            if len(self.themeMemo)>0:
                theme = self.themeMemo.pop()
                self.enjeuData[alea]['theme'] = theme
                list = []
                for i in reversed(range(len(self.enjeuData[alea]['list']))):
                    enjeu = self.enjeuData[alea]['list'][i]
                    if self.matchThema(enjeu, theme):
                        self.waitings[alea.actif()]['addicts'][enjeu.actif()] = {'done':False}
                        list.append(enjeu)
                        self.enjeuData[alea]['list'].pop(i)
                self.enjeuData[alea]['clear'] = list
                for enjeu in list:
                    self.getEnjeu(alea, enjeu)
                if len(list)==0:
                    self.getEnjeux(alea)
                return
        enjeu = self.enjeuData[alea]['list'].pop()
        self.waitings[alea.actif()]['addicts'][enjeu.actif()] = {'done':False}
        self.enjeuData[alea]['target'] = enjeu
        enjeu.layer = None
        self.getEnjeu(alea, enjeu)
    
    def getEnjeu(self, alea, enjeu):
        enjeu.cluster = self.dock.param.get('cluster')
        self.dock.psql.getExtractEnjeu(enjeu=enjeu, aleaList=[alea], callback=self.setExtract, args=(alea, enjeu))

    def setExtract(self, table, rdi, alea, enjeu):
        self.setProgress()
        vlayer, geomType = self.dock.psql.getLayer(table, rdi.layerName())
        rdi.geomType = geomType

        if self.mode.currentData()=='gpkg':
            name = enjeu.layerName()
            if self.regroup.currentData()=='one':
                file = self.getFile(alea=alea, enjeu=enjeu, ext="gpkg")
                self.saveGPKG(rdi, vlayer, file)
            else:
                file = self.getFile(alea=alea, ext="gpkg")
                if self.regroup.currentData()=='all':
                    name = alea.layerName() + ":" + name
                    file = self.getFile(ext="gpkg")
                self.saveGPKG(rdi, vlayer, file, name)
            self.waitingRoom(alea, enjeu)       

        if self.mode.currentData()=='pdf':
            rdi.affLayer(table, callback=self.waitingRoom, args=(alea, enjeu))              


    def waitingRoom(self, alea, enjeu=None, buffer=None):
        if enjeu is None:
            self.rect = alea.vlayer.extent()
            self.waitings[alea.actif()]['done'] = True
            if buffer:
                self.waitings[alea.actif()]['buffer'] = True
        else:
            self.waitings[alea.actif()]['addicts'][enjeu.actif()]['done'] = True
        if not self.waitings[alea.actif()]['done'] or not self.waitings[alea.actif()]['enjeu']:
            return
        if 'buffer' in self.waitings[alea.actif()] and not self.waitings[alea.actif()]['buffer']:
            return
        for e in self.waitings[alea.actif()]['addicts'].values():
            if not e['done']:
                return
        self.attemptStop()
        if self.mode.currentData()=='gpkg':
            pass
        if self.mode.currentData()=='pdf':
            self.setProgressInfo(f"{alea.layerName()}: génération PDF")
            
            if self.enjeuParallax():
                file = self.getFile(alea=alea, ext="pdf")
            elif self.split.isChecked() and self.enjeuData[alea]['theme']:
                file = self.getFile(alea=alea, ext="pdf", theme=self.enjeuData[alea]['theme'])
            else:
                file = self.getFile(alea=alea, enjeu=enjeu, ext="pdf")
            self.savePDF(file, alea.layerName())
            
        self.getEnjeux(alea)


    def saveGPKG(self, rdi, vlayer, file, name=None):
        rdi.affVLayer(vlayer)
        if name is not None:
            vlayer.setName(name)
        alg = processing.run("native:package", {
            'LAYERS': [vlayer],
            'OVARWRITE': True,
            'SAVE_STYLES': self.style.isChecked(),
            'OUTPUT': file
        })       

    def setLegendItems(self, node):
        for c in node.children():
            if isinstance(c, QgsLayerTreeGroup):
                group = c.name()
                if group not in self.excludeGroups:
                    self.rootLegend.addGroup(group)
                    self.setLegendItems(c)
            if isinstance(c, QgsLayerTreeLayer):
                group = c.parent().name()
                if group not in self.excludeGroups:
                    self.rootLegend.addLayer(c.layer())

    def getParam(self, key, type, default):
        val = self.dock.param.get(key)

    def setMasks(self):
        vlayer, geomType = self.dock.psql.getLayer(self.atlasTable, f"{self.atlasRdi.layerName()}_{self.atlasAlea.layerName()}_mask")
        self.atlasRdi.vlayer = vlayer
        self.atlasRdi.layer = self.dock.dyn.addInGroup(self.atlasRdi, 'mask')
        # self.atlasRdi.layer.setName("Pochoir")
        p = self.dock.param
        symbol = QgsFillSymbol().createSimple({'color':p.get('maskFill')})
        renderer = QgsRuleBasedRenderer(symbol)
        rules = renderer.rootRule()
        rules.removeChild(rules.children()[0])
        symbol = QgsFillSymbol().createSimple({'color':p.get('maskFill'), 'stroke':p.get('maskStroke'), 'stroke-width':str(p.get('maskWidth'))})
        if self.mask.isChecked():
            c = p.get('maskBurst')
            if not isinstance(c, list) or len(c)<2:
                c = ["#aa000000","#88ffffff"]
            sl = QgsShapeburstFillSymbolLayer(color=QtGui.QColor(c[0]), color2=QtGui.QColor(c[1]), blurRadius=15, useWholeShape=False, maxDistance=10)
            symbol.changeSymbolLayer(0, sl)
        rule = renderer.Rule(symbol=symbol, filterExp=f'"{self.atlasName.currentData()}"=@atlas_pagename', label="")
        rules.appendChild(rule)
        if self.mask.isChecked():
            r = QgsInvertedPolygonRenderer(renderer)
            self.atlasRdi.vlayer.setRenderer(r) 
        else:
            self.atlasRdi.vlayer.setRenderer(renderer)   

    def removeMasks(self):
        self.dock.dyn.closeAll('mask') 
    
    def autoLayout(self, project, name):
        layout = QgsPrintLayout(project)
        layout.initializeDefaults()
        map = QgsLayoutItemMap(layout)
        map.setRect(20, 20, 20, 20)
        map.setExtent(iface.mapCanvas().extent())
        map.attemptMove(QgsLayoutPoint(40, 25, QgsUnitTypes.LayoutMillimeters))
        map.attemptResize(QgsLayoutSize(250, 180, QgsUnitTypes.LayoutMillimeters))
        map.refresh()
        layout.addLayoutItem(map)
        legend = QgsLayoutItemLegend(layout)
        legend.setTitle("Légende")
        legend.attemptMove(QgsLayoutPoint(0, 20, QgsUnitTypes.LayoutMillimeters))
        layout.addLayoutItem(legend)
        title = QgsLayoutItemLabel(layout)
        title.setText("Enjeux en zone inondable")
        font = QtGui.QFont()
        font.setPointSize(20)
        font.setBold(True)
        title.setFont(font)
        title.attemptMove(QgsLayoutPoint(10, 3, QgsUnitTypes.LayoutMillimeters))
        title.attemptResize(QgsLayoutSize(280, 10, QgsUnitTypes.LayoutMillimeters))
        title.setHAlign(Qt.AlignCenter)
        layout.addLayoutItem(title)
        if self.atlasChoice.currentData() is not None:
            name += f' : [% "{self.atlasName.currentData()}" %]'
        title = QgsLayoutItemLabel(layout)
        title.setText(name)
        font = QtGui.QFont()
        font.setPointSize(16)
        title.setFont(font)
        title.attemptMove(QgsLayoutPoint(10, 13, QgsUnitTypes.LayoutMillimeters))
        title.attemptResize(QgsLayoutSize(280, 8, QgsUnitTypes.LayoutMillimeters))
        title.setHAlign(Qt.AlignCenter)
        layout.addLayoutItem(title)
        return layout
    
    def savePDF(self, file, name):
        project = QgsProject.instance()
        manager = project.layoutManager()
        layoutName = self.printer.currentData()
        if layoutName is None:
            layout = self.autoLayout(project, name)
        else:
            layout = manager.layoutByName(layoutName)
        map = layout.referenceMap()
        map.refresh()
        legend = None
        
        itemLabelMemo = {}
        for l in layout.items(): 
            if isinstance(l, QgsLayoutItemLegend):
                legend = l
            if isinstance(l, QgsLayoutItemLabel):
                txt = l.text()
                if txt.find(self.titleChars)>=0:
                    itemLabelMemo[l] = txt
                    txt = txt.replace(self.titleChars, name)
                    l.setText(txt)
        
        map.setFollowVisibilityPreset(False)
        if legend is not None:
            legend.setAutoUpdateModel(True)
            legend.setAutoUpdateModel(False)
            legend.refresh()
            self.rootLegend = QgsLayerTree()
            self.setLegendItems(project.layerTreeRoot())
            legend.model().setRootGroup(self.rootLegend)
            # legend.setAutoUpdateModel(True)
            
        
        exporter = QgsLayoutExporter(layout)
        settings = QgsLayoutExporter.PdfExportSettings()
        if self.multi.isChecked():
            settings.writeGeoPdf = True
            settings.includeGeoPdfFeatures = True
            if self.split.isChecked():
                self.setThemes()
                settings.exportThemes = self.coll.mapThemes()
            
        if self.atlasChoice.currentData() is not None:
            map.setAtlasDriven(True)
            map.setAtlasScalingMode(QgsLayoutItemMap.Auto)
            vlayer, geomType = self.dock.psql.getLayer(self.atlasTable, self.atlasRdi.layerName())
            atlas = layout.atlas()
            atlas.setCoverageLayer(vlayer)
            atlas.setEnabled(True)
            atlas.setPageNameExpression(f"{self.atlasName.currentData()}")
            atlas.setSortFeatures(True)
            atlas.setSortAscending(True)
            atlas.setSortExpression(f"{self.atlasName.currentData()}")
            self.setMasks()
            if self.multi.isChecked():
                base, f = os.path.split(file)
                f = f.replace(".pdf", "")
                base = os.path.join(base, f) + "/"
                if not os.path.exists(base):
                    os.mkdir(base)
                atlas.setFilenameExpression(f"'{f}_'||@atlas_pagename")
                exporter.exportToPdfs(atlas, base, settings)
            else:
                file = file.replace(".pdf", "_atlas.pdf")
                exporter.exportToPdf(atlas, file, settings)
            self.removeMasks()   
        else:
            map.zoomToExtent(self.rect)
            exporter.exportToPdf(file, settings)
            
        for l, txt in itemLabelMemo.items(): 
            l.setText(txt)
            

    def getFile(self, alea=None, enjeu=None, theme=None, ext="", occ=None):
        name = ""
        folder = self.folder
        if ext!="" and ext[:1]!=".":
            ext = "."+str(ext)
        if alea is None and enjeu is None:
            name = "rdi"
        if alea is not None:
            name = alea.layerName()
            bloc = alea.getValueFromMeta('bloc')
            if occ is not None:
                bloc = occ
            if bloc and bloc!="":
                folder = os.path.join(folder, bloc)
                if not os.path.exists(folder):
                    os.mkdir(folder)
        if theme is not None:
            if name != "":
                name += "_"
            name += theme
        elif enjeu is not None:
            if name != "":
                name += "_"
            name += enjeu.layerName()
        return os.path.join(folder, self.getName(name)+str(ext))
    
    def getName(self, name):
        name = name.replace(" ","_")
        name = "".join(x for x in name if x.isalnum() or x=='_')
        return name

    def remove(self, rdi):
        rdi.layer = self.dock.dyn.removeLayer(rdi)


    def initData(self):
        self.group = {
            'one': "Aucun",
            'alea': f"Par {self.dock.param.get('alea')}",
            'all': "Tout en un",
        }
        self.options = {
            'gpkg': "GeoPackage",
            'pdf': "Cartes PDF"
        }
        self.enjeu = {
            'separate': "Cartes thématiques",
            'dynamic': "PDF multicouches"
        }
        self.excludeCodes = ['enjeuLight', 'buff', 'stats', 'mask']
        self.excludeGroups = [self.dock.param.get(c) for c in self.excludeCodes]
        self.titleChars = "__rdiAuto__"
        
        
        
        
    def saveSynapse(self):
        list = self.getList('alea')
        if self.aleaCB.isChecked(): 
            list = self.dock.alea.getEltsSelected()
        for alea in list:
            vlayer, geomType = self.dock.psql.getLayer(alea.tableName(), alea.layerName(), alea.schemaName())
            name = alea.layerName()
            
            occ = alea.topAttr('name')
            file = self.getFile(alea=alea, ext="shp", occ=occ)
            
            alg = processing.run("native:dissolve", {
                'INPUT': vlayer,
                'OUTPUT': 'memory:'
            })
            
            l = alg['OUTPUT'].fields().names()
            for a in l:
                alg = processing.run("qgis:deletecolumn", {
                    'INPUT': alg['OUTPUT'],
                    'COLUMN': a,
                    'OUTPUT': 'memory:'
                })
                
            if self.dock.psql.context['simplify']>0:
                alg = processing.run("native:simplifygeometries", {
                    'INPUT': alg['OUTPUT'],
                    'TOLERANCE': self.dock.psql.context['simplify'],
                    'OUTPUT': 'memory:'
                })
            if self.dock.psql.context['buffer']>0:
                alg = processing.run("native:buffer", {
                    'INPUT': alg['OUTPUT'],
                    'DISTANCE': self.dock.psql.context['buffer'],
                    'OUTPUT': 'memory:'
                })
            
            
            alg = processing.run("native:savefeatures", {
                'INPUT': alg['OUTPUT'],
                'LAYER_NAME': name,
                'OUTPUT': file
            })  



   