# -*- coding: utf-8 -*-
"""
/***************************************************************************
 RAVIDialog
                                 A QGIS plugin
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                             -------------------
        begin                : 2024-10-24
        git sha              : $Format:%H$
        copyright            : (C) 2024 by Caio Arantes
        email                : github.com/caioarantes
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

import os
import sys
import importlib
import platform
import subprocess
import zipfile
import json
import webbrowser
import io
import array
from datetime import datetime, timedelta
from dateutil.relativedelta import relativedelta

# PyQt5 and QGIS imports
from PyQt5.QtCore import QDate, Qt, QVariant
from PyQt5.QtWidgets import (
    QApplication, QMainWindow, QLabel, QMessageBox, QFileDialog, QGridLayout, 
    QWidget, QDesktopWidget, QDialog, QVBoxLayout, QCheckBox, QDialogButtonBox, 
    QDateEdit, QScrollArea, QPushButton, QHBoxLayout, QToolButton, QTextBrowser
)
from qgis.PyQt import uic, QtWidgets
from qgis.core import (
    QgsProject, QgsRasterLayer, QgsRasterShader, QgsColorRampShader, 
    QgsSingleBandPseudoColorRenderer, QgsStyle, QgsRasterBandStats, 
    QgsMapLayer, QgsVectorLayer, QgsColorRamp, QgsLayerTreeLayer, 
    QgsCoordinateReferenceSystem, QgsCoordinateTransform, 
    QgsMultiBandColorRenderer, QgsContrastEnhancement, 
    QgsProcessingFeedback, QgsApplication, QgsRectangle, 
    QgsFeature, QgsGeometry, QgsField, QgsVectorFileWriter,
)
from qgis.utils import iface
import qgis

from PyQt5.QtWidgets import QSizePolicy

# Scientiffic and data processing libraries
import pandas as pd
import numpy as np
from scipy.signal import savgol_filter
import geopandas as gpd
import requests

# Plotly imports for visualization
import plotly.express as px
import plotly.graph_objects as go
from plotly.subplots import make_subplots

# Processing module for QGIS
import processing

# Earth Engine installation and import
def install_earthengine_api():
    try:
        import pip
        pip_args = ['install', 'earthengine-api==1.3.1']
        pip.main(pip_args)
        #print("Earth Engine API installed successfully.")
    except AttributeError:
        from pip._internal.cli.main import main as pip_main
        pip_main(['install', 'earthengine-api==1.3.1'])
        print("Earth Engine API installed successfully.")
    except Exception as e:
        print(f"An error occurred during installation: {e}")

# Check if the Earth Engine API is already installed
try:
    importlib.import_module('ee')
    print("Earth Engine API is already installed.")
    import ee
except ImportError:
    print("Earth Engine API not found. Installing...")
    install_earthengine_api()
    try:
        importlib.import_module('ee')
        print("Earth Engine API imported successfully.")
        import ee
    except ImportError:
        print("Earth Engine API could not be imported after installation.")


# This loads your .ui file so that PyQt can populate your plugin with the elements from Qt Designer
FORM_CLASS, _ = uic.loadUiType(os.path.join(
    os.path.dirname(__file__), 'ravi_dialog_base.ui'))

class RAVIDialog(QtWidgets.QDialog, FORM_CLASS):
    def __init__(self, parent=None):
        """Constructor."""
        super(RAVIDialog, self).__init__(parent)
        
        self.setWindowFlags(Qt.Window | Qt.WindowMinimizeButtonHint)
        super().__init__(None, Qt.WindowStaysOnTopHint | Qt.WindowMinimizeButtonHint | Qt.WindowCloseButtonHint)

        # #widgets-and-dialogs-with-auto-connect
        self.setupUi(self)  

        # Connect signals and slots
        self.autenticacao.clicked.connect(self.auth)
        self.desautenticacao.clicked.connect(self.auth_clear)
        self.update_vector.clicked.connect(self.load_vector_layers)
        self.update_vector_2.clicked.connect(self.load_vector_layers)
        self.tabWidget.currentChanged.connect(self.on_tab_changed)
        self.load_1index.clicked.connect(self.first_index)
        self.load_1rgb.clicked.connect(self.first_rgb)
        self.composicao.clicked.connect(self.composicao_clicked)
        self.clear_raster.clicked.connect(self.clear_all_raster_layers)
        self.hybrid.clicked.connect(self.hybrid_function)
        self.QPushButton_next.clicked.connect(self.next_clicked)
        self.QPushButton_next_2.clicked.connect(self.next_clicked)
        self.QPushButton_next_3.clicked.connect(self.next_clicked)
        self.QPushButton_next_4.clicked.connect(self.next_clicked)
        self.QPushButton_next_5.clicked.connect(self.next_clicked)
        self.QPushButton_next_6.clicked.connect(self.next_clicked)
        self.QPushButton_next_7.clicked.connect(self.next_clicked)
        self.QPushButton_back.clicked.connect(self.back_clicked)
        self.QPushButton_back_2.clicked.connect(self.back_clicked)
        self.QPushButton_back_3.clicked.connect(self.back_clicked)
        self.QPushButton_back_4.clicked.connect(self.back_clicked)
        self.QPushButton_back_5.clicked.connect(self.back_clicked)
        self.QPushButton_back_6.clicked.connect(self.back_clicked)
        self.QPushButton_back_7.clicked.connect(self.back_clicked)
        self.QPushButton_back_8.clicked.connect(self.back_clicked)
        self.loadtimeseries.clicked.connect(self.loadtimeseries_clicked)
        self.loadtimeseries_2.clicked.connect(self.loadtimeseries_clicked)
        self.navegador.clicked.connect(self.open_browser)
        self.datasrecorte.clicked.connect(self.datasrecorte_clicked)
        self.salvar.clicked.connect(self.salvar_clicked)
        self.salvar_nasa.clicked.connect(self.salvar_nasa_clicked)
        self.build_vector_layer.clicked.connect(self.build_vector_layer_clicked)

        self.QCheckBox_sav_filter.stateChanged.connect(self.plot_timeseries)
        self.filtro_grau.currentIndexChanged.connect(self.plot_timeseries)
        self.window_len.currentIndexChanged.connect(self.plot_timeseries)
        self.vector_layer_combobox.currentIndexChanged.connect(self.get_selected_layer_path)
        self.mQgsFileWidget.fileChanged.connect(self.on_file_changed)  

        
        self.radioButton_all.clicked.connect(self.all_clicked)
        self.radioButton_3months.clicked.connect(self.last_3m_clicked)
        self.radioButton_6months.clicked.connect(self.last_6m_clicked)
        self.radioButton_12months.clicked.connect(self.last_12m_clicked)
        self.radioButton_3years.clicked.connect(self.last_3_years_clicked)
        self.radioButton_5years.clicked.connect(self.last_5_years_clicked)  
        self.radioButton_select_year.clicked.connect(self.selected_year_clicked)
        self.combo_year.currentIndexChanged.connect(self.selected_year_clicked)
        self.horizontalSlider_local_pixel_limit.valueChanged.connect(self.update_labels)
        self.horizontalSlider_aio_cover.valueChanged.connect(self.update_labels)
        self.horizontalSlider_buffer.valueChanged.connect(self.update_labels)
        self.horizontalSlider_total_pixel_limit.valueChanged.connect(self.update_labels)
        self.series_indice_2.currentIndexChanged.connect(self.reload_update2)
        self.series_indice.currentIndexChanged.connect(self.reload_update)
        self.incioedit_2.dateChanged.connect(self.reload_update2)
        self.finaledit_2.dateChanged.connect(self.reload_update2)
        self.incioedit.dateChanged.connect(self.reload_update)
        self.finaledit.dateChanged.connect(self.reload_update)
        self.vector_layer_combobox_2.currentIndexChanged.connect(self.combobox_2_update) 


        self.nasapower.clicked.connect(self.open_nasapower)
        self.clear_nasa.clicked.connect(self.clear_nasa_clicked)
        self.textEdit.setReadOnly(True)  # Prevent editing
        self.textEdit.setTextInteractionFlags(Qt.TextBrowserInteraction)
        self.textEdit.anchorClicked.connect(self.open_link)
        self.textBrowser_valid_pixels.setReadOnly(True)
        self.textBrowser_valid_pixels.setTextInteractionFlags(Qt.TextBrowserInteraction)
        self.textBrowser_valid_pixels.anchorClicked.connect(self.open_link)

        self.series_indice.currentIndexChanged.connect(self.index_explain)
        
        # Set default dates
        self.autentication = False
        self.folder_set = False
        self.inicio = None
        self.final = None
        self.nuvem = None
        self.vector_path = None
        self.aoi = None
        self.aoi_ckecked = False
        self.df = None
        self.recorte_datas = None
        self.df_aux = None
        self.selected_dates = []
        self.output_folder = None
        self.df_nasa = None
        self.daily_precipitation = None

        self.last_12m_clicked()
        
        self.index_explain()

        self.tabWidget.setCurrentIndex(0)
        self.resizeEvent('small')

    def combobox_2_update(self):
        self.vector_layer_combobox.setCurrentIndex(self.vector_layer_combobox_2.currentIndex())

    def reload_update2(self):
        self.finaledit.setDate(self.finaledit_2.date())
        self.incioedit.setDate(self.incioedit_2.date())
        self.series_indice.setCurrentIndex(self.series_indice_2.currentIndex())

    def reload_update(self):
        self.finaledit_2.setDate(self.finaledit.date())
        self.incioedit_2.setDate(self.incioedit.date())
        self.series_indice_2.setCurrentIndex(self.series_indice.currentIndex())
        

    def clear_nasa_clicked(self):
        """
        Clears the NASA data and updates the timeseries plot.

        This method sets the `df_nasa` attribute to None, effectively clearing any
        existing NASA data. After clearing the data, it calls the `plot_timeseries`
        method to update the timeseries plot accordingly.
        """
        self.df_nasa = None
        self.plot_timeseries()

    def hybrid_function(self):
        """
        Adds a Google Hybrid XYZ tile layer to the QGIS project if it is not already present.
        This function performs the following steps:
        1. Checks if the "Google Hybrid" layer is already added to the project.
        2. If not, creates an XYZ tile layer with the Google Hybrid URL.
        3. Adds the layer to the QGIS project.
        4. Sets the project Coordinate Reference System (CRS) to EPSG:4326 (WGS 84).
        5. Adjusts the layer's visibility and adds it to the layer tree.
        6. Refreshes the map canvas and zooms to the full extent.
        Raises:
            Exception: If there is an error loading the Google Hybrid layer.
        Notes:
            - The function assumes that the QGIS environment is properly initialized.
            - The function uses the QGIS Python API (PyQGIS) to interact with the QGIS project and map canvas.
        """
        existing_layers = QgsProject.instance().mapLayers().values()
        layer_names = [layer.name() for layer in existing_layers]
        if "Google Hybrid" in layer_names:
            self.pop_aviso("Google Hybrid layer already added.")
            return
        
        google_hybrid_url = "type=xyz&zmin=0&zmax=20&url=https://mt1.google.com/vt/lyrs%3Dy%26x%3D{x}%26y%3D{y}%26z%3D{z}"
        layer_name = "Google Hybrid"
        provider_type = "wms"

        try:
            # Create the XYZ tile layer
            google_hybrid_layer = QgsRasterLayer(google_hybrid_url, layer_name, provider_type)

            if google_hybrid_layer.isValid():
                # Add the layer to the project
                QgsProject.instance().addMapLayer(google_hybrid_layer, False)

                # Set the project CRS to EPSG:4326 (WGS 84)
                crs_4326 = QgsCoordinateReferenceSystem("EPSG:4326")
                QgsProject.instance().setCrs(crs_4326)

                # Adjust visibility and add to the layer tree
                google_hybrid_layer.setOpacity(1)
                root = QgsProject.instance().layerTreeRoot()
                root.addLayer(google_hybrid_layer)

                # Refresh the canvas and zoom to extent
                iface.mapCanvas().refresh()
                iface.mapCanvas().zoomToFullExtent()
                print(f"{layer_name} layer added successfully in EPSG:4326.")
            else:
                print(f"Failed to load {layer_name}. Invalid layer.")
        except Exception as e:
            print(f"Error loading {layer_name}: {e}")
        project = QgsProject.instance()
        project.setCrs(QgsCoordinateReferenceSystem("EPSG:4326"))

    def update_labels(self):
        """
        Updates the text of several labels based on the values of horizontal sliders.

        This method updates the text of the following labels:
        - customfilter: Displays a custom filter message with the percentage of non-cloudy pixels within AOI.
        - label_coverage: Displays the percentage value of coverage.
        - label_cloud: Displays the percentage value of cloud coverage.

        The values are retrieved from the corresponding horizontal sliders:
        - horizontalSlider: Used for the custom filter percentage.
        - horizontalSlider_2: Used for the coverage percentage.
        - horizontalSlider_total_pixel_limit: Used for the cloud coverage percentage.
        """
        self.label_cloud_aoi.setText(f"{self.horizontalSlider_local_pixel_limit.value()}%")
        self.label_coverage.setText(f"{self.horizontalSlider_aio_cover.value()}%")
        self.label_cloud.setText(f"{self.horizontalSlider_total_pixel_limit.value()}%")
        self.label_buffer.setText(f"{self.horizontalSlider_buffer.value()}m")

    def custom_filter_clicked(self):
        """
        Slot method to handle the custom filter checkbox click event.

        This method is triggered when the custom filter checkbox is clicked.
        It enables or disables the horizontal slider based on the state of the checkbox.

        If the custom filter checkbox is checked, the horizontal slider is enabled.
        If the custom filter checkbox is unchecked, the horizontal slider is disabled.

        Returns:
            None
        """
        if self.customfilter.isChecked():
            self.horizontalSlider.setEnabled(True)
        else:
            self.horizontalSlider.setEnabled(False)

    def open_link(self, url):
        """Open the clicked link in the default web browser."""
        print(f"Opening URL: {url.toString()}")
        webbrowser.open(url.toString())

    def last_6m_clicked(self):
        today = datetime.today().strftime('%Y-%m-%d')
        one_month_ago = (datetime.today() - relativedelta(months=6)).strftime('%Y-%m-%d')
        self.finaledit.setDate(QDate.fromString(today, 'yyyy-MM-dd'))
        self.incioedit.setDate(QDate.fromString(one_month_ago, 'yyyy-MM-dd'))

    def last_3m_clicked(self):
        today = datetime.today().strftime('%Y-%m-%d')
        one_month_ago = (datetime.today() - relativedelta(months=3)).strftime('%Y-%m-%d')
        self.finaledit.setDate(QDate.fromString(today, 'yyyy-MM-dd'))
        self.incioedit.setDate(QDate.fromString(one_month_ago, 'yyyy-MM-dd'))

    def all_clicked(self):
        today = datetime.today().strftime('%Y-%m-%d')
        since = '2017-03-28'
        self.finaledit.setDate(QDate.fromString(today, 'yyyy-MM-dd'))
        self.incioedit.setDate(QDate.fromString(since, 'yyyy-MM-dd'))

    def last_12m_clicked(self):
        today = datetime.today().strftime('%Y-%m-%d')
        one_month_ago = (datetime.today() - relativedelta(months=12)).strftime('%Y-%m-%d')
        self.finaledit.setDate(QDate.fromString(today, 'yyyy-MM-dd'))
        self.incioedit.setDate(QDate.fromString(one_month_ago, 'yyyy-MM-dd'))

    def last_3_years_clicked(self):
        """
        Slot method to handle the event when the "Last 3 Years" button is clicked.

        This method sets the date range in the UI to the last 3 years from today.
        It updates the 'finaledit' widget to today's date and the 'incioedit' widget
        to the date 36 months (3 years) ago from today.
        """

        today = datetime.today().strftime('%Y-%m-%d')
        one_month_ago = (datetime.today() - relativedelta(months=36)).strftime('%Y-%m-%d')
        self.finaledit.setDate(QDate.fromString(today, 'yyyy-MM-dd'))
        self.incioedit.setDate(QDate.fromString(one_month_ago, 'yyyy-MM-dd'))

    def last_5_years_clicked(self):
        """
        Updates the date fields to reflect the current date and the date from five years ago.

        This method sets the 'finaledit' field to the current date and the 'incioedit' field to the date 
        exactly five years prior to the current date. The dates are formatted as 'YYYY-MM-DD'.
        """
        today = datetime.today().strftime('%Y-%m-%d')
        one_month_ago = (datetime.today() - relativedelta(months=60)).strftime('%Y-%m-%d')
        self.finaledit.setDate(QDate.fromString(today, 'yyyy-MM-dd'))
        self.incioedit.setDate(QDate.fromString(one_month_ago, 'yyyy-MM-dd'))

    def selected_year_clicked(self):
        """
        Slot method to handle the event when the "Select Year" radio button is clicked.

        This method sets the date range in the UI to the selected year from the combo box.
        It updates the 'finaledit' field to the end of the selected year and the 'incioedit' field to the start of the selected year.
        """
        year = self.combo_year.currentText()
        start = f"{year}-01-01"
        end = f"{year}-12-31"
        self.incioedit.setDate(QDate.fromString(start, 'yyyy-MM-dd'))
        self.finaledit.setDate(QDate.fromString(end, 'yyyy-MM-dd'))

    def build_vector_layer_clicked(self):
        """
        Handles the event when the "Build Vector Layer" button is clicked.

        This method performs the following steps:
        1. Sets the CRS (Coordinate Reference System) of the project to EPSG:4326.
        2. Checks if the output folder is selected. If not, displays a warning message and exits.
        3. Checks if the "Google Hybrid" layer is loaded. If not, displays a warning message and exits.
        4. Retrieves the extent of the current map canvas.
        5. Creates a polygon representing the extent and adds it to a new vector layer.
        6. Saves the vector layer as a shapefile in the specified output folder.
        7. Loads the shapefile into the map canvas.
        8. If the loaded layer's CRS is not EPSG:4326, reprojects the layer to EPSG:4326.
        9. Adds the reprojected layer to the project and optionally sets the canvas extent to the layer extent.
        """

        project = QgsProject.instance()
        project.setCrs(QgsCoordinateReferenceSystem("EPSG:4326"))
        if self.output_folder is None:
            self.pop_aviso_auth("Please select a output folder first.")
            return

        existing_layers = QgsProject.instance().mapLayers().values()
        layer_names = [layer.name() for layer in existing_layers]
        if "Google Hybrid" not in layer_names:
            self.pop_aviso_auth("Please load the Google Hybrid layer first.")
            return

        # Get the extent of the current map canvas
        canvas = iface.mapCanvas()
        extent = canvas.extent()

        # Create a polygon (rectangle) representing the extent
        rect = QgsRectangle(extent)

        crs = "EPSG:4326"  # Use CRS 4326
        # Create a vector layer with the extent as a polygon
        layer = QgsVectorLayer(f"Polygon?crs={crs}", "canvas_extent", "memory")
        pr = layer.dataProvider()

        # Add fields (if necessary)
        pr.addAttributes([QgsField("id", QVariant.Int)])
        layer.updateFields()

        # Create a feature for the extent
        feature = QgsFeature()
        geometry = QgsGeometry.fromRect(rect)
        feature.setGeometry(geometry)
        feature.setAttributes([1])  # You can set any attribute value here
        pr.addFeature(feature)

        # Update layer and add it to the map
        layer.updateExtents()

        shp_path = self.get_unique_filename("canvas_extent.shp")
        print(f"Shapefile path: {shp_path}")
        shp_name = os.path.basename(shp_path).replace(".shp", "")
        print(f"Shapefile name: {shp_name}")

        QgsVectorFileWriter.writeAsVectorFormat(layer, shp_path, "UTF-8", layer.crs(), "ESRI Shapefile")

        # Load the shapefile into the canvas
        loaded_layer = QgsVectorLayer(shp_path, shp_name, "ogr")
        if loaded_layer.isValid():
            # Check if the CRS is EPSG:4326
            if loaded_layer.crs().authid() != "EPSG:4326":
                # Reproject the layer to EPSG:4326
                crs_4326 = QgsCoordinateReferenceSystem("EPSG:4326")
                loaded_layer = processing.run("qgis:reprojectlayer", {
                    'INPUT': loaded_layer,
                    'TARGET_CRS': crs_4326,
                    'OUTPUT': 'memory:'
                })['OUTPUT']
                loaded_layer.setCrs(crs_4326)
                print(f"Layer reprojected to EPSG:4326")

            QgsProject.instance().addMapLayer(loaded_layer)
            #self.zoom_to_layer(shp_name)
            self.load_vector_layers()
            # iface.mapCanvas().setExtent(loaded_layer.extent())  # Optional: Set the canvas extent to the layer extent
        else:
            print("Failed to load the shapefile.")

    def salvar_clicked(self):
        """
        Handles the event when the save button is clicked.

        This method generates a filename based on the current selections in the
        series_indice and vector_layer_combobox widgets, opens a file dialog to
        allow the user to specify the save location, and writes the contents of
        the dataframe (df_aux) to a CSV file.

        The CSV file is encoded in Latin-1 and uses a comma as the separator.
        The file is saved with a '.csv' extension.

        Returns:
            None
        """
        df = self.df_aux
        try:
            df = df[['date','average_index','savitzky_golay_filtered', 'image_id']]
        except:
            df = df[['date','average_index','image_id']]      

        name = f"{self.series_indice.currentText()}_{self.vector_layer_combobox.currentText()}_time_series.csv"
        caminho, _ = QFileDialog.getSaveFileName(self, "Salvar", name, "CSV Files (*.csv)")
        if not caminho:
            return
        with open(caminho, 'w', encoding='latin-1') as arquivo:
            arquivo.write(df.to_csv(decimal='.', sep=',', index=False, encoding="latin-1", lineterminator='\n'))
        
        # Open the file after saving (platform-specific)
        if platform.system() == 'Windows':
            os.startfile(caminho)
        elif platform.system() == 'Darwin':  # macOS
            subprocess.call(['open', caminho])
        else:  # Linux and other Unix-like systems
            subprocess.call(['xdg-open', caminho])

    def salvar_nasa_clicked(self):

        if self.df_nasa is None:
            self.pop_aviso("No NASA data to save.")
            return
        
        df = self.daily_precipitation
        name = f"nasa_power_precipitation_{self.vector_layer_combobox.currentText()}.csv"

        caminho, _ = QFileDialog.getSaveFileName(self, "Salvar", name, "CSV Files (*.csv)")
        if not caminho:
            return
        with open(caminho, 'w', encoding='latin-1') as arquivo:
            arquivo.write(df.to_csv(decimal='.', sep=',', index=False, encoding="latin-1", lineterminator='\n'))
        
        # Open the file after saving (platform-specific)
        if platform.system() == 'Windows':
            os.startfile(caminho)
        elif platform.system() == 'Darwin':  # macOS
            subprocess.call(['open', caminho])
        else:  # Linux and other Unix-like systems
            subprocess.call(['xdg-open', caminho])

    def datasrecorte_clicked(self):
        """
        Opens a dialog for selecting specific dates for the time series.

        This method creates a dialog with checkboxes for each date, grouped by year and month.
        Users can select or deselect dates, months, or entire years for the time series.
        """
        dialog = QDialog(self)
        dialog.setWindowTitle("Date Selection for Time Series")
        dialog.setGeometry(100, 100, 400, 500)

        layout = QVBoxLayout(dialog)

        # Scroll Area for Checkboxes
        scroll_area = QScrollArea(dialog)
        scroll_area.setWidgetResizable(True)
        scroll_content = QWidget()
        scroll_layout = QVBoxLayout(scroll_content)

        self.checkboxes = []
        self.group_checkboxes = {}  # Store month group checkboxes
        self.year_checkboxes = {}  # Store year checkboxes
        self.group_widgets = {}  # Store month group content widgets

        # Group Dates by Year and Month
        self.df['date'] = pd.to_datetime(self.df['date'])  # Ensure dates are datetime objects
        grouped = self.df.groupby([self.df['date'].dt.year, self.df['date'].dt.month])

        # Organize by Year
        years = self.df['date'].dt.year.unique()
        for year in sorted(years):
            # Create a year-level widget
            year_widget = QWidget(dialog)
            year_layout = QVBoxLayout(year_widget)

            # Year-level checkbox (above all content for the year)
            year_checkbox = QCheckBox(f"Select All in {year}", dialog)
            year_checkbox.setChecked(
                True if self.recorte_datas is None else all(str(date.date()) in self.recorte_datas for date in self.df[self.df['date'].dt.year == year]['date'])
            )
            year_checkbox.stateChanged.connect(lambda state, yr=year: self.toggle_year_checkboxes(yr, state))
            scroll_layout.addWidget(year_checkbox)
            self.year_checkboxes[year] = year_checkbox

            # Indented content for the year
            year_content_widget = QWidget(dialog)
            year_content_layout = QVBoxLayout(year_content_widget)
            year_content_layout.setContentsMargins(20, 0, 0, 0)  # Add indentation for year content
            scroll_layout.addWidget(year_content_widget)

            # Add months under each year
            for (group_year, month), group in grouped:
                if group_year != year:
                    continue

                # Create a month-level widget
                group_label = f"{group_year}-{month:02d}"
                month_widget = QWidget(dialog)
                month_layout = QVBoxLayout(month_widget)

                # Month toggle button
                month_toggle_button = QToolButton(dialog)
                month_toggle_button.setText(f"▶ {group_label}")
                month_toggle_button.setCheckable(True)
                month_toggle_button.setChecked(True)
                month_toggle_button.setStyleSheet("text-align: left;")
                month_toggle_button.toggled.connect(
                    lambda checked, grp=month_widget, btn=month_toggle_button, lbl=group_label: self.toggle_group_visibility(grp, btn, lbl)
                )

                # Month-level checkbox
                group_checkbox = QCheckBox(f"Select All in {group_label}", dialog)
                group_checkbox.setChecked(
                    True if self.recorte_datas is None else all(str(date.date()) in self.recorte_datas for date in group['date'])
                )
                group_checkbox.stateChanged.connect(lambda state, grp=group_label: self.toggle_group_checkboxes(grp, state))
                month_layout.addWidget(group_checkbox)
                self.group_checkboxes[group_label] = group_checkbox

                # Add individual checkboxes with further indentation
                for date in group['date']:
                    date_str = str(date.date())
                    checkbox = QCheckBox(date_str, dialog)
                    checkbox.setChecked(True if self.recorte_datas is None else date_str in self.recorte_datas)
                    month_layout.addWidget(checkbox)
                    checkbox.setContentsMargins(20, 0, 0, 0)  # Further indentation for dates
                    self.checkboxes.append((checkbox, group_label, group_year))

                # Add month layout to the year content layout
                year_content_layout.addWidget(month_toggle_button)
                year_content_layout.addWidget(month_widget)
                self.group_widgets[group_label] = month_widget

        scroll_content.setLayout(scroll_layout)
        scroll_area.setWidget(scroll_content)
        layout.addWidget(scroll_area)

        # Buttons
        button_box = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel, dialog)
        button_layout = QHBoxLayout()
        apply_button = QPushButton("Apply", dialog)
        select_button = QPushButton("Select All", dialog)
        deselect_button = QPushButton("Deselect All", dialog)
        button_layout.addWidget(apply_button)
        button_layout.addWidget(select_button)
        button_layout.addWidget(deselect_button)
        layout.addLayout(button_layout)
        layout.addWidget(button_box)

        # Signal Connections
        button_box.accepted.connect(dialog.accept)
        button_box.rejected.connect(dialog.reject)
        apply_button.clicked.connect(self.apply_changes)  # Apply without closing
        select_button.clicked.connect(self.select_all_checkboxes)
        deselect_button.clicked.connect(self.deselect_all_checkboxes)

        if dialog.exec_() == QDialog.Accepted:
            self.apply_changes()  # Ensure changes are applied before closing
        else:
            print("Time series dialog canceled. No changes made.")

    def apply_changes(self):
        """
        Apply changes to the selected dates without closing the dialog.

        This method updates the selected dates based on the checked checkboxes
        and adjusts the time series plot accordingly.
        """
        self.selected_dates = [cb.text() for cb, _, _ in self.checkboxes if cb.isChecked()]
        self.recorte_datas = self.selected_dates
        print(f"Selected dates for time series (applied): {self.recorte_datas}")
        self.df_ajust()
        self.plot_timeseries()

    def toggle_group_visibility(self, group_widget, toggle_button, group_label):
        """
        Toggle the visibility of a group widget.

        Args:
            group_widget (QWidget): The widget representing the group.
            toggle_button (QToolButton): The button used to toggle the visibility.
            group_label (str): The label of the group.
        """
        group_widget.setVisible(toggle_button.isChecked())
        toggle_button.setText(f"▶ {group_label}" if not toggle_button.isChecked() else f"▼ {group_label}")

    def toggle_group_checkboxes(self, group_label, state):
        """
        Toggle all checkboxes in a month group.

        Args:
            group_label (str): The label of the group.
            state (int): The state of the checkbox (checked or unchecked).
        """
        for checkbox, group, _ in self.checkboxes:
            if group == group_label:
                checkbox.setChecked(state == Qt.Checked)

    def toggle_year_checkboxes(self, year, state):
        """
        Toggle all checkboxes in a year group.

        Args:
            year (int): The year to toggle.
            state (int): The state of the checkbox (checked or unchecked).
        """
        for checkbox, _, group_year in self.checkboxes:
            if group_year == year:
                checkbox.setChecked(state == Qt.Checked)
        for group_label, group_checkbox in self.group_checkboxes.items():
            if group_label.startswith(str(year)):
                group_checkbox.setChecked(state == Qt.Checked)

    def select_all_checkboxes(self):
        """Select all checkboxes."""
        for checkbox, _, _ in self.checkboxes:
            checkbox.setChecked(True)
        for group_checkbox in self.group_checkboxes.values():
            group_checkbox.setChecked(True)
        for year_checkbox in self.year_checkboxes.values():
            year_checkbox.setChecked(True)

    def deselect_all_checkboxes(self):
        """Deselect all checkboxes."""
        for checkbox, _, _ in self.checkboxes:
            checkbox.setChecked(False)
        for group_checkbox in self.group_checkboxes.values():
            group_checkbox.setChecked(False)
        for year_checkbox in self.year_checkboxes.values():
            year_checkbox.setChecked(False)

    def centralizar(self):
        """
        Centers the window on the screen without changing the screen the dialog is on.

        This method calculates the geometry of the window frame and moves it to the center of the available screen space on the current screen.
        """
        # 1. Get the current geometry of the window frame.
        qtRectangle = self.frameGeometry()
        # 2. Determine the center point of the available screen space on the current screen.
        screen = QDesktopWidget().screenNumber(self)
        centerPoint = QDesktopWidget().availableGeometry(screen).center()
        # 3. Move the center of the window frame to the center point of the screen.
        qtRectangle.moveCenter(centerPoint)
        # 4. Move the window to the new top-left position.
        self.move(qtRectangle.topLeft())

    def resizeEvent(self, size):
    
        self.setMinimumSize(0, 0)  # Remove minimum size constraint
        self.setMaximumSize(16777215, 16777215)  # Rem

        if size == 'small':
            self.resize(663, 373)
            self.setFixedSize(self.width(), self.height())  # Lock to small size
        elif size == 'big':
            self.resize(1145, 582)
            self.centralizar()
            self.setFixedSize(self.width(), self.height())  # Lock to big size

    def on_tab_changed(self, index):
        #print(f"Tab changed to index: {index}")
        if not self.autentication:
            self.tabWidget.setCurrentIndex(0)
            return
        if index in [0, 1, 2, 3, 4, 5, 6, 7, 8]:
            self.resizeEvent('small')
        elif index == 9:
            self.resizeEvent('big')
            self.centralizar()
        elif index == 9 and self.df is not None:
            self.resizeEvent('big')
            self.centralizar()
            self.plot_timeseries()

    def next_clicked(self):
        self.tabWidget.setCurrentIndex((self.tabWidget.currentIndex() + 1) % self.tabWidget.count())

    def back_clicked(self):
        self.tabWidget.setCurrentIndex((self.tabWidget.currentIndex() - 1) % self.tabWidget.count())

    def auth(self):
        """
        Authenticates Earth Engine and validates the default project.
        Warnings are displayed only if the default project is invalid.
        """
        try:
            # Step 1: Authenticate and initialize Earth Engine
            print("Authenticating Earth Engine...")
            ee.Authenticate()
            ee.Initialize()
            print("Authentication successful!")

            # Step 2: Test default project
            print("Testing default project...")
            default_project_path = "projects/earthengine-legacy/assets/"  # Replace with your default project's path if known

            # Attempt to list assets in the default project
            try:
                assets = ee.data.listAssets({'parent': default_project_path})
                print(f"Assets in default project: {assets}")

                if assets.get('assets') is not None:  # Valid project detected
                    print("Default project is valid.")
                    self.pop_aviso_auth("Authentication successful!")
                    self.autentication = True
                    self.load_vector_layers()
                    self.load_path_sugestion()
                    self.next_clicked()
                else:
                    print("Default project is valid but contains no assets.")  # No warning needed for this case
            except ee.EEException as e:
                # Invalid project or access issue
                print(f"Default project validation failed: {e}")
                self.pop_aviso_auth(f"Default project validation failed: {e}\nFollow the instructions to have a valid Google Cloud project.")
                self.auth_clear(True)


        except ee.EEException as e:
            # Handle Earth Engine-specific errors
            print(f"Earth Engine error: {e}")
            if "Earth Engine client library not initialized" in str(e):
                message = "Authentication failed. Please authenticate again."
                print(message)
                self.pop_aviso_auth(message)
            else:
                message = f"An error occurred during authentication or initialization: {e}"
                print(message)
                self.pop_aviso_auth(message)
                self.auth_clear(True)


        except Exception as e:
            # Handle unexpected errors
            message = f"An unexpected error occurred: {e}"
            print(message)
            self.pop_aviso_auth(message)

    def auth_clear(self, silent=False):
        #print('Desautenticando...')
        """Clears the Earth Engine authentication by deleting the credentials file."""
        
        system = platform.system()
        
        # Set the path for Earth Engine credentials based on the operating system
        if system == 'Windows':
            credentials_path = os.path.join(os.environ['USERPROFILE'], '.config', 'earthengine', 'credentials')
        elif system == 'Linux':
            credentials_path = os.path.join(os.environ['HOME'], '.config', 'earthengine', 'credentials')
        elif system == 'Darwin':  # MacOS
            credentials_path = os.path.join(os.environ['HOME'], 'Library', 'Application Support', 'earthengine', 'credentials')
        else:
            raise Exception(f"Unsupported operating system: {system}")

        # Check if the credentials file exists and delete it
        if os.path.exists(credentials_path):
            os.remove(credentials_path)
            if not silent:
                message = "Earth Engine authentication cleared successfully."
                self.pop_aviso_auth(message)

        else:
            message = "No Earth Engine credentials found to clear."
            self.pop_aviso_auth(message)

    def get_dates(self):
        # Get the date from the QDateEdit widget
        self.inicio = self.incioedit.date().toString("yyyy-MM-dd")

        self.final = self.finaledit.date().toString("yyyy-MM-dd")
        # print(f"Selected date: {self.inicio} to {self.final}")

                # Get the date from the QDateEdit widget

        # Get the selected text from the combobox
        self.nuvem = self.horizontalSlider_total_pixel_limit.value()
        # print(f"Nuvem limit: {self.nuvem}")

    def load_path_sugestion(self):
        """
        Load the path suggestion based on the user's operating system.
        """
        system = platform.system()
        if system == 'Windows':
            self.output_folder = os.path.join(os.environ['USERPROFILE'], 'Downloads')
        elif system == 'Linux':
            self.output_folder = os.path.join(os.environ['HOME'], 'Downloads')
        elif system == 'Darwin':  # MacOS
            self.output_folder = os.path.join(os.environ['HOME'], 'Downloads')

        # Pre-configure with a suggested directory
        self.mQgsFileWidget.setFilePath(self.output_folder)

    def pop_aviso_auth(self, aviso):
        """
        Displays a warning message box with the given message and Ok button.
        Args:
            aviso (str): The warning message to display in the message box.
        Returns:
            None
        Note:
            This method restores the override cursor before displaying the message box.
        """
        QApplication.restoreOverrideCursor()
        msg = QMessageBox(self)
        msg.setWindowTitle("Warning!")
        msg.setIcon(QMessageBox.Warning)
        msg.setText(aviso)
        
        # Set buttons with Ok on the right
        msg.setStandardButtons(QMessageBox.Ok)
        
        # Access the buttons to set custom text
        ok_button = msg.button(QMessageBox.Ok)
        ok_button.setText("Ok")
        
        msg.exec_()

    def pop_aviso(self, aviso):
        """
        Displays a warning message box with the given message and Ok/Cancel buttons.
        Args:
            aviso (str): The warning message to display in the message box.
        Returns:
            bool: True if the Ok button is clicked, False if the Cancel button is clicked.
        Note:
            This method restores the override cursor before displaying the message box.
        """
        QApplication.restoreOverrideCursor()
        msg = QMessageBox(self)
        msg.setWindowTitle("Warning!")
        msg.setIcon(QMessageBox.Warning)
        msg.setText(aviso)
        
        # Set buttons with Ok on the right and Cancel on the left
        msg.setStandardButtons(QMessageBox.Cancel | QMessageBox.Ok)
        
        # Access the buttons to set custom text
        cancel_button = msg.button(QMessageBox.Cancel)
        ok_button = msg.button(QMessageBox.Ok)
        cancel_button.setText("Cancel")
        ok_button.setText("Ok")
        
        ret = msg.exec_()  # Display the message box

        if ret == QMessageBox.Ok:
            print("Ok button clicked")
            return True
        elif ret == QMessageBox.Cancel:
            print("Cancel button clicked")
            return False

    def load_vector_layers(self):

        # Get all layers in the current QGIS project
        layers = list(QgsProject.instance().mapLayers().values())
        
        # Filter vector layers
        vector_layers = [layer for layer in layers if layer.type() == QgsMapLayer.VectorLayer]

        # Extract the names of vector layers
        current_layer_names = [self.vector_layer_combobox.itemText(i) for i in range(self.vector_layer_combobox.count())]

        # Identify newly added layers
        new_layers = [layer for layer in vector_layers if layer.name() not in current_layer_names]

        # Clear the combobox and the dictionary
        self.vector_layer_combobox.clear()
        self.vector_layer_ids = {}

        # Populate the combobox and update the dictionary
        for layer in vector_layers:
            layer_name = layer.name()
            self.vector_layer_combobox.addItem(layer_name)
            self.vector_layer_ids[layer_name] = layer.id()

        # If new layers are detected, set the combobox index to the most recent one
        if new_layers:
            last_added_layer = new_layers[-1]
            index = self.vector_layer_combobox.findText(last_added_layer.name())
            self.vector_layer_combobox.setCurrentIndex(index)

        self.vector_layer_combobox_2.clear()
        self.vector_layer_combobox_2.addItems(self.vector_layer_combobox.itemText(i) for i in range(self.vector_layer_combobox.count()))
        self.vector_layer_combobox_2.setCurrentIndex(self.vector_layer_combobox.currentIndex())

        # Call the method to handle the selected layer path
        self.get_selected_layer_path()

    def get_selected_layer_path(self):
        """
        Retrieves the path of the currently selected layer in the combobox and triggers further processing.
        """
        # Get the currently selected layer name from the combobox
        layer_name = self.vector_layer_combobox.currentText()
        self.zoom_to_layer(layer_name)
        print(f"Selected layer name: {layer_name}")  # Debug: Show selected layer name

        # Get the corresponding layer ID
        layer_id = self.vector_layer_ids.get(layer_name)

        # Get the layer using its ID
        layer = QgsProject.instance().mapLayer(layer_id)
        if layer:
            print(f"Layer found: {layer.name()}, ID: {layer_id}")  # Debug: Confirm layer is found
            self.selected_aio_layer_path = layer.dataProvider().dataSourceUri().split('|')[0]
            print(f"Selected layer path: {self.selected_aio_layer_path}")  # Debug: Show selected layer path
            
            # Trigger the processing function
            self.load_vector_function()

            # Enable next steps if necessary
            return None
        else:
            print(f"Layer '{layer_name}' with ID '{layer_id}' not found in the project.")
            return None

    def first_index(self):
        """
        Calculates a vegetation index, downloads the GeoTIFF, and adds it to the QGIS project
        as a styled raster layer, ensuring unique names for each layer.
        """
        try:
            print("First index clicked")
            QApplication.setOverrideCursor(Qt.WaitCursor)  # Set wait cursor for user feedback

            # Retrieve vegetation index and date inputs
            vegetation_index = self.imagem_unica_indice.currentText()
            date = [self.dataunica.currentText()]
            
            first_image = self.sentinel2.filter(
                    ee.Filter.inList('date', date)
                ).first()

            # Clip image to AOI
            first_image = first_image.clip(self.aoi)

            # Calculate the selected vegetation index
            if vegetation_index == 'NDVI':
                index_image = first_image.normalizedDifference(['B8', 'B4']).rename('NDVI')
            elif vegetation_index == 'GNDVI':
                index_image = first_image.normalizedDifference(['B8', 'B3']).rename('GNDVI')
            elif vegetation_index == 'EVI':
                index_image = first_image.expression(
                    '2.5 * ((NIR / 10000 - RED / 10000) / (NIR / 10000 + 6 * RED / 10000 - 7.5 * BLUE / 10000 + 1))', {
                        'NIR': first_image.select('B8'),
                        'RED': first_image.select('B4'),
                        'BLUE': first_image.select('B2')
                    }
                ).rename('EVI')
            elif vegetation_index == 'SAVI':
                L = 0.5  # Soil brightness correction factor
                index_image = first_image.expression(
                    '(1 + L) * ((NIR / 10000) - (RED / 10000)) / ((NIR / 10000) + (RED / 10000) + L)', {
                        'NIR': first_image.select('B8'),
                        'RED': first_image.select('B4'),
                        'L': L
                    }
                ).rename('SAVI')

            # Prepare download URL and output filename
            url = index_image.getDownloadUrl({
                'scale': 10,
                'region': self.aoi.geometry().bounds().getInfo(),
                'format': 'GeoTIFF'
            })
            base_output_file = f'{vegetation_index}_{date[0]}.tiff'
            output_file = self.get_unique_filename(base_output_file)

            # Download the image
            response = requests.get(url)
            if response.status_code == 200:
                with open(output_file, 'wb') as f:
                    f.write(response.content)
                print(f"{vegetation_index} image downloaded as {output_file}")
            else:
                print(f"Failed to download image. HTTP Status: {response.status_code}")
                return

            # Prepare unique layer name
            layer_name = f'{vegetation_index} {date[0]}'
            base_name = layer_name
            i = 1
            while QgsProject.instance().mapLayersByName(layer_name):
                layer_name = f"{base_name}_{i}"
                i += 1
            print(f"Layer name adjusted to '{layer_name}' to ensure uniqueness.")

            # Add raster layer with styling
            self.load_raster_layer_colorful(output_file, layer_name, vegetation_index)

        except Exception as e:
            print(f"An error occurred: {e}")
        finally:
            QApplication.restoreOverrideCursor()  # Restore the default cursor

    def first_rgb(self):
        """
        Fetches the first Sentinel-2 image for the selected date, downloads it,
        and adds it as an RGB layer in QGIS without duplication.
        """
        # Set the cursor to indicate processing
        QApplication.setOverrideCursor(Qt.WaitCursor)

        try:
            date = [self.dataunica.currentText()]
            print(f"Selected date: {date}")

            first_image = self.sentinel2.filter(
                                ee.Filter.inList('date', date)
                            ).first()

            # Print the number of days in the collection
            # num_days = first_image.size().getInfo()
            # print(f"Number of days in the collection (unique?): {num_days}")

            # Clip image to AOI
            first_image = first_image.clip(self.aoi)

            # Available bands: [B1, B2, B3, B4, B5, B6, B7, B8, B8A, B9, B11, B12, AOT, WVP, SCL, TCI_R, TCI_G, TCI_B, MSK_CLDPRB, MSK_SNWPRB, QA10, QA20, QA60, MSK_CLASSI_OPAQUE, MSK_CLASSI_CIRRUS, MSK_CLASSI_SNOW_ICE]
            bands = ['B1', 'B2', 'B3', 'B4', 'B5', 'B6', 'B7', 'B8', 'B8A', 'B9', 'B11', 'B12']
            first_image = first_image.select(bands)

            # Get the acquisition date and define download region
            region = self.aoi.geometry().bounds().getInfo()['coordinates']

            # Generate download URL
            try:
                url = first_image.getDownloadUrl({
                    'scale': 10,
                    'region': region,
                    'format': 'GeoTIFF'
                })
            except Exception as e:
                self.pop_aviso_auth(f"Failed to generate download URL: {e}")
                return

            # Define output file
            base_output_file = f'Sentinel2_AllBands_{date[0]}.tiff'
            output_file = self.get_unique_filename(base_output_file)

            # Download the image
            try:
                response = requests.get(url, stream=True)
                response.raise_for_status()
                with open(output_file, 'wb') as f:
                    for chunk in response.iter_content(chunk_size=8192):
                        if chunk:
                            f.write(chunk)
                print(f"Image downloaded to {output_file}")
            except requests.exceptions.RequestException as e:
                self.pop_aviso_auth(f"Error downloading image: {e}")
                return

            # Add the image as a raster layer in QGIS
            layer_name = f'Sentinel-2 RGB {date[0]}'
            existing_layers = QgsProject.instance().mapLayersByName(layer_name)

            layer = QgsRasterLayer(output_file, layer_name)
            if not layer.isValid():
                self.pop_aviso_auth(f"Failed to load the layer: {output_file}")
                return

            # Set min and max values for each band (Red, Green, Blue)
            min_val, max_val = 200, 2300

            # Create a new MultiBandColorRenderer for RGB
            renderer = QgsMultiBandColorRenderer(
                layer.dataProvider(),
                4,  # Red band (B4, 1-based index)
                3,  # Green band (B3)
                2   # Blue band (B2)
            )

            # Set contrast enhancement for each band (Red, Green, Blue)
            try:
                red_ce = QgsContrastEnhancement(layer.dataProvider().dataType(4))
                red_ce.setMinimumValue(min_val)
                red_ce.setMaximumValue(max_val)
                red_ce.setContrastEnhancementAlgorithm(QgsContrastEnhancement.StretchToMinimumMaximum)

                green_ce = QgsContrastEnhancement(layer.dataProvider().dataType(3))
                green_ce.setMinimumValue(min_val)
                green_ce.setMaximumValue(max_val)
                green_ce.setContrastEnhancementAlgorithm(QgsContrastEnhancement.StretchToMinimumMaximum)

                blue_ce = QgsContrastEnhancement(layer.dataProvider().dataType(2))
                blue_ce.setMinimumValue(min_val)
                blue_ce.setMaximumValue(max_val)
                blue_ce.setContrastEnhancementAlgorithm(QgsContrastEnhancement.StretchToMinimumMaximum)

                renderer.setRedContrastEnhancement(red_ce)
                renderer.setGreenContrastEnhancement(green_ce)
                renderer.setBlueContrastEnhancement(blue_ce)
            except AttributeError as ae:
                print(f"Error configuring renderer: {ae}")
                return
            except Exception as e:
                print(f"Unexpected error configuring renderer: {e}")
                return

            # Set the renderer to the layer
            layer.setRenderer(renderer)

            # Add the raster layer to the QGIS project and insert it at the top of the layer tree
            QgsProject.instance().addMapLayer(layer, False)
            root = QgsProject.instance().layerTreeRoot()
            root.insertChildNode(0, QgsLayerTreeLayer(layer))
            iface.setActiveLayer(layer)
        except Exception as e:
            self.pop_aviso_auth(f"An error occurred: {e}")
        finally:
            QApplication.restoreOverrideCursor()

    def zoom_to_layer(self, layer_name, margin_ratio=0.1):
        """
        Zoom to the specified layer with an optional margin.

        :param layer_name: Name of the layer to zoom to.
        :param margin_ratio: Fraction of the extent to add as margin (default is 0.1, or 10%).
        """
        project = QgsProject.instance()
        layers = project.mapLayersByName(layer_name)  # Get layers matching the name
        
        if not layers:
            print(f"Layer '{layer_name}' not found.")
            return
        
        layer = layers[0]  # Use the first matching layer
        iface = qgis.utils.iface  # Access the QGIS interface
        canvas = iface.mapCanvas()  # Get the active map canvas
        
        # Ensure the canvas CRS matches the layer CRS
        canvas.setDestinationCrs(layer.crs())
        
        # Get the layer's extent and add a margin
        layer_extent = layer.extent()
        x_margin = layer_extent.width() * margin_ratio
        y_margin = layer_extent.height() * margin_ratio
        
        expanded_extent = QgsRectangle(
            layer_extent.xMinimum() - x_margin,
            layer_extent.yMinimum() - y_margin,
            layer_extent.xMaximum() + x_margin,
            layer_extent.yMaximum() + y_margin
        )
        
        # Set the expanded extent to the canvas
        canvas.setExtent(expanded_extent)
        canvas.refresh()
        
        print(f"Zoomed to layer extent with margin: {expanded_extent.toString()}")

    def get_unique_filename(self, base_file_name):
        name, extension = os.path.splitext(base_file_name)
        output_file = os.path.join(self.output_folder, base_file_name)
        counter = 1

        while os.path.exists(output_file):
            output_file = os.path.join(self.output_folder, f"{name}_{counter}{extension}")
            counter += 1

        print(f"Unique filename: {output_file}")
        return output_file
    
    def composicao_clicked(self):
        QApplication.setOverrideCursor(Qt.WaitCursor)
        print("Composição de índices vegetativos")
        indice_vegetacao = self.indice_composicao.currentText()
        metrica = self.metrica.currentText()
        """
        Calculates and downloads an image from Sentinel-2 data for a specified 
        area of interest, time range, vegetation index, and metric. 
        If a file with the same name exists, it adds a numerical suffix 
        to the filename to avoid overwriting. Then loads the image into QGIS.
        """

        Date_list_selection = [date.strftime('%Y-%m-%d') for date in pd.to_datetime(self.df_aux['date']).tolist()] if 'date' in self.df_aux.columns else []
        print(f"Selected dates for time series: {Date_list_selection}")


        print("Final number of images before:", self.sentinel2.size().getInfo())
        # Print the dates of the images in the collection for debugging purposes
        dates_in_collection = self.sentinel2.aggregate_array('date').getInfo()
        print(f"Dates in the collection: {dates_in_collection}")

        print(f"Selected dates (timestamps): {Date_list_selection}")

        # Filter the collection by the dates
        sentinel2_selected_dates = self.sentinel2.filter(
            ee.Filter.inList('date', ee.List(Date_list_selection))
        )

        print("Final number of images after:", sentinel2_selected_dates.size().getInfo())

        # Calculate the specified vegetation index for each image in the collection
        def calculate_index(image):
            if indice_vegetacao == 'NDVI':
                return image.normalizedDifference(['B8', 'B4']).rename('NDVI')
            elif indice_vegetacao == 'EVI':
                return image.expression(
                    '2.5 * ((NIR / 10000 - RED / 10000) / (NIR / 10000 + 6 * RED / 10000 - 7.5 * BLUE / 10000 + 1))', {
                        'NIR': image.select('B8'),
                        'RED': image.select('B4'),
                        'BLUE': image.select('B2')
                    }).rename('EVI')
            elif indice_vegetacao == 'SAVI':
                return image.expression(
                    '(1 + L) * ((NIR / 10000) - (RED / 10000)) / ((NIR / 10000) + (RED / 10000) + L)', {
                        'NIR': image.select('B8'),
                        'RED': image.select('B4'),
                        'L': 0.5  # Adjust L value as needed
                    }).rename('SAVI')
            elif indice_vegetacao == 'GNDVI':
                return image.normalizedDifference(['B8', 'B3']).rename('GNDVI')         
            else:
                raise ValueError(f"Invalid indice_vegetacao: {indice_vegetacao}")

        index_collection = sentinel2_selected_dates.map(calculate_index)

        # Apply the specified metric to the vegetation index collection
        if metrica == 'Mean':
            final_image = index_collection.mean()
        elif metrica == 'Max':
            final_image = index_collection.max()
        elif metrica == 'Min':  
            final_image = index_collection.min()
        elif metrica == 'Median':
            final_image = index_collection.median()
        elif metrica == 'Amplitude':
            final_image = index_collection.max().subtract(index_collection.min())
        elif metrica == 'Standard Deviation':
            final_image = index_collection.reduce(ee.Reducer.stdDev())

        
        final_image = final_image.clip(self.aoi.geometry())

        url = final_image.getDownloadUrl({
            'scale': 10,
            'region': self.aoi.geometry().bounds().getInfo(),
            'format': 'GeoTIFF'
        })

        base_output_file = f'{metrica}_{indice_vegetacao}.tiff' 
        output_file = self.get_unique_filename(base_output_file)

        response = requests.get(url)
        with open(output_file, 'wb') as f:
            f.write(response.content)

        print(f"{indice_vegetacao} image downloaded as {output_file}")

        layer_name = f'{indice_vegetacao} {metrica}'
        
        base_output_file = f'{metrica}_{indice_vegetacao}_clipped.tiff' 
        output_file_clipped = self.get_unique_filename(base_output_file)

        self.clip_raster_by_vector(output_file, self.selected_aio_layer_path, output_file_clipped, layer_name)

        self.load_raster_layer_colorful(output_file_clipped, layer_name, indice_vegetacao)

    def clip_raster_by_vector(self, raster_path, shapefile_path, output_path, layer_name):
        print(f"Clipping raster {raster_path} by vector {shapefile_path} to {output_path}")
        # Load layers
        shapefile_layer = QgsVectorLayer(shapefile_path, "Clip Layer", "ogr")
        raster_layer = QgsRasterLayer(raster_path, "Raster Layer")

        # Check if layers loaded successfully
        if not shapefile_layer.isValid():
            print("Failed to load shapefile.")
        if not raster_layer.isValid():
            print("Failed to load raster.")

        # Clip raster using the shapefile
        result = processing.run(
            "gdal:cliprasterbymasklayer",
            {
                'INPUT': raster_layer,
                'MASK': shapefile_layer,
                'NODATA': -9999,  # Change to appropriate NoData value if needed
                'CROP_TO_CUTLINE': True,
                'KEEP_RESOLUTION': True,
                'OUTPUT': output_path
            },
            feedback=QgsProcessingFeedback()
        )

        print(f"Clipping result: {result}")

        # # Load the clipped raster into QGIS
        # clipped_layer = QgsRasterLayer(output_path, "Clipped Raster")
        # if clipped_layer.isValid():
        #     QgsProject.instance().addMapLayer(clipped_layer)
        #     print(f"Clipped raster saved and added to QGIS: {output_path}")
        # else:
        #     print("Failed to add the clipped raster to QGIS.")
        
    def on_file_changed(self, file_path):
        """Slot called when the selected file changes."""
        print(f"File selected: {file_path}")
        self.output_folder = file_path
        self.folder_set = True
        self.aoi_ckecked_function()
        # self.check_next_button()

    def index_explain(self):
        vegetation_indices = {
            "NDVI": """
                <h3>Normalized Difference Vegetation Index (NDVI)</h3>
                <p>
                    The Normalized Difference Vegetation Index (NDVI) is a widely used and well-established 
                    indicator of vegetation health and vigor. It exploits the contrasting spectral 
                    reflectance properties of plant pigments, particularly chlorophyll. 
                    Healthy vegetation strongly absorbs visible red light for photosynthesis while 
                    reflecting a significant portion of near-infrared (NIR) radiation. 
                    Conversely, non-vegetated areas like soil and water tend to reflect both red and 
                    NIR light more equally. 
                </p>
                <p>
                    The NDVI formula is calculated as follows:
                </p>
                <pre>
        NDVI = (NIR - RED) / (NIR + RED)
                </pre>
                <p>
                    where:
                    <ul>
                        <li><b>NIR</b>: Reflectance in the near-infrared band</li>
                        <li><b>RED</b>: Reflectance in the red band</li>
                    </ul>
                    By calculating the difference between NIR and red reflectance and normalizing it 
                    by their sum, NDVI effectively enhances the vegetation signal while minimizing 
                    the influence of factors like variations in illumination and atmospheric conditions. 
                    NDVI values typically range from -1 to 1. 
                    Higher values (closer to 1) generally indicate denser, healthier vegetation with 
                    higher leaf area and chlorophyll content. 
                    Lower values (closer to -1) often correspond to bare soil, water, or senescent 
                    (dying) vegetation.
                </p>
            """,
            "GNDVI": """
                <h3>Green Normalized Difference Vegetation Index (GNDVI)</h3>
                <p>
                    The Green Normalized Difference Vegetation Index (GNDVI) is a modification of NDVI 
                    that utilizes the green band of the electromagnetic spectrum instead of the red band. 
                    Chlorophyll, the primary pigment involved in photosynthesis, strongly absorbs 
                    blue and red light while reflecting green light. 
                    Therefore, GNDVI is particularly sensitive to variations in chlorophyll content 
                    within plant canopies.
                </p>
                <p>
                    The GNDVI formula is calculated as:
                </p>
                <pre>
        GNDVI = (NIR - GREEN) / (NIR + GREEN)
                </pre>
                <p>
                    where:
                    <ul>
                        <li><b>NIR</b>: Reflectance in the near-infrared band</li>
                        <li><b>GREEN</b>: Reflectance in the green band</li>
                    </ul>
                    This sensitivity makes GNDVI a valuable tool for:
                    <ul>
                        <li>Monitoring plant stress and nutrient deficiencies</li>
                        <li>Detecting early signs of disease or pest infestations</li>
                        <li>Assessing crop vigor and yield potential</li>
                        <li>Studying the impact of environmental factors on plant growth</li>
                    </ul>
                </p>
            """,
            "EVI": """
                <h3>Enhanced Vegetation Index (EVI)</h3>
                <p>
                    The Enhanced Vegetation Index (EVI) was developed to address some of the limitations 
                    of NDVI, particularly in areas of high biomass or atmospheric interference. 
                    EVI incorporates a blue band in its calculation, which helps to minimize the 
                    influence of atmospheric aerosols and soil background noise. 
                    Additionally, EVI uses a canopy background adjustment term to improve sensitivity 
                    in areas of high biomass and to better discriminate vegetation from non-vegetated 
                    surfaces.
                </p>
                <p>
                    The EVI formula is calculated as:
                </p>
                <pre>
        EVI = 2.5 * ((NIR - RED) / (NIR + 6 * RED - 7.5 * BLUE + 1))
                </pre>
                <p>
                    where:
                    <ul>
                        <li><b>NIR</b>: Reflectance in the near-infrared band</li>
                        <li><b>RED</b>: Reflectance in the red band</li>
                        <li><b>BLUE</b>: Reflectance in the blue band</li>
                    </ul>
                    EVI has proven to be highly effective in:
                    <ul>
                        <li>Monitoring vegetation dynamics in diverse ecosystems</li>
                        <li>Estimating biomass and productivity</li>
                        <li>Assessing the impact of climate change on vegetation</li>
                        <li>Mapping vegetation cover and land use/land cover change</li>
                    </ul>
                </p>
            """,
            "SAVI": """
                <h3>Soil-Adjusted Vegetation Index (SAVI)</h3>
                <p>
                    The Soil-Adjusted Vegetation Index (SAVI) is specifically designed to minimize 
                    the influence of soil background reflectance, particularly in areas with sparse 
                    vegetation cover. 
                    In such areas, soil reflectance can significantly impact the accuracy of 
                    vegetation indices like NDVI.
                </p>
                <p>
                    SAVI incorporates a soil brightness correction factor (L) into its calculation. 
                    This factor adjusts the sensitivity of the index to soil background, 
                    allowing for more accurate assessment of vegetation in areas with varying 
                    soil conditions. SAVI is particularly useful in:
                    <ul>
                        <li>Arid and semi-arid regions</li>
                        <li>Agricultural areas with low plant cover</li>
                        <li>Disturbed or degraded ecosystems</li>
                    </ul>
                </p>
                <p>
                    The SAVI formula is calculated as:
                </p>
                <pre>
        SAVI = (1 + L) * ((NIR - RED) / (NIR + RED + L))
                </pre>
                <p>
                    where:
                    <ul>
                        <li><b>NIR</b>: Reflectance in the near-infrared band</li>
                        <li><b>RED</b>: Reflectance in the red band</li>
                        <li><b>L</b>: Soil brightness correction factor (typically set to 0.5)</li>
                    </ul>
                </p>
                <p><b>Note:</b> For this plugin, the soil brightness correction factor (L) is set to 0.5.</p>
            """
        }


        explanation = vegetation_indices.get(self.series_indice.currentText())
        self.textBrowser_index_explain.setHtml(explanation)



    def load_vector_function(self):
        shapefile_path = self.selected_aio_layer_path
        
        # Check if the path is a .zip file
        if shapefile_path.endswith('.zip'):
            # Try to read shapefile from a zip archive
            try:
                # Check if the .zip file exists and open it
                with zipfile.ZipFile(shapefile_path, 'r') as zip_ref:
                    zip_ref.printdir()  # Optional: Print contents of the zip to debug
                    # Try to find the .shp file inside the zip
                    shapefile_found = False
                    for file in zip_ref.namelist():
                        if file.endswith('.shp'):
                            shapefile_found = True
                            shapefile_within_zip = file
                            break
                    
                    if shapefile_found:
                        # Read shapefile directly from the zip file
                        self.aoi = gpd.read_file(f'zip://{shapefile_path}/{shapefile_within_zip}')
                        print(f"Successfully loaded shapefile from {shapefile_path}.")
                    else:
                        print("No .shp file found inside the zip archive.")
                        return
            except Exception as e:
                print(f"Error reading shapefile from zip archive: {e}")
                return

        else:
            # If not a .zip, assume it is a regular shapefile
            try:
                # Read the shapefile normally
                # Use the project's CRS to read the shapefile
                self.aoi = gpd.read_file(shapefile_path).to_crs("EPSG:4326")
                print(f"Successfully loaded shapefile from {shapefile_path}.")
            except Exception as e:
                print(f"Error reading shapefile: {e}")
                return
        
        # After loading, check if the GeoDataFrame is not empty
        if not self.aoi.empty:
            # If the GeoDataFrame contains multiple geometries, dissolve them into one
            if len(self.aoi) > 1:
                self.aoi = self.aoi.dissolve()

            # Extract the first geometry from the dissolved GeoDataFrame
            geometry = self.aoi.geometry.iloc[0]

            # Check if the geometry is a Polygon or MultiPolygon
            if geometry.geom_type in ['Polygon', 'MultiPolygon']:
                # Convert the geometry to GeoJSON format
                geojson = geometry.__geo_interface__

                # Remove the third dimension from the coordinates if it exists
                if geojson['type'] == 'Polygon':
                    geojson['coordinates'] = [list(map(lambda coord: coord[:2], ring)) for ring in geojson['coordinates']]
                elif geojson['type'] == 'MultiPolygon':
                    geojson['coordinates'] = [[list(map(lambda coord: coord[:2], ring)) for ring in polygon] for polygon in geojson['coordinates']]

                # Create an Earth Engine geometry object from the GeoJSON coordinates
                ee_geometry = ee.Geometry(geojson)

                # Convert the Earth Engine geometry to a Feature
                feature = ee.Feature(ee_geometry)

                # Create a FeatureCollection with the feature
                self.aoi = ee.FeatureCollection([feature])

                print("AOI defined successfully.")
                self.vector_layer_combobox_2.setCurrentIndex(self.vector_layer_combobox.currentIndex())
                self.find_area()
                self.find_centroid()
                
                self.aoi_ckecked  = True
                self.aoi_ckecked_function()

            else:
                
                print("The geometry is not a valid type (Polygon or MultiPolygon).")
                self.pop_aviso("The geometry is not a valid type (Polygon or MultiPolygon).")
        else:
            print("The shapefile does not contain any geometries.")
            self.pop_aviso("The shapefile does not contain any geometries.")

    def find_centroid(self):
        centroid = self.aoi.geometry().centroid()
        self.lat = centroid.getInfo().get('coordinates')[1]
        self.lon = centroid.getInfo().get('coordinates')[0]
        print(f"{round(self.lat,4)},{round(self.lon,4)}")
        area = self.aoi.geometry().area().getInfo() / 1e6  # Convert from square meters to square kilometers
        print(f"Area: {area:.2f} km²")
        if area >= 100:
            self.aoi = None
            self.aoi_ckecked = False
            self.aoi_ckecked_function()
        return area

    def open_nasapower(self):
        longitude = str(self.lon)
        latitude = str(self.lat)
        start = self.df_aux.date.tolist()[0]
        end = self.df_aux.date.tolist()[-1]
        print(f"Latitude: {latitude}, Longitude: {longitude}")
        print(f"Start date: {start}, End date: {end}")
        print("Opening NASA POWER data for the selected location...")

        start_date = datetime.strptime(str(start).split()[0], "%Y-%m-%d")
        end_date = datetime.strptime(str(end).split()[0], "%Y-%m-%d")
        # start_date = datetime.strptime(str(start), "%Y-%m-%d")
        # end_date = datetime.strptime(str(end), "%Y-%m-%d")

        # Adjust the start date to the first day of the month
        new_start = start_date.replace(day=1).strftime("%Y%m%d")
        
        # Adjust the end date to the last day of the month
        new_end = (end_date.replace(day=1) + timedelta(days=32)).replace(day=1) - timedelta(days=1)
        new_end = new_end.strftime("%Y%m%d")
        
        # Print the adjusted start and end dates for debugging
        print(new_start, new_end)


        base_url = (f"https://power.larc.nasa.gov/api/temporal/daily/point?parameters=PRECTOTCORR&community=RE&longitude={longitude}&latitude={latitude}&start={new_start}&end={new_end}&format=JSON")
        api_request_url = base_url.format(longitude=longitude, latitude=latitude)
        response = requests.get(url=api_request_url, verify=True, timeout=1000)
        content = json.loads(response.content.decode('utf-8'))
        df = pd.DataFrame.from_dict(content['properties']['parameter'])
        df[df < 0] = 0
        
        # Convert the index to datetime
        df.index = pd.to_datetime(df.index, format='%Y%m%d')
        self.daily_precipitation = df.reset_index().rename(columns={'index': 'Date'}).copy()
        

        # Resample the data to monthly frequency and sum the values
        monthly_sum = df.resample('M').sum()
        print(monthly_sum)
        self.df_nasa = monthly_sum
        self.plot_timeseries()

    def find_area(self):
        area_km2 = self.aoi.geometry().area().getInfo() / 1e6  # Convert from square meters to square kilometers
        area_ha = area_km2 * 100  # Convert from square kilometers to hectares
        print(f"Area: {area_km2:.2f} km² ({area_ha:.2f} ha)")
        self.aoi_area.setText(f"AOI Total Area: {area_km2:.2f} km² ({area_ha:.2f} hectares)")
        return area_km2

    def aoi_ckecked_function(self):
        if self.aoi_ckecked and self.folder_set:
            self.QPushButton_next.setEnabled(True)
        else:
            self.QPushButton_next.setEnabled(False)

    def resetting(self):
        self.recorte_datas = None
        self.load_vector_function()
        self.get_dates()
        # self.customfilter.setChecked(False)
        #self.remove_cloudy.setChecked(False)
        self.QCheckBox_sav_filter.setChecked(False)
        self.filtro_grau.setCurrentIndex(0)
        self.window_len.setCurrentIndex(0)
        self.df_nasa = None

    def loadtimeseries_clicked(self):
        # Find the centroid of the AOI and check if the area is within the limit
        area = self.find_centroid()
        if area >= 100:
            self.pop_aviso(f"Area too large ({area:.2f} km²). The limit is 100 km².")
            return
        
        # Reset settings and set the cursor to indicate processing
        self.resetting()
        QApplication.setOverrideCursor(Qt.WaitCursor)

        # Retrieve user inputs for date range, cloud percentage, and AOI
        inicio = self.inicio
        final = self.final
        nuvem = self.nuvem
        aoi = self.aoi
        coverage_threshold = self.horizontalSlider_aio_cover.value() / 100
        local_pixel_limit = self.horizontalSlider_local_pixel_limit.value()
        print(f"Coverage threshold: {coverage_threshold}")

        # Define the Sentinel-2 image collection with filtering by date, bounds, and cloud percentage
        sentinel2 = ee.ImageCollection('COPERNICUS/S2_SR_HARMONIZED') \
            .filterDate(inicio, final) \
            .filterBounds(aoi) \
            .filter(ee.Filter.lt('CLOUDY_PIXEL_PERCENTAGE', nuvem)) \
            .map(lambda image: image.set('date', image.date().format('YYYY-MM-dd')))
        
        # Check if the collection is empty
        if sentinel2.size().getInfo() == 0:
            self.pop_aviso("No images found for the selected criteria. Please select a larger date range or less strick filtering and try again.")
            QApplication.restoreOverrideCursor()
            return
              
        # Apply AOI coverage filter to the image collection
        if coverage_threshold > 0:
            sentinel2 = self.AOI_coverage_filter(sentinel2, aoi, coverage_threshold)
            if sentinel2.size().getInfo() == 0:
                self.pop_aviso("No images found for the selected criteria. Please select a larger date range or less strick filtering and try again.")
                QApplication.restoreOverrideCursor()
                return            

        if local_pixel_limit > 0:
            # Apply local pixel limit filter to the image collection
            sentinel2 = self.filter_within_AOI(sentinel2, aoi, local_pixel_limit)
            if sentinel2.size().getInfo() == 0:
                self.pop_aviso("No images found for the selected criteria. Please select a larger date range or less strick filtering and try again.")
                QApplication.restoreOverrideCursor()
                return

        # Apply cloud and shadow mask if the checkbox is checked
        if self.mask.isChecked():
            sentinel2 = self.SCL_filter(sentinel2, aoi)
            if sentinel2.size().getInfo() == 0:
                self.pop_aviso("No images found for the selected criteria. Please select a larger date range or less strick filtering and try again.")
                QApplication.restoreOverrideCursor()
                return

        sentinel2 = self.uniqueday_collection(sentinel2)

        # Store the filtered image collection in the instance variable
        self.sentinel2 = sentinel2

        # Calculate the time series and plot it
        self.calculate_timeseries()
        self.plot_timeseries()

    def uniqueday_collection(self, sentinel2):
        # Step 1: Aggregate timestamps from the ImageCollection
        original_timestamps = sentinel2.aggregate_array('system:time_start').getInfo()

        # Step 2: Convert timestamps to formatted dates
        formatted_dates = [datetime.fromtimestamp(ts / 1000).strftime('%Y-%m-%d') for ts in original_timestamps]

        # Step 3: Identify unique dates and map them back to the original timestamps
        df = pd.DataFrame(list(zip(original_timestamps, formatted_dates)), columns=['timestamp', 'date'])
        first_timestamps_per_date = df.groupby('date')['timestamp'].min().tolist()

        # Step 4: Filter the collection to include only the first image for each unique date
        return sentinel2.filter(
            ee.Filter.inList('system:time_start', ee.List(first_timestamps_per_date))
        )
    
    def AOI_coverage_filter(self, sentinel2, aoi, coverage_threshold):
        #Coverage Ratio Function
        aoi_geometry = aoi.first().geometry()
        aoi_area = aoi_geometry.area()

        def calculate_coverage_ratio(image):
            """
            Calculates the ratio of the AOI area covered by the image.
            
            Args:
                image (ee.Image): The Sentinel-2 image.
            
            Returns:
                ee.Image: The original image with an added 'coverage_ratio' property.
            """
            # Compute the intersection geometry between AOI and image footprint
            intersection = aoi_geometry.intersection(image.geometry(), ee.ErrorMargin(1))
            
            # Calculate the area of the intersection
            intersection_area = intersection.area()
            
            # Calculate the coverage ratio (intersection area / AOI area)
            coverage_ratio = intersection_area.divide(aoi_area)
            
            # Set the coverage ratio as a property of the image
            return image.set('coverage_ratio', coverage_ratio)

        # -------------------------------
        # Step 6: Apply Coverage Ratio Calculation
        # -------------------------------

        # Map the coverage ratio function over the Sentinel-2 collection
        sentinel2_with_ratio = sentinel2.map(calculate_coverage_ratio)

        # -------------------------------
        # Step 7: Filter Based on Coverage Ratio
        # -------------------------------

        # Define a filter to keep images with coverage_ratio >= coverage_threshold
        coverage_filter = ee.Filter.gte('coverage_ratio', coverage_threshold)

        # Apply the filter to get the final collection
        covering_colection = sentinel2_with_ratio.filter(coverage_filter)

        # Get the number of images before filtering
        initial_count = sentinel2.size().getInfo()

        # Get the number of images after coverage filtering
        filtered_count = covering_colection.size().getInfo()

        print(f"Number of images before coverage filtering: {initial_count}")
        print(f"Number of images with >= {coverage_threshold*100}% AOI coverage: {filtered_count}")  

        return covering_colection

    def filter_within_AOI(self, sentinel2, aoi, valid_pixel_threshold):
        scl_classes_behavior = {
            0: self.mask_class0.isChecked(),  # No data
            1: self.mask_class1.isChecked(),  # Saturated/defective
            2: self.mask_class2.isChecked(),  # Dark features
            3: self.mask_class3.isChecked(),  # Cloud shadows
            4: self.mask_class4.isChecked(),  # Vegetation
            5: self.mask_class5.isChecked(),  # Bare soils
            6: self.mask_class6.isChecked(),  # Water
            7: self.mask_class7.isChecked(),  # Cloud low probability
            8: self.mask_class8.isChecked(),  # Cloud medium probability
            9: self.mask_class9.isChecked(),  # Cloud high probability
            10: self.mask_class10.isChecked(), # Thin cirrus
            11: self.mask_class11.isChecked()  # Snow or ice
        }
        def mask_cloud_and_shadows(image):
            scl = image.select('SCL')
            # Start with an all-inclusive mask
            mask = ee.Image.constant(1)
            # Apply exclusions
            for class_value, include in scl_classes_behavior.items():
                if include:
                    mask = mask.And(scl.neq(class_value))
            
            masked_image = image.updateMask(mask)
            
            # Calculate the percentage of valid pixels
            total_pixels = image.select(0).reduceRegion(
                reducer=ee.Reducer.count(),
                geometry=aoi,
                scale=10
            ).get('B1')
            
            valid_pixels = masked_image.select(0).reduceRegion(
                reducer=ee.Reducer.count(),
                geometry=aoi,
                scale=10
            ).get('B1')
            
            percentage_valid = ee.Number(valid_pixels).divide(total_pixels).multiply(100)
            
            # Add the percentage of valid pixels as a property
            return masked_image.set('percentage_valid_pixels', percentage_valid)

            # Apply the cloud and shadow mask function to the image collection
        sentinel2_masked =  sentinel2.map(mask_cloud_and_shadows)

        # Filter the collection based on the valid pixel threshold
        filtered_collection = sentinel2_masked.filter(ee.Filter.gte('percentage_valid_pixels', valid_pixel_threshold))

        # Get the number of images in the filtered collection
        filtered_count = filtered_collection.size().getInfo()

        masked_timestamps = filtered_collection.aggregate_array('system:time_start').getInfo()

        return sentinel2.filter(
            ee.Filter.inList('system:time_start', ee.List(masked_timestamps))
        )

    def SCL_filter(self, sentinel2, aoi):
        scl_classes_behavior = {
            0: self.mask_class0.isChecked(),  # No data
            1: self.mask_class1.isChecked(),  # Saturated/defective
            2: self.mask_class2.isChecked(),  # Dark features
            3: self.mask_class3.isChecked(),  # Cloud shadows
            4: self.mask_class4.isChecked(),  # Vegetation
            5: self.mask_class5.isChecked(),  # Bare soils
            6: self.mask_class6.isChecked(),  # Water
            7: self.mask_class7.isChecked(),  # Cloud low probability
            8: self.mask_class8.isChecked(),  # Cloud medium probability
            9: self.mask_class9.isChecked(),  # Cloud high probability
            10: self.mask_class10.isChecked(), # Thin cirrus
            11: self.mask_class11.isChecked()  # Snow or ice
        }
        def mask_cloud_and_shadows(image):
            scl = image.select('SCL')
            # Start with an all-inclusive mask
            mask = ee.Image.constant(1)
            # Apply exclusions
            for class_value, include in scl_classes_behavior.items():
                if include:
                    mask = mask.And(scl.neq(class_value))
            
            return image.updateMask(mask)      
            # Apply the cloud and shadow mask function to the image collection
        return sentinel2.map(mask_cloud_and_shadows)
 
    def calculate_timeseries(self):
        vegetation_index = self.series_indice.currentText()

        # Buffer the AOI geometry inward by 10 meters (adjust distance as needed)
        buffer_distance = -self.horizontalSlider_buffer.value()
        print(f"Buffer distance: {buffer_distance} meters")
        aoi = self.aoi.map(lambda feature: feature.buffer(buffer_distance))

        # Define the vegetation index calculation in a function
        def calculate_index(image):
            if vegetation_index == 'NDVI':
                index_image = image.normalizedDifference(['B8', 'B4']).rename('index')
            elif vegetation_index == 'EVI':
                index_image = image.expression(
                    '2.5 * ((NIR / 10000 - RED / 10000) / (NIR / 10000 + 6 * RED / 10000 - 7.5 * BLUE / 10000 + 1))', {
                        'NIR': image.select('B8'),
                        'RED': image.select('B4'),
                        'BLUE': image.select('B2')
                    }
                ).rename('index')
            elif vegetation_index == 'SAVI':
                L = 0.5
                index_image = image.expression(
                    '(1 + L) * ((NIR / 10000) - (RED / 10000)) / ((NIR / 10000) + (RED / 10000) + L)', {
                        'NIR': image.select('B8'),
                        'RED': image.select('B4'),
                        'L': L
                    }
                ).rename('index')
            elif vegetation_index == 'GCI':
                index_image = image.expression(
                    'NIR / GREEN - 1', {
                        'NIR': image.select('B8'),
                        'GREEN': image.select('B3')
                    }
                ).rename('index')
            elif vegetation_index == 'GNDVI':
                index_image = image.normalizedDifference(['B8', 'B3']).rename('index')
            else:
                raise ValueError(f"Unsupported vegetation index: {vegetation_index}")
            
            # Calculate mean value for the index over AOI
            mean_index = index_image.reduceRegion(
                reducer=ee.Reducer.mean(),
                geometry=aoi,
                scale=10,
                bestEffort=True
            ).get('index')
            
            return image.set({'mean_index': mean_index})

        # Map the calculation function over the collection and get results
        result = self.sentinel2.map(calculate_index)
        result = result.filter(ee.Filter.notNull(['mean_index']))

        # Retrieve dates and mean index values separately using aggregate_array
        dates = result.aggregate_array('date').getInfo()
        mean_indices = result.aggregate_array('mean_index').getInfo()
        image_ids = result.aggregate_array('system:index').getInfo()

        # Combine the dates, mean indices, and image IDs into a DataFrame
        df = pd.DataFrame({'date': dates, 'average_index': mean_indices, 'image_id': image_ids})

        print(df)

        # Optional: Smoothing or further processing
        self.df = df.copy()
        self.df_aux = df.copy()
        self.load_dates()
        self.plot_timeseries()

    def clear_all_raster_layers(self):
        # Get the current project instance
        project = QgsProject.instance()

        # Create a copy of the layer list to avoid issues with removing during iteration
        layers_to_remove = list(project.mapLayers().values())  

        # Iterate over the copied list
        for layer in layers_to_remove:
            if layer.type() == QgsMapLayer.RasterLayer and layer.name() != "Google Hybrid":
                layer_name = layer.name()  # Store the layer name before removing it
                project.removeMapLayer(layer.id())  # Use layer.id() for removal
                print(f"Removed raster layer: {layer_name}")
                iface.mapCanvas().refresh()

    def df_ajust(self):
        df = self.df.copy()
        if self.recorte_datas:
            df = df[df['date'].isin(self.recorte_datas)]
            self.df_aux = df.copy()

    def df_run_filter(self):
        df = self.df_aux.copy()
        try:

            if self.window_len.count() == 0:
                        self.window_len.clear()
                        self.window_len.addItems(list(map(str, list(range(7, len(df)+1)))))
                        self.window_len.setCurrentIndex(0)

            window_length = int(self.window_len.currentText())
            polyorder = int(self.filtro_grau.currentText().split('%')[0])
            print(f'Window length: {window_length}, Polyorder: {polyorder}')

            if window_length > len(df):
                window_length = len(df)
                self.window_len.setCurrentIndex(len(df) - 5)
                print(f'Window length too large. Using maximum value: {window_length}')

            # Apply Savitzky-Golay filter to smooth the time series

            
            df['savitzky_golay_filtered'] = savgol_filter(df['average_index'], window_length=window_length, polyorder=polyorder)
            self.df_aux = df.copy()
            return True
        except Exception as e:
            self.pop_aviso(f"Not enough images to apply the Savitzky-Golay filter. Please select a larger date range or less strick filtering.")
            self.QCheckBox_sav_filter.setChecked(False)
            self.plot_timeseries()
            return False

        

    def plot_timeseries(self):
        print('plot1 started')
        self.df_ajust()
        
        # Prepare to plot
        myFile = io.StringIO()
        if self.QCheckBox_sav_filter.isChecked() and self.df_run_filter():
            df = self.df_aux
            try:
                self.fig = go.Figure()
                self.fig.add_trace(go.Scatter(x=df['date'], y=df['average_index'], mode='lines', name=self.series_indice.currentText(), line=dict(color='green')))
                self.fig.add_trace(go.Scatter(x=df['date'], y=df['savitzky_golay_filtered'], mode='lines', name=f"{self.series_indice.currentText()} filtered", line=dict(color='purple')))
            except Exception as e:
                self.pop_aviso(f"An error occurred while plotting: {e}")
                self.QCheckBox_sav_filter.setChecked(False)
        else:
            df = self.df_aux
            self.fig = go.Figure()
            self.fig.add_trace(go.Scatter(x=df['date'], y=df['average_index'], mode='lines', name=self.series_indice.currentText(), line=dict(color='green')))
       
        self.fig.update_layout(
            xaxis_title='Date', 
            yaxis_title=self.series_indice.currentText(), 
            title=f"Time Series - {self.series_indice.currentText()} - {self.vector_layer_combobox.currentText()}               Image count: {len(df)}"
        )
    
        self.fig.update_traces(hovertemplate='date = %{x|%Y-%m-%d}<br>average_ndvi = %{y:.2f}<extra></extra>')
        
        # Configurations for the Plotly plot
        config = {
            'displaylogo': False,
            'modeBarButtonsToRemove': [
                "toImage", "sendDataToCloud", "zoom2d", "pan2d", "select2d",
                "lasso2d", "zoomIn2d", "zoomOut2d", "autoScale2d", "resetScale2d",
                "hoverClosestCartesian", "hoverCompareCartesian", "zoom3d",
                "pan3d", "orbitRotation", "tableRotation", "resetCameraLastSave",
                "resetCameraDefault3d", "hoverClosest3d", "zoomInGeo", 
                "zoomOutGeo", "resetGeo", "hoverClosestGeo", "hoverClosestGl2d",
                "hoverClosestPie", "toggleHover", "toggleSpikelines", "resetViews"
            ]
        }

        if isinstance(self.df_nasa, pd.DataFrame):
            # Add bar plot (set below the line explicitly)
            self.fig.add_trace(go.Bar(
                x=self.df_nasa.index, 
                y=self.df_nasa['PRECTOTCORR'], 
                name='Monthly Precipitation', 
                yaxis='y2', 
                marker_color='blue',
                opacity=0.4
            ))

            # Ensure correct layout and layering
            self.fig.update_layout(
                yaxis=dict(
                    title=self.series_indice.currentText(),
                ),
                yaxis2=dict(
                    title='Precipitation (mm)',
                    overlaying='y',
                    side='right',
                ),
                xaxis=dict(title='Date'),
            )

        
        # Update layout and render the plot
        self.fig.write_html(myFile, config=config)
        html = myFile.getvalue()  
        self.webView.setHtml(html)
        print('ok plot1')

    
        self.tabWidget.setCurrentIndex(9)
        # self.showNormal()
        QApplication.restoreOverrideCursor()

    def open_browser(self):
        self.fig.show()

    def load_dates(self):
        datas = self.df.date.unique().astype(str).tolist()
        self.dataunica.clear()
        self.dataunica.addItems(datas)
        self.dataunica.setCurrentIndex(self.dataunica.count() - 1)

    def load_raster_layer_colorful(self, raster_file_path, layer_name, metrica=None):
        print(f"Loading raster layer color: {metrica}")

        # Load the raster layer
        raster_layer = QgsRasterLayer(raster_file_path, layer_name)
        if not raster_layer.isValid():
            print("Failed to load raster layer!")
        else:
            QgsProject.instance().addMapLayer(raster_layer, False)
            root = QgsProject.instance().layerTreeRoot()
            root.insertChildNode(0, QgsLayerTreeLayer(raster_layer))
            print("Raster layer loaded successfully!")

            # Create a color ramp shader
            color_ramp_shader = QgsColorRampShader()
            color_ramp_shader.setColorRampType(QgsColorRampShader.Interpolated)

            # Load the predefined color ramp (e.g., RdYlGn) from the QGIS style manager
            style = QgsStyle().defaultStyle()
            color_ramp = style.colorRamp('RdYlGn')

            # Check if the color ramp is successfully loaded
            if color_ramp:
                # Define the number of color stops
                num_stops = 5
                min_val = raster_layer.dataProvider().bandStatistics(1).minimumValue
                max_val = raster_layer.dataProvider().bandStatistics(1).maximumValue
                step = (max_val - min_val) / (num_stops - 1)

                # Create color ramp items by interpolating the color ramp
                color_ramp_items = []
                for i in range(num_stops):
                    value = min_val + i * step
                    color = color_ramp.color(i / (num_stops - 1))  # Interpolates color along the ramp
                    color_ramp_items.append(QgsColorRampShader.ColorRampItem(value, color))

                # Set the color ramp items to the color ramp shader
                color_ramp_shader.setColorRampItemList(color_ramp_items)
            else:
                print("Color ramp 'RdYlGn' not found in the QGIS style library.")

            # Create a raster shader and set it to use the color ramp shader
            raster_shader = QgsRasterShader()
            raster_shader.setRasterShaderFunction(color_ramp_shader)

            # Apply the raster shader to the raster layer renderer
            renderer = QgsSingleBandPseudoColorRenderer(raster_layer.dataProvider(), 1, raster_shader)
            raster_layer.setRenderer(renderer)

            # Refresh the layer to update the visualization
            raster_layer.triggerRepaint()

        if metrica == 'NDVI':
            # Clone the current renderer
            newrend = raster_layer.renderer().clone()

            # Set the classification range (min and max values)
            # newrend.setClassificationMin(min_val)
            # newrend.setClassificationMax(max_val)
            newrend.setClassificationMin(0)
            newrend.setClassificationMax(1)

            # Apply the new renderer to the layer
            raster_layer.setRenderer(newrend)
            # Refresh the map canvas to reflect the changes
            iface.mapCanvas().refresh()
        else:
            # Clone the current renderer
            newrend = raster_layer.renderer().clone()

            # Set the classification range (min and max values)
            newrend.setClassificationMin(min_val)
            newrend.setClassificationMax(max_val)
            # newrend.setClassificationMin(0)
            # newrend.setClassificationMax(1)

            # Apply the new renderer to the layer
            raster_layer.setRenderer(newrend)
            # Refresh the map canvas to reflect the changes
            iface.mapCanvas().refresh()

        QApplication.restoreOverrideCursor()