# -*- coding: utf-8 -*-
"""
/***************************************************************************
 RasterVision: A QGIS plugin
 Work with Raster Vision data in QGIS
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

import os

from PyQt5.QtCore import QSettings, QTranslator, qVersion, QCoreApplication
from PyQt5.QtGui import QIcon
from PyQt5.QtWidgets import QAction

from .resources import * # Initialize Qt resources from file resources.py

from .registry import Registry
from .experiment_dialog import ExperimentDialogController
from .predict_dialog import PredictDialogController
from .profiles_dialog import ProfilesDialogController
from .config_dialog import ConfigDialogController

class RasterVisionPlugin:
    """Main entry point for the Raster Vision QGIS Plugin."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """

        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'RasterVisionPlugin_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)

            if qVersion() > '4.3.3':
                QCoreApplication.installTranslator(self.translator)

        # Create the dialog (after translation) and keep reference
        self.experiment_controller = ExperimentDialogController(self.iface)
        self.predict_controller = PredictDialogController(self.iface)
        self.profiles_controller = ProfilesDialogController()
        self.config_controller = ConfigDialogController()

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Raster Vision')
        # TODO: We are going to let the user set this up in a future iteration
        self.toolbar = self.iface.addToolBar(u'RasterVisionPlugin')
        self.toolbar.setObjectName(u'RasterVisionPlugin')

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('RasterVisionPlugin', message)

    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            self.toolbar.addAction(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        experiment_icon_path = ':/plugins/raster_vision_qgis/icon-experiment.png'
        predict_icon_path = ':/plugins/raster_vision_qgis/icon-predict.png'
        profiles_icon_path = ':/plugins/raster_vision_qgis/icon-profiles.png'
        config_icon_path = ':/plugins/raster_vision_qgis/icon-configure.png'

        self.add_action(
            experiment_icon_path,
            text=self.tr(u'Load &Experiment'),
            callback=self.run_load_experiment,
            parent=self.iface.mainWindow())

        self.add_action(
            predict_icon_path,
            text=self.tr(u'&Predict'),
            callback=self.run_predict,
            parent=self.iface.mainWindow())

        self.add_action(
            profiles_icon_path,
            text=self.tr(u'&Style Profiles'),
            callback=self.run_profiles,
            parent=self.iface.mainWindow())

        self.add_action(
            config_icon_path,
            text=self.tr(u'&Configure'),
            callback=self.run_configure,
            parent=self.iface.mainWindow())


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&Raster Vision'),
                action)
            self.iface.removeToolBarIcon(action)
        # remove the toolbar
        del self.toolbar

    def run_load_experiment(self):
        self.experiment_controller.run()

    def run_predict(self):
        self.predict_controller.run()

    def run_profiles(self):
        self.profiles_controller.run()

    def run_configure(self):
        self.config_controller.run()
