import traceback
import webbrowser
import tempfile
import subprocess
import numpy as np
from os.path import dirname, basename, join, splitext, exists
from os import makedirs, chdir, listdir
from qgis.core import *
from qgis.gui import *
from qgis.PyQt.QtWidgets import *
from qgis.PyQt.QtCore import *
from qgis.PyQt.QtGui import *
from osgeo import gdal

from .utils import loadUIFormClass


class RasterTimeseriesManager(QObject):

    pluginName = 'RasterTimeseriesManager'

    class Setting(object):
        FFMPEG_BIN = 'ffmpeg_bin'
        IMAGE_MAGICK_BIN = 'imageMagick_bin'

    def __init__(self, iface):
        QObject.__init__(self)
        self.iface = iface
        self.ui = RasterTimeseriesManagerUi()
        self.ui.setWindowIcon(self.icon())

        self.widgetsToBeEnabled = [self.ui.tab(), self.ui.layerSplit(), self.ui.info(), self.ui._time]

        # init widgets
        self.ui.status().hide()
        self.ui.fileFfmpeg().setFilePath(self.setting(key=self.Setting.FFMPEG_BIN, default=''))
        self.ui.fileImageMagick().setFilePath(self.setting(key=self.Setting.IMAGE_MAGICK_BIN, default=''))
        self.ui.fileImageMagick().setFilePath(self.setting(key=self.Setting.IMAGE_MAGICK_BIN, default=''))
        self.ui.fileTimeseries().setFilePath(join(tempfile.gettempdir(), self.pluginName, 'timeseries.vrt'))
        self.ui.stepUnit().setCurrentIndex(4) # indices

        # todo: ts creation has issues, hide it for now
        self.ui._tab.removeTab(2)

        # disable widgets
        for w in self.widgetsToBeEnabled:
            w.setEnabled(False)

        # init state variables
        self._timeseries = RasterTimeseries(None)
        self._timeseriesBuddy = dict()
        self.nbuddy = 10
        self.currentDate = QDate()
        self.currentIndex = -1
        self.currentVisible = True

        self.currentIndexBuddy = -1
#        self.animationFrameLength = 250

        # connect signals
        self.ui.date().dateChanged.connect(self.setDate)
        self.ui.slider().valueChanged.connect(self.onDateNumberChanged)  #self.setNumber)
        self.ui.play().toggled.connect(self.onPlayToggled)
        self.ui.layer().layerChanged.connect(self.setTimeseries)
        self.ui.next().clicked.connect(lambda *args: self.setDate(self.nextDate()))
        self.ui.previous().clicked.connect(lambda *args: self.setDate(self.nextDate(reversed=True)))
        self.ui.layerSplit().clicked.connect(self.onLayerSplitClicked)
        self.ui.layerMerge().clicked.connect(self.onLayerMergeClicked)
        self.ui.saveFrames().clicked.connect(self.onSaveFrames)
        self.ui.saveMp4().clicked.connect(self.onSaveMp4)
        self.ui.saveGif().clicked.connect(self.onSaveGif)
        self.ui.openFolder().clicked.connect(lambda *args: webbrowser.open(self.ui.saveFolder().filePath()))
        self.ui.clearFolder().clicked.connect(self.onClearFolder)
        self.ui.info().clicked.connect(self.onInfoClicked)
        for i in range(self.nbuddy):
            self.ui.layerBuddy(i).layerChanged.connect(self.onTimeseriesBuddyChanged)
        self.canvas().renderComplete.connect(self.drawDate)

        # add to plugins toolbar
        if isinstance(self.iface, QgisInterface):
            self.action1 = QAction(self.icon(), 'RasterTimeseriesManager', self.iface.mainWindow())
            self.action1.triggered.connect(self.toggleUiVisibility)
            self.iface.addToolBarIcon(self.action1)
            self.action2 = QAction('Load test data', self.iface.mainWindow())
            self.action2.triggered.connect(self.loadTestdata)
            self.iface.addPluginToMenu('Raster Timeseries Manager', self.action2)

    def pluginFolder(self):
        return join(dirname(__file__), '..')

    def icon(self):
        return QIcon(join(self.pluginFolder(), 'icon.png'))

    def drawDate(self, painter):
        assert isinstance(painter, QPainter)

        if self.timeseries().isValid():

            font = QFont('Courier', pointSize=18)
            font.setPointSize(18)
            painter.setFont(font)

            painter.setFont(font)
            self.canvas().center()
            ptLabel = QPointF(10, 50)

            labelText = self.ui.date().text()

            brush = self.canvas().backgroundBrush()
            c = QColor('#ffffff')
            c.setAlpha(170)
            brush.setColor(c)
            painter.setBrush(brush)
            fm = QFontMetrics(font)
            backGroundSize = QSizeF(fm.size(Qt.TextSingleLine, labelText))
            backGroundSize = QSizeF(backGroundSize.width() + 10, -1 * (backGroundSize.height() + 10))
            backGroundPos = QPointF(ptLabel.x() - 5, ptLabel.y() + 10)
            background = QPolygonF(QRectF(backGroundPos, backGroundSize))
            painter.drawPolygon(background)

            pen = QPen(Qt.SolidLine)
            #pen.setWidth(20)
            #pen.setWidthF(500)
            pen.setColor(QColor('#000000'))
            painter.setPen(Qt.NoPen)
            painter.setPen(pen)
            #painter.scale(2,2)
            painter.drawText(ptLabel, labelText)



    def showWaitCursor(self):
        QApplication.setOverrideCursor(Qt.WaitCursor)

    def hideWaitCursor(self):
        QApplication.restoreOverrideCursor()

    def handleException(self, error):
        assert isinstance(error, Exception)
        traceback.print_exc()
        self.setStatus()
        self.ui.status().show()
        QTimer.singleShot(5000, lambda: self.ui.status().hide())

    def setStatus(self):
        self.ui.status().setText('Unexprected error: see Python Console [Ctrl+Alt+P] log for details. ')

    def setSetting(self, key, value):
        s = QgsSettings()
        s.setValue('{}/{}'.format(self.pluginName, key), value)

    def setting(self, key, default=None):
        s = QgsSettings()
        return s.value('{}/{}'.format(self.pluginName, key), default)

    def toggleUiVisibility(self):
        self.ui.setVisible(not self.ui.isVisible())

    def loadTestdata(self):
        layer = self.iface.addRasterLayer(join(dirname(__file__), '..', 'testdata', 'timeseries.bsq'))
        self.ui.layer().setLayer(layer)
        self.setTimeseries(layer=layer)

    def unload(self):
        """Unload the plugin"""
        self.iface.removeDockWidget(self.ui)
        self.iface.removePluginMenu('Raster Timeseries Manager', self.action1)

    def canvas(self):
        canvas = self.iface.mapCanvas()
        assert isinstance(canvas, QgsMapCanvas)
        return canvas

    def show(self):
        self.ui.show()

    def setTimeseries(self, layer):

        self._timeseries = RasterTimeseries(layer=layer)

        if self.timeseries().isValid():
            self.ui.slider().setRange(1, self.timeseries().numberOfObservations)
            self.ui.dateRangeStart().setDate(self.timeseries().dates[0])
            self.ui.dateRangeEnd().setDate(self.timeseries().dates[-1])
            self.setDateIndex(0)
            for w in self.widgetsToBeEnabled:
                w.setEnabled(True)
        else:
            for w in self.widgetsToBeEnabled:
                w.setEnabled(False)



    def onTimeseriesBuddyChanged(self, layer):
        # add selected layer
        self.setTimeseriesBuddy(layer=layer)

        # filter layers to remove deselected layers (not the most elegant way to do it, but it is not time-critical)
        currentLayers = list()
        for i in range(self.nbuddy):
            currentLayers.append(self.ui.layerBuddy(i).currentLayer())
        self._timeseriesBuddy = {layer: self._timeseriesBuddy[layer] for layer in currentLayers if layer is not None}

    def setTimeseriesBuddy(self, layer):
        if layer not in self._timeseriesBuddy:
            self._timeseriesBuddy[layer] = RasterTimeseries(layer=layer)

    def timeseries(self):
        assert isinstance(self._timeseries, RasterTimeseries)
        return self._timeseries

    def timeseriesBuddies(self):
        return list(self._timeseriesBuddy.values())

    def setDateNumber(self, number):
        try:
            self.setDateIndex(index=number - 1)
        except Exception as error:
            self.handleException(error)

    def setDateIndex(self, index):
        self.setDate(date=self.timeseries().dates[index])

    def setDate(self, date):

        assert isinstance(date, QDate)

        if not self.timeseries().isValid():
            return

        if date > self.ui.dateRangeEnd().date():
            date = self.ui.dateRangeStart().date()
        elif date < self.ui.dateRangeStart().date():
            date = self.ui.dateRangeEnd().date()

        self.currentDate = date
        snap = self.ui.snap().currentText()
        index = self.timeseries().findDateIndex(date, snap)

        # update the gui without emitting signals
        self.ui.date().blockSignals(True)
        self.ui.date().setDate(self.currentDate)
        off = self.currentDate.daysTo(self.timeseries().dates[index])
        sign = '+' if off >= 0 else '-'
        self.ui.date().setDisplayFormat('yyyy-MM-dd ({}) {}{}'.format(index+1, sign, abs(off)))
        self.ui.date().blockSignals(False)
        self.ui.slider().blockSignals(True)
        self.ui.slider().setValue(index+1)
        self.ui.slider().blockSignals(False)

        # update timeseries renderer
        maxoff = self.ui.maxoff().value()
        if self.ui.maxoff().isEnabled():
            isOff = abs(self.timeseries().dates[index].daysTo(date)) > maxoff
        else:
            isOff = False

        if isOff:
            self.timeseries().setDateIndex(index=-1) # turn off layer
            self.currentVisible = False
        else:
            if index != self.currentIndex  or not self.currentVisible:
                self.timeseries().setDateIndex(index=index)
                #self.timeseries().layer.triggerRepaint()
            self.currentVisible = True

        self.currentIndex = index

        # update timeseries buddy renderer
        for timeseriesBuddy in self.timeseriesBuddies():
            if timeseriesBuddy.isValid():
                index = timeseriesBuddy.findDateIndex(date, snap)
                if index != timeseriesBuddy.currentIndex:
                    timeseriesBuddy.currentIndex = index
                    timeseriesBuddy.setDateIndex(index=index)

                    # needed for linked map views!
                    timeseriesBuddy.layer.triggerRepaint()

        self.canvas().refreshAllLayers()

    def nextDate(self, reversed=False):

        unit = self.ui.stepUnit().currentText()
        step = self.ui.stepSize().value()
        if reversed:
            step = -step

        if unit == 'days':
            date = self.currentDate.addDays(step)
        elif unit == 'weeks':
            date = self.currentDate.addDays(7 * step)
        elif unit == 'months':
            date = self.currentDate.addMonths(step)
        elif unit == 'years':
            date = self.currentDate.addYears(step)
        elif unit == 'indices':
            index = self.timeseries().findDateIndex(self.currentDate, snap=self.ui.snap().currentText()) + step
            if index < 0:
                date = self.ui.dateRangeStart().date().addDays(-1)
            elif index >= len(self.timeseries().dates):
                date = self.ui.dateRangeEnd().date().addDays(1)
            else:
                date = self.timeseries().dates[index]
        else:
            raise Exception('unknown unit')
        return date

    def play(self):
        self.setDate(date=self.nextDate())
        self.canvas().waitWhileRendering()
        if self.ui.play().isChecked():
            animationFrameLength = 1000 / self.ui.fps().value()
            QTimer.singleShot(animationFrameLength, self.play)

    def onSaveFrames(self):

        self.showWaitCursor()
        folder = self.saveFrames()
        self.hideWaitCursor()

        import webbrowser
        webbrowser.open(folder)

    def saveFrames(self):
        folder = self.ui.saveFolder().filePath()

        if not exists(folder):
            makedirs(folder)

        size = self.canvas().size()
        image = QImage(size, QImage.Format_RGB32)

        step = self.ui.stepSize().value()
        if step > 0:
            dateStart = self.ui.dateRangeStart().date()
            dateEnd = self.ui.dateRangeEnd().date()
        elif step < 0:
            dateEnd = self.ui.dateRangeStart().date()
            dateStart = self.ui.dateRangeEnd().date()
        else:
            return

#        self.ui.setEnabled(False)
        date = dateStart
        i = 0
        while True:

            if (step > 0 and date > dateEnd) or (step < 0 and date < dateStart):
                break

            self.setDate(date)
            self.canvas().waitWhileRendering()

            image.fill(QColor('white'))
            painter = QPainter(image)
            settings = self.canvas().mapSettings()
            job = QgsMapRendererCustomPainterJob(settings, painter)
            job.renderSynchronously()
            self.drawDate(painter=painter)

            painter.end()

            filename = join(folder, 'frame{}.png'.format(str(i).zfill(10)))
            image.save(filename)

            date = self.nextDate()
            i += 1

#        self.ui.setEnabled(True)
        return folder

    def onSaveMp4(self):

        self.showWaitCursor()

        try:
            ffmpeg_bin = self.ui.fileFfmpeg().filePath()

            if not exists(str(ffmpeg_bin)) or not basename(ffmpeg_bin).startswith('ffmpeg'):
                msg = QMessageBox(parent=self.ui,
                                  text='Select the FFmpeg binary under settings.')
                msg.setWindowTitle('Wrong or missing FFmpeg binary.')
                msg.exec()
            else:
                self.setSetting(key=self.Setting.FFMPEG_BIN, value=ffmpeg_bin)


                filename = self.saveMp4(ffmpeg_bin=ffmpeg_bin)
                if filename is not None:
                    webbrowser.open(filename)
        except Exception as error:
            self.handleException(error)

        self.hideWaitCursor()

    def saveMp4(self, fps=10, ffmpeg_bin="ffmpeg"):

        folder = self.ui.saveFolder().filePath()
        video = 'video.mp4'
        cmd = [ffmpeg_bin, '-r', str(fps), '-i', 'frame%10d.png', '-vcodec', 'libx264', '-y', '-an',
               video, '-vf', '"pad=ceil(iw/2)*2:ceil(ih/2)*2"']

#        cmd = [ffmpeg_bin, '-c:v', 'libx264' '-pix_fmt', 'yuv420p', '-vf', '"pad=width={W}:height={H}:x=0:y=0:color=black"']

        print(' '.join(cmd))

        chdir(folder)
        res = subprocess.call(cmd, stdin=subprocess.PIPE, shell=True)#, stdout=f, stderr=f)

        if res != 0:
            return None
        else:
            return join(folder, video)

    def onSaveGif(self):

        imageMagick_bin = self.ui.fileImageMagick().filePath()

        if not exists(str(imageMagick_bin)) or not (basename(imageMagick_bin).startswith('magick') or basename(imageMagick_bin).startswith('convert')):
            msg = QMessageBox(parent=self.ui,
                              text='Select the ImageMagick binary under settings.')
            msg.setWindowTitle('Wrong or missing ImageMagick binary.')
            msg.exec()
        else:
            self.setSetting(key=self.Setting.IMAGE_MAGICK_BIN, value=imageMagick_bin)

            self.showWaitCursor()
            filename = self.saveGif(imageMagick_bin=imageMagick_bin)
            self.hideWaitCursor()

            if filename is not None:
                webbrowser.open(filename)

    def saveGif(self, imageMagick_bin='magick'):

        folder = self.ui.saveFolder().filePath()
        video = join(folder, 'video.gif')
        pngs = join(folder, '*.png')
        cmd = [imageMagick_bin, '-delay', '20', pngs, '-loop', '0', video]
        print(' '.join(cmd))

        chdir(dirname(imageMagick_bin))
        res = subprocess.call(cmd, stdin=subprocess.PIPE, shell=True)#, stdout=f, stderr=f)

        if res != 0:
            return None
        else:
            return join(folder, video)

    def onClearFolder(self):
        folder = self.ui.saveFolder().filePath()
        if exists(folder):
            import os, shutil
            for file in listdir(folder):
                filename = join(folder, file)
                try:
                    if os.path.isfile(filename):
                        os.unlink(filename)
                except Exception as error:
                    self.handleException(error)

    def onDateNumberChanged(self, number):
        self.setDate(date=self.timeseries().dates[number-1])

    def onDateChanged(self, date):
        self.setDate(date=date)

    def onInfoClicked(self):
        msg = QDialog(parent=self.ui)
        msg.setWindowTitle('Timeseries information')
        msg.resize(QSize(800, 600))
        layout = QVBoxLayout()
        msg.setLayout(layout)
        text = QTextEdit()
        layout.addWidget(text)
        date2str = lambda d: '{}-{}-{}'.format(d.year(), str(d.month()).zfill(2), str(d.day()).zfill(2))
        text.setText('bands ({}): {}\n\ndates ({}): {}'.format(
            self.timeseries().numberOfBands, ', '.join(self.timeseries().names), self.timeseries().numberOfObservations,
            ', '.join([date2str(date) for date in self.timeseries().dates])))
        msg.exec_()

    def onPlayToggled(self, checked):
        if checked:
            self.ui.play().setText('Pause')
            self.play()
        else:
            self.ui.play().setText('Play')

    def onLayerSplitClicked(self):
        self.splitTimeseries(dirname=join('/vsimem/{}/splitted'.format(self.pluginName),
                                          str(np.random.randint(100000, 999999))))

    def splitTimeseries(self, dirname):
        if not exists(dirname):
            makedirs(dirname)

        tsFilename = self.timeseries().source()
        ds = gdal.Open(tsFilename)
        filenames = list()
        for i, name in enumerate(self.timeseries().names):
            filename = join(dirname, '{}.vrt'.format(name))
            bandList = list(range(i + 1, self.timeseries().numberOfObservations * self.timeseries().numberOfBands + 1, self.timeseries().numberOfBands))
            vrt = gdal.BuildVRT(destName=filename, srcDSOrSrcDSTab=ds, options=gdal.BuildVRTOptions(bandList=bandList))
            for i, bandNumber in enumerate(bandList):
                description = ds.GetRasterBand(bandNumber).GetDescription()
                vrt.GetRasterBand(i+1).SetDescription(description)
            filenames.append(filename)
            vrt = None

        layers = [self.iface.addRasterLayer(filename) for filename in reversed(filenames)]
        return list(reversed(layers))

    def onLayerMergeClicked(self):
        layers = QgsProject.instance().layerTreeRoot().checkedLayers()
        layers = [layer for layer in layers if isinstance(layer, QgsRasterLayer)]
        print(layers)
        if len(layers) > 1:
            layer = self.mergeTimeseries(filename=self.ui.fileTimeseries().filePath(),
                                         layers=layers)
            layer = self.iface.addRasterLayer(layer.source())
            self.ui.layer().setLayer(layer=layer)

    def mergeTimeseries(self, filename, layers):
        assert 0 # todo
        if not exists(dirname(filename)):
            makedirs(dirname(filename))

        dss = dict()
        infoss = dict()

        ds0 = None
        for layer in layers:

            ds = gdal.Open(layer.source())
            if ds0 is None: ds0 = ds

            if (ds0.RasterXSize != ds.RasterXSize or
                ds0.RasterYSize != ds.RasterYSize or
                ds0.RasterCount != ds.RasterCount):
                raise Exception('Can not merge timeseries, raster sizes do not match.')

            infoss[layer] = RasterTimeseriesManager.deriveInformation(layer=layer)
            dss[layer] = ds

        vrt = gdal.GetDriverByName('VRT').Create(filename, ds0.RasterXSize, ds0.RasterYSize,
                                                 ds0.RasterCount * len(layers), ds0.GetRasterBand(1).DataType)

        vrt.SetProjection(ds0.GetProjection())
        vrt.SetGeoTransform(ds0.GetGeoTransform())

        xml = '''<SimpleSource>
              <SourceFilename relativeToVRT="0">{}</SourceFilename>
              <SourceBand>{}</SourceBand>
            </SimpleSource>'''

        vrtBandNumber = 1
        for i in range(ds0.RasterCount):
            for layer in layers:
                vrt.GetRasterBand(vrtBandNumber).SetMetadataItem("source_0",
                                                                 xml.format(layer.source(), i+1),
                                                                 "new_vrt_sources")
                #description = ds.GetRasterBand(i+1).GetDescription()
                dates, names, numberOfObservations, numberOfBands = infoss[layer]
                date = dates[i]
                assert isinstance(date, QDate)
                sdate = '{}-{}-{}'.format(str(date.year()).zfill(4), str(date.month()).zfill(2), str(date.day()).zfill(2))
                description = '{} - {}'.format(sdate, names[i % numberOfBands])
                print(description)
                vrt.GetRasterBand(vrtBandNumber).SetDescription(description)
                vrtBandNumber += 1
        vrt = None

        layer = QgsRasterLayer(filename)
        assert layer.isValid()
        return layer

    def onLayerToMemoryClicked(self):
        pass


class RasterTimeseries(object):

    def __init__(self, layer):

        if isinstance(layer, QgsRasterLayer):
            self.layer = layer
            self.dates, self.names, self.numberOfObservations, self.numberOfBands = self.deriveInformation(layer=layer)
            self.currentIndex = -1
            self.valid = True
        else:
            self.valid = False

    def isValid(self):
        return self.valid

    @staticmethod
    def deriveInformation(layer):
        assert isinstance(layer, QgsRasterLayer)

        def deriveFromDescriptions(descriptions):
            dates = list()
            names = list()
            for description in descriptions:
                sep = ' - '
                i = description.find(sep)
                if i == -1:
                    return None
                name = description[i + len(sep):].strip()
                date = description[:i].strip()
                y, m, d = map(int, date.split('-'))
                date = QDate(y, m, d)
                dates.append(date)
                names.append(name)
            return dates, names

        def deriveFromFallback():
            date0 = QDate(2000, 1, 1)
            dates = [date0.addDays(i) for i in range(layer.bandCount())]
            names = [splitext(basename(layer.source()))[0]]
            return dates, names

        def deriveFromMetadata(metadata):

            def toArray(s, dtype=str):
                if s is None:
                    return None
                else:
                    return [dtype(v.strip()) for v in s.replace('{', '').replace('}', '').split(',')]

            if metadata.get('wavelength') is None:
                return None
            else:
                units = metadata.get('wavelength_units', 'decimal years')
                if units.lower() == 'decimal years':
                    dyears = toArray(metadata['wavelength'], float)
                    dates = [QDate(int(dy), 1, 1).addDays(round((dy - int(dy)) * 366) - 1) for dy in dyears]
                else:
                    raise Exception('unknown time unit: "wavelength units = {}"'.format(units))

            if metadata.get('band_names') is None:
                raise Exception('use "band names" item in "ENVI" domain to specify the band names')
            else:
                names = toArray(metadata['band_names'])
            return dates, names

        ds = gdal.Open(layer.source())
        info = deriveFromMetadata(ds.GetMetadata('ENVI'))
        if info is None:
            info = deriveFromDescriptions([ds.GetRasterBand(i + 1).GetDescription() for i in range(ds.RasterCount)])
            if info is None:
                info = deriveFromFallback()
        dates_, names_ = info


        names = [names_[0]]
        for name in names_[1:]:
            if name == names_[0]:
                break
            else:
                names.append(name)

        numberOfBands = len(names)
        numberOfObservations = int(ds.RasterCount / numberOfBands)
        dates = dates_[::numberOfBands]

        for date in dates:
            assert isinstance(date, QDate)
        return dates, names, numberOfObservations, numberOfBands

    def findDateIndex(self, date, snap):

        if snap == 'nearest':
            dist = [abs(date.daysTo(d)) for d in self.dates]
        elif snap == 'next':
            dist = np.array([date.daysTo(d) for d in self.dates], dtype=np.float32)
            dist[dist < 0] = np.inf
        elif snap == 'previous':
            dist = np.array([-date.daysTo(d) for d in self.dates], dtype=np.float32)
            dist[dist < 0] = np.inf
        else:
            raise Exception('unknown mode: {}'.format(snap))

        return np.argmin(dist)

    def setDateIndex(self, index):
        renderer = self.layer.renderer()
        bandNumber = lambda number: ((number - 1) % self.numberOfBands + 1) + (index * self.numberOfBands)
        if isinstance(renderer, QgsMultiBandColorRenderer):
            renderer.setRedBand(bandNumber(renderer.redBand()))
            renderer.setGreenBand(bandNumber(renderer.greenBand()))
            renderer.setBlueBand(bandNumber(renderer.blueBand()))
        elif isinstance(renderer, QgsPalettedRasterRenderer):
            # renderer.setBand(bandNumber(renderer.band()))
            assert NotImplementedError()  # question posted https://gis.stackexchange.com/questions/315160/how-to-set-a-new-raster-band-to-a-qgspalettedrasterrenderer-object
        elif isinstance(renderer, QgsSingleBandGrayRenderer):
            renderer.setGrayBand(bandNumber(renderer.grayBand()))
        elif isinstance(renderer, QgsSingleBandPseudoColorRenderer):
            renderer.setBand(bandNumber(renderer.band()))
        elif isinstance(renderer, QgsHillshadeRenderer):
            renderer.setBand(bandNumber(renderer.band()))
        elif renderer is None:
            pass
        else:
            raise NotImplementedError('renderer not supported: {}'.format(renderer))


class RasterTimeseriesManagerUi(QgsDockWidget, loadUIFormClass(pathUi=join(join(dirname(__file__)), 'dockwidget.ui'))):

    def __init__(self, parent=None):
        QgsDockWidget.__init__(self, parent)
        self.setupUi(self)

        # init widgets
        defaultRoot = join(tempfile.gettempdir(), 'frames')
        self.saveFolder().setDefaultRoot(defaultRoot)
        self.saveFolder().setFilePath(defaultRoot)
        self.saveFolder().setStorageMode(QgsFileWidget.GetDirectory)

        # set icons
        iconPlay = QIcon()
        iconPlay.addFile(join(dirname(__file__), 'icon_play.svg'), QSize(32, 32), QIcon.Normal, QIcon.Off)
        iconPlay.addFile(join(dirname(__file__), 'icon_pause.svg'), QSize(32, 32), QIcon.Normal, QIcon.On)
        self.play().setIcon(iconPlay)
        self.next().setIcon(QIcon(join(dirname(__file__), 'icon_next.svg')))
        self.previous().setIcon(QIcon(join(dirname(__file__), 'icon_previous.svg')))


        self.saveFrames().setIcon(QIcon(join(dirname(__file__), 'icon_frame.svg')))
        self.saveMp4().setIcon(QIcon(join(dirname(__file__), 'icon_mp4.svg')))
        self.saveGif().setIcon(QIcon(join(dirname(__file__), 'icon_gif.svg')))

    def fps(self):
        assert isinstance(self._fps, QSpinBox)
        return self._fps

    def date(self):
        assert isinstance(self._date, QDateEdit)
        return self._date

    def dateRangeStart(self):
        assert isinstance(self._dateRangeStart, QDateEdit)
        return self._dateRangeStart

    def dateRangeEnd(self):
        assert isinstance(self._dateRangeEnd, QDateEdit)
        return self._dateRangeEnd

    def maxoff(self):
        assert isinstance(self._maxoff, QSpinBox)
        return self._maxoff

    def snap(self):
        assert isinstance(self._snap, QComboBox)
        return self._snap

    def stepSize(self):
        assert isinstance(self._stepSize, QSpinBox)
        return self._stepSize

    def stepUnit(self):
        assert isinstance(self._stepUnit, QComboBox)
        return self._stepUnit

    def slider(self):
        assert isinstance(self._dateSlider, QSlider)
        return self._dateSlider

    def play(self):
        assert isinstance(self._play, QToolButton)
        return self._play

    def next(self):
        assert isinstance(self._dateNext, QToolButton)
        return self._dateNext

    def previous(self):
        assert isinstance(self._datePrevious, QToolButton)
        return self._datePrevious

    def layer(self):
        assert isinstance(self._layer, QgsMapLayerComboBox)
        return self._layer

    def layerBuddy(self, i):
        layerBuddy = getattr(self, '_layerBuddy' + str(i))
        assert isinstance(layerBuddy, QgsMapLayerComboBox)
        return layerBuddy

    def layerSplit(self):
        assert isinstance(self._layerSplit, QToolButton)
        return self._layerSplit

    def layerMerge(self):
        assert isinstance(self._layerMerge, QToolButton)
        return self._layerMerge

    def saveFrames(self):
        assert isinstance(self._saveFrames, QToolButton)
        return self._saveFrames

    #def viewFrames(self):
    #    assert isinstance(self._viewFrames, QToolButton)
    #    return self._viewFrames

    def saveMp4(self):
        assert isinstance(self._saveMp4, QToolButton)
        return self._saveMp4

    def saveGif(self):
        assert isinstance(self._saveGif, QToolButton)
        return self._saveGif

    def saveFolder(self):
        assert isinstance(self._saveFolder, QgsFileWidget)
        return self._saveFolder

    def fileFfmpeg(self):
        assert isinstance(self._fileFfmpeg, QgsFileWidget)
        return self._fileFfmpeg

    def fileImageMagick(self):
        assert isinstance(self._fileImageMagick, QgsFileWidget)
        return self._fileImageMagick

    def fileTimeseries(self):
        assert isinstance(self._fileTimeseries, QgsFileWidget)
        return self._fileTimeseries

    def openFolder(self):
        assert isinstance(self._openFolder, QToolButton)
        return self._openFolder

    def clearFolder(self):
        assert isinstance(self._clearFolder, QToolButton)
        return self._clearFolder

    def info(self):
        assert isinstance(self._info, QToolButton)
        return self._info

    def status(self):
        assert isinstance(self._status, QLineEdit)
        return self._status

    def tab(self):
        assert isinstance(self._tab, QTabWidget)
        return self._tab
