import traceback
import numpy as np
import math
from qgis.core import *
from os.path import join, dirname
from qgis.gui import *
from PyQt5.QtWidgets import *
from PyQt5.QtCore import *
from PyQt5.QtGui import *
from collections import namedtuple
from . import utils
from .site import pyqtgraph as pg
from .rasterdataplottingdock import RasterDataPlottingDock

debug = True

class RasterDataPlotting(QObject):

    pluginName = 'Raster Data Plotting'
    pluginId = pluginName.replace(' ', '')

    class PlotType(object):
        Scatter = 0
        SpectralProfile = 1
        TemporalProfile = 2

    PlotRange = namedtuple('PlotRange', ('minX', 'maxX', 'minY', 'maxY'))

    def __init__(self, iface, parent=None):
        QObject.__init__(self, parent)
        assert isinstance(iface, QgisInterface)
        self.iface = iface

        # init ui
        self.ui = RasterDataPlottingDock()
        self.ui.setWindowIcon(self.icon())
        self.ui.imageView().view.invertY(False)

        # hide
        self.ui.plotType().setMaxCount(1)
        #self.ui.g1.hide()

        self.ui.bandsX().hide()

        # init plots
        self.profiles = dict()

        # state
        self.cacheBandX = dict()
        self.cachePercentilesH = dict()
        self.cacheData = None
        #self.cacheOverlayBuffer = None
        #self.cacheOverlay = QImage()
        self.oldSources = None
        self.oldBinning = None
        self.oldRois = None
        self._painter = None # Set painter after first map canvas renderCompleted signal was handled. Can we get it directly???

        self._connectSignals()
        self._integratePlugin()

    def _connectSignals(self):
        #self.canvas().extentsChanged.connect(self.onSomethingChanged)
        self.canvas().renderComplete.connect(self.onCanvasRenderCompleted)
        self.ui.layerX().layerChanged.connect(self.onLayerXChanged)
        self.ui.layerY().layerChanged.connect(self.onSomethingChanged)
        self.ui.bandXRaster().bandChanged.connect(self.onSomethingChanged)
        self.ui.bandYRaster().bandChanged.connect(self.onSomethingChanged)
        self.ui.bandXRenderer().currentIndexChanged.connect(self.onSomethingChanged)
        self.ui.bandYRenderer().currentIndexChanged.connect(self.onSomethingChanged)
        #self.ui.binsX().valueChanged.connect(self.onSomethingChanged)
        #self.ui.binsY().valueChanged.connect(self.onSomethingChanged)
        self.ui.bandsX().valueChanged.connect(self.onBandsXChanged)
        self.ui.percentageH().valueChanged.connect(self.onPercentageHChanged)

        # group Mask
        self.ui.doMask().toggled.connect(self.canvas().refresh)
        self.ui.doMaskColoring().toggled.connect(self.canvas().refresh)
        self.ui.layerMask().layerChanged.connect(self.onLayerMaskChanged)
        self.ui.layerMaskSelectedOnly().toggled.connect(self.canvas().refresh)

        # group ROI
        self.ui.roiCircle().clicked.connect(self.onCreateRoi)
        self.ui.roiRectangle().clicked.connect(self.onCreateRoi)
        self.ui.roiPolygon().clicked.connect(self.onCreateRoi)
        self.ui.roiOpacity().valueChanged.connect(self.canvas().refresh)
        self.ui.doRoi().toggled.connect(self.canvas().refresh)

        self.ui.refresh().clicked.connect(self.onSomethingChanged)
        self.ui.plotType().currentIndexChanged.connect(self.onPlotTypeChanged)
        #for w in [self.ui.minX(), self.ui.minY(), self.ui.maxX(), self.ui.maxY(), self.ui.minH(), self.ui.maxH()]:
        #    w.textChanged.connect(self.onSomethingChanged)

        self.ui.imageView().plotItem().sigRangeChanged.connect(self.onPlotRangeChanged)

        #self.ui.imageView().sigRangeChanged.connect(self.onPlotRangeChanged)
        self.ui.imageView().sigHistogramLevelsChanged.connect(self.onPlotHistogramLevelsChanged)

    def _integratePlugin(self):
        if isinstance(self.iface, QgisInterface):
            self.action1 = QAction(self.icon(), self.pluginName.replace(' ', ''), self.iface.mainWindow())
            self.action1.triggered.connect(self.toggleUiVisibility)
            self.iface.addToolBarIcon(self.action1)

    def show(self):
        self.ui.show()

    def toggleUiVisibility(self):
        self.ui.setVisible(not self.ui.isVisible())

    def pluginFolder(self):
        return join(dirname(__file__), '..')

    def icon(self):
        return QIcon(join(self.pluginFolder(), 'icon.png'))

    def showWaitCursor(self):
        QApplication.setOverrideCursor(Qt.WaitCursor)

    def hideWaitCursor(self):
        QApplication.restoreOverrideCursor()

    def handleException(self, error):
        assert isinstance(error, Exception)
        traceback.print_exc()

    def unload(self):
        """Unload the plugin"""
        self.iface.removeDockWidget(self.ui)
        self.iface.removePluginMenu(self.pluginName, self.action1)

    def canvas(self):
        canvas = self.iface.mapCanvas()
        assert isinstance(canvas, QgsMapCanvas)
        return canvas

    def onPercentageHChanged(self, *args):
        minH, maxH = self.plotHistogramLevelsFromPercentile()
        self.setPlotHistogramLevels(minH, maxH)
        self.ui.imageView().plotItem().blockSignals(True)
        self.ui.imageView().setLevels(minH, maxH)
        self.ui.imageView().ui.histogram.item.setHistogramRange(minH, maxH)

    def onCreateRoi(self):
        color = self.ui.roiColor().color()
        width = 1.0
        cosmetic = True
        pen = QPen(QBrush(color), width)
        pen.setCosmetic(cosmetic)
        r = self.plotRange()
        s = [r.maxX - r.minX,  r.maxY - r.minY] # span
        m = [v * 0.1 for v in s] # margin

        if self.sender() == self.ui.roiCircle():
            roi = pg.EllipseROI(pos=(r.minX + m[0], r.minY + m[1]), size=[s[0] - 2*m[0], s[1] - 2*m[1]],
                                removable=True, pen=pen, rotatable=True, movable=True)
        elif self.sender() == self.ui.roiRectangle():
            roi = pg.RectROI(pos=(r.minX + m[0], r.minY + m[1]), size=[s[0] - 2*m[0], s[1] - 2*m[1]],
                             removable=True, pen=pen, rotatable=True, movable=True)
            roi.addRotateHandle(pos=(1, 0), center=(0.5, 0.5))
        elif self.sender() == self.ui.roiPolygon():
            roi = pg.PolyLineROI(positions=[(r.minX + m[0], r.minY + m[1]),
                                            (r.maxX - m[0], r.minY + m[1]),
                                            (r.minX/2 + r.maxX/2, r.maxY - m[1])],
                                 closed=True, removable=True, pen=pen, rotatable=False, movable=True)
        else:
            raise Exception()

        roi.sigRegionChangeFinished.connect(self.canvas().refresh)
        self.ui.imageView().addRoi(roi=roi)
        self.canvas().refresh()

    def createRoi(self, roi):
        self.ui.imageView().addRoi(roi=roi)

    def onCanvasRenderCompleted(self, painter):
        if debug: print('onCanvasRenderCompleted')

        assert isinstance(painter, QPainter)
        self.painter = painter
        self.plot()
        self.drawOverlay()

    def onBandsXChanged(self, nbands):
        if debug: print('onBandsXChanged')
        self.cacheBandX[self.ui.layerX().currentLayer()] = nbands
        self.clearProfiles()
        self.plot()

    def onLayerXChanged(self):
        if debug: print('onLayerXChanged')
        if (self.plotType() == self.PlotType.SpectralProfile) or (self.plotType() == self.PlotType.TemporalProfile):
            self.clearProfiles()
            layer = self.ui.layerX().currentLayer()
            if layer is not None:
                assert isinstance(layer, QgsRasterLayer)
                self.ui.bandsX().setMaximum(layer.bandCount())
                if self.plotType() == self.PlotType.SpectralProfile:
                    self.ui.bandsX().setValue(self.cacheBandX.get(layer, layer.bandCount()))
                if self.plotType() == self.PlotType.TemporalProfile:
                    self.ui.bandsX().setValue(self.cacheBandX.get(layer, 1))
        self.plot()

    def onLayerMaskChanged(self):
        if debug: print('onLayerMaskChanged')
        layer = self.ui.layerMask().currentLayer()
        if isinstance(layer, QgsMapLayer):
            layer.rendererChanged.connect(self.onSomethingChanged)
        self.canvas().refresh()

    def onPlotTypeChanged(self, index):
        if debug: print('onPlotTypeChanged')
        self.initPlot(plotType=index)
        self.onLayerXChanged()

    def onPlotRangeChanged(self, *args):
        if debug: print('onPlotRangeChanged')

        imageItem = self.ui.imageView().getImageItem()
        assert isinstance(imageItem, pg.ImageItem)
        print('imageItem pixelSize', imageItem.pixelSize())



        # turn-off auto-scaling
        self.ui.updateRangeX().setChecked(False)
        self.ui.updateRangeY().setChecked(False)
        ((minX, maxX), (minY, maxY)) = args[1]
        self.setPlotRange(minX, maxX, minY, maxY)

        #self.plot()
        #QTimer.singleShot(1, self.canvas().refresh)

    def setPlotRange(self, minX=None, maxX=None, minY=None, maxY=None):
        widgets = (self.ui.minX(), self.ui.maxX(), self.ui.minY(), self.ui.maxY())
        values = (minX, maxX, minY, maxY)
        for w, v in zip(widgets, values):
            if v is None:
                continue

            w.blockSignals(True)
            w.setText(str(utils.roundPlotRangeValue(v)))
            w.blockSignals(False)

    def plotRange(self):
        widgets = (self.ui.minX(), self.ui.maxX(), self.ui.minY(), self.ui.maxY())
        values = [utils.toFloat(w.text()) for w in widgets]
        return self.PlotRange(*values)

    def setPlotXRange(self, min, max):
        self.setPlotRange(minX=min, maxX=max)

    def setPlotYRange(self, min, max):
        self.setPlotRange(minY=min, maxY=max)

    def onPlotHistogramLevelsChanged(self, *args):
        print('onPlotHistogramLevelsChanged')
        minH, maxH = self.ui.imageView().ui.histogram.item.getLevels()
        self.setPlotHistogramLevels(minH, maxH)

    def setPlotHistogramLevels(self, minH, maxH):
        widgets = (self.ui.minH(), self.ui.maxH())
        values = (minH, maxH)

        self.ui.imageView().blockSignals(True)

        for w, v in zip(widgets, values):
            #if not w.isEnabled():
            #    continue
            w.blockSignals(True)
            w.setText(str(utils.roundPlotHistogramRangeValue(v)))
            w.blockSignals(False)

        self.ui.imageView().blockSignals(False)

    def scatterPlotBands(self):

        if self.ui.bandMode().currentIndex() == 0: # raster band selection
            bandX = self.ui.bandXRaster().currentBand()
            bandY = self.ui.bandYRaster().currentBand()
        else: # renderer band selection

            def getRendererBand(uiLayer, uiIndex):
                assert isinstance(uiLayer, QgsMapLayerComboBox)
                assert isinstance(uiIndex, QComboBox)
                renderer = uiLayer.currentLayer().renderer()
                if not isinstance(renderer, QgsMultiBandColorRenderer):
                    if uiIndex.currentIndex() != 0:
                        uiIndex.setCurrentIndex(0)

                return renderer.usesBands()[uiIndex.currentIndex()]

            bandX = getRendererBand(uiLayer=self.ui.layerX(), uiIndex=self.ui.bandXRenderer())
            bandY = getRendererBand(uiLayer=self.ui.layerY(), uiIndex=self.ui.bandYRenderer())

        return bandX, bandY

    def onSomethingChanged(self, *args):
        print('onSomethingChanged')
        self.plot()

    def initPlot(self, plotType):
        self.ui.imageView().setHistogramVisibility(visible=plotType == self.PlotType.Scatter)

    def plotReset(self):

        # clear scatter image
        self.ui.imageView().setLevels(0, 0)
        self.ui.imageView().getHistogramWidget().item.setHistogramRange(0, 0)

        # clear overlay image
        self.cacheOverlay = QImage()

    def plotType(self):
        return self.ui.plotType().currentIndex()

    def plotHistogramLevelsFromPercentile(self):
        maxP = int(self.ui.percentageH().value() * 10)
        minP = 1000 - maxP
        return self.cachePercentilesH[minP],self.cachePercentilesH[maxP]

    def doPlot(self):
        return self.ui.doPlot().isChecked() and self.ui.isVisible()

    def doSpatialRoi(self):
        return self.ui.doMask().isChecked() and self.ui.layerMask().currentLayer() is not None

    def plot(self):

        if self.doPlot():

            if self.plotType() == self.PlotType.Scatter:
                self.plotScatter()
            elif self.plotType() == self.PlotType.SpectralProfile:
                self.plotProfile()
            elif self.plotType() == self.PlotType.TemporalProfile:
                self.plotProfile()

    def scatterPlotSources(self):
        return (self.ui.layerX().currentLayer(), self.ui.layerY().currentLayer(),
                self.ui.bandXRaster().currentBand(), self.ui.bandYRaster().currentBand(),
                self.scatterPlotBands(),
                self.ui.doMask().isChecked(), self.ui.layerMask().currentLayer(),
                self.ui.doMaskColoring().isChecked(),
                0 if self.ui.layerMask().currentLayer() is None else self.ui.layerMask().currentLayer().selectedFeatureIds(),
                self.ui.layerMask().currentLayer(),
                self.ui.layerMaskSelectedOnly().isChecked(),
                self.canvas().extent())

    def scatterPlotBinning(self):
        return self.plotRange(), self.ui.binsX().value(), self.ui.binsY().value()

    def scatterPlotRois(self):
        result = [self.ui.roiOpacity().value()]
        for roi in self.ui.imageView().rois:
            assert isinstance(roi, pg.ROI)
            result.extend((roi, roi.size().x(), roi.size().y(), roi.pos().x(), roi.pos().y()))
        return tuple(result)

    def plotScatter(self):

        # Check if data sources changed.
        newSources = self.scatterPlotSources()
        sourcesChanged = newSources != self.oldSources or self.cacheData is None
        self.oldSources = newSources

        # Check if data binning changed.
        newBinning = self.scatterPlotBinning()
        binningChanged = (newBinning != self.oldBinning)
        self.oldBinning = newBinning

        # Check if rois changed.
        newRois = self.scatterPlotRois()
        roisChanged = (newRois != self.oldRois)
        self.oldRois = newRois

        # return if layers are not valid
        if self.ui.layerX().currentLayer() is None or self.ui.layerY().currentLayer() is None:
            self.plotReset()
            return

        # derive band indices
        bandX, bandY = self.scatterPlotBands()

        # read data and mask
        if sourcesChanged:

            x, mask1, xsize, ysize = self.readArray(layer=self.ui.layerX().currentLayer(), band=bandX)
            y, mask2, xsize, ysize = self.readArray(layer=self.ui.layerY().currentLayer(), band=bandY)
            maskColor = self.readMaskColor()
            mask = maskColor != -1

            if x is None or x.size == 0:
                self.plotReset()
                return

            if y is None or y.size == 0:
                self.plotReset()
                return

            # apply mask
            np.logical_and(mask, mask1, out=mask)
            np.logical_and(mask, mask2, out=mask)

            try:
                x = x[mask]
                y = y[mask]
                maskColor = maskColor[mask]
            except AttributeError: # handle full masked case
                self.plotReset()
                return

            # check for empty data
            if x.size == 0:
                self.plotReset()
                return

            if y.size == 0:
                self.plotReset()
                return
            self.cacheData = x, y, maskColor, mask, xsize, ysize
            if debug: print('Read data.')
        else:
            x, y, maskColor, mask, xsize, ysize = self.cacheData
            if debug: print('Use cached data.')

        # get bins
        if binningChanged or sourcesChanged:

            # Calculate number of bins to match native screen size or use as is.
            if self.ui.updateBinsX().isChecked() or self.ui.updateBinsY().isChecked():
                #self.ui.imageView().setImage(np.zeros(shape=(1, 1)))  # init zero image to calculate native screen resolution
                resX, resY = self.ui.imageView().getImageItem().pixelSize()
                sizeX, sizeY = self.ui.imageView().getImageItem().width(), self.ui.imageView().getImageItem().height()

            if self.ui.updateBinsX().isChecked():
                binsX = min(max(math.floor(sizeX * resX), 10), 999)
            else:
                binsX = self.ui.binsX().value()

            if self.ui.updateBinsY().isChecked():
                binsY = min(max(math.floor(sizeY * resY), 10), 999)
            else:
                binsY = self.ui.binsY().value()

            for w, v in zip([self.ui.binsX(), self.ui.binsY()], [binsX, binsY]):
                w.blockSignals(True)
                w.setValue(v)
                w.blockSignals(False)

            # get ranges
            autoRangeX = not self.ui.minX().isEnabled() and not self.ui.maxX().isEnabled()
            if not autoRangeX:
                minX = utils.toFloat(self.ui.minX().text(), default=0)
                maxX = utils.toFloat(self.ui.maxX().text(), default=1)
            else:
                minX = x.min()
                maxX = x.max()
            self.setPlotXRange(minX, maxX)

            autoRangeY = not self.ui.minY().isEnabled() and not self.ui.maxY().isEnabled()
            if not autoRangeY:
                minY = utils.toFloat(self.ui.minY().text())
                maxY = utils.toFloat(self.ui.maxY().text())
            else:
                minY = y.min()
                maxY = y.max()
            self.setPlotYRange(minY, maxY)

            rangeX = maxX - minX
            rangeY = maxY - minY

            # Calculate histogram image.
            scaleX = rangeX / float(binsX)
            scaleY = rangeY / float(binsY)

            # Try to calculate the histogram. May fail, if the user settings are not consistent or in edge cases.
            # We just ignore this and wait for the next valid setup.
            try:

                from scipy.stats import binned_statistic_2d
                r = binned_statistic_2d(x=x, y=y, values=None, statistic='count', bins=[binsX, binsY],
                                        range=[[minX, maxX], [minY, maxY]], expand_binnumbers=True)
                h = r.statistic.astype(np.int64)
                binIndices = (1000000 + (r.binnumber[0] - 1) * 1000 + r.binnumber[1] - 1).flatten()

            except Exception as error:
                self.handleException(error)
                return

            if self.doSpatialRoi() and self.ui.doMaskColoring().isChecked():

                def hByMaskColor(color): # color format is 1RRRGGGBBB
                    valid = maskColor == color
                    r = binned_statistic_2d(x=x[valid], y=y[valid], values=None, statistic='count', bins=[binsX, binsY],
                                            range=[[minX, maxX], [minY, maxY]], expand_binnumbers=True)
                    h = r.statistic
                    return h

                h_sum = np.zeros(shape=(binsX, binsY))
                r = np.zeros(shape=(binsX, binsY))
                g = np.zeros(shape=(binsX, binsY))
                b = np.zeros(shape=(binsX, binsY))

                for color in np.unique(maskColor):
                    if color == -1: continue

                    h_fid = hByMaskColor(color)
                    h_sum += h_fid
                    valid = h_fid != 0

                    r[valid] += h_fid[valid] * int(color % 1000000000 / 1000000)
                    g[valid] += h_fid[valid] * int(color % 1000000 / 1000)
                    b[valid] += h_fid[valid] * int(color % 1000)

                r /= h_sum
                g /= h_sum
                b /= h_sum

                height, width = h.shape
                image = np.empty(shape=(height, width, 3), dtype=np.uint8)
                image[:, :, 0] = r.astype(np.uint8)
                image[:, :, 1] = g.astype(np.uint8)
                image[:, :, 2] = b.astype(np.uint8)
                minH, maxH = 0, 255
                self.ui.imageView().setHistogramVisibility(False)

            else:
                # get the contrast stretch
                autoRangeH = not self.ui.minH().isEnabled() and not self.ui.maxH().isEnabled()
                if not autoRangeH:
                    minH = utils.toFloat(self.ui.minH().text())
                    maxH = utils.toFloat(self.ui.maxH().text())
                else:
                    q = range(1001)
                    p = np.percentile(h, [v / 10 for v in q])
                    self.cachePercentilesH = dict(zip(q, p))
                    minH, maxH = self.plotHistogramLevelsFromPercentile()

                self.setPlotHistogramLevels(minH, maxH)

                image = h
                self.ui.imageView().setHistogramVisibility(True)

            # Set image.
            self.ui.imageView().plotItem().blockSignals(True)
            self.ui.imageView().getImageItem().blockSignals(True)

            try:
                self.ui.imageView().setImage(image, pos=[minX, minY], scale=[scaleX, scaleY])
                self.ui.imageView().setLevels(minH, maxH)
                self.ui.imageView().getHistogramWidget().item.setHistogramRange(minH, maxH)
            except Exception as error:
                self.handleException(error)
                return
            self.ui.imageView().plotItem().blockSignals(False)
            self.ui.imageView().getImageItem().blockSignals(False)

            self.cacheBinning = binsX, binsY, binIndices, h
            if debug: print('Bin data.')
        else:
            binsX, binsY, binIndices, h = self.cacheBinning
            if debug: print('Use cached bin data.')

        # Create buffer for overlay image calculations
        if sourcesChanged:
            r = np.zeros_like(mask, dtype=np.uint8)
            g = np.zeros_like(mask, dtype=np.uint8)
            b = np.zeros_like(mask, dtype=np.uint8)
            a = np.zeros_like(mask, dtype=np.uint8)
            self.cacheOverlayBuffer = r, g, b, a
            if debug: print('Create buffer.')
        else:
            r, g, b, a = self.cacheOverlayBuffer
            if debug: print('Use cached buffer.')

        # Create lookup dict holding image locations grouped by bin locations
        if binningChanged or sourcesChanged:
            imageIndices = np.where(mask)[0]
            binIndicesDict = {bin: list() for bin in binIndices}
            [binIndicesDict[bin].append(i) for bin, i in zip(binIndices, imageIndices)]
            binXIndexArray = (np.array(range(binsX)).reshape(1, binsX) * np.full(shape=(binsY, 1), fill_value=1))
            binYIndexArray = (np.array(range(binsY)).reshape(binsY, 1) * np.full(shape=(1, binsX), fill_value=1))

            self.cacheLocationLookup = binIndicesDict, binXIndexArray, binYIndexArray
            if debug: print('Create image location lookup.')
        else:
            binIndicesDict, binXIndexArray, binYIndexArray = self.cacheLocationLookup
            if debug: print('Use cached image location lookup.')

        # Create overlay image from ROIs.
        rois = self.ui.imageView().rois
        if roisChanged or binningChanged or sourcesChanged:

            # Pseudo-erase all pixels by setting setting the opacity to zero.
            a *= 0

            # Update overlay image with each individually colored roi
            img = self.ui.imageView().getImageItem()
            for roi in rois:
                assert isinstance(roi, pg.ROI)
                selBinXIndexArray = roi.getArrayRegion(binXIndexArray.T, img=img).astype(np.int64)
                selBinYIndexArray = roi.getArrayRegion(binYIndexArray.T, img=img).astype(np.int64)
                selBinIndexArray = (1000000 + selBinXIndexArray * 1000. + selBinYIndexArray).astype(np.int64)
                selHArray = roi.getArrayRegion(h, img=img).astype(np.int64)
                nonEmptyBins = selBinIndexArray[selHArray != 0]

                color = roi.pen.color()
                imageIndices = list()
                errorPrinted = False
                for bin in nonEmptyBins:
                    imageIndicesForTheBin = binIndicesDict.get(bin, None)
                    if imageIndicesForTheBin is None:
                        #if not errorPrinted:
                        #    print('unexpected missing bin:', bin)
                        #    errorPrinted = True
                        continue
                    imageIndices.extend(imageIndicesForTheBin)
                print(color.red())
                print(color.green())
                print(color.blue())

                r[imageIndices] = color.red()
                g[imageIndices] = color.green()
                b[imageIndices] = color.blue()
                a[imageIndices] = int(2.5 * self.ui.roiOpacity().value())

            r = r.reshape(ysize, xsize)
            g = g.reshape(ysize, xsize)
            b = b.reshape(ysize, xsize)
            a = a.reshape(ysize, xsize)

            rgba = np.array([b, g, r, a]).transpose([1,2,0])
            rgba = np.require(rgba, np.uint8, 'C')
            qrgba = QImage(rgba.data, rgba.shape[1], rgba.shape[0], rgba.strides[0], QImage.Format_ARGB32)
            self.cacheOverlay = qrgba

    def drawOverlay(self):
        if self.ui.doRoi().isChecked() and self.doPlot():
            self.painter.drawImage(QRect(QPoint(0, 0), self.cacheOverlay.size()), self.cacheOverlay)

    def plotProfile(self):

        values = self.readProfile(layer=self.ui.layerX().currentLayer())
        if values is not None:
            x = range(len(values[0]))
            for i, y in enumerate(values):
                if not self.profileInitialized(i):
                    self.initProfile(i)
                self.profiles[i].setData(x, y)

    def initProfile(self, index):
        assert isinstance(index, int)
        self.profiles[index] = self.ui.imageView().plotItem().plot([0, 0],[0, 0], pen=pg.mkPen(color=(255, 0, 0), width=2, style=QtCore.Qt.SolidLine))
        self.ui.imageView().clear()

    def clearProfiles(self):
        for profile in self.profiles.values():
            profile.clear()

    def profileInitialized(self, index):
        assert isinstance(index, int)
        return self.profiles.get(index) is not None

    def readProfile(self, layer):

        if layer is None:

            values = None

        else:

            # get center
            assert isinstance(layer, QgsRasterLayer), repr(layer)
            provider = layer.dataProvider()
            assert isinstance(provider, QgsRasterDataProvider)
            center = self.canvas().center()
            assert isinstance(center, QgsPointXY)
            extent = self.canvas().extent()
            assert isinstance(extent, QgsRectangle)

            # reproject center if needed
            canvasCrs = self.canvas().mapSettings().destinationCrs()
            layerCrs = layer.crs()
            if canvasCrs != layerCrs:
                tr = QgsCoordinateTransform(canvasCrs, layerCrs, QgsProject.instance())
                center = tr.transform(center)
                extent = tr.transform(extent)

            # read data
            size = self.canvas().size()
            identifyResult = provider.identify(point=center, format=QgsRaster.IdentifyFormatValue,
                                               boundingBox=extent, width=size.width(), height=size.height())
            results = identifyResult.results()
            values = [results[i+1] for i in range(len(results))]
            values = [v if v is not None else np.nan for v in values]

            # split data
            nbands = self.ui.bandsX().value()
            if (layer.bandCount() % nbands) != 0:
                return None
            nobservations = int(layer.bandCount() / nbands)
            if self.plotType() == self.PlotType.TemporalProfile:
                values = np.array(values).reshape((nobservations, nbands)).T
            elif self.plotType() == self.PlotType.SpectralProfile:
                values = np.array(values).reshape((nobservations, nbands))

        return values

    def readArray(self, layer, band):

        if layer is None:
            array = mask = xsize = ysize = None
        else:
            # get extent
            assert isinstance(layer, QgsRasterLayer), repr(layer)
            provider = layer.dataProvider()
            assert isinstance(provider, QgsRasterDataProvider)
            extent = self.canvas().extent()
            assert isinstance(extent, QgsRectangle)

            # setup on-the-fly resampling if needed
            canvasCrs = self.canvas().mapSettings().destinationCrs()
            layerCrs = layer.crs()

            if canvasCrs != layerCrs:
                projector = QgsRasterProjector()
                projector.setCrs(layerCrs, canvasCrs)
                pipe = QgsRasterPipe()
                if not pipe.set(provider.clone()):
                    raise Exception('Cannot set pipe provider')

                projector = QgsRasterProjector()
                projector.setCrs(layerCrs, canvasCrs)
                if not pipe.insert(2, projector):
                    raise Exception('Cannot set pipe projector')

            else:
                projector = provider

            # read data
            size = self.canvas().size()
            xsize, ysize = size.width(), size.height()
            block = projector.block(band, extent, xsize, ysize)
            assert isinstance(block, QgsRasterBlock)
            array = np.frombuffer(np.array(np.array(block.data())),
                                  dtype=utils.qgisDataTypeToNumpyDataType(block.dataType()))

            # calculate mask from band layer
            mask = np.full_like(array, fill_value=True, dtype=np.bool)
            noDataValues = [obj.min() for obj in provider.userNoDataValues(band)]
            if provider.sourceHasNoDataValue(band) and provider.useSourceNoDataValue(band):
                noDataValues.append(provider.sourceNoDataValue(band))

            for noDataValue in noDataValues:
                mask[array == noDataValue] = False

        return array, mask, xsize, ysize

    def readMaskColor(self):
        layer = self.ui.layerMask().currentLayer()
        size = self.canvas().size()
        extent = self.canvas().extent()
        assert isinstance(extent, QgsRectangle)
        canvasCrs = self.canvas().mapSettings().destinationCrs()
        assert isinstance(canvasCrs, QgsCoordinateReferenceSystem)

        if layer is None or not self.ui.doMask().isChecked():
            array = np.full(shape=(size.height(), size.width()), fill_value=1255000000, dtype=np.int32)
        else:
            onlySelectedFeatures = self.ui.layerMaskSelectedOnly().isChecked() and layer.selectedFeatureCount() > 0
            array = utils.colorArray(layer=layer, crs=canvasCrs, extent=extent, size=size,
                                     onlySelectedFeatures=onlySelectedFeatures)

        return array.flatten()

