# -*- coding: utf-8 -*-
"""
/***************************************************************************
 RasterStatsPlus
                                 A QGIS plugin
 Calculates detailed statistics of a selected raster layer, allows you to choose the band via a menu, and generates histogram and Gaussian curve plots.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-01-16
        git sha              : $Format:%H$
        copyright            : (C) 2025 by Dr. Geol. Faustino Cetraro
        email                : geol-faustino@libero.it
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .raster_stats_plus_dialog import RasterStatsPlusDialog
import os.path

from PyQt5.QtWidgets import QAction, QFileDialog, QMessageBox
from PyQt5.QtCore import Qt
from PyQt5.QtWidgets import QColorDialog
import numpy as np
import matplotlib.pyplot as plt
from osgeo import gdal
import csv
from qgis.core import QgsRasterLayer, QgsRasterHistogram, QgsProject 
from qgis.core import QgsRasterBandStats
import random

class RasterStatsPlus:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        
        # initialize locale path
        locale_path = os.path.join(self.plugin_dir, 'i18n', f'raster_stats_plus_{locale}.qm')

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Raster Stats Plus')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('RasterStatsPlus', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToRasterMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = os.path.join(self.plugin_dir,'icon.png')
        self.add_action(
            icon_path,
            text=self.tr(u'Raster Stats Plus'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginRasterMenu(
                self.tr(u'&Raster Stats Plus'),
                action)
            self.iface.removeToolBarIcon(action)


    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            self.first_start = False
            self.dlg = RasterStatsPlusDialog()
        
        try:
            # at the start
            self.dlg.progressBar.setValue(0)

            # Populates the list of loaded rasters
            self.populate_raster_list()  
            self.populate_fields()

            # Default colors
            self.dlg.label_10.setStyleSheet(f"background-color: {'#51f3f3'};")  # Default color for histogram
            self.dlg.label_12.setStyleSheet(f"background-color: {'red'};")      # Default color for vertical line

            # Event Linking: Run Once
            if not hasattr(self, "_events_connected") or not self._events_connected:
                # Reconnect events
                self.dlg.cmb_raster_load.currentIndexChanged.connect(self.populate_fields)
                self.dlg.pushButton_colore_histogram.clicked.connect(self.select_histogram_color)
                self.dlg.pushButton_colore_curve.clicked.connect(self.select_curve_color)
                self.dlg.pushButton_run.clicked.connect(self.on_pushButton_calculate)
                self.dlg.pushButton_cancel.clicked.connect(self.on_pushButton_cancel)
                self.dlg.pushButton_hystogram.clicked.connect(self.on_pushButton_hystogram)
                self.dlg.pushButton_gauss_graph.clicked.connect(self.on_pushButton_gauss_graph)
                self.dlg.pushButton_close.clicked.connect(self.on_pushButton_close)
                self.dlg.pushButton_export.clicked.connect(self.on_pushButton_export)
          
                self._events_connected = True  

        except Exception as e:
            QMessageBox.critical(self.dlg, self.tr("Error"), self.tr(f"An error occurred: {str(e)}"))

        # show the dialog
        self.dlg.show()
        # Run the dialog event loop
        result = self.dlg.exec_()
        # See if OK was pressed
        if result:
            # Do something useful here - delete the line containing pass and
            # substitute with your code.
            pass
    
    def populate_raster_list(self):
        self.dlg.cmb_raster_load.clear()
        layers = [layer for layer in QgsProject.instance().mapLayers().values() if isinstance(layer, QgsRasterLayer)]
        for layer in layers:
            self.dlg.cmb_raster_load.addItem(layer.name(), layer.id())

    def populate_fields(self):
        raster_id = self.dlg.cmb_raster_load.currentData()
        layer = QgsProject.instance().mapLayer(raster_id)
        if layer:
            provider = layer.dataProvider()
            self.dlg.cmb_field.clear()
            for band in range(1, provider.bandCount() + 1):
                self.dlg.cmb_field.addItem(f"Band {band}", band)

    def select_histogram_color(self):
        """Manages the color selection window for the histogram"""
        if not hasattr(self, 'color_dialog_histogram') or self.color_dialog_histogram is None:
            self.color_dialog_histogram = QColorDialog(self.dlg)
        
        # If the dialogue is already visible, do not open it again.
        if self.color_dialog_histogram.isVisible():
            return

        color = self.color_dialog_histogram.getColor()
        if color.isValid():
            self.selected_histogram_color = color.name()  # Get the color in hexadecimal format
            # Change the label background color
            self.dlg.label_10.setStyleSheet(f"background-color: {self.selected_histogram_color};")

    def select_curve_color(self):
        """Manages the color selection window for the class"""
        if not hasattr(self, 'color_dialog_curve') or self.color_dialog_curve is None:
            self.color_dialog_curve = QColorDialog(self.dlg)
        
        # If the dialogue is already visible, do not open it again.
        if self.color_dialog_curve.isVisible():
            return

        color = self.color_dialog_curve.getColor()
        if color.isValid():
            self.selected_curve_color = color.name()  # Get the color in hexadecimal format
            # Change the label background color
            self.dlg.label_12.setStyleSheet(f"background-color: {self.selected_curve_color};")
    
    def get_label_background_color(self, label):
        """Retrieve the background color of a QLabel."""
        style = label.styleSheet()
        color_start = style.find("background-color:")
        if color_start == -1:
            return None  # No color set
        color_end = style.find(";", color_start)
        return style[color_start + len("background-color:"):color_end].strip()
        
    def on_pushButton_calculate(self):
        """
        Calcola le statistiche per il layer raster selezionato e aggiorna i campi QLineEdit corrispondenti.
        """
        try:
            self.dlg.progressBar.setValue(0)  # progressbar

            # Get the selected raster layer
            raster_id = self.dlg.cmb_raster_load.currentData()
            raster_layer = QgsProject.instance().mapLayer(raster_id)

            if not raster_layer or not isinstance(raster_layer, QgsRasterLayer):
                QMessageBox.critical(self.dlg, self.tr("Error"), self.tr("Invalid or not selected raster layer."))
                return

            self.dlg.progressBar.setValue(10)

            # Get the selected band
            band_index = self.dlg.cmb_field.currentData()
            if band_index is None:
                self.dlg.progressBar.setValue(0)
                QMessageBox.critical(self.dlg, self.tr("Error"), self.tr("No band selected."))
                return

            self.dlg.progressBar.setValue(20)

            # Mappa dei tipi di dati GDAL a tipi di dati numpy
            gdal_to_numpy = {
                'Byte': np.uint8,
                'UInt16': np.uint16,
                'Int16': np.int16,
                'UInt32': np.uint32,
                'Int32': np.int32,
                'Float32': np.float32,
                'Float64': np.float64
            }

            # Ripulisci i valori precedenti (reset variabili)
            self.dlg.lineEdit_sum.setText("0.00")
            self.dlg.lineEdit_count.setText("0.00")
            self.dlg.lineEdit_mean.setText("0.00")
            self.dlg.lineEdit_median.setText("0.00")
            self.dlg.lineEdit_std.setText("0.00")
            self.dlg.lineEdit_variance.setText("0.00")
            self.dlg.lineEdit_min.setText("0.00")
            self.dlg.lineEdit_max.setText("0.00")
            self.dlg.lineEdit_minority.setText("0.00")
            self.dlg.lineEdit_majority.setText("0.00")
            self.dlg.lineEdit_unique.setText("0.00")

            # Ottieni il percorso del file raster
            file_path = raster_layer.dataProvider().dataSourceUri()
            dataset = gdal.Open(file_path)
            if not dataset:
                QMessageBox.critical(self.dlg, self.tr("Error"), self.tr("Unable to open raster file with GDAL."))
                return

            band = dataset.GetRasterBand(band_index)

            # Usa il tipo di dato corretto dalla mappa gdal_to_numpy
            data_type = gdal.GetDataTypeName(band.DataType)
            if data_type not in gdal_to_numpy:
                QMessageBox.critical(self.dlg, self.tr("Error"), self.tr(f"Unsupported raster data type: {data_type}"))
                dataset = None  # Close the dataset to release memory
                return

            # Leggi i dati del raster e maschera i valori NoData
            raster_data = band.ReadAsArray().astype(gdal_to_numpy[data_type])
            no_data_value = band.GetNoDataValue()

            # Ripuliamo la memoria escludendo i valori NoData dal calcolo
            if no_data_value is not None:
                raster_data = np.ma.masked_equal(raster_data, no_data_value)

            self.dlg.progressBar.setValue(40)

            # Calcola le statistiche su tutto il raster
            min_value, max_value = raster_data.min(), raster_data.max()
            mean_value, std_dev = raster_data.mean(), raster_data.std()

            self.dlg.progressBar.setValue(70)

            # Calcola altre statistiche
            sum_value = raster_data.sum()
            count = raster_data.count()
            median = np.ma.median(raster_data)
            variance = raster_data.var()
            unique, counts = np.unique(raster_data.compressed(), return_counts=True)
            minority = unique[np.argmin(counts)] if unique.size > 0 else None
            majority = unique[np.argmax(counts)] if unique.size > 0 else None
            unique_count = len(unique)

            self.dlg.progressBar.setValue(90)

            # Visualizza le statistiche nei QLineEdit
            self.dlg.lineEdit_sum.setText(f"{sum_value:.2f}")
            self.dlg.lineEdit_count.setText(str(count))
            self.dlg.lineEdit_mean.setText(f"{mean_value:.2f}")
            self.dlg.lineEdit_median.setText(f"{median:.2f}")

            self.dlg.progressBar.setValue(100)

            self.dlg.lineEdit_std.setText(f"{std_dev:.2f}")
            self.dlg.lineEdit_variance.setText(f"{variance:.2f}")
            self.dlg.lineEdit_min.setText(f"{min_value:.2f}")
            self.dlg.lineEdit_max.setText(f"{max_value:.2f}")

            self.dlg.lineEdit_minority.setText(f"{minority:.2f}")
            self.dlg.lineEdit_majority.setText(f"{majority:.2f}")
            self.dlg.lineEdit_unique.setText(f"{unique_count:.2f}")
            
            self.dlg.progressBar.setValue(0)

            # Cleanup: Free memory by deleting the dataset
            dataset = None

        except Exception as e:
            QMessageBox.critical(self.dlg, self.tr("Error"), self.tr(f"Error while calculating statistical values: {str(e)}"))
            self.dlg.progressBar.setValue(0)  # Reset progressbar on error

    def on_pushButton_hystogram(self):
        """
        Genera l'istogramma per i dati raster.
        """
        try:
            self.dlg.progressBar.setValue(0)

            raster_id = self.dlg.cmb_raster_load.currentData()
            raster_layer = QgsProject.instance().mapLayer(raster_id)

            if not raster_layer or not isinstance(raster_layer, QgsRasterLayer):
                QMessageBox.critical(self.dlg, self.tr("Error"), self.tr("Invalid or not selected raster layer."))
                return

            # Get the selected band
            band_index = self.dlg.cmb_field.currentData()
            if band_index is None:
                QMessageBox.critical(self.dlg, self.tr("Error"), self.tr("No band selected."))
                return

            self.dlg.progressBar.setValue(30)

            # Get Raster Data Provider
            provider = raster_layer.dataProvider()
            file_path = provider.dataSourceUri()
            dataset = gdal.Open(file_path)
            band = dataset.GetRasterBand(band_index)
            
            data_type = gdal.GetDataTypeName(band.DataType)
            gdal_to_numpy = {
                'Byte': np.uint8,
                'UInt16': np.uint16,
                'Int16': np.int16,
                'UInt32': np.uint32,
                'Int32': np.int32,
                'Float32': np.float32,
                'Float64': np.float64
            }

            self.dlg.progressBar.setValue(50)

            raster_data = band.ReadAsArray().astype(gdal_to_numpy[data_type])
            no_data_value = band.GetNoDataValue()

            # Mask NoData values
            if no_data_value is not None:
                raster_data = np.ma.masked_equal(raster_data, no_data_value)

            # Retrieve colors from label backgrounds
            histogram_color = self.get_label_background_color(self.dlg.label_10)

            self.dlg.progressBar.setValue(70)

            # Plot histogram
            plt.figure(num=self.tr("Histogram"))  # Set window title to "Histogram"
            plt.hist(raster_data.compressed(), bins=200, color=histogram_color, alpha=0.7)
            plt.title(self.tr("Histogram"), pad=10, fontweight='bold', fontsize=12)
            plt.xlabel(self.tr("Value"), labelpad=5, fontweight='bold', fontsize=10)
            plt.ylabel(self.tr("Frequency"), labelpad=5, fontweight='bold', fontsize=10)
            plt.grid(True, linestyle='--', alpha=0.5)  # Optional: Add gridlines for better readability
            plt.tight_layout(pad=0.5)  # Adjust layout with a small padding

            self.dlg.progressBar.setValue(100)

            plt.show()

            self.dlg.progressBar.setValue(0)

        except Exception as e:
            QMessageBox.critical(self.dlg, self.tr("Error"), self.tr(f"Error while generating histogram: {str(e)}"))


    def on_pushButton_gauss_graph(self):
        """
        Genera la curva di Gauss per i dati raster.
        """
        try:
            self.dlg.progressBar.setValue(0)

            raster_id = self.dlg.cmb_raster_load.currentData()
            raster_layer = QgsProject.instance().mapLayer(raster_id)

            if not raster_layer or not isinstance(raster_layer, QgsRasterLayer):
                QMessageBox.critical(self.dlg, self.tr("Error"), self.tr("Invalid or not selected raster layer."))
                return

            # Get the selected band
            band_index = self.dlg.cmb_field.currentData()
            if band_index is None:
                QMessageBox.critical(self.dlg, self.tr("Error"), self.tr("No band selected."))
                return

            self.dlg.progressBar.setValue(30)

            # Get Raster Data Provider
            provider = raster_layer.dataProvider()
            file_path = provider.dataSourceUri()
            dataset = gdal.Open(file_path)
            band = dataset.GetRasterBand(band_index)
            
            data_type = gdal.GetDataTypeName(band.DataType)
            gdal_to_numpy = {
                'Byte': np.uint8,
                'UInt16': np.uint16,
                'Int16': np.int16,
                'UInt32': np.uint32,
                'Int32': np.int32,
                'Float32': np.float32,
                'Float64': np.float64
            }

            self.dlg.progressBar.setValue(50)

            raster_data = band.ReadAsArray().astype(gdal_to_numpy[data_type])
            no_data_value = band.GetNoDataValue()

            # Mask NoData values
            if no_data_value is not None:
                raster_data = np.ma.masked_equal(raster_data, no_data_value)

            self.dlg.progressBar.setValue(70)

            # Gaussian Curve Plot
            mean = raster_data.mean()
            std_dev = raster_data.std()
            x = np.linspace(raster_data.min(), raster_data.max(), 100)
            gaussian = (1 / (std_dev * np.sqrt(2 * np.pi))) * np.exp(-0.5 * ((x - mean) / std_dev) ** 2)

            curve_color = self.get_label_background_color(self.dlg.label_12)

            self.dlg.progressBar.setValue(90)

            plt.figure(num=self.tr("Gaussian curve"))
            plt.plot(x, gaussian, color=curve_color)
            plt.title(self.tr("Gaussian curve"), pad=10, fontweight='bold', fontsize=12)
            plt.xlabel(self.tr("Value"), labelpad=5, fontweight='bold', fontsize=10)
            plt.ylabel(self.tr("Probability Density"), labelpad=5, fontweight='bold', fontsize=10)
            plt.grid(True, linestyle='--', alpha=0.5)  # Optional: Add gridlines for better readability
            plt.tight_layout(pad=0.5)  # Adjust layout with a small padding

            self.dlg.progressBar.setValue(100)

            plt.show()

            self.dlg.progressBar.setValue(0)

        except Exception as e:
            QMessageBox.critical(self.dlg, self.tr("Error"), self.tr(f"Error while generating Gaussian graph: {str(e)}"))

    def on_pushButton_cancel(self):
        """Clear all QLineEdit fields after confirmation if they contain values."""
        # List of all QLineEdit fields to check
        line_edits = [
            self.dlg.lineEdit_sum,
            self.dlg.lineEdit_count,
            self.dlg.lineEdit_mean,
            self.dlg.lineEdit_median,
            self.dlg.lineEdit_std,
            self.dlg.lineEdit_variance,
            self.dlg.lineEdit_min,
            self.dlg.lineEdit_max,
            self.dlg.lineEdit_minority,
            self.dlg.lineEdit_majority,
            self.dlg.lineEdit_unique
        ]
        
       # Check if there are any fields with values different from "0.00"
        has_non_zero_values = any(line_edit.text() != "0.00" for line_edit in line_edits)
        
        if not has_non_zero_values:
            QMessageBox.information(self.dlg, self.tr("No Values"), self.tr("All fields are already empty or zero."))
            return

        # Confirmation dialog
        reply = QMessageBox.question(
            self.dlg,
            self.tr("Confirm Reset"),
            self.tr("Some fields contain values. Are you sure you want to clear all fields?"),
            QMessageBox.Yes | QMessageBox.No,
            QMessageBox.No
        )
        
        if reply == QMessageBox.Yes:
            # Clear each field
            for line_edit in line_edits:
                #line_edit.clear()
                line_edit.setText("0.00") 
            
    
    def on_pushButton_close(self):
        """Close the UI. Show confirmation if fields contain non-zero values."""
        # List of all QLineEdit fields to check
        line_edits = [
            self.dlg.lineEdit_sum,
            self.dlg.lineEdit_count,
            self.dlg.lineEdit_mean,
            self.dlg.lineEdit_median,
            self.dlg.lineEdit_std,
            self.dlg.lineEdit_variance,
            self.dlg.lineEdit_min,
            self.dlg.lineEdit_max,
            self.dlg.lineEdit_minority,
            self.dlg.lineEdit_majority,
            self.dlg.lineEdit_unique
        ]
        
        # Check if any field has a non-zero value
        has_non_zero_values = any(line_edit.text().strip() and float(line_edit.text()) != 0 for line_edit in line_edits)
        
        if has_non_zero_values:
            # Confirmation dialog
            reply = QMessageBox.question(
                None,  # Use None or self.dlg if it inherits from QWidget
                self.tr("Confirm Close"),
                self.tr("Some fields contain values. Are you sure you want to close and clear all fields?"),
                QMessageBox.Yes | QMessageBox.No,
                QMessageBox.No
            )
            if reply == QMessageBox.No:
                return  # Do nothing if the user selects "No"

        # Clear each field
        for line_edit in line_edits:
            # line_edit.clear()
            line_edit.setText("0.00") 
        
        # Close the dialog
        self.dlg.close()
    

    def on_pushButton_export(self):
        """Export the calculated values to a .csv or .txt file with user-selected delimiter."""
        # List of the calculated values
        stats = {
            self.tr("Sum"): self.dlg.lineEdit_sum.text(),
            self.tr("Count"): self.dlg.lineEdit_count.text(),
            self.tr("Mean"): self.dlg.lineEdit_mean.text(),
            self.tr("Median"): self.dlg.lineEdit_median.text(),
            self.tr("Standard deviation"): self.dlg.lineEdit_std.text(),
            self.tr("Variance"): self.dlg.lineEdit_variance.text(),
            self.tr("Minimum"): self.dlg.lineEdit_min.text(),
            self.tr("Maximum"): self.dlg.lineEdit_max.text(),
            self.tr("Minority"): self.dlg.lineEdit_minority.text(),
            self.tr("Majority"): self.dlg.lineEdit_majority.text(),
            self.tr("Unique values"): self.dlg.lineEdit_unique.text()
        }

        # Check if any value is non-zero and non-empty
        has_non_zero_values = any(value.strip() and float(value) != 0 for value in stats.values())
        
        if not has_non_zero_values:
            QMessageBox.warning(self.dlg, self.tr("Warning"), self.tr("There are no calculated values to export."), QMessageBox.Ok)
            return  # If no values are calculated, exit the function

        # Get the selected delimiter from the comboBox
        delimiter = self.dlg.comboBox_col_delim.currentText()

        # Map the selected delimiter text to actual characters
        if delimiter == "T":
            delimiter = "\t"  # Tab character
        elif delimiter == ";":
            delimiter = ";"
        elif delimiter == ",":
            delimiter = ","
        elif delimiter == "|":
            delimiter = "|"
        else:
            # Default to comma if an unknown option is selected
            delimiter = ";"

        # Ask user to select a file path to save the data
        file_dialog = QFileDialog(self.dlg)
        file_dialog.setDefaultSuffix("csv")
        file_path, _ = file_dialog.getSaveFileName(self.dlg, self.tr("Save File"), "", self.tr("CSV Files (*.csv);;Text Files (*.txt)"))

        if not file_path:
            return  # If no file path is selected, return without saving

        try:
            # Determine file extension and write data to file
            file_extension = file_path.split('.')[-1].lower()
            
            if file_extension == 'csv':
                # Save data as CSV with the selected delimiter
                with open(file_path, mode='w', newline='') as file:
                    writer = csv.writer(file, delimiter=delimiter)
                    writer.writerow([self.tr("Statistic"), self.tr("Value")])  # Write header
                    for key, value in stats.items():
                        writer.writerow([key, value])  # Write each row
            elif file_extension == 'txt':
                # Save data as a text file with the selected delimiter
                with open(file_path, mode='w') as file:
                    for key, value in stats.items():
                        file.write(f"{key}{delimiter}{value}\n")  # Write each line with delimiter
            else:
                raise ValueError(self.tr("Unsupported file format"))

            QMessageBox.information(self.dlg, self.tr("Success"), self.tr("File successfully saved!"), QMessageBox.Ok)

        except Exception as e:
            QMessageBox.critical(self.dlg, self.tr("Error"), self.tr(f"An error occurred while saving the file: {str(e)}"), QMessageBox.Ok)

