# -*- coding: utf-8 -*-
"""
/***************************************************************************
 RasterReclassifier
                                 A QGIS plugin
 Reclassification of the raster layer using a table to assign new values. It is also possible to display a histogram of the classes according to the minimum and maximum values ​​of the reference band.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-01-03
        git sha              : $Format:%H$
        copyright            : (C) 2025 by Dr. Geol. Faustino Cetraro
        email                : geol-faustino@libero.it
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .raster_reclassifier_dialog import RasterReclassifierDialog
import os.path
import os

from PyQt5.QtWidgets import QAction, QTableWidgetItem, QFileDialog, QMessageBox
from PyQt5.QtWidgets import QGraphicsScene
from PyQt5.QtCore import QThreadPool, QRunnable, pyqtSignal, QObject
from PyQt5 import QtCore, QtGui, QtWidgets
from PyQt5.QtWidgets import QItemDelegate, QLineEdit
from PyQt5.QtGui import QIntValidator, QDoubleValidator
from PyQt5.QtCore import QRegExp
from PyQt5.QtGui import QRegExpValidator
import re
from PyQt5.QtCore import Qt
from PyQt5.QtWidgets import QColorDialog
from PyQt5.QtGui import QColor

import numpy as np
import matplotlib.pyplot as plt
from qgis.PyQt.QtCore import pyqtSlot
from qgis.PyQt.QtGui import QPixmap

from qgis.core import QgsRasterLayer, QgsRasterHistogram, QgsProject 
from qgis.core import QgsRasterBandStats

from matplotlib.backends.backend_qt5agg import FigureCanvasQTAgg as FigureCanvas
from osgeo import gdal
import csv
import locale


class RasterReclassifier:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        
        # initialize locale path
        locale_path = os.path.join(self.plugin_dir, 'i18n', f'raster_reclassifier_{locale}.qm')

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Raster Reclassifier')
        
        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('RasterReclassifier', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToRasterMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""
        icon_path = os.path.join(self.plugin_dir, 'icon.png')
        self.add_action(icon_path, text=self.tr(u'Raster Reclassifier'), callback=self.run, parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginRasterMenu(
                self.tr(u'&Raster Reclassifier'),
                action)
            self.iface.removeToolBarIcon(action)


    def run(self):
        """Run method that performs all the real work"""
        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start:
            self.first_start = False
            self.dlg = RasterReclassifierDialog()

        try:

            # Populates the list of loaded rasters
            self.populate_raster_list()  
            self.populate_fields()

            # Set the selected index of the QTabWidget to the first tab (index 0)
            self.dlg.tabWidget.setCurrentIndex(0)

            # Default colors
            self.dlg.label_10.setStyleSheet(f"background-color: {'#51f3f3'};")  # Default color for histogram
            self.dlg.label_12.setStyleSheet(f"background-color: {'red'};")      # Default color for vertical line

            # Event Linking: Run Once
            if not hasattr(self, "_events_connected") or not self._events_connected:
                # Reconnect events
                self.dlg.cmb_raster_load.currentIndexChanged.connect(self.populate_fields)
                self.dlg.pushButton_save_raster.clicked.connect(self.select_output_file)
                self.dlg.pushButton_colore_histogram.clicked.connect(self.select_histogram_color)
                self.dlg.pushButton_colore_class.clicked.connect(self.select_class_color)
                self.dlg.pushButton_extract.clicked.connect(self.create_histogram)
                self.dlg.pushButton_import.clicked.connect(self.import_values)
                self.dlg.pushButton_run.clicked.connect(self.on_pushButton_calculate)
                self.dlg.pushButton_cancel.clicked.connect(self.on_pushButton_cancel)
                self.dlg.pushButton_save_table.clicked.connect(self.on_pushButton_save_table)
                self.dlg.pushButton_graph.clicked.connect(self.on_pushButton_graph)
                self.dlg.pushButton_close.clicked.connect(self.on_pushButton_close)
                
                # Connessione del nuovo pulsante
                self.dlg.pushButton_new.clicked.connect(self.pushButton_new_value)

                self._events_connected = True  

            # Set the width of the columns
            self.dlg.tableWidget_value.setColumnWidth(0, 90)
            self.dlg.tableWidget_value.setColumnWidth(1, 90)
            self.dlg.tableWidget_value.setColumnWidth(2, 90)         

            # Connect the keypress event to the numerical control function
            self.dlg.tableWidget_value.itemChanged.connect(self.on_item_changed)

            # Initial histogram
            self.create_empty_histogram()

            # Show the dialog
            self.dlg.show()
            # Run the dialog event loop
            result = self.dlg.exec_()
            if result:
                pass
        except Exception as e:
            QMessageBox.critical(self.dlg, self.tr("Error"), self.tr(f"An error occurred: {str(e)}"))

    def on_item_changed(self, item):
        """Handles the item (cell) change event."""
        if item is not None:
            text = item.text()

            # Apply numeric control while editing cell
            if not self.is_valid_number(text):
                item.setText("")  # Sets a blank cell if the value is not numeric

    def is_valid_number(self, value):
        """Function that tests whether a string represents an integer or decimal, including the negative sign."""
        # Regular expression that allows integers and decimals, with negative sign
        pattern = r"^-?\d+([.,]\d+)?$"
        
        # Check if the value matches the pattern
        return bool(re.match(pattern, value))
    
    def populate_raster_list(self):
        self.dlg.cmb_raster_load.clear()
        layers = [layer for layer in QgsProject.instance().mapLayers().values() if isinstance(layer, QgsRasterLayer)]
        for layer in layers:
            self.dlg.cmb_raster_load.addItem(layer.name(), layer.id())

    def populate_fields(self):
        raster_id = self.dlg.cmb_raster_load.currentData()
        layer = QgsProject.instance().mapLayer(raster_id)
        if layer:
            provider = layer.dataProvider()
            self.dlg.cmb_field.clear()
            for band in range(1, provider.bandCount() + 1):
                self.dlg.cmb_field.addItem(f"Band {band}", band)
    
    def select_output_file(self):
        # Open a file dialog to select a save location with a filter for GeoTIFF files
        output_file, _ = QFileDialog.getSaveFileName(
            self.dlg, 
            self.tr("Select Output File"), 
            "", 
            "GeoTIFF Files (*.tif *.tiff);;All Files (*)"
        )
        
        # If a file was selected, update the output line edit with the file path
        if output_file:
            # Ensure the file has a .tif or .tiff extension
            if not output_file.endswith(('.tif', '.tiff')):
                output_file += '.tif'
            self.dlg.lineEdit_output.setText(output_file)
    
    def select_histogram_color(self):
        """Manages the color selection window for the histogram"""
        if not hasattr(self, 'color_dialog_histogram') or self.color_dialog_histogram is None:
            self.color_dialog_histogram = QColorDialog(self.dlg)
        
        # If the dialogue is already visible, do not open it again.
        if self.color_dialog_histogram.isVisible():
            return

        color = self.color_dialog_histogram.getColor()
        if color.isValid():
            self.selected_histogram_color = color.name()  # Get the color in hexadecimal format
            # Change the label background color
            self.dlg.label_10.setStyleSheet(f"background-color: {self.selected_histogram_color};")

    def select_class_color(self):
        """Manages the color selection window for the class"""
        if not hasattr(self, 'color_dialog_class') or self.color_dialog_class is None:
            self.color_dialog_class = QColorDialog(self.dlg)
        
        # If the dialogue is already visible, do not open it again.
        if self.color_dialog_class.isVisible():
            return

        color = self.color_dialog_class.getColor()
        if color.isValid():
            self.selected_class_color = color.name()  # Get the color in hexadecimal format
            # Change the label background color
            self.dlg.label_12.setStyleSheet(f"background-color: {self.selected_class_color};")
       
    def get_label_background_color(self, label):
        """Retrieve the background color of a QLabel."""
        style = label.styleSheet()
        color_start = style.find("background-color:")
        if color_start == -1:
            return None  # No color set
        color_end = style.find(";", color_start)
        return style[color_start + len("background-color:"):color_end].strip()

    def create_histogram(self):
        try:
            # progressbar
            self.dlg.progressBar.setValue(0)

            # State variable to prevent double messaging
            if not hasattr(self, "_table_cleared"):
                self._table_cleared = False

           # Initial check to see if the table is empty
            table_widget = self.dlg.tableWidget_value
            is_table_empty = all(
                table_widget.item(row, col) is None or table_widget.item(row, col).text().strip() == ""
                for row in range(table_widget.rowCount())
                for col in range(table_widget.columnCount())
            )
            
            if not is_table_empty and not self._table_cleared:
                # Confirmation message to overwrite values
                response = QMessageBox.question(
                    self.dlg,
                    self.tr("Confirm"),
                    self.tr("Do you want to overwrite the values in the table?"),
                    QMessageBox.Yes | QMessageBox.No,
                    QMessageBox.No
                )
                if response == QMessageBox.No:
                    return  # Stop execution if user chooses No

            # Reset the table
            table_widget.clearContents()
            table_widget.setRowCount(0)
            self._table_cleared = False  # Reset status after deletion
       
            # Get the selected raster layer
            raster_id = self.dlg.cmb_raster_load.currentData()
            raster_layer = QgsProject.instance().mapLayer(raster_id)

            if not raster_layer or not isinstance(raster_layer, QgsRasterLayer):
                QMessageBox.critical(self.dlg, self.tr("Error"), self.tr("Invalid or not selected raster layer."))
                return
            
            # Indicates that the table has been confirmed for deletion
            self._table_cleared = True
            
            # progressbar
            self.dlg.progressBar.setValue(10)

            # Get the selected band
            band_index = self.dlg.cmb_field.currentData()
            if band_index is None:
                QMessageBox.critical(self.dlg, self.tr("Error"), self.tr("No band selected."))
                return

            # progressbar
            self.dlg.progressBar.setValue(20)

            # Get Raster Data Provider
            provider = raster_layer.dataProvider()

            # Control which function to use based on the combobox index
            index = self.dlg.cmb_statist_raster.currentIndex()
            
            # progressbar
            self.dlg.progressBar.setValue(30)

            if index == 0:  # Full raster
                file_path = raster_layer.dataProvider().dataSourceUri()
                dataset = gdal.Open(file_path)
                if not dataset:
                    QMessageBox.critical(self.dlg, self.tr("Error"), self.tr("Unable to open raster file with GDAL."))
                    return

                band = dataset.GetRasterBand(band_index)
                stats = band.GetStatistics(True, True)
                min_value, max_value, mean_value, std_dev = stats[0], stats[1], stats[2], stats[3]
                
                raster_data = band.ReadAsArray().astype(np.float32)
                no_data_value = band.GetNoDataValue()
            else:  # Current view
                stats = provider.bandStatistics(band_index, QgsRasterBandStats.All)
                if stats.elementCount == 0:
                    QMessageBox.critical(self.dlg, self.tr("Error"), self.tr("No data available in the selected band."))
                    return

                min_value, max_value, mean_value, std_dev = stats.minimumValue, stats.maximumValue, stats.mean, stats.stdDev
                extent = raster_layer.extent()
                cols, rows = raster_layer.width(), raster_layer.height()
                block = provider.block(band_index, extent, cols, rows)
                raster_data = np.array([[block.value(col, row) for col in range(cols)] for row in range(rows)], dtype=float)
                no_data_value = provider.sourceNoDataValue(band_index)

            # progressbar
            self.dlg.progressBar.setValue(40)

            # Mask NoData values
            masked_data = np.ma.masked_equal(raster_data, no_data_value)

            # progressbar
            self.dlg.progressBar.setValue(50)
            
            # Counts valid cells (cells that are not masked as NoData)
            valid_cells = np.count_nonzero(~masked_data.mask)

            # progressbar
            self.dlg.progressBar.setValue(60)

            # Update values in linesEdit
            self.dlg.lineEdit_count.setText(str(valid_cells))
            self.dlg.lineEdit_min.setText(f"{min_value:.2f}")  
            self.dlg.lineEdit_max.setText(f"{max_value:.2f}")
            self.dlg.lineEdit_mean.setText(f"{mean_value:.2f}")
            self.dlg.lineEdit_std.setText(f"{std_dev:.2f}")

            # progressbar
            self.dlg.progressBar.setValue(70)

            # Get the number of classes and index of the selected method
            num_classes = self.dlg.spinBox_classes.value()
            method_index = self.dlg.cmb_method.currentIndex()

            # Calculate ranges based on the selected method
            breaks = None
            try:
                # Calculating intervals
                if method_index == 0:  # Equal Interval
                    breaks = np.linspace(min_value, max_value, num_classes + 1)
                elif method_index == 1:  # Quantile
                    breaks = np.percentile(masked_data.compressed(), np.linspace(0, 100, num_classes + 1))
                elif method_index == 2:  # Manual
                    breaks = [None] * (num_classes + 1)  # Lista vuota per indicare che non ci sono intervalli da calcolare
                else:
                    QMessageBox.critical(self.dlg, self.tr("Error"), self.tr("Classification method not supported."))
                    return
                
            except Exception as e:
                QMessageBox.critical(self.dlg, self.tr("Error"), self.tr(f"Error while calculating intervals: {str(e)}"))
                return
                  
            # progressbar
            self.dlg.progressBar.setValue(80)

            # Update tableWidget_value
            self.dlg.tableWidget_value.setRowCount(num_classes)
            for i in range(num_classes):
                if method_index != 2:
                    # We handle the lower and upper bound of the range separately
                    if isinstance(breaks[i], tuple):
                        start_item = QTableWidgetItem(f"{breaks[i][0]:.2f}")  # Lower limit
                        end_item = QTableWidgetItem(f"{breaks[i][1]:.2f}")    # Upper limit
                    else:
                        start_item = QTableWidgetItem(f"{breaks[i]:.2f}")    # For methods like Equal Interval and Quantile
                        end_item = QTableWidgetItem(f"{breaks[i + 1]:.2f}")  # Next limit
               
                else:  # Manual Method, empty cells only
                    start_item = QTableWidgetItem("")
                    end_item = QTableWidgetItem("")

                new_item = QTableWidgetItem("")  # Empty field to enter new values
                self.dlg.tableWidget_value.setItem(i, 0, start_item)
                self.dlg.tableWidget_value.setItem(i, 1, end_item)
                self.dlg.tableWidget_value.setItem(i, 2, new_item)

                # Align text to the center for each cell
                start_item.setTextAlignment(QtCore.Qt.AlignCenter)
                end_item.setTextAlignment(QtCore.Qt.AlignCenter)
                new_item.setTextAlignment(QtCore.Qt.AlignCenter)

                self.dlg.tableWidget_value.setItem(i, 0, start_item)
                self.dlg.tableWidget_value.setItem(i, 1, end_item)
                self.dlg.tableWidget_value.setItem(i, 2, new_item)
                     
            # progressbar
            self.dlg.progressBar.setValue(90)

            # Retrieve colors from label backgrounds
            histogram_color = self.get_label_background_color(self.dlg.label_10)
            class_color = self.get_label_background_color(self.dlg.label_12)

            # Create the histogram
            fig = plt.figure(figsize=(4.41, 3.41))
            ax = fig.add_subplot(111)
            ax.grid(True)
            ax.hist(masked_data.compressed(), bins=200, color=histogram_color, alpha=0.7)
            ax.set_title(self.tr("Histogram"), pad=10, fontweight='bold', fontsize=12)
            ax.set_xlabel(self.tr("Raster Values"), labelpad=5, fontweight='bold', fontsize=10)
            ax.set_ylabel(self.tr("Count"), labelpad=5, fontweight='bold', fontsize=10)

            # progressbar
            self.dlg.progressBar.setValue(100)

            if method_index != 2:
                # Draw red dashed vertical lines for the values ​​in the `Start` column
                for i in range(num_classes):  # Tutte le righe, dalla prima all'ultima
                    start_value = float(self.dlg.tableWidget_value.item(i, 0).text())
                    ax.axvline(x=start_value, color=class_color, linestyle='--', linewidth=1)

                # Add the last vertical line for the `End` value of the last row
                last_end_value = float(self.dlg.tableWidget_value.item(num_classes - 1, 1).text())
                ax.axvline(x=last_end_value, color=class_color, linestyle='--', linewidth=1)

            plt.tight_layout(pad=0.5)
            self.canvas = FigureCanvas(fig)
            self.canvas.setFixedSize(441, 341)
            scene = QGraphicsScene()
            scene.addWidget(self.canvas)
            self.dlg.graphicsView_hystogram.setScene(scene)
            self.canvas.draw()

            # progressbar
            self.dlg.progressBar.setValue(0)

        except Exception as e:
            QMessageBox.critical(self.dlg, self.tr("Error"), self.tr(f"Error generating histogram: {str(e)}"))
    
    def create_empty_histogram(self):
        try:
            # Close any existing figures to prevent multiple figures from stacking up in memory
            plt.close('all')  # Chiude tutte le figure attive

            # Create the figure and the axis
            fig = plt.figure(figsize=(4.41, 3.41))  # Sized to fit QGraphicsView
            ax = fig.add_subplot(111)

            # Clean the chart from any previous data
            ax.clear()

            # Set the axis title and labels
            ax.set_title(self.tr("Histogram"), pad=10, fontweight='bold', fontsize=12)
            ax.set_xlabel(self.tr("Raster Values"), labelpad=5, fontweight='bold', fontsize=10)
            ax.set_ylabel(self.tr("Count"), labelpad=5, fontweight='bold', fontsize=10)

            # Add a grid to make the graph visible
            ax.grid(True)

            # Setting x-axis and y-axis limits to display blank chart correctly
            ax.set_xlim(0, 1000)   # X-axis limits
            ax.set_ylim(0, 1000)   # Y-axis limits (to display the grid)

            # Use tight_layout to arrange chart elements
            plt.tight_layout(pad=0.5)

            # Create the FigureCanvas to display the graph inside the QGraphicsView
            self.canvas = FigureCanvas(fig)
            self.canvas.setFixedSize(441, 341)  # Fixed size to fit QGraphicsView

            # Create a graphic scene and add the canvas
            scene = QGraphicsScene()
            scene.addWidget(self.canvas)

            # Add scene to graphicsView_hystogram
            self.dlg.graphicsView_hystogram.setScene(scene)

            # Draw the canvas (view the graph)
            self.canvas.draw()

        except Exception as e:
            QMessageBox.critical(self.dlg, self.tr("Error"), self.tr(f"Error creating empty histogram: {str(e)}"))
    
    def on_pushButton_cancel(self):
        try:
            # Check if the table is empty
            row_count = self.dlg.tableWidget_value.rowCount()
            if row_count == 0:
                return  
            
            # Show a confirmation message
            reply = QMessageBox.question(self.dlg, self.tr('Confirm'), 
                                        self.tr("Are you sure you want to delete your data?"), 
                                        QMessageBox.Yes | QMessageBox.No, 
                                        QMessageBox.No)

            # If the user chooses 'No', exit the function without doing anything
            if reply == QMessageBox.No:
                return
        
            # 1. Restore linesEdit
            self.dlg.lineEdit_count.setText("0.00") 
            self.dlg.lineEdit_min.setText("0.00")      
            self.dlg.lineEdit_max.setText("0.00")      
            self.dlg.lineEdit_mean.setText("0.00")     
            self.dlg.lineEdit_std.setText("0.00")      
            self.dlg.cmb_statist_raster.setCurrentIndex(0)
            self.dlg.lineEdit_output.setText("")
            self.dlg.cmb_method.setCurrentIndex(0)
            self.dlg.spinBox_classes.setValue(5) 

            # 2. Clean up the tableWidget_value (table)
            self.dlg.tableWidget_value.clearContents()  # Clear the contents of the table
            self.dlg.tableWidget_value.setRowCount(0)   # Removes all rows from the table
            
            # 3. Reset the graph (returning to an initial state, for example empty)
            self.create_empty_histogram()

        except Exception as e:
            QMessageBox.critical(self.dlg, self.tr("Error"), self.tr(f"Error during reset: {str(e)}"))
    
    def on_pushButton_save_table(self):
        try:
            # Check if the table is empty
            row_count = self.dlg.tableWidget_value.rowCount()
            if row_count == 0:
                QMessageBox.warning(self.dlg, self.tr("Empty Table"), self.tr("The table is empty. There is no data to save."))
                return  # Exits the function if the table is empty

            # Get the selected delimiter from the comboBox
            selected_delimiter = self.dlg.comboBox_col_delim.currentText()

            # Delimiter map based on comboBox selection
            delimiter_map = {
                ";" : ";",    # Semicolon
                "," : ",",    # Comma
                "|" : "|",    # Pipe
                "T" : "\t"    # Tabulation
            }

            # Get the matching delimiter
            delimiter = delimiter_map.get(selected_delimiter, ";")  # Default ';' if not found

            # Get the selected decimal symbol from the comboBox
            decimal_symbol = self.dlg.comboBox_decimals.currentText()  # It can be either "." or ","

            # Ask the user to choose the path and name of the file to save
            options = QFileDialog.Options()
            file_path, _ = QFileDialog.getSaveFileName(self.dlg, self.tr("Save table"), "", self.tr("CSV Files (*.csv);;Text Files (*.txt)"), options=options)

            if not file_path:  # If the user has not chosen a file, exit
                return

            # Open the file for writing
            with open(file_path, mode='w', newline='', encoding='utf-8') as file:
                writer = csv.writer(file, delimiter=delimiter)

                # Add header (column titles)
                writer.writerow([self.tr('Start'), self.tr('End'), self.tr('New')])  # Customize by table

                # Add data from table
                for row in range(row_count):
                    start_item = self.dlg.tableWidget_value.item(row, 0)
                    end_item = self.dlg.tableWidget_value.item(row, 1)
                    new_item = self.dlg.tableWidget_value.item(row, 2)

                    # Format the numerical values with the selected decimal symbol
                    def format_decimal(value):
                        try:
                            if value:
                                # Convert to float
                                value = float(value)
                                # Format with two decimal places
                                formatted_value = f"{value:.2f}"
                                # Replace the decimal point with the selected symbol
                                return formatted_value.replace('.', decimal_symbol)
                            else:
                                return ""
                        except ValueError:
                            return value  # Return original value if not a number

                    # Write each row of the table into the file, formatting all numerical columns
                    writer.writerow([
                        format_decimal(start_item.text()) if start_item else '', 
                        format_decimal(end_item.text()) if end_item else '', 
                        new_item.text() if new_item else ''
                    ])

            # Show a confirmation message
            QMessageBox.information(self.dlg, self.tr("Save completed"), self.tr("Table saved successfully!"))

        except Exception as e:
            QMessageBox.critical(self.dlg, self.tr("Error"), self.tr(f"Error saving file: {str(e)}"))

    
    def on_pushButton_graph(self):
        try:
             # Check if the table is empty
            row_count = self.dlg.tableWidget_value.rowCount()
            if row_count == 0:
                QMessageBox.warning(self.dlg, self.tr("Error"), self.tr("No histogram to save."))
                return  
            
            # Ask the user to choose the path and file name to save the image
            options = QFileDialog.Options()
            file_path, _ = QFileDialog.getSaveFileName(self.dlg, self.tr("Save histogram"), "", self.dlg, self.tr("Image Files(*.png *.jpg *.jpeg *.bmp *.tiff);;PNG Files (*.png)"), options=options)

            if not file_path:  # If the user has not chosen a file, exit
                return

            # Save the graph as an image (PNG, JPEG, etc.)
            self.canvas.figure.savefig(file_path, dpi=300, bbox_inches='tight')  # Save in high resolution

            # Show a confirmation message
            QMessageBox.information(self.dlg, self.tr("Save completed"), self.tr("Histogram saved successfully!"))

        except Exception as e:
            QMessageBox.critical(self.dlg, self.tr("Error"), self.tr(f"Error saving image: {str(e)}"))
    
    def on_pushButton_close(self):
        try:
            # Check if the table contains non-empty data
            table_widget = self.dlg.tableWidget_value
            has_non_empty_rows = any(
                table_widget.item(row, col) is not None and table_widget.item(row, col).text().strip() != ""
                for row in range(table_widget.rowCount())
                for col in range(table_widget.columnCount())
            )

            # If the table is empty, close the window directly
            if not has_non_empty_rows:
                self.dlg.close()
                return

            # Show a confirmation message
            reply = QMessageBox.question(
                self.dlg,
                self.tr('Confirm'),
                self.tr("Are you sure you want to delete your data?"),
                QMessageBox.Yes | QMessageBox.No,
                QMessageBox.No
            )

            # If the user chooses 'No', exit the function without doing anything
            if reply == QMessageBox.No:
                return
            
            # Clear the tableWidget_value
            self.dlg.tableWidget_value.clearContents()
            self.dlg.tableWidget_value.setRowCount(0)

            # Release the graphic scene if it exists
            if hasattr(self, 'canvas') and self.canvas:
                self.canvas.close()
                self.canvas.deleteLater()
                self.canvas = None

            if self.dlg.graphicsView_hystogram.scene():
                self.dlg.graphicsView_hystogram.scene().clear()
                self.dlg.graphicsView_hystogram.setScene(None)

            # Reset progress bar
            self.dlg.progressBar.setValue(0)

            # Reset lineEdit
            self.dlg.lineEdit_count.clear()
            self.dlg.lineEdit_min.clear()
            self.dlg.lineEdit_max.clear()
            self.dlg.lineEdit_mean.clear()
            self.dlg.lineEdit_std.clear()
            self.dlg.lineEdit_output.clear()

            # Close the dialogue
            self.dlg.close()

            # Destroy the color selectors to prevent them from being reopened next time
            if hasattr(self, 'color_dialog_histogram'):
                self.color_dialog_histogram.deleteLater()
                self.color_dialog_histogram = None
            
            if hasattr(self, 'color_dialog_class'):
                self.color_dialog_class.deleteLater()
                self.color_dialog_class = None

            # Close any existing figures to prevent multiple figures from stacking up in memory
            plt.close('all')  # Chiude tutte le figure attive

        except Exception as e:
            QMessageBox.critical(self.dlg, self.tr("Error"), self.tr(f"Error during cleanup: {str(e)}"))


    def import_values(self):
        # Get the selected delimiter from the comboBox
        selected_delimiter = self.dlg.comboBox_col_delim.currentText()

        # Delimiter map based on comboBox selection
        delimiter_map = {
            ";" : ";",    # Semicolon
            "," : ",",    # Comma
            "T" : "\t",   # Tabulation
            "|" : "|"     # Pipe
        }
        
        # Get the matching delimiter
        delimiter = delimiter_map.get(selected_delimiter, ";")  # Default ';' if not found

        # Open a dialog box to select the file
        file_path, _ = QFileDialog.getOpenFileName(
            self.dlg, self.tr("Import values"), "", self.tr("Text file (*.txt);;File CSV (*.csv);;All files (*)")
        )

        if not file_path:
            return  # If user cancels, exit the function

        try:
            # Open the selected file and read the data
            with open(file_path, newline='', encoding='utf-8') as file:
                reader = csv.reader(file, delimiter=delimiter)
                rows = list(reader)

            # Clear the tableWidget before inserting data
            self.dlg.tableWidget_value.clearContents()
            self.dlg.tableWidget_value.setRowCount(0)

            # Check if the first row contains numeric values
            def is_numeric_row(row_data):
                try:
                    # Try to convert the first 3 values to numbers (float)
                    for value in row_data[:3]:  # Only the first 3 values
                        float(value.replace(',', '.'))  # # Convert using "." as the decimal separator
                    return True
                except ValueError:
                    return False

            # If the first line is numeric, it is considered, otherwise it is skipped.
            start_index = 0
            if not is_numeric_row(rows[0]):
                start_index = 1  # Ignore the first line if it is not numeric

            # Populate the tableWidget with data starting from the correct row
            for row_data in rows[start_index:]:
                if len(row_data) >= 3:  # Make sure there are at least 3 columns
                    row_index = self.dlg.tableWidget_value.rowCount()
                    self.dlg.tableWidget_value.insertRow(row_index)
                    for col_index in range(3):  # Solo le prime 3 colonne
                        # Replace commas with periods in numeric values
                        value = row_data[col_index].replace(',', '.') if self.is_numeric(row_data[col_index]) else row_data[col_index]
                        item = QTableWidgetItem(value)
                        item.setTextAlignment(Qt.AlignCenter)  # Allinea al centro
                        self.dlg.tableWidget_value.setItem(row_index, col_index, item)

            # Set the spinBox_classes value to the number of rows in the table
            row_count = self.dlg.tableWidget_value.rowCount()
            self.dlg.spinBox_classes.setValue(row_count)

        except Exception as e:
            # Show an error message if there is a problem
            QMessageBox.critical(self.dlg, self.tr("Error"), self.tr(f"Unable to import data: {str(e)}"))

    def is_numeric(self, value):
        """Helper method to check if a value is numeric."""
        try:
            # Try to convert the value to float
            float(value.replace(',', '.'))  # Convert using "." as the decimal separator
            return True
        except ValueError:
            return False

    def on_pushButton_calculate(self):
        try:
            self.dlg.progressBar.setValue(0)  # progressbar

            # Get the selected raster layer
            raster_id = self.dlg.cmb_raster_load.currentData()
            raster_layer = QgsProject.instance().mapLayer(raster_id)

            if not raster_layer or not isinstance(raster_layer, QgsRasterLayer):
                QMessageBox.critical(self.dlg, self.tr("Error"), self.tr("Invalid or not selected raster layer."))
                return

            self.dlg.progressBar.setValue(10)

            # Get the selected band
            band_index = self.dlg.cmb_field.currentData()
            if band_index is None:
                self.dlg.progressBar.setValue(0)
                QMessageBox.critical(self.dlg, self.tr("Error"), self.tr("No band selected."))
                return

            self.dlg.progressBar.setValue(20)

            # Get Raster Data Provider
            provider = raster_layer.dataProvider()
            file_path = provider.dataSourceUri()
            dataset = gdal.Open(file_path)
            if not dataset:
                self.dlg.progressBar.setValue(0)
                QMessageBox.critical(self.dlg, self.tr("Error"), self.tr("Unable to open raster file with GDAL."))
                return

            band = dataset.GetRasterBand(band_index)
            raster_data = band.ReadAsArray().astype(np.float32)
            no_data_value = band.GetNoDataValue()

            self.dlg.progressBar.setValue(40)

            # Mask NoData values
            if no_data_value is not None:
                raster_data = np.ma.masked_equal(raster_data, no_data_value)

            self.dlg.progressBar.setValue(50)

            # Get the save path of the new raster
            output_path = self.dlg.lineEdit_output.text()
            if not output_path:
                self.dlg.progressBar.setValue(0)
                QMessageBox.warning(self.dlg, self.tr("Error"), self.tr("Specify the save path for the GeoTIFF file."))
                return

            self.dlg.progressBar.setValue(60)

            # Retrieve minimum and maximum values
            min_value = float(self.dlg.lineEdit_min.text())
            max_value = float(self.dlg.lineEdit_max.text())

            # Iterate over the rows of the table and apply the reclassification
            row_count = self.dlg.tableWidget_value.rowCount()
            for row in range(row_count):
                start_item = self.dlg.tableWidget_value.item(row, 0)
                end_item = self.dlg.tableWidget_value.item(row, 1)
                new_item = self.dlg.tableWidget_value.item(row, 2)

                if start_item and end_item and new_item:
                    try:
                        start_value = float(start_item.text().strip())
                        end_value = float(end_item.text().strip())
                        new_value = float(new_item.text().strip())

                        if start_value < min_value or end_value > max_value:
                            QMessageBox.warning(
                                self.dlg, self.tr("Range Error"),
                                self.tr(f"The start values ({start_value}) and end ({end_value}) must be between the minimum and maximum values.")
                            )
                            return

                        raster_data = self.reclassify_raster(raster_data, start_value, end_value, new_value)

                    except ValueError:
                        QMessageBox.warning(
                            self.dlg, self.tr("Error"), self.tr(f"Non-numeric values in row cells {row + 1}.")
                        )
                        return

            self.dlg.progressBar.setValue(80)

            # Save the new raster as GeoTIFF
            driver = gdal.GetDriverByName('GTiff')
            if driver is None:
                self.dlg.progressBar.setValue(0)
                QMessageBox.warning(self.dlg, self.tr("Error"), self.tr("Driver per GeoTIFF non disponibile."))
                return

            output_dataset = driver.Create(output_path, raster_data.shape[1], raster_data.shape[0], 1, gdal.GDT_Float32)
            output_band = output_dataset.GetRasterBand(1)
            output_band.WriteArray(raster_data)

            self.dlg.progressBar.setValue(90)

            # Copy metadata from the original raster
            output_dataset.SetGeoTransform(dataset.GetGeoTransform())
            output_dataset.SetProjection(dataset.GetProjection())

            if no_data_value is not None:
                output_band.SetNoDataValue(no_data_value)

            output_dataset = None  # Close the dataset

            self.dlg.progressBar.setValue(100)

            QMessageBox.information(self.dlg, self.tr("Save completed"), self.tr(f"Reclassified raster saved as GeoTIFF: {os.path.basename(output_path)}"))

            # Load the raster into the project if required
            if self.dlg.checkBox_load_prg.isChecked():
                new_raster_layer = QgsRasterLayer(output_path, os.path.basename(output_path))
                if new_raster_layer.isValid():
                    QgsProject.instance().addMapLayer(new_raster_layer)
                else:
                    QMessageBox.warning(self.dlg, self.tr("Error"), self.tr("Unable to load new raster into project."))

            self.dlg.progressBar.setValue(0)

        except Exception as e:
            QMessageBox.critical(self.dlg, self.tr("Error"), self.tr(f"Error while calculating reclassification: {str(e)}"))
            self.dlg.progressBar.setValue(0)  # Reset progressbar on error


    def reclassify_raster(self, raster_data, start_value, end_value, new_value):
        """
        Reclassifies the raster values by replacing those in the range [start_value, end_value]
        with the new_value value.
        """
        return np.where((raster_data >= start_value) & (raster_data <= end_value), new_value, raster_data)

    def pushButton_new_value(self):
        """
        Fills the third column of the tableWidget with increasing integer values, starting at 1.
        Ensures the table has at least one row and three columns before proceeding.

        The function assumes that `self.dlg.tableWidget_value` is a valid QTableWidget instance.
        """
        # Verify that the table exists in the dialog
        if not hasattr(self.dlg, 'tableWidget_value'):
            QMessageBox.critical(self.dlg, self.tr("Error"), self.tr("The tableWidget_value is not defined in the dialog."))
            return

        # Verify that the table has rows
        row_count = self.dlg.tableWidget_value.rowCount()
        if row_count == 0:
            QMessageBox.warning(self.dlg, self.tr("Warning"), self.tr("The table is empty. Add some rows before proceeding."))
            return     

        # Fills the third column with increasing integer values starting from 1
        for row in range(row_count):
            item = QTableWidgetItem(str(row + 1))  # Crea un nuovo item con il valore
            item.setTextAlignment(Qt.AlignCenter)  # Allinea il testo al centro
            self.dlg.tableWidget_value.setItem(row, 2, item)  # Aggiungi l'item alla cella
        
        # Mostra un messaggio di conferma
        QMessageBox.information(self.dlg, self.tr("Operation completed"), self.tr("The new values have been added successfully."))
