# -*- coding: utf-8 -*-
# Name: raster_MCDA
# Author: Wallner Andreas
# Date: 05.01.2020
# description:  Plugins main script where all the action takes place
# TO DO:  
#    Check weights dynamically
#    benefit / cost table input options
#    block criterion table column 0 (layername)
#    check if LineEdit_Output is valid Path

    

"""
/***************************************************************************
 Rastermcda
                                 A QGIS plugin
 Plugin implementing MCDA, for raster datasets
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2019-10-25
        git sha              : $Format:%H$
        copyright            : (C) 2019 by Wallner Andreas Georg
        email                : andreasgeorg.wallner@edu.fh-kaernten.ac.at
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QTableWidget, QTableWidgetItem, QMessageBox, QFileDialog
from qgis.core import QgsProject, QgsRasterLayer, Qgis, QgsLayerTreeLayer 
from osgeo import gdal # for conversion raster to array to raster
import numpy as np # for Array calculation
from osgeo import osr # for Spatial reference System
import sys
from fractions import Fraction


# addition QGIS Task (05.01.2020)
from qgis.core import QgsTask, QgsMessageLog
MESSAGE_CATEGORY = 'CalculationTask'

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .raster_MCDA_dialog import RastermcdaDialog
from .raster_MCDA_dialog import RastermcdaWeightingPCDialog
import os.path



class Rastermcda:
    
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        
        # Set variables and default values --------------------------------------------------------------
        # Save reference to the QGIS interface
        self.iface = iface
        # list for the inputlayers (Will come from QGIS Layer tree)
        self.inputlayers = []
        # list for crterion layers (will come from criteriontable)
        self.criterionRasters = []
        # list for constraint layers (will come from constrainttable)
        self.constraintRasters = []
        # noData Value for the output Raster (= standart noData for float32:https://gis.stackexchange.com/questions/334073/qgis-3-does-not-recognize-3-40282e38-float-raster-values-as-nodata/338388)
        self.outNoData = -3.40282e+38 
        #number of digits which have to stay the same for eigenvector calculation in Pairwise comparision weighting (4 digits = 10000)
        self.numSameDigits_WPC = 10000
        # default active criterion table (self.dlg.activetable) will be initialized in run()
        # instance to UI
        self.dlg_weighting_PC = RastermcdaWeightingPCDialog()
        #------------------------------------------------------------------------------------------------
        
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'Rastermcda_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&rasterMCDA')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('Rastermcda', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/raster_MCDA/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'MCDA - Raster'),
            callback=self.run,
            parent=self.iface.mainWindow())
        
        self.add_action(
            icon_path,
            text=self.tr(u'MCDA - Weighting (Pairwise)'),
            callback=self.runWeightingPC,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&rasterMCDA'),
                action)
            self.iface.removeToolBarIcon(action)

    def run(self):
        self.dlg = RastermcdaDialog()
        self.weightsLoaded = False
        # default active criterion table
        self.activeTable = self.dlg.tableWidget_wlc
        self.activeTable.setEnabled(True)
        """Run method that performs all the real work"""
        # Create the dialog with elements (after translation) and keep reference
        # connect Qt widget events with functions
        self.dlg.pushButton_input.clicked.connect(self.addCriterion)
        self.dlg.pushButton_deleteRow.clicked.connect(self.removeCriteria)
        self.dlg.pushButton_help.clicked.connect(self.dlg.showHelp)
        self.dlg.pushButton_output.clicked.connect(self.select_output_file)
        
        self.dlg.pushButton_input_con.clicked.connect(self.addConstraint)
        self.dlg.pushButton_deleteRow_con.clicked.connect(self.removeConstraint)
        self.dlg.pushButton_ok.clicked.connect(self.run_mcda)
        self.dlg.pushButton_cancel.clicked.connect(self.exit_mcda)
        
        self.dlg.tabWidget_decision_rule.currentChanged.connect(self.setActiveTable)
        
        self.dlg.pushButton_loadWeights.clicked.connect(self.loadWeights)
        
        #VALIDATION_1----------------------------------------------------------------------
        # check for min 2 valid rasterlayer and add only valid raster Layers to plugin
        # Clear the contents of inputlayers from previous runs
        self.inputlayers.clear()
        # Funktion to get all the Layer nodes from a QGIS Layer Tree to self.inputLayers
        self.setInputLayers(QgsProject.instance().layerTreeRoot())
        # Funktion checks if #inputLayers min 2
        self.checkInputLayers() 
        #VALIDATION_1_end--------------------------------------------------------------------
        
        # Put al elements of self.inputLayers to comboBox--------------------------------------
        # Clear the contents of the comboBox from previous runs
        self.dlg.comboBox_input.clear()
        # Populate the comboBox with names of all the loaded layers
        self.dlg.comboBox_input.addItems([layer.name() for layer in self.inputlayers])
        # create Tables (number columns, headers, size..)
        self.dlg.createQTables()
        #to comboBox_end---------------------------------------------------------------------------------------
        
        # show the dialog
        self.dlg.show()

    def run_mcda(self):
        try:
            # check output location -----------------------------------------------------------------------
            # if outputPath is empty or not a valid path call funktion self.select_output_file() -> QtFileDialog 
            if (len(self.dlg.lineEdit_output.text()) == 0) or not os.path.exists(self.dlg.lineEdit_output.text()): 
                self.select_output_file()
            #check output location_end------------------------------------------------------------------------------------------------
            
            # Get layers from criteriontable into self.criterionRasters -----------------------------------
            # Clear the contents of criterionRasters from previous runs
            self.criterionRasters.clear()
            for row in range(0, self.activeTable.rowCount()):
                # get rastername form table column 0
                rastername = QTableWidgetItem(self.activeTable.item(row, 0)).text()
                # get raster from QGIS Project according to the name
                rlayer = QgsProject.instance().mapLayersByName(rastername)[0]
                # add raster to list self.criterionRasters
                self.criterionRasters.append(rlayer)
            # Get layers from criteriontable_end-----------------------------------------------------------------
            
            # Get layers from constrainttable into self.constraintRasters -----------------------------------
            if (self.dlg.tableWidget_con.rowCount()>0):
                # Clear the contents of constraintRasters from previous runs
                self.constraintRasters.clear()
                for row in range(0, self.dlg.tableWidget_con.rowCount()):
                    # get rastername form table column 0
                    rastername = QTableWidgetItem(self.dlg.tableWidget_con.item(row, 0)).text()
                    # get raster from QGIS Project according to the name
                    rlayer = QgsProject.instance().mapLayersByName(rastername)[0]
                    # add raster to list self.criterionRasters
                    self.constraintRasters.append(rlayer)
                # Get layers from constrainttable_end-----------------------------------------------------------------
            
            
            # VALIDATION 2 -----------------------------------------------------------------------------------
            # Call function to check criteria Layer properties (cellsize, extend, SRS)
            self.checkCriteriaLayerProperties()
            #VALIDATION 2_end-----------------------------------------------------------------------------------
            
            #Run WLC or IP
            if self.dlg.tabWidget_decision_rule.currentIndex() == 0:
                self.runWLC()
            if self.dlg.tabWidget_decision_rule.currentIndex() == 1:
                self.runIP()
            if self.dlg.tabWidget_decision_rule.currentIndex() == 2:
                self.runOWA()
        except Exception as error:
            # Error, User gets information through Messagebox and self.iface.messageBar().pushMessage()
            self.iface.messageBar().pushMessage(
            "Something went wrong", "Calculation not possible: "+repr(error),
            level=Qgis.Critical, duration=10)
        

#-----------------------------------Decision Rules logic Methods----------------------------------

    # WLC decision Rule
    def runWLC(self):
        # Preperations--------------------------------------------------------------------------
        # Funktion to convert all the inputrasters into NumPy arrays
        rArrays = self.rastersToArray(self.criterionRasters)
        # create Array with same shape as first input Array (used for output)
        outputArray = np.zeros(rArrays[0].shape)
        # get weights out of criteriontable
        weights = self.getWeights()
        # Preperarions_end-------------------------------------------------------------------------
        
        # Calculation WLC --------------------------------------------------------------------------
        # loop through every Array
        for i, array in enumerate(rArrays):
            # multiply the current array with the dedicated weight (calculate like: Matrix*skalar)
            # and add the result to outputArray
            outputArray += array * weights[i]
        
        # set np.nan to Output No data Value
        outputArray[outputArray == np.nan] = self.outNoData
        # Calculation WLC_end---------------------------------------------------------------------
        
        #call function to convert Output Array to a raster file
        self.outArrayToRaster(outputArray)


    # IP (TOPSIS) decision Rule
    def runIP(self):
        # Preperations --------------------------------------------------------------------------
        # P Value for calculation (dinstance prinziple)
        pValue = 2
        # Funktion to convert all the inputrasters into NumPy arrays
        rArrays = self.rastersToArray(self.criterionRasters)
        # create Arrays with same shape as first input Array (used for output)
        # sPlus for distance to idel point
        sPlus = np.zeros(rArrays[0].shape)
        # sNeg for distance to negative idel point
        sNeg = np.zeros(rArrays[0].shape)
        # outputArray for the result of TOPSIS
        outputArray = np.zeros(rArrays[0].shape)
        # get weights out of criteriontable
        weights = self.getWeights()
        # Preperarions_end-------------------------------------------------------------------------
        
        # Calculation TOPSIS ---------------------------------------------------------------------
        # loop through every Array
        for i, array in enumerate(rArrays):
            # calculate sPlus for active array and add result to old sPlus
            sPlus += (weights[i] ** pValue) * ((array - np.nanmax(array)) ** pValue)
            # calculate sNeg for active array and add result to old sNeg
            sNeg += (weights[i] ** pValue) * ((array - np.nanmin(array)) ** pValue)
        # Summ of all sPlus ^(1/p)
        sPlus = (sPlus ** (1/pValue))
        # Summ of all sNeg ^(1/p)
        sNeg = (sNeg ** (1/pValue))
        # calculate TOPSIS result
        outputArray = sNeg / (sPlus + sNeg)
        # set np.nan to Output No data Value
        outputArray[outputArray == np.nan] = self.outNoData
        # Calculation TOPSIS_end---------------------------------------------------------------------
        
        # call function to convert Output Array to a raster file
        self.outArrayToRaster(outputArray)

#------------------------Updates OWA--------------------------------------------------

    def runOWA(self):
        #Rasters to NumPy Arrays, get weights and ordered weights
        rArrays = self.rastersToArray(self.criterionRasters)
        weights = self.getWeights()
        orderedWeights = self.getOrderedWeights()
        # calculate OWA with NumPy Arrays and weights
        outputArray = self.calculateOWA(rArrays, weights, orderedWeights)
        # set np.nan to Output No data Value
        outputArray[outputArray == np.nan] = self.outNoData
        #call function to convert Output Array to a raster file
        self.outArrayToRaster(outputArray)
        
    def calculateOWA(self, rArrays, weights, orderedWeights):
        # create output Array with same shape as input
        result = np.zeros(rArrays[0].shape)
        # calculate tradeoff
        n = len(orderedWeights)
        tradeoff = self.calculateTradeoff(orderedWeights, n)
        # new Factor Weights
        nFW =np.array( (1/n)+(weights-(1/n))*tradeoff)
        
        for row in range(rArrays[0].shape[0]):
            for column in range(rArrays[0].shape[1]):
                # initialize arrays
                factors = np.zeros(n)
                factor_nFW = np.zeros(n)
                sort_indizes = np.zeros(n)
                
                for index, array in enumerate(rArrays):
                    # F:
                    factors[index] = array[row,column]
                    # F*NFw:
                    factor_nFW[index] = (nFW[index]*array[row,column])
                
                #ranking (https://stackoverflow.com/questions/9007877/sort-arrays-rows-by-another-array-in-python)
                sort_indizes = np.argsort(factor_nFW) # in: [5, 2, 6] out: [1, 0, 2] (rank index)
                
                # R F*NFw:
                ranked_factor_nFW = factor_nFW[sort_indizes[::1]]
                # RNFw:
                ranked_nFW = nFW[sort_indizes[::1]]
                # RF:
                ranked_factors = factors[sort_indizes[::1]]
                
                # CW/SCW
                cW = ranked_nFW * orderedWeights
                summCW = np.sum(cW)
                sCW = cW/summCW
               
                # OWA result (like WLC: weight(sCW)*value(ranked_factors)
                owaResult = 0
                for index, r_factor in enumerate(ranked_factors):
                    owaResult += sCW[index]*r_factor
                result[row, column] = owaResult
        
        return result

    def calculateTradeoff(self, orderedWeights, n):
        # initialize array (for inner part of the sqrt)
        innerParts = np.zeros(n)
        
        for index, ow in enumerate(orderedWeights):
            innerParts[index] = (ow-(1/n)) ** 2
        
        # total part within sqrt
        withinSqrt = (n * np.sum(innerParts))/(n-1)
        
        return 1 - (withinSqrt ** (1/2))
    
    def getOrderedWeights(self):
        oWeights = []
        for col in range(0, self.dlg.tableWidget_owa_ow.columnCount()):
            # get weight value of column 2
            oWeights.append(float(QTableWidgetItem(self.dlg.tableWidget_owa_ow.item(0, col)).text()))
        
        oWeights = np.asarray(oWeights).astype(np.float32)
        #print(oWeights)
        #print(np.all(np.diff(oWeights) < 0))
        #print(np.all(np.diff(oWeights) > 0))
        
        if oWeights.sum(axis=0) != 1:
            old_oWeights = oWeights
            oWeights = oWeights/oWeights.sum(axis=0)
            self.dlg.showMessageBox(QMessageBox.Information, "Information" , "Information: Ordered weights not Valid", "Input ordered weights are getting Normalized", "Sum of Weights equals 1!\nChanged:\n"+str(old_oWeights)+"\nTO:\n"+str(oWeights))
        
        #if( (oWeights != np.sort(oWeights)).all() and (oWeights != np.sort(oWeights)[::-1]).all() ):
        if not ((np.all(np.diff(oWeights) < 0) ) or (np.all(np.diff(oWeights) > 0) )):
            self.dlg.showMessageBox(QMessageBox.Warning, "Exeption" , "Exeption: Ordered Weights not valid", "Ordered Weights must be strictly increasing or decreasing!", "Example:\n(0.1, 0.2, 0.7) -> OK\n(0.1, 0.7, 0.2) -> NOT OK")
            raise Exception("Ordered Weights not valid")
            
        return oWeights
    

#--------------------------------------------OWA end-----------------------------------------

    # Funktion: converts the input rasters to NumPy arrays
    def rastersToArray(self, targetArray):
        # empty list (will be filled with NumPy arrays) 
        rArrays = []
        
        #loop through every raster in self.criterionRasters
        for row, raster in enumerate(targetArray):
            # Raster to NumPy-----------------------------------------------------
            #open raster as gdal 
            gd = gdal.Open(str(raster.source()))
            # get raster band 1
            band = gd.GetRasterBand(1)
            
            # convert the band of the gdal raster to a NumPy array (datatype float 32)
            numpyArray = band.ReadAsArray().astype(np.float32)
            
            
            #Raster to NumPy_end-------------------------------------------------------
            
            # No Data Values handling-----------------------------------------------------
            # set the bands no dataValues to np.nan (important for calculation)
            # save the bands no dataValues to nd (datatype float 32)
            nd = np.float32(band.GetNoDataValue())
            # sets all values in array which are equal to nd to np.nan
            numpyArray[numpyArray == nd] = np.nan
            # No Data Values handling_end-----------------------------------------------------
            
            
            #check if binary for constraints-----------------------------
            if (targetArray == self.constraintRasters):
                checkBooleanArray = np.array([0,1,np.nan]).astype(np.float32)
                diff = np.setdiff1d(numpyArray, checkBooleanArray)
                if diff.size:
                    name = raster.name()
                    self.dlg.showMessageBox(QMessageBox.Warning, "Exeption" , "Exeption: Constraint layer not Boolean", "Please only add boolean rasters (0,1) as a constraint layer", "Layer: "+str(name))
                    raise Exception("Constraint layer not Boolean")
            
            
            # Standardize --------------------------------------------------------------
            # get sfactor (benefit (1) or cost (0) criterion)
            sfactor = int(QTableWidgetItem(self.activeTable.item(row, 1)).text())
            # if array range is not in [0,1]
            if (np.nanmax(numpyArray) > 1) or (np.nanmin(numpyArray) < 0):
                # standardize according to sfactor
                numpyArray = self.standardizeArray(numpyArray, sfactor)
            # if standardized cost criterion exists in IP table (not monontoly increasing)
            elif (sfactor == 0) and (self.activeTable == self.dlg.tableWidget_ip):
                # function will switch values (0->1, 1->0) we need  monontoly increasing layers for TOPSIS
                numpyArray = self.standardizeArray(numpyArray, 2)
            # Standardize_end --------------------------------------------------------------
            
            # add NumPy array to list rArrays
            rArrays.append(numpyArray)
        
        # function returns list of Numpy arrays
        return rArrays


    # Function: get Weights out of criterion Table
    def getWeights(self):
        # empty list (for weights)
        weights = []
        # loop through every row in criteriontable
        for row in range(0, self.activeTable.rowCount()):
            # get weight value of column 2
            weights.append(float(QTableWidgetItem(self.activeTable.item(row, 2)).text()))
        
        # convert list to NumPy array (important for calculations)
        weights = np.asarray(weights).astype(np.float32)
        
        # if sum of weights is not 1
        if not self.checkWeights():
            old_weights = weights
            # normalize weights
            weights = weights/weights.sum(axis=0)
            if not self.weightsLoaded:
                # user feedback
                self.dlg.showMessageBox(QMessageBox.Information, "Information" , "Information: Weights not Valid", "Input Weights are getting Normalized", "Sum of Weights equals 1!\nChanged:\n"+str(old_weights)+"\nTO:\n"+str(weights))
        # function returns a NumPy array of weights
        return weights
    
    # Function: To Standartize a array according to the sfactor (1=benefit, 0=cost, 2=manipulate array (switch values)
    def standardizeArray(self, array, sfactor):
        # standardize the Array according to the sfactor (benefit / cost)
        # benefit
        if sfactor == 1: 
            array = ((array - np.nanmin(array)) / (np.nanmax(array) - np.nanmin(array)))
        # standartized cost but IP --> make monotonically increasing: switch values (0->1, 1->0)
        elif sfactor == 2: 
            array = (array * (-1)) + 1
        # cost
        else: 
            array = (np.nanmax(array) - array) / (np.nanmax(array) - np.nanmin(array))
        
        # function returns standardized array
        return array


    # Function: converts output Array to a raster file
    def outArrayToRaster(self, outputArray):
        conRArrays = self.rastersToArray(self.constraintRasters)
        if (len(conRArrays) > 0):
            for con in conRArrays:
                con[con == 0] = np.nan
                outputArray = outputArray * con
        
        # Preperations -----------------------------------------------------------------------------
        # get output location of new raster
        outputRaster = self.dlg.lineEdit_output.text()
        # create gdal dataset
        # source: https://www.hatarilabs.com/ih-en/how-to-convert-a-matrix-in-a-geospatial-tiff-in-qgis-with-pyqgis-tutorial
        # Rows and columns of new raster
        numrows = len(outputArray)
        numcols = len(outputArray[0])
        # driver to create certain datatype
        driver = gdal.GetDriverByName("GTiff")
        # create new dataset with driver(output location, Rows, columns, number of bands, datatype)
        dataset = driver.Create(outputRaster, numcols, numrows, 1, gdal.GDT_Float32)
        # Preperations_end -----------------------------------------------------------------------------
        
        # Spatial Parameters ----------------------------------------------------------------------
        # get Spatial properties from first input raster
        e = self.criterionRasters[0].extent()
        # coordinates
        xmin = e.xMinimum()
        xmax = e.xMaximum()
        ymin = e.yMinimum()
        ymax = e.yMaximum()
        # cellsize
        xres = (xmax-xmin)/float(numcols)
        yres = (ymax-ymin)/float(numrows)
        # Spatial reference System
        lyrCRS = str(self.criterionRasters[0].crs().authid()).split(":")[1]
        # Parameters for transformation:
        # (coord X (up-left corner) , cellsize in x direction, angle (default=0), coord Y (up-left corner), angle (default=0), cellsize in y direction)
        geotransform = [xmin,xres,0,ymax,0,-yres]
        # execute transformation
        dataset.SetGeoTransform(geotransform)
        # set Coordinate System
        out_srs = osr.SpatialReference()
        out_srs.ImportFromEPSG(int(lyrCRS))
        dataset.SetProjection(out_srs.ExportToWkt())
        # Spatial Parameters_end -----------------------------------------------------------------
        
        # Write Values ----------------------------------------------------------------------
        # set no Data Values
        dataset.GetRasterBand(1).SetNoDataValue(self.outNoData)
        # Write values of output Array
        dataset.GetRasterBand(1).WriteArray(outputArray)
        # close dataset
        dataset = None
        # Write Values_end -------------------------------------------------------------------
        
        #--PLUGIN END----------PLUGIN END-----------PLUGIN END---------------
        #add raster to QGIS
        self.iface.addRasterLayer(outputRaster,"OUTPUT","gdal")
        # user feedback
        self.iface.messageBar().clearWidgets()
        self.iface.messageBar().pushMessage(
                "Success", "Output file written at " + outputRaster,
                level=Qgis.Success, duration=3)
        self.dlg.close()
        #--PLUGIN END----------PLUGIN END-----------PLUGIN END---------------


#-----------------------------------Plugin Logic Methods----------------------------------

    # Funktion to get all the Layer nodes from a QGIS Layer Tree to self.inputLayers
    def setInputLayers(self, group):
        # from root goe to every child in tree structure
        for child in group.children():
            # if child is a layer node
            if isinstance(child, QgsLayerTreeLayer):
                # and if the layer is a valid raster layer
                if (type(child.layer()) == QgsRasterLayer and child.layer().isValid()):
                    #and if is singleband:
                    if (child.layer().rasterType() == 0):
                        # add the layer to list self.inputlayers
                        self.inputlayers.append(child.layer())
            # else: (child is a group node)
            else:
                # call funktion again, new root is the active group node
                self.setInputLayers(child)


    #Function: opens a Qt File dialoge to set output location
    def select_output_file(self):
        filename, _filter = QFileDialog.getSaveFileName(
           self.dlg, "Select   output file ","",'*.tiff')
        self.dlg.lineEdit_output.setText(filename)


    # Function adds a selected criterion to the criteion table
    # Example: https://stackoverflow.com/questions/24044421/how-to-add-a-row-in-a-tablewidget-pyqt
    def addCriterion(self):
        # add a new row to the actual number of rows
        actualRowCount = self.activeTable.rowCount()
        self.activeTable.insertRow(actualRowCount)
        # get the selected Layer from the ComboBox_input (index in ComboBox = index in self.inputlayers)
        selectedLayerIndex = self.dlg.comboBox_input.currentIndex()
        selectedLayer = self.inputlayers[selectedLayerIndex]
        # add layername and benefit/cost (default = 1) to new table row
        self.activeTable.setItem(actualRowCount, 0, QTableWidgetItem(selectedLayer.name()))
        self.activeTable.setItem(actualRowCount, 1, QTableWidgetItem("1")) # 1 for benefit criterion
        # delete element from combobox and possible inputlayers (self.inputlayers)
        self.dlg.comboBox_input.removeItem(selectedLayerIndex)
        del self.inputlayers[selectedLayerIndex]
        # calculating suitable weights automatically after adding new inputlayer
        self.autoCalcWeights(actualRowCount)  
        # if IP --> autofill IP (insert ideal/negative ideal point)
        if self.dlg.tabWidget_decision_rule.currentIndex() == 0:
            pass
        if self.dlg.tabWidget_decision_rule.currentIndex() == 1:
            self.autoFillIp(actualRowCount)
        if self.dlg.tabWidget_decision_rule.currentIndex() == 2:
            orderedWeitghtsCount = self.dlg.tableWidget_owa_ow.columnCount()
            self.dlg.tableWidget_owa_ow.insertColumn(orderedWeitghtsCount)
            self.dlg.tableWidget_owa_ow.horizontalHeader().hide();


    # Function removes selected criterioa from the criteion table
    def removeCriteria(self):
        # get selected row indices
        indices = self.activeTable.selectionModel().selectedRows()
        # empty list for the removed layers
        removedRasterLayers = []
        
        # loop through every selected index
        for index in sorted(indices):
            i = index.row()
            #save ramoved layername to get the removed raster layer from QGIS project by its name
            removedLayerName = QTableWidgetItem(self.activeTable.item(i, 0)).text()
            removedRasterLayers.append(QgsProject.instance().mapLayersByName(removedLayerName)[0])
            # remove row from criterion table
            self.activeTable.removeRow(index.row())
        
        #add removed elements back to comboBox and to self.inputlayers
        self.dlg.comboBox_input.addItems([layer.name() for layer in removedRasterLayers])
        for removedItem in removedRasterLayers:
            self.inputlayers.append(removedItem)
        
        #autocalc weights again (if min 1 element in table)
        if self.activeTable.rowCount() > 0:
            self.autoCalcWeights(int(self.activeTable.rowCount())-1)

    def addConstraint(self):
        # add a new row to the actual number of rows
        actualRowCount = self.dlg.tableWidget_con.rowCount()
        self.dlg.tableWidget_con.insertRow(actualRowCount)
        # get the selected Layer from the ComboBox_input (index in ComboBox = index in self.inputlayers)
        selectedLayerIndex = self.dlg.comboBox_input.currentIndex()
        selectedLayer = self.inputlayers[selectedLayerIndex]
        
        # add layername and benefit/cost (default = 1) to new table row
        self.dlg.tableWidget_con.setItem(actualRowCount, 0, QTableWidgetItem(selectedLayer.name()))
        # delete element from combobox and possible inputlayers (self.inputlayers)
        self.dlg.comboBox_input.removeItem(selectedLayerIndex)
        del self.inputlayers[selectedLayerIndex]
    
    def removeConstraint(self):
        # get selected row indices
        indices = self.dlg.tableWidget_con.selectionModel().selectedRows()
        # empty list for the removed layers
        removedRasterLayers = []
        
        # loop through every selected index
        for index in sorted(indices):
            i = index.row()
            #save ramoved layername to get the removed raster layer from QGIS project by its name
            removedLayerName = QTableWidgetItem(self.dlg.tableWidget_con.item(i, 0)).text()
            removedRasterLayers.append(QgsProject.instance().mapLayersByName(removedLayerName)[0])
            # remove row from criterion table
            self.dlg.tableWidget_con.removeRow(index.row())
        
        #add removed elements back to comboBox and to self.inputlayers
        self.dlg.comboBox_input.addItems([layer.name() for layer in removedRasterLayers])
        for removedItem in removedRasterLayers:
            self.inputlayers.append(removedItem)
    
    # Function: calculating suitable weights automatically after adding new inputlayer
    def autoCalcWeights(self, actualRowCount):
        # calculate equal weigths
        autoWeight = 1/(actualRowCount+1)
        # loop through every row
        for row in range(0, actualRowCount+1):
            # put equal weigths in column 2
            self.activeTable.setItem(row, 2, QTableWidgetItem(str(autoWeight)))
        
        # Function checks if sum of weights = 1
        self.checkWeights()


    # Function: gets the min and max value of the raster layer and fills it into the IP table
    # How to work with rasters example: https://docs.qgis.org/testing/en/docs/pyqgis_developer_cookbook/raster.html
    def autoFillIp(self, actualRowCount):
        # get layername form column 0
        rastername = QTableWidgetItem(self.activeTable.item(actualRowCount, 0)).text()
        # get raster from QGIS Project
        rlayer = QgsProject.instance().mapLayersByName(rastername)[0]
        # calculate band statistics
        stats = rlayer.dataProvider().bandStatistics(1)
        rmin = stats.minimumValue
        rmax = stats.maximumValue
        # put max and min value to column 3 and 4
        self.activeTable.setItem(actualRowCount, 3, QTableWidgetItem(str(rmax)))
        self.activeTable.setItem(actualRowCount, 4, QTableWidgetItem(str(rmin)))


    # Function: changes active criterion table when user changes criterion table Tab
    def setActiveTable(self):
        # get index of Tab
        currentDecisionRule = self.dlg.tabWidget_decision_rule.currentIndex()
        # if active Tab = WLC --> set self.active table to WLC
        if currentDecisionRule == 0:
            self.activeTable = self.dlg.tableWidget_wlc
        # if active Tab = IP --> set self.active table to IP
        elif currentDecisionRule == 1:
            self.activeTable = self.dlg.tableWidget_ip
        elif currentDecisionRule == 2:
            self.activeTable = self.dlg.tableWidget_owa
    
#-------------------------------------------VALIDATION BLOCK-----------------------------------------------------
    
    # Function: checks if sum of weights = 1
    def checkWeights(self):
        # "empty" sum for calculation
        sum = 0
        # loop through everyy row in criteriontable
        for row in range(0, self.activeTable.rowCount()):
            # get the weight from column 2
            rowWeight = QTableWidgetItem(self.activeTable.item(row, 2))
            # add weight to old sum
            sum = sum + float(rowWeight.text())
        
        # Put sum into textBrowser_checkWeights
        self.dlg.textBrowser_checkWeights.setText(str(sum))
        # if weights normalized
        if sum == 1:
            # texboxbackground --> lightgreen
            self.dlg.textBrowser_checkWeights.setStyleSheet("background-color: lightgreen;")
            # Function returns boolean
            return True
        else:
            # texboxbackground --> red
            self.dlg.textBrowser_checkWeights.setStyleSheet("background-color: red;")
            # Function returns boolean
            return False


    # Function: checks if number of possible inputlayers (from QGIS project) is min 2
    def checkInputLayers(self):
        # if not --> user feedback
        if len(self.inputlayers) < 2:
            self.dlg.showMessageBox(QMessageBox.Information, "Information" , "Not Enough valid Rasters in QGIS Project", "Please add at least two valid raster Files to QGIS Map Layers", "The Plugin checked for at least two valid raster layers.")

    
    # Function: checks properties o different criteria layers
    # min 2 criteria, same Spatial reference System, same cell size, same origin and extend
    def checkCriteriaLayerProperties(self):
        # 0. number of criteria min 2
        if len(self.criterionRasters) < 2:
            self.dlg.showMessageBox(QMessageBox.Warning, "Exeption" , "Exeption: Not Enough criterion Layers", "Please add at least 2 criteria", "Otherwise it is not a MULTI criteria decision problem")
            raise Exception("Not Enough criterion Layers")
        
        # Get all relevant parameters of first criterion raster
        # 1. Spatial Reference System
        lyrCRS = str(self.criterionRasters[0].crs().authid()).split(":")[1]
        # 2. cell size
        pixelSizeX = self.criterionRasters[0].rasterUnitsPerPixelX()
        pixelSizeY = self.criterionRasters[0].rasterUnitsPerPixelY()
        # 3. origin and number of columns/rows
        e = self.criterionRasters[0].extent()
        # get Origin out of extend, round to get a tolerance
        # in testphase we had rasters, which could be calculated with raster calculator, but not with the Plugin because of differenzes in high decimals. so we decided to use a tolerance
        orXmax = e.xMaximum()
        orYmin = e.yMinimum()
        orXmax = np.round(orXmax, 6)
        orYmin = np.round(orYmin, 6)
        #get number of columns and rows
        orColNr = self.criterionRasters[0].width()
        orRowNr = self.criterionRasters[0].height()
        # set controll variable (different_Properties) False
        different_Properties = False
        
        #loop through every other raster and compare parameters
        for raster in self.criterionRasters[1:]:
            # get relevant parameters form raster
            re = raster.extent()
            xmax = re.xMaximum()
            ymin = re.yMinimum()
            xmax = np.round(xmax, 6)
            ymin = np.round(ymin, 6)
            colNr = raster.width()
            rowNr = raster.height()
            # check Spatial reference System
            if (str(raster.crs().authid()).split(":")[1] != lyrCRS):
                different_Properties = True
                detailedinfo = "CRITERION: Different Coordinate Systems"
            # check Cellsize
            elif (raster.rasterUnitsPerPixelX() != pixelSizeX) or (raster.rasterUnitsPerPixelY() != pixelSizeY):
                different_Properties = True
                detailedinfo = "CRITERION: Different Cellsizes"
            # check Raster Origins and Extend
            elif not ( (np.isclose(xmax, orXmax)) and (np.isclose(ymin, orYmin)) ) or (orColNr != colNr) or (orRowNr != rowNr):
                different_Properties = True
                detailedinfo = "CRITERION: Different Raster Origins or Extends"
            else:
                pass
        
        #loop through every other raster and compare parameters
        if self.constraintRasters:
            for raster in self.constraintRasters:
                # get relevant parameters form raster
                re = raster.extent()
                xmax = re.xMaximum()
                ymin = re.yMinimum()
                xmax = np.round(xmax, 6)
                ymin = np.round(ymin, 6)
                colNr = raster.width()
                rowNr = raster.height()
                # check Spatial reference System
                if (str(raster.crs().authid()).split(":")[1] != lyrCRS):
                    different_Properties = True
                    detailedinfo = "CONSTRAINT: Different Coordinate Systems"
                # check Cellsize
                elif (raster.rasterUnitsPerPixelX() != pixelSizeX) or (raster.rasterUnitsPerPixelY() != pixelSizeY):
                    different_Properties = True
                    detailedinfo = "CONSTRAINT: Different Cellsizes"
                # check Raster Origins and Extend
                elif not ( (np.isclose(xmax, orXmax)) and (np.isclose(ymin, orYmin)) ) or (orColNr != colNr) or (orRowNr != rowNr):
                    different_Properties = True
                    detailedinfo = "CONSTRAINT: Different Raster Origins or Extends"
                else:
                    pass
        
        # if controll variable (different_Properties) = True show user feedback. calculation not possible
        if different_Properties:
            self.dlg.showMessageBox(QMessageBox.Warning, "Exeption" , "Exeption: Different raster properties", "The relevant raster properties are different.", detailedinfo)
            raise Exception("Different raster properties: "+detailedinfo)

    def exit_mcda(self):
        self.dlg.close()
#----------------------------MAIN UPDATES Pairwise Comparision-----------------------------------------------------------------

    def loadWeights(self):
        try:
            filename, _filter = QFileDialog.getOpenFileName(
               self.dlg, "Open pairwise comparision output ","",'*.wpc')
            wpcFile = open(filename, "r")
            weights = []
            
            for line in wpcFile:
                lineFound = False
                factorname = line.split(":")[0]
                weight = line.split(":")[1]
                if factorname == "Consistency":
                    self.weightsLoaded = True
                    self.dlg.showMessageBox(QMessageBox.Information, "Information" , "Weights loaded!", "Consistency ratio of weighting was: "+str(weight), "")
                    self.activeTable.setEnabled(False)
                    break
                
                for row in range(0, self.activeTable.rowCount()):
                    rowName = QTableWidgetItem(self.activeTable.item(row, 0)).text()
                    #print(rowName)
                    if rowName == factorname:
                        lineFound = True
                        self.activeTable.setItem(row, 2, QTableWidgetItem(weight))
                        break
                if lineFound == False:
                    raise ValueError("Different criteria names in weighting process and actual criterion table")
        except:
            self.dlg.showMessageBox(QMessageBox.Warning, "Exeption" , "Exeption: Loading weights not possible", "Something went wrong while loading the weights from pairwise comparision", "Caused by:\nInvalid wpc file\nOR\nDifferent criteria names in weighting process and actual criterion table")
        
    
#------------------------------------ Weighthing Pairwise Comp--------------------------------------------------------------
    def runWeightingPC(self):
        self.dlgW = RastermcdaWeightingPCDialog()
        self.dlgW.pushButton_addFactor.clicked.connect(self.addCriterion_WPC)
        self.dlgW.pushButton_removeFactor.clicked.connect(self.removeCriterion_WPC)
        self.dlgW.pushButton_output.clicked.connect(self.select_output_file_WPC)
        self.dlgW.pushButton_ok.clicked.connect(self.calculate_WPC)
        self.dlgW.pushButton_cancel.clicked.connect(self.exit_WPC)
        
        #VALIDATION_1----------------------------------------------------------------------
        # check for min 2 valid rasterlayer and add only valid raster Layers to plugin
        # Clear the contents of inputlayers from previous runs
        self.inputlayers.clear()
        # Funktion to get all the Layer nodes from a QGIS Layer Tree to self.inputLayers
        self.setInputLayers(QgsProject.instance().layerTreeRoot())
        # Funktion checks if #inputLayers min 2
        self.checkInputLayers() 
        #VALIDATION_1_end--------------------------------------------------------------------
        
        # Put al elements of self.inputLayers to comboBox--------------------------------------
        # Clear the contents of the comboBox from previous runs
        self.dlgW.comboBox_inputWeighting.clear()
        # Populate the comboBox with names of all the loaded layers
        self.dlgW.comboBox_inputWeighting.addItems([layer.name() for layer in self.inputlayers])
        
        self.dlgW.show()
    
    def calculate_WPC(self):
        if self.dlgW.tableWidget_pairwCompMatrix.rowCount() > 1:
            # if outputPath is empty or not a valid path call funktion self.select_output_file() -> QtFileDialog 
            if (len(self.dlgW.lineEdit_output.text()) == 0) or not os.path.exists(self.dlgW.lineEdit_output.text()): 
                self.select_output_file_WPC()
            initPairwCompMatrix = self.getInitPairwCompMatrix()
            pairwCompMatrix = self.getPairwiseCompMatrix(initPairwCompMatrix)
            eigenvector = self.calculateEigenvector(pairwCompMatrix, np.zeros(pairwCompMatrix[0].shape))
            consistency = self.calculateCosistencyRatio(pairwCompMatrix, eigenvector)
            messageText = ""
            outputText = ""
            for index in range(eigenvector.shape[0]):
                factorname =  self.dlgW.tableWidget_pairwCompMatrix.horizontalHeaderItem(index).text()
                messageText += ""+str(factorname)+": \t"+str(eigenvector[index])+"\n"
                outputText += ""+str(factorname)+":"+str(eigenvector[index])+"\n"
            outputText += "Consistency:"+str(consistency)
            
            outputTextFile = self.dlgW.lineEdit_output.text()
            text_file = open(outputTextFile, "w")
            text_file.write(outputText)
            text_file.close()
            
            if consistency <= 0.1:
                RastermcdaDialog().showMessageBox(QMessageBox.Information, "Result Weighting" , "Result Weights: ", messageText, "Consistency Ratio: "+str(consistency))
                self.iface.messageBar().clearWidgets()
                self.iface.messageBar().pushMessage(
                        "Success", "Output file written at " + outputTextFile,
                        level=Qgis.Success, duration=3)
                self.dlgW.close()
            else: 
                RastermcdaDialog().showMessageBox(QMessageBox.Warning, "CONSISTENCY TO HIGH", "Consitency Ratio: "+str(consistency),"\nConsitency Ratio is to high due to inconsistency within Pairwise Comparision Matrix. Please check input and calculate again.", "Result was saved nevertheless.\nWeights: \n"+str( messageText))
        else:
            RastermcdaDialog().showMessageBox(QMessageBox.Warning, "Exeption" , "Exeption: Not Enough criterion Layers", "Please add at least 2 criteria", "Otherwise pairwise Comparision makes no sense.\nWeight = 1")
    
    def getInitPairwCompMatrix(self):
        actualRowCount = self.dlgW.tableWidget_pairwCompMatrix.rowCount()
        initPairwCompMatrix = np.zeros((actualRowCount, actualRowCount)).astype(np.float32)
        
        for row in range(initPairwCompMatrix.shape[0]):
            for col in range(initPairwCompMatrix.shape[1]):
                if row == col:
                    initPairwCompMatrix[row,col] = 1
                elif row < col:
                    initPairwCompMatrix[row,col] = 0
                else:
                    initPairwCompMatrix[row,col] = Fraction(QTableWidgetItem(self.dlgW.tableWidget_pairwCompMatrix.item(row, col)).text())
        
        self.checkInput_WPC(initPairwCompMatrix)
        return initPairwCompMatrix
    
    def getPairwiseCompMatrix(self, origPairwCompMatrix):
        pairwCompMatrix = np.zeros(origPairwCompMatrix.shape).astype(np.float32)
        
        for row in range(origPairwCompMatrix.shape[0]):
            for column in range(origPairwCompMatrix.shape[1]):
                if origPairwCompMatrix[row, column] == 0:
                    pairwCompMatrix[row, column] = 1/origPairwCompMatrix[column, row]
                else:
                    pairwCompMatrix[row, column] = origPairwCompMatrix[row, column]
        
        return pairwCompMatrix
    
    def calculateEigenvector(self, pairwCompMatrix, initEigenvector):
        #square Matrix
        squareMatrix = np.matmul(pairwCompMatrix, pairwCompMatrix).astype(np.float32)
        rowsumms = squareMatrix.sum(axis=1)
        # for some reason we need to "double normalize" otherwise vector was not normalized correct
        eigenvector_nearly_normalized = (rowsumms/rowsumms.sum())
        eigenvector_normalized = (eigenvector_nearly_normalized/eigenvector_nearly_normalized.sum())
        
        #self.numSameDigits_WPC = 10000 
        # Multiply eigenvector with 10000 and transfer it to INT to compare the old eigenvector (initEigenvector) with the new one 
        # check 5 digit numbers
        check1 = ((eigenvector_normalized*self.numSameDigits_WPC).astype(np.int32))
        check2 = ((initEigenvector*self.numSameDigits_WPC).astype(np.int32))
        
        #if eigenvector has not changed, return; else repeat with squared Matrix (recursuion) 
        if np.array_equal(check1,check2):
            return eigenvector_normalized
        else:
            return self.calculateEigenvector(squareMatrix, eigenvector_normalized)
    
    def calculateCosistencyRatio(self, pairwCompMatrix, eigenvector):
        # Source rancomIndex: http://rad.ihu.edu.gr/fileadmin/labsfiles/decision_support_systems/lessons/ahp/AHP_Lesson_1.pdf
        randomIndex=np.array([0.00, 0.00, 0.58, 0.90, 1.12, 1.24, 1.32, 1.41, 1.45, 1.49, 1.51, 1.48, 1.56, 1.57, 1.59]).astype(np.float32)
        # initialize Matrix
        consistencyMatrix = np.zeros(pairwCompMatrix.shape).astype(np.float32)
        #Multiply every column with according weight in eigenvector
        for index, elem in enumerate(eigenvector):
            consistencyMatrix[:,index] = pairwCompMatrix[:,index]*eigenvector[index]
        #calculate rwsums
        rowsumms = consistencyMatrix.sum(axis=1).astype(np.float32)
        #consistency vector and average
        consVector = (rowsumms / eigenvector).astype(np.float32)
        consVectorAverage = np.average(consVector).astype(np.float32)
        #consistency Index and ratio
        consIndex = (consVectorAverage-(len(eigenvector)))/(len(eigenvector)-1)
        consRatio = consIndex/randomIndex[len(eigenvector)-1]
        
        return consRatio
    
    def addCriterion_WPC(self):
        actualRowCount = self.dlgW.tableWidget_pairwCompMatrix.rowCount()
        selectedLayerIndex = self.dlgW.comboBox_inputWeighting.currentIndex()
        selectedLayer = self.inputlayers[selectedLayerIndex]
        self.dlgW.tableWidget_pairwCompMatrix.insertRow(actualRowCount)
        self.dlgW.tableWidget_pairwCompMatrix.insertColumn(actualRowCount)
        
        self.dlgW.tableWidget_pairwCompMatrix.setHorizontalHeaderItem(actualRowCount, QTableWidgetItem(selectedLayer.name()))
        self.dlgW.tableWidget_pairwCompMatrix.setVerticalHeaderItem(actualRowCount, QTableWidgetItem(selectedLayer.name()))
        
        # delete element from combobox and possible inputlayers (self.inputlayers)
        self.dlgW.comboBox_inputWeighting.removeItem(selectedLayerIndex)
        del self.inputlayers[selectedLayerIndex]
        
        self.dlgW.tableWidget_pairwCompMatrix.setItem(actualRowCount, actualRowCount, QTableWidgetItem("1"))
    
    def removeCriterion_WPC(self):
        actualRowCount = self.dlgW.tableWidget_pairwCompMatrix.rowCount()
        remLayerName =  self.dlgW.tableWidget_pairwCompMatrix.horizontalHeaderItem(actualRowCount-1).text()
        self.dlgW.tableWidget_pairwCompMatrix.removeRow(actualRowCount-1)
        self.dlgW.tableWidget_pairwCompMatrix.removeColumn(actualRowCount-1)
        
        #add removed elements back to comboBox and to self.inputlayers
        remLayer = QgsProject.instance().mapLayersByName(remLayerName)[0]
        self.dlgW.comboBox_inputWeighting.addItem(remLayer.name())
        self.inputlayers.append(remLayer)
    
    def select_output_file_WPC(self):
        filename, _filter = QFileDialog.getSaveFileName(
           self.dlgW, "Select   output file ","",'*.wpc')
        self.dlgW.lineEdit_output.setText(filename)

    
    def checkInput_WPC(self, initPairwCompMatrix):
        if np.amax(initPairwCompMatrix) > 7 or np.amin(initPairwCompMatrix) < 0:
            RastermcdaDialog().showMessageBox(QMessageBox.Warning, "Exeption" , "Exeption: Invalid Input", "Only use values within the scale of Saaty", "Calculation of weights not possible")
            raise Exception("Not Enough criterion Layers")
    
    def exit_WPC(self):
        self.dlgW.close()
    
