# -*- coding: utf-8 -*-
"""
/***************************************************************************
 RasterDivider
                                 A QGIS plugin
 This plugin divides raster into equal sized parts.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2022-02-10
        git sha              : $Format:%H$
        copyright            : (C) 2022 by Murat ÇALIŞKAN
        email                : caliskan.murat.20@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QFileDialog, QMessageBox
from qgis.utils import iface

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .raster_divider_dialog import RasterDividerDialog
import os.path
from qgis.core import QgsProject, QgsVectorLayer, Qgis

import os
import numpy as np
import pandas as pd
from osgeo import gdal, ogr
from datetime import date

class RasterDivider:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'RasterDivider_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Raster Divider')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('RasterDivider', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToRasterMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/raster_divider/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'Raster Divider'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginRasterMenu(
                self.tr(u'&Raster Divider'),
                action)
            self.iface.removeToolBarIcon(action)
            
    def selectInputFile(self):
        sender = self.dlg.sender()
        oname = sender.objectName()
        
        if oname == "tb_browse_2":
            self.dlg.le_footprint_path.setText("")
            gpkgPath, _filter = QFileDialog.getOpenFileName(self.dlg, "Select footprint file", "", 'Geopackage (*.gpkg)')
            self.dlg.le_footprint_path.setText(gpkgPath)
            
    def selectOutputFile(self):
        self.dlg.le_outputFile.setText("")
        self.gpkgPath, self._filter = QFileDialog.getSaveFileName(self.dlg, "Select output file", "", 'Geopackage (*.gpkg)')
        self.dlg.le_outputFile.setText(self.gpkgPath)
    
    def selectOutputFolder(self):
        self.dlg.le_outputFolder.setText("")
        output_dir = QFileDialog.getExistingDirectory(None, 'Select working directory', "", QFileDialog.ShowDirsOnly)
        self.dlg.le_outputFolder.setText(output_dir)
    
    def fillCompressWidget(self):
        self.dlg.cb_compressions.clear()
        if self.dlg.rb_numpy.isChecked():
            self.dlg.cb_compressions.addItems(["NONE", "npz"])
            
        elif self.dlg.rb_vrt.isChecked():
            self.dlg.cb_compressions.addItem("NONE")
            
        elif self.dlg.rb_gtiff.isChecked():
            compression_list = ["DEFLATE", "NONE", "LZW", "JPEG", "PACKBITS", "CCITTRLE", "CCITTFAX3", "CCITTFAX4", "LZMA", "ZSTD", "LERC", 
                                "LERC_DEFLATE", "LERC_ZSTD", "WEBP", "JXL"]
            self.dlg.cb_compressions.addItems(compression_list)
    
    def getRasterSize(self):
        try:
            rasterName = self.dlg.cb_in_raster.currentText()
            raster_path = self.raster_layers[rasterName].source()
            
            raster = gdal.Open(raster_path)        
            width = raster.RasterXSize
            height = raster.RasterYSize
            band_count = raster.RasterCount
            band_1_nodataval = raster.GetRasterBand(1).GetNoDataValue()
            
            nodataval = 0 if band_1_nodataval is None else band_1_nodataval
            
            self.dlg.sb_init_val.setValue(nodataval)
            
            self.dlg.sb_overlap_x.setMaximum(width-1)
            self.dlg.sb_overlap_y.setMaximum(height-1)
            
            self.dlg.sb_rasterWidth.setMaximum(width)
            self.dlg.sb_rasterHeight.setMaximum(height)  
            
            self.dlg.lbl_rasterWidth.setStyleSheet("color:black;")
            self.dlg.lbl_rasterHeight.setStyleSheet("color:black;")
            
            self.dlg.lbl_rasterWidth.setText(str(width))
            self.dlg.lbl_rasterHeight.setText(str(height))
            self.dlg.lbl_rasterBand.setText(str(band_count))
            
        except:
            self.dlg.lbl_rasterWidth.setStyleSheet("color:red; font-weight:bold;")
            self.dlg.lbl_rasterHeight.setStyleSheet("color:red; font-weight:bold;")
            self.dlg.lbl_rasterBand.setStyleSheet("color:red; font-weight:bold;")
            
            self.dlg.lbl_rasterWidth.setText("Invalid Raster!")
            self.dlg.lbl_rasterHeight.setText("Invalid Raster!")
            self.dlg.lbl_rasterBand.setText("Invalid Raster!")
            
            self.dlg.pushButton.setEnabled(False)
    
    def createGPKG(self, data, srs, out_gpkg):
        lyr_name = os.path.basename(out_gpkg).split(".")[0]
        
        driver = ogr.GetDriverByName('GPKG')
        ds = driver.CreateDataSource(out_gpkg)        
        layer = ds.CreateLayer(lyr_name, srs, ogr.wkbPolygon)
        
        for name, dtype in data.dtypes.items():
            if str(dtype) == "object":
                layer.CreateField(ogr.FieldDefn(name, ogr.OFTString))
            elif str(dtype).startswith("int"):
                layer.CreateField(ogr.FieldDefn(name, ogr.OFTInteger))
            elif str(dtype).startswith("float"):
                layer.CreateField(ogr.FieldDefn(name, ogr.OFTReal))
                
        defn = layer.GetLayerDefn()
    
        for _, row in data.iterrows():
            feat = ogr.Feature(defn)
            
            for col, val in row.items():
                feat.SetField(col, val)
            
            minx = row["minx"]
            miny = row["miny"]
            maxx = row["maxx"]
            maxy = row["maxy"]
            
            wkt = f"POLYGON (({minx} {maxy}, {maxx} {maxy}, {maxx} {miny}, {minx} {miny}, {minx} {maxy}))"
            geom = ogr.CreateGeometryFromWkt(wkt)
        
            # Geometry
            feat.SetGeometry(geom)
            layer.CreateFeature(feat)
        
        ds = layer = defn = feat = None
        
        if self.dlg.chb_addToMap.isChecked():
            try:
                layer = QgsVectorLayer(out_gpkg, lyr_name, "ogr")
                QgsProject.instance().addMapLayer(layer)
            except:
                pass

    def getPatches(self, raster_path, geot, xSize, ySize, split_x_size, split_y_size, min_overlap_x=0, min_overlap_y=0, auto_overlap=True):
        start_x_arr = np.array([], dtype=np.int32)
        start_val_x = 0
        
        while True:
            start_x_arr = np.append(start_x_arr, start_val_x)
            start_val_x += split_x_size - min_overlap_x
            if (start_x_arr[-1] + split_x_size) >= xSize:
                break
        
        start_y_arr = np.array([], dtype=np.int32)
        start_val_y = 0
        while True:
            start_y_arr = np.append(start_y_arr, start_val_y)
            start_val_y += split_y_size - min_overlap_y
            if (start_y_arr[-1] + split_y_size) >= ySize:
                break
        
        end_x_arr = start_x_arr + split_x_size
        end_y_arr = start_y_arr + split_y_size
        
        if auto_overlap:
            overlap_x = np.zeros_like(start_x_arr)
            num_of_patch_x = overlap_x.shape[0]
            total_res_x = int(end_x_arr[-1] - xSize)
            
            overlap_y = np.zeros_like(start_y_arr)
            num_of_patch_y = overlap_y.shape[0]
            total_res_y = int(end_y_arr[-1] - ySize)
            
            if total_res_x > 0:
                res_patch_x, res_x = divmod(total_res_x, num_of_patch_x-1)        
                overlap_x[1:res_x+1] = 1
                overlap_x[1:] = overlap_x[1:] + res_patch_x
                
            shift_x = overlap_x.cumsum()
            start_x_arr = start_x_arr - shift_x
            
            end_x_arr = start_x_arr + split_x_size
            
            if total_res_y > 0:
                res_patch_y, res_y = divmod(total_res_y, num_of_patch_y-1)
                overlap_y[1:res_y+1] = 1
                overlap_y[1:] = overlap_y[1:] + res_patch_y
                
            shift_y = overlap_y.cumsum()
            start_y_arr = start_y_arr - shift_y
            
            end_y_arr = start_y_arr + split_y_size
            
            overlap_x[1:] = overlap_x[1:] + min_overlap_x
            overlap_y[1:] = overlap_y[1:] + min_overlap_y
            
        else:
            overlap_x = np.full_like(start_x_arr, min_overlap_x)
            overlap_y = np.full_like(start_y_arr, min_overlap_y)  
        
        overlap_x[0] = 0
        overlap_y[0] = 0
        
        col = np.hstack([start_x_arr.reshape(-1,1), end_x_arr.reshape(-1,1)])
        col_idx = np.arange(col.shape[0]).reshape(-1,1)
        col = np.hstack([col_idx, col, overlap_x.reshape(-1,1)])
        
        row = np.hstack([start_y_arr.reshape(-1,1), end_y_arr.reshape(-1,1)])
        row_idx = np.arange(row.shape[0]).reshape(-1,1)
        row = np.hstack([row_idx, row, overlap_y.reshape(-1,1)])
        
        
        cols = np.repeat(col, row.shape[0], axis=0)
        rows = np.tile(row, (col.shape[0], 1))
        
        grid = pd.DataFrame(np.hstack([cols, rows]), columns=["col_id", "start_col","end_col", "overlap_col", "row_id", "start_row","end_row", "overlap_row"])
        minx = geot[0] + grid["start_col"] * geot[1]
        maxx = minx + split_x_size * geot[1]
        maxy = geot[3] + grid["start_row"] * geot[-1]
        miny = maxy + split_y_size * geot[-1]
        
        file_name = os.path.basename(raster_path).split(".")[0]
        name = file_name + "_"  + grid["col_id"].astype(str) + "_" + grid["row_id"].astype(str)
        
        grid["name"] = name
        grid["width_px"] = split_x_size
        grid["height_px"] = split_y_size
        grid["minx"] = minx
        grid["miny"] = miny
        grid["maxx"] = maxx
        grid["maxy"] = maxy
        grid["original_raster_path"] = raster_path
        
        if self.dlg.rb_strict.isChecked():
            grid["init_val"] = self.dlg.sb_init_val.value()
            grid = grid.loc[:,["name", "col_id", "row_id", "width_px", "height_px", "overlap_col", "overlap_row", "start_col", "start_row", "minx", "miny", "maxx", "maxy", "original_raster_path", "init_val"]]
        else:
            grid = grid.loc[:,["name", "col_id", "row_id", "width_px", "height_px", "overlap_col", "overlap_row", "start_col", "start_row", "minx", "miny", "maxx", "maxy", "original_raster_path"]]
            
        return grid.rename(columns = {"start_col":"ul_col", "start_row":"ul_row"})
    
    def generate_footprints(self):
        if not self.dlg.le_outputFile.text().endswith(".gpkg"):
            QMessageBox.critical(None, "ERROR", """Invalid output file name""")
            return        
        
        self.dlg.setEnabled(False)
        self.dlg.pushButton.setText("RUNNING...")
        self.dlg.processEvents()
        

        split_x_size = self.dlg.sb_rasterWidth.value()
        split_y_size = self.dlg.sb_rasterHeight.value()
        min_overlap_x = self.dlg.sb_overlap_x.value()
        min_overlap_y = self.dlg.sb_overlap_y.value()
        auto_overlap = True if self.dlg.rb_auto.isChecked() else False
        
        rasterName = self.dlg.cb_in_raster.currentText()
        raster_path = self.raster_layers[rasterName].source()
        
        raster = gdal.Open(raster_path)
        xSize = raster.RasterXSize
        ySize = raster.RasterYSize
        geot = geot  = raster.GetGeoTransform()
        srs = raster.GetSpatialRef()

        out_file_path = self.dlg.le_outputFile.text()
        
        footprints = self.getPatches(raster_path, geot, xSize, ySize, split_x_size, split_y_size, min_overlap_x, min_overlap_y, auto_overlap)
        self.createGPKG(footprints, srs, out_file_path)
        
        self.iface.messageBar().pushMessage("Successful", "Footprints created!" , level=Qgis.Success, duration=5)
        
        self.dlg.setEnabled(True)
        self.dlg.pushButton.setText("RUN")
        self.dlg.processEvents()
    
    
    def splitAsTiff(self, idx):
        en, feat, compress, creationOptions, out_folder_path = self.param_list[idx]
        
        if en == 0:
            self.raster = gdal.Open(feat.GetField("original_raster_path"))
            self.xSize = self.raster.RasterXSize
            self.ySize = self.raster.RasterYSize
            self.nodataval = self.raster.GetRasterBand(1).GetNoDataValue()
            
        name = feat.GetField("name")
        minX = feat.GetField("minx")
        minY = feat.GetField("miny")
        maxX = feat.GetField("maxx")
        maxY = feat.GetField("maxy")
        try:
            init_val = feat.GetField("init_val")
        except:
            init_val = None
        
        newRasterSource = os.path.join(out_folder_path, name+".tif")
        
        if init_val:
            out_raster = gdal.Warp(newRasterSource,
                                   self.raster,
                                   srcNodata=self.nodataval,
                                   dstNodata = init_val,
                                   outputBounds = (minX, minY, maxX, maxY),
                                   format = "GTiff",
                                   creationOptions = creationOptions
            )
        else:
            out_raster = gdal.Warp(newRasterSource,
                                   self.raster,
                                   outputBounds = (minX, minY, maxX, maxY),
                                   format = "GTiff",
                                   creationOptions = creationOptions
            )
        
        out_raster = None
    
    def splitAsVRT(self, idx):
        en, feat, compress, creationOptions, out_folder_path = self.param_list[idx]
        
        if en == 0:
            self.raster = gdal.Open(feat.GetField("original_raster_path"))
            self.xSize = self.raster.RasterXSize
            self.ySize = self.raster.RasterYSize
            self.nodataval = self.raster.GetRasterBand(1).GetNoDataValue()
            
        name = feat.GetField("name")
        minX = feat.GetField("minx")
        minY = feat.GetField("miny")
        maxX = feat.GetField("maxx")
        maxY = feat.GetField("maxy")
        try:
            init_val = feat.GetField("init_val")
        except:
            init_val = None
            
        newRasterSource = os.path.join(out_folder_path, name+".vrt")
            
        out_raster = gdal.BuildVRT(newRasterSource,
                               self.raster,
                               srcNodata=self.nodataval,
                               VRTNodata = init_val,
                               outputBounds = (minX, minY, maxX, maxY),
                               creationOptions = creationOptions
        )
        
        out_raster = None
    
    def splitAsNumpy(self, idx):
        en, feat, compress, creationOptions, out_folder_path = self.param_list[idx]
        
        if en == 0:
            self.raster = gdal.Open(feat.GetField("original_raster_path"))
            self.xSize = self.raster.RasterXSize
            self.ySize = self.raster.RasterYSize
            self.band_count = self.raster.RasterCount
            
        name = feat.GetField("name")
        ul_col = feat.GetField("ul_col")
        ul_row = feat.GetField("ul_row")
        width_px = feat.GetField("width_px")
        height_px = feat.GetField("height_px")
        try:
            init_val = feat.GetField("init_val")
        except:
            init_val = None
        
        if (ul_col+width_px-1 > self.xSize) or (ul_row+height_px-1 > self.ySize):
            width_px_new = min(self.xSize - ul_col, width_px)
            height_px_new = min(self.ySize - ul_row, height_px)        
            arr_temp = self.raster.ReadAsArray(xoff=ul_col, yoff=ul_row, xsize=width_px_new, ysize=height_px_new)
            
            arr = np.full((self.band_count,height_px,width_px), init_val)
            arr[:self.band_count, :height_px_new, :width_px_new] = arr_temp
        else:
            arr = self.raster.ReadAsArray(xoff=ul_col, yoff=ul_row, xsize=width_px, ysize=height_px)
                    
        if compress == "npz":
            newRasterSource = os.path.join(out_folder_path, name+".npz")
            np.savez_compressed(newRasterSource, arr)
        else:
            newRasterSource = os.path.join(out_folder_path, name+".npy")
            np.save(newRasterSource, arr)
        
    
    def split_raster(self):
        if not os.path.isfile(self.dlg.le_footprint_path.text()):
            QMessageBox.critical(None, "ERROR", """Invalid footprint file path""")
            return
        
        if not os.path.isdir(self.dlg.le_outputFolder.text()):
            QMessageBox.critical(None, "ERROR", """Invalid folder path""")
            return
        
        self.dlg.setEnabled(False)
        self.dlg.pushButton.setText("RUNNING...")
        self.dlg.processEvents() 
        
        out_folder_path = self.dlg.le_outputFolder.text()
        footprint_path = self.dlg.le_footprint_path.text()        
        
        footprints = ogr.Open(footprint_path)
        layer = footprints.GetLayer()
        
        compress = None if self.dlg.cb_compressions.currentText() == "NONE" else self.dlg.cb_compressions.currentText()
        creationOptions = [f"COMPRESS={compress}", *[i.strip() for i in self.dlg.le_options.text().split(",") if not i.lower().startswith("compress")]]
                
        self.param_list = [(en, feat, compress, creationOptions, out_folder_path) for en, feat in enumerate(layer)]
        idx_list = range(len(self.param_list))
        feat_cnt = len(idx_list)
        
        if self.dlg.rb_gtiff.isChecked():
            cnt=1
            for idx in idx_list:
                self.splitAsTiff(idx)
                pb_val = int((cnt/feat_cnt)*100)
                self.dlg.progressBar.setValue(pb_val)
                cnt+=1                
        
        elif self.dlg.rb_numpy.isChecked():
            cnt=1
            for idx in idx_list:
                self.splitAsNumpy(idx)
                pb_val = int((cnt/feat_cnt)*100)
                self.dlg.progressBar.setValue(pb_val)
                cnt+=1
        
        elif self.dlg.rb_vrt.isChecked():
            cnt=1
            for idx in idx_list:
                self.splitAsVRT(idx)
                pb_val = int((cnt/feat_cnt)*100)
                self.dlg.progressBar.setValue(pb_val)
                cnt+=1
        
        self.raster = footprints = layer = None
        
        self.iface.messageBar().pushMessage("Successful", "Raster patchs created!" , level=Qgis.Success, duration=5)
        
        self.dlg.progressBar.setValue(0)
        self.dlg.setEnabled(True)
        self.dlg.pushButton.setText("RUN")
        self.dlg.processEvents() 
    
    
    def run(self):
        """Run method that performs all the real work"""
        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            # self.first_start = False
            self.dlg = RasterDividerDialog()
            
            self.dlg.lbl_info.setText("""<html><head/><body><a href="https://github.com/caliskanmurat/qgis_raster_divider_plugin"><img width="20" height="20" src=":/plugins/raster_divider/info.svg"/></a></body></html>""")
            
            if any([
                (date.today().day == 18 and date.today().month == 3),
                (date.today().day == 23 and date.today().month == 4),
                (date.today().day == 19 and date.today().month == 5),
                (date.today().day == 30 and date.today().month == 8),
                (date.today().day == 29 and date.today().month == 10),
                (date.today().day == 10 and date.today().month == 11)
                ]):
                
                self.dlg.setWindowIcon(QIcon(':/plugins/raster_divider/mka.png'))
            else:
                self.dlg.setWindowIcon(QIcon(':/plugins/raster_divider/icon.png'))
                
            self.out_folder_check = False
            self.create_outtype_check = False
            
            self.raster_layers = {}
            for layer in QgsProject.instance().mapLayers().values():
                if layer.type()== 1:
                    src = layer.source()
                    try:
                        _ = gdal.Open(src)
                        self.raster_layers[layer.name()] = layer
                    except:
                        pass
                    
            
            compression_list = ["DEFLATE", "NONE", "LZW", "JPEG", "PACKBITS", "CCITTRLE", "CCITTFAX3", "CCITTFAX4", "LZMA", "ZSTD", "LERC", 
                                "LERC_DEFLATE", "LERC_ZSTD", "WEBP", "JXL"]
            
            self.dlg.cb_compressions.addItems(compression_list)
            
            self.dlg.cb_in_raster.clear()
            self.dlg.cb_in_raster.addItems(self.raster_layers.keys())
            self.getRasterSize()
            
            self.dlg.cb_in_raster.currentTextChanged.connect(self.getRasterSize)
            
            self.dlg.tb_browse_2.clicked.connect(self.selectInputFile)
            self.dlg.tb_browse.clicked.connect(self.selectOutputFile)
            self.dlg.tb_browse_3.clicked.connect(self.selectOutputFolder)
            
            self.dlg.pushButton.clicked.connect(self.generate_footprints)
            self.dlg.pushButton_2.clicked.connect(self.split_raster)
            
            self.dlg.rb_numpy.toggled.connect(self.fillCompressWidget)
            self.dlg.rb_vrt.toggled.connect(self.fillCompressWidget)
            self.dlg.rb_gtiff.toggled.connect(self.fillCompressWidget)

        self.dlg.show()