# -*- coding: utf-8 -*-
"""
/***************************************************************************
 RasterDivider
                                 A QGIS plugin
 This plugin divides raster into equal sized parts.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2022-02-10
        git sha              : $Format:%H$
        copyright            : (C) 2022 by Murat ÇALIŞKAN
        email                : caliskan.murat.20@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QFileDialog, QMessageBox
from qgis.utils import iface

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .raster_divider_dialog import RasterDividerDialog
import os.path

import os
from osgeo import gdal, ogr, osr
from datetime import date
from glob import glob
from qgis.core import QgsProject, Qgis

class RasterDivider:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'RasterDivider_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Raster Divider')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('RasterDivider', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToRasterMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/raster_divider/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'Raster Divider'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginRasterMenu(
                self.tr(u'&Raster Divider'),
                action)
            self.iface.removeToolBarIcon(action)
            
        
    def check(self):        
        if os.path.isdir(self.dlg.le_outputFolder.text()):
            self.out_folder_check = True
        else:
            self.out_folder_check = False
        
        if any([self.dlg.chb_createOverview.isChecked(), self.dlg.chb_createChunk.isChecked()]):
            self.dlg.chb_addToMap.setEnabled(True)
            self.create_outtype_check = True
        else:
            self.dlg.chb_addToMap.setEnabled(False)
            self.create_outtype_check = False  
        
        if all((self.input_type_check, self.out_folder_check, self.create_outtype_check)):
            self.dlg.btn_run.setEnabled(True)
        else:
            self.dlg.btn_run.setEnabled(False)
    
    def selectOutputFolder(self):
        self.dlg.le_outputFolder.setText("")
        output_dir = QFileDialog.getExistingDirectory(None, 'Open working directory', "", QFileDialog.ShowDirsOnly)
        self.dlg.le_outputFolder.setText(output_dir)
                
    def checkFolderPath(self):
        if  os.path.isdir(self.dlg.le_outputFolder.text()):
            self.dlg.lbl_message.setStyleSheet("color:black;")
            self.dlg.lbl_message.setText("")
        else:
            self.dlg.lbl_message.setStyleSheet("color:red; font-weight:bold;")
            self.dlg.lbl_message.setText("Invalid output folder path!")
        
        self.check()
    
    def getRasterSize(self):
        try:
            rasterName = self.dlg.cb_in_raster.currentText()
            raster_path = self.raster_layers[rasterName].source()
            
            raster = gdal.Open(raster_path)        
            width = raster.RasterXSize
            height = raster.RasterYSize
            
            self.dlg.sb_overlap_x.setMaximum(width-1)
            self.dlg.sb_overlap_y.setMaximum(height-1)
            
            self.dlg.sb_rasterWidth.setMaximum(width)
            self.dlg.sb_rasterHeight.setMaximum(height)  
            
            self.dlg.lbl_rasterWidth.setStyleSheet("color:black;")
            self.dlg.lbl_rasterHeight.setStyleSheet("color:black;")
            
            self.dlg.lbl_rasterWidth.setText(str(width))
            self.dlg.lbl_rasterHeight.setText(str(height))
            
            self.input_type_check = True
            
        except:
            self.dlg.lbl_rasterWidth.setStyleSheet("color:red; font-weight:bold;")
            self.dlg.lbl_rasterHeight.setStyleSheet("color:red; font-weight:bold;")
            
            self.dlg.lbl_rasterWidth.setText("Invalid Raster!")
            self.dlg.lbl_rasterHeight.setText("Invalid Raster!")
            
            self.input_type_check = False
        
        self.check()
    
    def getFormat(self):
        format_ = self.formats.get(self.dlg.cb_format.currentText())
        self.dlg.cb_format_2.clear()
        self.dlg.cb_format_2.addItems(format_[1].split(" "))    
        
    def createGeom(self, minx, miny, maxx, maxy):
        ring = ogr.Geometry(ogr.wkbLinearRing)
        ring.AddPoint_2D(minx, maxy)
        ring.AddPoint_2D(maxx, maxy)
        ring.AddPoint_2D(maxx, miny)
        ring.AddPoint_2D(minx, miny)
        ring.AddPoint_2D(minx, maxy)
        
        geom = ogr.Geometry(ogr.wkbPolygon)
        geom.AddGeometry(ring)
        
        return geom
    
    def createShp(self, data, geoms, srs, out_folder):
        out_shp = os.path.join(out_folder,"footprints.shp")
        driver = ogr.GetDriverByName('Esri Shapefile')
        ds = driver.CreateDataSource(out_shp)        
        layer = ds.CreateLayer("footprints", srs, ogr.wkbPolygon)
        
        layer.CreateField(ogr.FieldDefn('name', ogr.OFTString))
        layer.CreateField(ogr.FieldDefn('fx', ogr.OFTInteger))
        layer.CreateField(ogr.FieldDefn('fy', ogr.OFTInteger))
        layer.CreateField(ogr.FieldDefn('width_px', ogr.OFTString))
        layer.CreateField(ogr.FieldDefn('height_px', ogr.OFTString))
        layer.CreateField(ogr.FieldDefn('overlap_x', ogr.OFTString))
        layer.CreateField(ogr.FieldDefn('overlap_y', ogr.OFTString))
        defn = layer.GetLayerDefn()
    
        for e,(i,g) in enumerate(zip(data,geoms)):
            feat = ogr.Feature(defn)
            feat.SetField('id', e)
            feat.SetField('name', i[0])
            feat.SetField('fx', i[1])
            feat.SetField('fy', i[2])
            feat.SetField('width_px', i[3])
            feat.SetField('height_px', i[4])
            feat.SetField('overlap_x', i[5])
            feat.SetField('overlap_y', i[6])
        
            # Geometry
            feat.SetGeometry(g)
            layer.CreateFeature(feat)
        
        ds = layer = defn = feat = None
    
    def execute(self):
        
        self.dlg.btn_run.setText("RUNNING...")
        self.dlg.processEvents()
        
        if not any([self.dlg.chb_createOverview.isChecked(), self.dlg.chb_createChunk.isChecked()]):
            QMessageBox.critical(None, "ERROR", "Select at least one output type!")
        else:            
            prev_start_x, prev_start_y = 0, 0
            data, geoms = [], []   
            
            split_size_X = int(self.dlg.sb_rasterWidth.value())
            split_size_Y = int(self.dlg.sb_rasterHeight.value())
            overlap_x = int(self.dlg.sb_overlap_x.value())
            overlap_y = int(self.dlg.sb_overlap_y.value())
            creation_options = [i.strip() for i in self.dlg.le_creation_options.text().split(",")]
            out_folder = self.dlg.le_outputFolder.text()
            
            currentFormat = self.formats.get(self.dlg.cb_format.currentText())[0]
            currentExt = self.dlg.cb_format_2.currentText()
            
            opt = {
               "format" : currentFormat,
               "creationOptions" : creation_options
            }
            
            rasterName = self.dlg.cb_in_raster.currentText()
            raster_path = self.raster_layers[rasterName].source()
            
            if not os.path.isdir(out_folder):
                QMessageBox.critical(None, "ERROR", "Invalid folder path!")
                return
            else:
                raster = gdal.Open(raster_path)
                crs = raster.GetSpatialRef()
                
                if not crs:
                    QMessageBox.critical(None, "ERROR", "Input raster does not have Coordinate Reference System!")
                    return
                    
                width = raster.RasterXSize
                height = raster.RasterYSize
                                
                band = raster.GetRasterBand(1)
                noDataValue = band.GetNoDataValue()
                g1, g2, g3, g4, g5, g6 = raster.GetGeoTransform()
                
                if (width < split_size_X) or (height < split_size_Y):
                    QMessageBox.critical(None, "ERROR", "Width of chunk size must be lower than {} and height of chunk size must be lower than {}.".format(self.width+1, self.height+1))
                    return
                
                if (overlap_x > split_size_X) or (overlap_y > split_size_Y):
                    QMessageBox.critical(None, "ERROR", "Overlap_X size must be lower than {} and Overlap_Y size must be lower than {}.".format(split_size_X, split_size_Y))
                    return
                
                number_of_frame_X = int(width / split_size_X) if width % split_size_X == 0 else (width // split_size_X) + 1
                total_residual_X = ((split_size_X - (width % split_size_X)) % split_size_X)
                residual_X = total_residual_X // (number_of_frame_X - 1) if number_of_frame_X > 1 else 0
                res_of_res_X = total_residual_X % (number_of_frame_X - 1) if number_of_frame_X > 1 else 0
                
                if (overlap_x > 0) and (overlap_x > residual_X) and (number_of_frame_X > 1):
                    nX = number_of_frame_X
                    tresX = nX * split_size_X - width
                    resX = overlap_x
                    
                    while tresX // (nX-1) < resX:
                        nX += 1        
                        tresX = nX * split_size_X - width
                        
                    resX = tresX // (nX - 1) if nX > 1 else 1
                    rresX = tresX % (nX - 1) if nX > 1 else 1
                    
                    residual_X = resX
                    number_of_frame_X = nX
                    res_of_res_X = rresX    

                number_of_frame_Y = int(height / split_size_Y) if height % split_size_Y == 0 else (height // split_size_Y) + 1
                total_residual_Y = ((split_size_Y - (height % split_size_Y)) % split_size_Y)
                residual_Y = total_residual_Y // (number_of_frame_Y - 1) if number_of_frame_Y > 1 else 0
                res_of_res_Y = total_residual_Y % (number_of_frame_Y - 1) if number_of_frame_Y > 1 else 0

                if (overlap_y > 0) and (overlap_y > residual_Y) and (number_of_frame_Y > 1):
                    nY = number_of_frame_Y
                    tresY = nY * split_size_Y - height
                    resY = overlap_y
                    
                    while tresY // (nY-1) < resY:
                        nY += 1        
                        tresY = nY * split_size_Y - height
                        
                    resY = tresY // (nY - 1) if nY > 1 else 1
                    rresY = tresY % (nY - 1) if nY > 1 else 1
                    
                    residual_Y = resY
                    number_of_frame_Y = nY
                    res_of_res_Y = rresY
                
                for fx in range(number_of_frame_X):
                    if fx == 0:
                        prev_start_x=0
                        x0 = 0
                    else:
                        prev_start_x=x0
                        if fx < res_of_res_X + 1:
                            x0 = xend - residual_X - 1
                        else:
                            x0 = xend - residual_X
                    
                    xend = x0 + split_size_X
                    for fy in range(number_of_frame_Y):
                        if fy == 0:
                            prev_start_y=0
                            y0 = 0
                        else:
                            prev_start_y=y0
                            if fy < res_of_res_Y + 1:
                                y0 = yend - residual_Y - 1
                            else:
                                y0 = yend - residual_Y
                        
                        yend = y0 + split_size_Y
                        
                        ng1 = g1 + x0 * g2
                        ng4 = g4 + y0 * g6
                        
                        minX = ng1
                        maxY = ng4
                        maxX = minX + split_size_X * g2
                        minY = maxY + split_size_Y * g6
                        
                        raster_name = os.path.split(raster_path)[-1].split(".")[0]
                        name = "{}_{}_{}".format(raster_name, fx, fy)
                        
                        if self.dlg.chb_createOverview.isChecked():
                            geoms.append(self.createGeom(minX, minY, maxX, maxY))
                            data.append([name, fx, fy, split_size_X, split_size_Y, (split_size_X+prev_start_x-x0)%split_size_X, (split_size_Y+prev_start_y-y0)%split_size_Y])
                        
                        if self.dlg.chb_createChunk.isChecked():
                            newRasterSource = os.path.join(out_folder, name + "." + currentExt)
                            out_raster = gdal.Warp(newRasterSource, raster, **{"outputBounds" : (minX, minY, maxX, maxY), "dstNodata" : noDataValue, "xRes":g2, "yRes":g6, **opt})
                            out_raster = None
                            del out_raster
                    
                    if self.dlg.chb_createOverview.isChecked():
                        self.createShp(data=data, geoms=geoms, srs=crs, out_folder=out_folder)
                        
                self.iface.messageBar().pushMessage("Success", "Raster splitted successfully!" , level=Qgis.Success, duration=5)
                
                if self.dlg.chb_addToMap.isChecked():
                    for p in glob(os.path.join(out_folder, "{}_[0-9]_[0-9].{}".format(raster_name, currentExt))):
                        layer_name = os.path.split(p)[-1].split(".")[0]
                        iface.addRasterLayer(p, layer_name)
                    
                    path_vector = os.path.join(f"{out_folder}", "footprints.shp")
                    layer_name_vector = os.path.split(path_vector)[-1].split(".")[0]
                    iface.addVectorLayer(path_vector, layer_name_vector, "ogr")
    
        self.dlg.btn_run.setText("RUN")
        self.dlg.processEvents()
    
    def run(self):
        """Run method that performs all the real work"""
        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            # self.first_start = False
            self.dlg = RasterDividerDialog()
            
            if any([
                (date.today().day == 18 and date.today().month == 3),
                (date.today().day == 23 and date.today().month == 4),
                (date.today().day == 19 and date.today().month == 5),
                (date.today().day == 30 and date.today().month == 8),
                (date.today().day == 29 and date.today().month == 10),
                (date.today().day == 10 and date.today().month == 11)
                ]):
                
                self.dlg.setWindowIcon(QIcon(':/plugins/raster_divider/mka.png'))
            
            self.input_type_check = False    
            self.out_folder_check = False
            self.create_outtype_check = False
            
            self.raster_layers = {layer.name():layer for layer in QgsProject.instance().mapLayers().values() if layer.type()== 1}
            self.dlg.cb_in_raster.clear()
            self.dlg.cb_in_raster.addItems(self.raster_layers.keys())
            self.dlg.cb_format.clear()
            self.formats = {gdal.GetDriver(i).LongName:(gdal.GetDriver(i).ShortName, gdal.GetDriver(i).GetMetadataItem(gdal.DMD_EXTENSIONS)) for i in range(gdal.GetDriverCount()) if (gdal.GetDriver(i).GetMetadataItem('DCAP_RASTER') == "YES") and (gdal.GetDriver(i).GetMetadataItem(gdal.DMD_EXTENSIONS))}
            self.dlg.cb_format.addItems(sorted(self.formats))
            self.dlg.cb_format.setCurrentText("GeoTIFF")
            self.dlg.cb_format_2.addItems(["tif","tiff"])
            self.getRasterSize()
            
            self.dlg.cb_in_raster.currentTextChanged.connect(self.getRasterSize)
            self.dlg.tb_browse.clicked.connect(self.selectOutputFolder)
            self.dlg.le_outputFolder.textChanged.connect(self.checkFolderPath)
            self.dlg.cb_format.currentTextChanged.connect(self.getFormat)
            self.dlg.chb_createOverview.stateChanged.connect(self.check)
            self.dlg.chb_createChunk.stateChanged.connect(self.check)
            
            self.dlg.btn_run.clicked.connect(self.execute)

        self.dlg.show()