# -*- coding: utf-8 -*-
"""
/***************************************************************************
 LazyRasterCalculatorDockWidget
                                 A QGIS plugin
 A lazy evalutation raster calculator using raster-tools.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                             -------------------
        begin                : 2025-05-01
        git sha              : $Format:%H$
        copyright            : (C) 2025 by Tim Van Driel
        email                : timothy.vandriel@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

import os

from qgis.PyQt import QtWidgets, uic
from qgis.PyQt.QtCore import pyqtSignal
from qgis.PyQt.QtWidgets import QAction
from qgis.core import (
    QgsProject,
    QgsMapLayerType,
    QgsCoordinateReferenceSystem,
    QgsRasterLayer,
    QgsMapLayer,
)
from qgis.gui import QgsProjectionSelectionDialog
from qgis.utils import iface
from PyQt5.QtWidgets import QFileDialog, QMessageBox, QInputDialog

try:
    from .backend import *
except ImportError:
    QMessageBox.critical(
        None,
        "Import Error",
        "Failed to import backend modules. Please ensure all dependencies are installed.",
    )
    raise ImportError("Backend modules could not be imported.")
import traceback


FORM_CLASS, _ = uic.loadUiType(
    os.path.join(os.path.dirname(__file__), "lazy_raster_calculator_dockwidget_base.ui")
)


class LazyRasterCalculatorDockWidget(QtWidgets.QDockWidget, FORM_CLASS):

    closingPlugin = pyqtSignal()

    def __init__(self, parent=None):
        """Constructor."""
        super(LazyRasterCalculatorDockWidget, self).__init__(parent)
        # Set up the user interface from Designer.
        # After setupUI you can access any designer object by doing
        # self.<objectname>, and you can use autoconnect slots - see
        # http://doc.qt.io/qt-5/designer-using-a-ui-file.html
        # #widgets-and-dialogs-with-auto-connect
        self.setupUi(self)

        # Hooking into the context menu of the layer tree
        self.layer_tree_view = iface.layerTreeView()
        self.layer_tree_view.contextMenuAboutToShow.connect(self.on_context_menu)

        # check if user removed the lazy layer
        QgsProject.instance().layerWillBeRemoved.connect(self.on_layer_removed)

        # check if layers were added or removed
        QgsProject.instance().layersAdded.connect(self.populate_raster_layer_list)
        QgsProject.instance().layerRemoved.connect(self.populate_raster_layer_list)

        # Connect double-click event to populate expression box
        self.rasterLayerListWidget.itemDoubleClicked.connect(
            self.handle_layer_double_click
        )

        # buttons for raster calculator
        self.plusButton.clicked.connect(lambda: self.insert_operator("+"))
        self.minusButton.clicked.connect(lambda: self.insert_operator("-"))
        self.multiplyButton.clicked.connect(lambda: self.insert_operator("*"))
        self.divideButton.clicked.connect(lambda: self.insert_operator("/"))
        self.openParenButton.clicked.connect(lambda: self.insert_operator("("))
        self.closeParenButton.clicked.connect(lambda: self.insert_operator(")"))
        self.ltButton.clicked.connect(lambda: self.insert_operator("<"))
        self.gtButton.clicked.connect(lambda: self.insert_operator(">"))
        self.ltEqButton.clicked.connect(lambda: self.insert_operator("<="))
        self.gtEqButton.clicked.connect(lambda: self.insert_operator(">="))
        self.notEqButton.clicked.connect(lambda: self.insert_operator("!="))
        self.equalButton.clicked.connect(lambda: self.insert_operator("=="))
        self.andButton.clicked.connect(lambda: self.insert_operator("&"))
        self.orButton.clicked.connect(lambda: self.insert_operator("|"))
        self.notButton.clicked.connect(lambda: self.insert_operator("~"))
        self.clearButton.clicked.connect(self.clear_expression)

        # crs button and combobox
        self.crsSelectButton.clicked.connect(self.open_crs_dialog)
        self.populate_crs_combobox()

        # dtypes combobox
        self.populate_dtypes_combobox()

        # okay and cancel buttons
        self.okButton.clicked.connect(self.on_ok_clicked)
        self.cancelButton.clicked.connect(self.on_cancel_clicked)

        # check for valid expression
        self.expressionBox.textChanged.connect(self.on_expression_changed)

        # Initialize managers
        self.layer_manager = LayerManager()
        self.raster_manager = RasterManager(self.layer_manager)
        self.expression_evaluator = ExpressionEvaluator(self.raster_manager)
        self.lazy_registry = get_lazy_layer_registry()
        self.raster_saver = RasterSaver()

    def closeEvent(self, event):
        self.clear_expression()
        self.closingPlugin.emit()
        event.accept()

    def showEvent(self, event):
        super().showEvent(event)
        self.populate_raster_layer_list()

    def on_context_menu(self, menu):
        """Adds custom actions to the context menu for lazy raster layers only."""
        layer = self.layer_tree_view.currentLayer()
        if not layer:
            return  # leave the default menu alone

        # Only modify the menu for lazy raster layers
        if layer.type() == QgsMapLayer.RasterLayer and layer.customProperty(
            "is_lazy", False
        ):
            # Add custom actions if not already present
            existing_actions = [action.text() for action in menu.actions()]

            if "Compute Lazy Layer" not in existing_actions:
                compute = QAction("Compute Lazy Layer", menu)
                compute.triggered.connect(lambda: self.compute_lazy_layer(layer))
                menu.addAction(compute)

            if "Compute and Export Lazy Layer..." not in existing_actions:
                export = QAction("Compute and Export Lazy Layer...", menu)
                export.triggered.connect(lambda: self.export_lazy_layer(layer))
                menu.addAction(export)

            # Optionally remove non-lazy-specific actions *only for lazy layers*
            for action in menu.actions()[:]:  # make a copy of the list
                if (
                    action.text()
                    not in [
                        "Compute Lazy Layer",
                        "Compute and Export Lazy Layer...",
                    ]
                    and not action.isSeparator()
                ):
                    menu.removeAction(action)

        # For non-lazy layers, do nothing — QGIS will show its default menu

    def compute_lazy_layer(self, layer):
        """Computes the lazy layer by retrieving it from the lazy registry, and adds it to the project as a new raster layer using a temporary path.
        Args:
            layer (QgsRasterLayer): The lazy raster layer to compute.
        """
        layer_name = layer.customProperty("lazy_name", None)
        if not layer_name:
            QMessageBox.warning(
                self,
                "Error",
                "This layer does not have a valid lazy name. Cannot compute.",
            )
            return

        try:
            # Retrieve and copy to break from Dask graph
            lazy_layer = self.lazy_registry.get(layer_name)
            raster = lazy_layer.copy()

            # Save computed result to temporary location and get new layer
            _, _ = self.raster_saver.temp_output(raster, layer_name)

            # Remove the old placeholder (not the new one)
            QgsProject.instance().removeMapLayer(layer.id())

            del raster  # Free memory
            del lazy_layer
            self.clear_expression()  # Clear the expression box

        except Exception as e:
            QMessageBox.critical(
                self,
                "Compute Error",
                f"An error occurred while computing the lazy layer:\n{str(e)}",
            )

    def export_lazy_layer(self, layer):
        """Exports the lazy layer to a GeoTIFF file and path specified by the user.
        Args:
            layer (QgsRasterLayer): The lazy raster layer to export."""
        layer_name = layer.customProperty("lazy_name", None)
        if not layer_name:
            QMessageBox.warning(
                self,
                "Error",
                "This layer does not have a valid lazy name. Cannot export.",
            )
            return

        # Try to copy raster safely
        try:
            lazy_layer = self.lazy_registry.get(layer_name)
            raster = lazy_layer.copy()
            del lazy_layer  # Free memory

        except Exception as e:
            QMessageBox.critical(
                self, "Error", f"Failed to prepare raster for export:\n{str(e)}"
            )
            return

        # Get file path from user
        suggested_filename = f"{layer_name}.tif"
        file_path, _ = QFileDialog.getSaveFileName(
            self,
            "Export Lazy Layer",
            suggested_filename,
            "GeoTIFF (*.tif *.tiff)",
        )
        if not file_path:  # User cancelled the dialog
            return
        if not os.path.isfile(file_path):  # Check if the file path is valid
            QMessageBox.warning(
                self,
                "Invalid File Path",
                "The specified file path does not exist or is not a valid file.",
            )
            return

        # Determine driver
        ext = os.path.splitext(file_path)[-1].lower()
        if ext in [".tif", ".tiff"]:
            driver = "GTiff"
        else:
            QMessageBox.warning(
                self,
                "Unsupported Format",
                "The selected file format is not supported for export.",
            )
            return

        try:
            raster.save(file_path, driver=driver, tiled=True)

            QMessageBox.information(
                self,
                "Export Successful",
                f"Lazy layer '{layer_name}' exported successfully to:\n{file_path}",
            )

            # clean up after successful save
            QgsProject.instance().removeMapLayer(
                layer.id()
            )  # Remove the placeholder layer
            del raster  # free memory
            self.clear_expression()  # Clear the expression box

        except Exception as e:
            tb = traceback.format_exc()
            QMessageBox.critical(
                self,
                "Export Error",
                f"An error occurred while exporting the lazy layer:\n{str(e)}\n\nTraceback:\n{tb}",
            )

    def on_layer_removed(self, layer_id):
        """
        Handle the removal of a layer from the project.
        This method:
        1. Removes the lazy layer from the registry if it's a lazy layer.
        2. Deletes any associated temporary file if tracked.
        Args:
            layer_id (str): The ID of the layer that was removed.
        """
        layer = QgsProject.instance().mapLayer(layer_id)
        if not layer:
            return

        # 1. Remove from lazy registry if applicable
        lazy_name = layer.customProperty("lazy_name", None)
        if lazy_name and self.lazy_registry.has(lazy_name):
            self.lazy_registry.remove(lazy_name)

    def populate_raster_layer_list(self):
        """Populate the list widget with visible raster layers, including band-specific entries for multi-band rasters."""
        self.rasterLayerListWidget.clear()

        for layer in QgsProject.instance().mapLayers().values():
            if layer.type() != QgsMapLayerType.RasterLayer:
                continue

            is_lazy = layer.customProperty("is_lazy", False)
            base_name = (
                layer.customProperty("lazy_name", layer.name())
                if is_lazy
                else layer.name()
            )
            band_count = layer.bandCount()
            if is_lazy:
                display_name = f"{base_name} (Lazy)"
                self.rasterLayerListWidget.addItem(display_name)
                band_count = int(layer.customProperty("band_count", band_count))
                if band_count == 1:
                    continue
                else:
                    # Add band-specific entries for lazy layers
                    for i in range(1, band_count + 1):
                        display_name = f"{base_name} (Lazy)@{i}"
                        self.rasterLayerListWidget.addItem(display_name)

            else:
                self.rasterLayerListWidget.addItem(base_name)
                if band_count == 1:
                    continue  # Single-band layers only show the base name
                else:
                    # Add band-specific entries
                    for i in range(1, band_count + 1):
                        display_name = f"{base_name}@{i}"
                        self.rasterLayerListWidget.addItem(display_name)

    def handle_layer_double_click(self, item):
        """Handle double-click event on a layer name in the list widget.
        This method is called when the user double-clicks on a layer name in the list.
        It inserts the layer name into the expression box at the current cursor position.
        """
        layer_name = '"' + item.text() + '"'
        self.insert_text_into_text_edit(layer_name)

    def insert_operator(self, operator):
        """Insert an operator at the current cursor position in the expression box.
        This method is called when the user clicks on an operator button (e.g., +, -, *, /).
        """
        self.insert_text_into_text_edit(operator)

    def clear_expression(self):
        """Clear the expression box."""
        self.expressionBox.clear()

    def insert_text_into_text_edit(self, text):
        """Insert text at the current cursor position in the expression box."""
        cursor = self.expressionBox.textCursor()
        cursor.insertText(text)
        self.expressionBox.setTextCursor(cursor)
        self.expressionBox.setFocus()

    def update_expression_status(self, is_valid):
        """Update the status label based on the validity of the expression."""
        if is_valid:
            self.expressionStatusLabel.setText("Valid expression")
            self.expressionStatusLabel.setStyleSheet("color: green;")
        else:
            self.expressionStatusLabel.setText("Invalid expression")
            self.expressionStatusLabel.setStyleSheet("color: red;")

    def on_expression_changed(self):
        """Handle changes in the expression box."""
        if self.expressionBox.toPlainText().strip() == "":
            self.expressionStatusLabel.setText("Waiting for input...")
            self.expressionStatusLabel.setStyleSheet("")
        else:
            text = self.expressionBox.toPlainText().strip()
            valid = ExpressionEvaluator.is_valid_expression(text)
            self.update_expression_status(valid)

    def open_crs_dialog(self):
        """Open the CRS selection dialog and set the selected CRS."""
        initial_crs = QgsCoordinateReferenceSystem("EPSG:4326")

        dlg = QgsProjectionSelectionDialog(self)
        dlg.setCrs(initial_crs)

        if dlg.exec_() == 1:  # User pressed OK
            selected_crs = dlg.crs()
            selected_authid = selected_crs.authid()
            selected_label = f"{selected_authid} - {selected_crs.description()}"

            # Check if it's already in the combo box
            found = False
            for i in range(self.crsComboBox.count()):
                if self.crsComboBox.itemData(i) == selected_authid:
                    self.crsComboBox.setCurrentIndex(i)
                    found = True
                    break

            # If not found, add it
            if not found:
                self.crsComboBox.addItem(selected_label, selected_authid)
                self.crsComboBox.setCurrentIndex(self.crsComboBox.count() - 1)

    def populate_crs_combobox(self):
        """Populate the CRS combo box with available coordinate reference systems."""
        self.crsComboBox.clear()

        # Default CRS 4326
        default_crs = QgsCoordinateReferenceSystem("EPSG:4326")
        self.crsComboBox.addItem(
            f"{default_crs.authid()} - {default_crs.description()}",
            default_crs.authid(),
        )

        # Project CRS
        project_crs = QgsProject.instance().crs()
        self.crsComboBox.addItem(
            f"Project CRS - {project_crs.authid()} - {project_crs.description()}",
            project_crs.authid(),
        )

    def populate_dtypes_combobox(self):
        dtypes = [
            "<AUTO>",
            "Byte",
            "Int8",
            "UInt16",
            "Int16",
            "UInt32",
            "Int32",
            "UInt64",
            "Int64",
            "Float32",
            "Float64",
            "CInt16",
            "CInt32",
            "CFloat32",
            "CFloat64",
        ]
        for dtype in dtypes:
            self.dtypeComboBox.addItem(dtype)
        self.dtypeComboBox.setCurrentIndex(0)  # Set default to <AUTO>

    def on_ok_clicked(self):
        """Handle the OK button click event.
        This method evaluates the expression entered by the user, checks if it is valid,
        and adds the resulting raster layer to the QGIS project.
        If the expression is lazy, it prompts for a name and adds a placeholder layer.
        """
        # Get user inputs
        expression = self.expressionBox.toPlainText().strip()
        is_lazy = self.lazyCheckBox.isChecked()
        crs_index = self.crsComboBox.currentIndex()
        target_crs_authid = self.crsComboBox.itemData(crs_index)
        d_type = self.dtypeComboBox.currentText()

        # Validate inputs
        if not expression:
            QMessageBox.warning(
                self,
                "Missing Information",
                "Please enter an expression before proceeding.",
            )
            return

        try:

            # Prompt for name if lazy
            result_name = None
            if is_lazy:
                result_name, ok = QInputDialog.getText(
                    self, "Lazy Layer Name", "Enter a name for the lazy layer:"
                )
                if not ok or not result_name.strip():
                    QMessageBox.warning(
                        self, "Invalid Name", "Lazy layer name cannot be empty."
                    )
                    return
            else:
                result_name, ok = QInputDialog.getText(
                    self, "Raster Layer Name", "Enter a name for the raster layer:"
                )
                if not ok or not result_name.strip():
                    QMessageBox.warning(
                        self, "Invalid Name", "Raster layer name cannot be empty."
                    )
                    return
            result_name = result_name.strip()

            # Evaluate
            result = self.expression_evaluator.evaluate(
                expression,
                target_crs_authid,
                d_type=d_type,
            )

            if is_lazy:
                # Add placeholder fake QgsRasterLayer
                self.raster_manager.add_lazy_layer(result_name.strip(), result)
                uri = f"NotComputed:{result_name}"
                fake_layer = QgsRasterLayer(uri, f"{result_name} (Lazy)")
                fake_layer.setCustomProperty("is_lazy", True)
                fake_layer.setCustomProperty("lazy_name", result_name)
                fake_layer.setCustomProperty("lazy_expression", expression)
                fake_layer.setCustomProperty("lazy_crs", target_crs_authid)
                fake_layer.setCustomProperty("lazy_dtype", str(result.dtype))
                fake_layer.setCustomProperty("band_count", str(result.nbands))
                QgsProject.instance().addMapLayer(fake_layer)

                QMessageBox.information(
                    self,
                    "Lazy Evaluation",
                    f"Lazy layer '{result_name}' has been created and added as a placeholder.",
                )
                self.clear_expression()
                return
            try:
                # Save the raster to a temporary file and add it to the project
                _, _ = self.raster_saver.temp_output(result, result_name)
                QMessageBox.information(
                    self,
                    "Success",
                    f"Raster added to project",
                )
                self.clear_expression()
            except RasterSaveError as e:
                print(f"Error saving file: {str(e)}")
                return

        except BandMismatchError as e:
            QMessageBox.critical(self, "Band Mismatch", str(e))
        except InvalidExpressionError as e:
            QMessageBox.critical(self, "Invalid Expression", str(e))
        except LayerNotFoundError as e:
            QMessageBox.critical(self, "Missing Layers", str(e))
        except RasterToolsUnavailableError as e:
            QMessageBox.critical(self, "Raster Tools Error", str(e))
        except RasterSaveError as e:
            QMessageBox.critical(self, "Save Error", str(e))
        except Exception as e:
            tb = traceback.format_exc()
            QMessageBox.critical(
                self,
                "Unexpected Error",
                f"An unexpected error occurred:\n{str(e)}\n\nTraceback:\n{tb}",
            )

    def on_cancel_clicked(self):
        """Handle cancel button click event"""
        self.clear_expression()
        self.crsComboBox.setCurrentIndex(0)
        self.lazyCheckBox.setChecked(True)
        self.dtypeComboBox.setCurrentIndex(0)
        self.populate_raster_layer_list()
        self.populate_crs_combobox()
        self.update_expression_status(False)
