"""
/***************************************************************************
 RasterTools
                                 A QGIS plugin
 This plugin provides a raster calculator and delivered cost calculator.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-07-31
        git sha              : $Format:%H$
        copyright            : (C) 2025 by Tim Van Driel
        email                : timothy.vandriel@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

import re
import traceback
from qgis.core import QgsMessageLog, Qgis
from .exceptions import InvalidExpressionError
from .raster_manager import RasterManager
from .safe_evaluator import SafeEvaluator
import ast

try:
    from raster_tools.clipping import clip_box
except ImportError:
    "raster_tools.clipping module not found. Ensure raster-tools is installed."


class ExpressionEvaluator:
    """
    Responsible for parsing and evaluating raster expressions using layer names and Raster objects.
    """

    def __init__(self, raster_manager: RasterManager):
        """
        Initializes the ExpressionEvaluator with a reference to a RasterManager.

        Args:
            raster_manager (RasterManager): Manages loading raster layers as Raster objects.
        """
        self.raster_manager = raster_manager

    @staticmethod
    def extract_layer_names(expression: str) -> list[str]:
        """
        Extracts all raster layer names enclosed in double quotes from the expression.

        Args:
            expression (str): A mathematical expression with quoted raster layer names.

        Returns:
            list[str]: A list of layer names found in the expression.
        """
        pattern = r'"([^"]+)"'
        return re.findall(pattern, expression)

    @staticmethod
    def is_valid_expression(expression: str) -> bool:
        """
        Validates the expression syntax using Python's AST parser.
        Replaces quoted layer names with valid identifiers before parsing.
        Ensures only valid operators and known identifiers are used.
        """
        if not expression:
            return False

        # Check for adjacent quoted layer names without operator
        if re.search(r'"[^"]+"\s*"[^"]+"', expression):
            return False

        # Disallow any double operator except exponent (**)
        # This checks the raw expression before replacements
        if re.search(r"([+\-*/^%]{2,})", expression):
            matches = re.findall(r"([+\-*/^%]{2,})", expression)
            for m in matches:
                if m != "**":
                    return False

        # Replace quoted layer names with dummy identifiers
        dummy_names = {}

        def replacer(match):
            name = match.group(1)
            if name not in dummy_names:
                dummy_names[name] = f"r_{len(dummy_names)}"
            return dummy_names[name]

        expr_cleaned = re.sub(r'"([^"]+)"', replacer, expression)

        # Try parsing into AST
        try:
            tree = ast.parse(expr_cleaned, mode="eval")
        except SyntaxError:
            return False

        # Allowed variable names = only our dummy raster names
        allowed_names = set(dummy_names.values())

        # Walk the AST and validate names
        for node in ast.walk(tree):
            if isinstance(node, ast.Name):
                if node.id not in allowed_names:
                    return False

        return True

    def evaluate(
        self,
        expression: str,
        target_crs_authid: str = None,
        d_type: str = "<AUTO>",
    ):
        """
        Evaluates a raster expression by:
        - Extracting layer names.
        - Validating their presence in the QGIS project.
        - Validating the rasters have the same number of bands.
        - Reprojecting rasters to a target CRS if specified.
        - Aligning rasters to the smallest extent.
        - Creating a safe evaluation context.
        - Validating the expression syntax.
        - Replacing names with safe variable names.
        - Evaluating the expression using `eval()`.
        - Casting the result to the specified data type.

        Args:
            expression (str): The raster math expression, with layer names in quotes.
            target_crs_authid (str, optional): The target CRS authority ID for reprojection.
            d_type (str, optional): The data type to cast the resulting raster to. Defaults to "<AUTO>".

        Returns:
            raster_tools.Raster: The resulting lazily-evaluated raster object.

        Raises:
            LayerNotFoundError: If any raster layers are missing from the project.
            InvalidExpressionError: If the expression syntax is invalid or fails evaluation.
        """
        # Step 1: Extract layer names from the expression
        layer_names = self.extract_layer_names(expression)

        # Step 2: Validate that all layer names exist in the project
        self.raster_manager.layer_manager.validate_layer_names(layer_names)

        # Step 3: Validate the expression syntax
        if not self.is_valid_expression(expression):
            raise InvalidExpressionError("Expression syntax is invalid.")

        # Step 4: Retrieve Raster objects for all layers
        raster_objects = self.raster_manager.get_rasters(layer_names)
        self.raster_manager.check_bands(raster_objects)  # check for consistent bands

        # Step 4.5a: Reproject rasters if needed to target CRS
        if target_crs_authid:
            raster_objects = {
                name: self.raster_manager.reproject_if_needed(raster, target_crs_authid)
                for name, raster in raster_objects.items()
            }
        # check for overlaps after each raster matches the target CRS raise error if one or more rasters do not overlap
        self.raster_manager.raster_overlap(raster_objects)

        # Step 4.5b: Align rasters to the smallest extent
        # ref_name, raster_objects = self.raster_manager._align_to_smallest_extent(
        #     raster_objects
        # )
        ref_name, raster_objects = self.raster_manager.align_to_overlap(raster_objects)

        # Step 5: Create a safe evaluation context
        context = {}  # maps safe variable names to Raster objects
        name_map = {}  # maps original layer names to safe variable names

        for i, (name, raster) in enumerate(raster_objects.items()):
            safe_name = f"r_{i}"  # Create safe variable name
            context[safe_name] = raster
            name_map[name] = safe_name

        # Replace layer names in expression with safe variable names
        safe_expression = re.sub(
            r'"([^"]+)"', lambda m: name_map.get(m.group(1), m.group(0)), expression
        )

        try:
            # Step 6: Evaluate the expression
            evaluator = SafeEvaluator(
                context
            )  # Initialize the safe evaluator with context
            result = evaluator.evaluate(
                safe_expression
            )  # Evaluate the expression safely

            d_type = self.raster_manager.get_dtype(d_type)
            result = result.astype(d_type) if d_type != "<AUTO>" else result

            return result
        except Exception as e:
            QgsMessageLog.logMessage(
                f"Error evaluating expression: {str(e)}\n{traceback.format_exc()}",
                "Lazy Raster Calculator",
                Qgis.Critical,
            )
            raise InvalidExpressionError(str(e))
