# -*- coding: utf-8 -*-
"""
/***************************************************************************
 DeliveredCostDockWidget
                                 A QGIS plugin
 This plugin provides delivered cost for an area of interest.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                             -------------------
        begin                : 2025-07-15
        git sha              : $Format:%H$
        copyright            : (C) 2025 by Tim
        email                : timothy.vandriel@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

import os

from qgis.PyQt import QtGui, QtWidgets, uic
from qgis.PyQt.QtCore import pyqtSignal
from qgis.core import (
    QgsProject,
    QgsRasterLayer,
    Qgis,
    QgsRectangle,
    QgsVectorLayer,
    QgsFeature,
    QgsCoordinateReferenceSystem,
    QgsCoordinateTransform,
    QgsGeometry,
    QgsPointXY,
    QgsRasterShader,
    QgsColorRampShader,
    QgsSingleBandPseudoColorRenderer,
    QgsRasterBandStats,
    QgsMarkerSymbol,
    QgsWkbTypes,
)
from qgis.gui import QgsMapToolPan
from qgis.utils import iface
from PyQt5.QtWidgets import QMessageBox
from PyQt5.QtCore import QTimer, QThreadPool
from .draw_polygon_tool import DrawPolygonTool
from .pick_point_tool import PickPointTool
from PyQt5.QtGui import QColor


FORM_CLASS, _ = uic.loadUiType(
    os.path.join(os.path.dirname(__file__), "delivered_cost_dockwidget_base.ui")
)


class DeliveredCostDockWidget(QtWidgets.QDockWidget, FORM_CLASS):

    closingPlugin = pyqtSignal()

    def __init__(self, parent=None):
        """Constructor."""
        super(DeliveredCostDockWidget, self).__init__(parent)
        # Set up the user interface from Designer.
        # After setupUI you can access any designer object by doing
        # self.<objectname>, and you can use autoconnect slots - see
        # http://doc.qt.io/qt-5/designer-using-a-ui-file.html
        # #widgets-and-dialogs-with-auto-connect
        self.setupUi(self)
        self.threadpool = QThreadPool.globalInstance()
        # Make log textbox read-only
        self.plainTextEdit.setReadOnly(True)
        # Manage the OSM layer
        self.osm_layer_id = None
        # Check if layers are removed
        QgsProject.instance().layerWillBeRemoved.connect(self.on_layer_removed)
        # Trigger action pan tool
        iface.actionPan().trigger()

        # Background layers
        self.layer_configs = {
            self.esriCheckBox: {
                "name": "ESRI World Imagery",
                "url": "type=xyz&zmin=0&zmax=19&url=https://services.arcgisonline.com/ArcGIS/rest/services/World_Imagery/MapServer/tile/{z}/{y}/{x}",
                "provider": "wms",
                "layer_id": None,
            },
            self.burnProbCheckBox: {
                "name": "Burn Probability",
                "url": "url=https://apps.fs.usda.gov/fsgisx03/rest/services/wo_spf_fam/Nat_BurnProbability/ImageServer",
                "provider": "arcgismapserver",
                "layer_id": None,
            },
            self.evtCheckBox: {
                "name": "EVT",
                "url": "url=https://lfps.usgs.gov/arcgis/rest/services/Landfire_LF230/US_230EVT/ImageServer",
                "provider": "arcgismapserver",
                "layer_id": None,
            },
            self.pctCheckBox: {
                "name": "PCL",
                "url": "url=https://apps.fs.usda.gov/fsgisx03/rest/services/wo_spf_fam/Potential_Control_Location/ImageServer",
                "provider": "arcgismapserver",
                "layer_id": None,
            },
        }
        # Connect checkboxes to their respective layer loading functions
        for checkbox in self.layer_configs.keys():
            checkbox.toggled.connect(self.on_layer_checkbox_toggled)

        # Connect sliders to spinboxes
        self.rtSkidderSpeedSlider.valueChanged.connect(
            lambda value: self.update_spinbox_from_slider(
                self.rtSkidderSpeedSlider, self.rtSpdSpinBox
            )
        )
        self.rtSpdSpinBox.valueChanged.connect(
            lambda value: self.update_slider_from_spinbox(
                self.rtSpdSpinBox, self.rtSkidderSpeedSlider
            )
        )
        self.skylineSpeedSlider.valueChanged.connect(
            lambda value: self.update_spinbox_from_slider(
                self.skylineSpeedSlider, self.skylineSpdSpinBox
            )
        )
        self.skylineSpdSpinBox.valueChanged.connect(
            lambda value: self.update_slider_from_spinbox(
                self.skylineSpdSpinBox, self.skylineSpeedSlider
            )
        )
        self.rtSkidderMonSlider.valueChanged.connect(self.rtSkidderMonSpinBox.setValue)
        self.rtSkidderMonSpinBox.valueChanged.connect(
            lambda val: self.rtSkidderMonSlider.setValue(int(val))
        )

        self.skylineMonSlider.valueChanged.connect(self.skylineMonSpinBox.setValue)
        self.skylineMonSpinBox.valueChanged.connect(
            lambda val: self.skylineMonSlider.setValue(int(val))
        )

        self.fellerbuncherRateSlider.valueChanged.connect(
            self.fellerbunchRateSpinBox.setValue
        )
        self.fellerbunchRateSpinBox.valueChanged.connect(
            lambda val: self.fellerbuncherRateSlider.setValue(int(val))
        )

        self.handFellingSlider.valueChanged.connect(
            self.handfellingRateSpinBox.setValue
        )
        self.handfellingRateSpinBox.valueChanged.connect(
            lambda val: self.handFellingSlider.setValue(int(val))
        )

        self.processingSlider.valueChanged.connect(self.processingSpinBox.setValue)
        self.processingSpinBox.valueChanged.connect(
            lambda val: self.processingSlider.setValue(int(val))
        )

        self.haulingSlider.valueChanged.connect(self.haulingSpinBox.setValue)
        self.haulingSpinBox.valueChanged.connect(
            lambda val: self.haulingSlider.setValue(int(val))
        )

        self.handTreatmentSlider.valueChanged.connect(
            self.handTreatmentSpinBox.setValue
        )
        self.handTreatmentSpinBox.valueChanged.connect(
            lambda val: self.handTreatmentSlider.setValue(int(val))
        )

        self.prescribedFireSlider.valueChanged.connect(
            self.prescribedFireSpinBox.setValue
        )
        self.prescribedFireSpinBox.valueChanged.connect(
            lambda val: self.prescribedFireSlider.setValue(int(val))
        )

        self.rtSkidderPayloadSlider.valueChanged.connect(
            lambda value: self.update_spinbox_from_slider(
                self.rtSkidderPayloadSlider, self.rtSkidderPayloadSpinBox
            )
        )
        self.rtSkidderPayloadSpinBox.valueChanged.connect(
            lambda value: self.update_slider_from_spinbox(
                self.rtSkidderPayloadSpinBox, self.rtSkidderPayloadSlider
            )
        )
        self.skylinePayloadSlider.valueChanged.connect(
            lambda value: self.update_spinbox_from_slider(
                self.skylinePayloadSlider, self.skylinePayloadSpinBox
            )
        )
        self.skylinePayloadSpinBox.valueChanged.connect(
            lambda value: self.update_slider_from_spinbox(
                self.skylinePayloadSpinBox, self.skylinePayloadSlider
            )
        )
        self.logTruckPayloadSlider.valueChanged.connect(
            lambda value: self.update_spinbox_from_slider(
                self.logTruckPayloadSlider, self.logTruckPayloadSpinBox
            )
        )
        self.logTruckPayloadSpinBox.valueChanged.connect(
            lambda value: self.update_slider_from_spinbox(
                self.logTruckPayloadSpinBox, self.logTruckPayloadSlider
            )
        )
        # Initialize AOI and Facility comboboxes
        self.setup_layer_listeners()
        self.populate_layer_comboboxes()

        # Initialize the draw polygon tool
        self.drawTool = DrawPolygonTool(iface.mapCanvas())
        self.drawTool.polygonCompleted.connect(self.handle_polygon_completed)
        self.drawPolygonButton.clicked.connect(self.activate_draw_tool)
        self.aoi_geometry = None  # Store drawn polygon geometry
        self.aoi_layer_id = None  # Store AOI layer ID

        # Initialize pick point tool
        self.pointTool = PickPointTool(iface.mapCanvas())
        self.pointTool.pointPicked.connect(self.handle_point_picked)
        self.pickPointButton.clicked.connect(self.activate_point_picker)
        self.facility_coords = []  # Store picked point coordinates
        self.facility_layer = None  # Store facility layer object
        self.facility_layer_id = None  # Store facility layer ID

        # Connect run button
        self.runButton.clicked.connect(self.run_delivered_cost)

    def update_spinbox_from_slider(self, slider, spinbox):
        """Update spinbox value based on slider value."""
        spinbox.setValue(slider.value() / 10.0)

    def update_slider_from_spinbox(self, spinbox, slider):
        """Update slider value based on spinbox value."""
        slider.setValue(int(spinbox.value() * 10))

    def add_osm_basemap(self):
        """Add OpenStreetMap layer if not already added."""
        if getattr(self, "osm_layer_id", None) is None:
            # Set project CRS to 3857 (Web Mercator)
            QMessageBox.information(
                None, "Adding OSM Basemap", "Setting project CRS to EPSG:3857"
            )
            crs = QgsCoordinateReferenceSystem("EPSG:3857")
            QgsProject.instance().setCrs(crs)
            layer_name = "OSM Standard"
            url = "type=xyz&zmin=0&zmax=19&url=http://tile.openstreetmap.org/{z}/{x}/{y}.png"
            layer = QgsRasterLayer(url, layer_name, "wms")
            if layer.isValid():
                QgsProject.instance().addMapLayer(layer, addToLegend=False)
                QgsProject.instance().layerTreeRoot().insertLayer(-1, layer)
                self.osm_layer_id = (
                    layer.id()
                )  # Store only the layer ID, not the layer object
            else:
                QMessageBox.critical(
                    None,
                    "Layer Load Error",
                    f"Failed to load layer: {layer_name}",
                )

    def zoom_to_us_extent_3857(self):
        """Zoom the map canvas to an approximate extent of the continental U.S. in EPSG:3857."""
        # Approximate extent for continental U.S. in EPSG:3857 (Web Mercator)
        extent_3857 = QgsRectangle(-14000000, 2800000, -7000000, 6300000)

        # Set extent and refresh canvas
        canvas = iface.mapCanvas()
        canvas.setExtent(extent_3857)
        canvas.refresh()

    def on_layer_removed(self, layer_id):
        """Handle layer removal event.
        Args:
            layer_id (str): The ID of the layer being removed.
        """
        if getattr(self, "osm_layer_id", None) == layer_id:
            self.osm_layer_id = None
        if self.facility_layer_id == layer_id:
            self.facility_layer_id = None
            self.facility_coords = []
            self.facility_layer = None
        if self.aoi_layer_id == layer_id:
            self.aoi_layer_id = None
            self.aoi_geometry = None

    def populate_layer_comboboxes(self):
        """Populate AOI and Facility comboboxes with available layers."""
        # Save current selections
        aoi_id = self.aoiComboBox.currentData()
        point_id = self.pointComboBox.currentData()
        roads_id = self.roadsComboBox.currentData()
        barriers_id = self.barriersComboBox.currentData()

        # Clear and re-add default items
        self.aoiComboBox.clear()
        self.pointComboBox.clear()
        self.roadsComboBox.clear()
        self.barriersComboBox.clear()

        self.aoiComboBox.addItem("Select AOI Layer or create new", None)
        self.pointComboBox.addItem("Select Facility Layer or create new", None)
        self.roadsComboBox.addItem("Select Roads Layer (Optional)", None)
        self.barriersComboBox.addItem("Select Barriers Layer (Optional)", None)

        # Track mapping from layer ID to combo index
        aoi_index_map = {None: 0}
        point_index_map = {None: 0}
        roads_index_map = {None: 0}
        barriers_index_map = {None: 0}

        for layer in QgsProject.instance().mapLayers().values():
            if isinstance(layer, QgsVectorLayer):
                layer_id = layer.id()
                layer_name = layer.name()
                geom_type = layer.geometryType()

                # Add vector layers to roads and barriers combo boxes
                self.roadsComboBox.addItem(layer_name, layer_id)
                roads_index_map[layer_id] = self.roadsComboBox.count() - 1
                self.barriersComboBox.addItem(layer_name, layer_id)
                barriers_index_map[layer_id] = self.barriersComboBox.count() - 1

                # Add AOI and Facility layers based on geometry type
                if geom_type == QgsWkbTypes.PolygonGeometry:
                    self.aoiComboBox.addItem(layer_name, layer_id)
                    aoi_index_map[layer_id] = self.aoiComboBox.count() - 1
                elif geom_type == QgsWkbTypes.PointGeometry:
                    self.pointComboBox.addItem(layer_name, layer_id)
                    point_index_map[layer_id] = self.pointComboBox.count() - 1

        # Restore previous selections if still present
        self.aoiComboBox.setCurrentIndex(aoi_index_map.get(aoi_id, 0))
        self.pointComboBox.setCurrentIndex(point_index_map.get(point_id, 0))
        self.roadsComboBox.setCurrentIndex(roads_index_map.get(roads_id, 0))
        self.barriersComboBox.setCurrentIndex(barriers_index_map.get(barriers_id, 0))

    def setup_layer_listeners(self):
        """Connect signals to update layer comboboxes when layers are added or removed."""
        QgsProject.instance().layersAdded.connect(self.populate_layer_comboboxes)
        QgsProject.instance().layersRemoved.connect(self.populate_layer_comboboxes)
        QgsProject.instance().layerWasAdded.connect(self.populate_layer_comboboxes)

    def get_selected_aoi_layer(self):
        """Get the currently selected AOI layer from the combobox.
        Returns:
            QgsVectorLayer: The selected AOI layer, or None if no valid layer is selected.
        """
        layer_id = self.aoiComboBox.currentData()
        if layer_id is None:
            return None
        return QgsProject.instance().mapLayer(layer_id)

    def get_selected_aoi_geometry(self):
        """Sets self.aoi_geometry to the geometry of the selected AOI layer.
        If no AOI layer is selected or the layer is invalid, returns None.
        Returns:
            QgsCoordinateReferenceSystem: The CRS of the AOI layer, or None if no valid layer is selected.
        """
        # Get the selected AOI layer
        aoi_layer = self.get_selected_aoi_layer()
        if aoi_layer is None or not aoi_layer.isValid():  # Check if layer is valid
            return None

        # Get the first feature's geometry
        feature = next(aoi_layer.getFeatures())
        self.aoi_geometry = feature.geometry()
        return aoi_layer.crs()  # Return the CRS of the AOI layer

    def get_selected_facility_layer(self):
        """Get the currently selected Facility layer from the combobox.
        Returns:
            QgsVectorLayer: The selected Facility layer, or None if no valid layer is selected.
        """
        # Get the selected layer ID from the combobox
        layer_id = self.pointComboBox.currentData()
        if layer_id is None:  # If no layer is selected, return None
            return None
        return QgsProject.instance().mapLayer(layer_id)

    def get_selected_facility_coords(self):
        """Sets self.facility_coords to the coordinates of the selected Facility layer.
        If no Facility layer is selected or the layer is invalid, returns None.
        Returns:
            QgsCoordinateReferenceSystem: The CRS of the Facility layer, or None if no valid layer is selected.
        """
        # Get the selected Facility layer
        self.facility_coords = []  # Reset coordinates list
        facility_layer = self.get_selected_facility_layer()
        if (
            facility_layer is None or not facility_layer.isValid()
        ):  # Check if layer is valid
            return None

        # Get all features' geometries
        for feature in facility_layer.getFeatures():
            geom = feature.geometry()
            self.facility_coords.append((geom.asPoint().x(), geom.asPoint().y()))
        return facility_layer.crs()

    def on_layer_checkbox_toggled(self, checked):
        """Handle toggling of background layer checkboxes.
        Args:
            checked (bool): True if the checkbox is checked, False if unchecked.
        """
        # Get the checkbox that sent this signal
        checkbox = self.sender()
        config = self.layer_configs.get(checkbox)  # Get the config for this checkbox
        if config is None:
            return

        if checked:
            # Add layer if not already added
            if not config["layer_id"]:
                layer = QgsRasterLayer(
                    config["url"], config["name"], config["provider"]
                )
                if layer.isValid():
                    QgsProject.instance().addMapLayer(layer)
                    config["layer_id"] = layer.id()
                else:
                    QMessageBox.critical(
                        None,
                        "Layer Load Error",
                        f"Failed to load layer: {config['name']}",
                    )
        else:
            # Remove layer if exists
            layer_id = config.get("layer_id")
            if layer_id:
                layer = QgsProject.instance().mapLayer(layer_id)
                if layer:
                    QgsProject.instance().removeMapLayer(layer)
                    iface.mapCanvas().refresh()
                config["layer_id"] = None

    def activate_draw_tool(self):
        """Activate the draw polygon tool to create an AOI polygon."""
        # If AOI layer already exists, remove it from project
        if hasattr(self, "aoi_layer_id") and self.aoi_layer_id:
            layer = QgsProject.instance().mapLayer(self.aoi_layer_id)
            if layer:
                QgsProject.instance().removeMapLayer(layer)
            self.aoi_layer_id = None
            iface.mapCanvas().refresh()

        # Set up and activate the drawing tool
        iface.mapCanvas().setMapTool(self.drawTool)

        # Show instructions
        iface.messageBar().pushMessage(
            "Instructions for drawing a polygon",
            "Click to add points, right-click to finish the polygon. AOI will be removed if button is clicked again.",
            level=Qgis.Info,
            duration=15,
        )

    def handle_polygon_completed(self, geom):
        """Handle the completion of a polygon drawing.
        Args:
            geom (QgsGeometry): The drawn polygon geometry.
        """
        project_crs = QgsProject.instance().crs()
        geom = QgsGeometry(geom)

        # Create the layer
        aoi_layer = QgsVectorLayer(
            f"Polygon?crs={project_crs.authid()}", "AOI", "memory"
        )
        QgsProject.instance().addMapLayer(aoi_layer)
        self.aoi_layer_id = aoi_layer.id()  # Store only the ID
        self.aoiComboBox.setCurrentIndex(
            self.aoiComboBox.findData(self.aoi_layer_id)
        )  # update combobox selection

        # Add the geometry to the layer
        pr = aoi_layer.dataProvider()
        feat = QgsFeature()
        feat.setGeometry(geom)
        pr.addFeatures([feat])
        aoi_layer.updateExtents()
        aoi_layer.triggerRepaint()
        iface.mapCanvas().refresh()

        iface.actionPan().trigger()

    def activate_point_picker(self):
        """Activate the point picking tool to select facility locations. Adds a new memory layer if not already created."""
        if self.facility_layer is None:
            # Create a memory layer for facility points
            crs = QgsProject.instance().crs()
            self.facility_layer = QgsVectorLayer(
                f"Point?crs={crs.authid()}", "Facilities", "memory"
            )
            QgsProject.instance().addMapLayer(self.facility_layer)
            self.facility_layer_id = self.facility_layer.id()
            self.pointComboBox.setCurrentIndex(
                self.pointComboBox.findData(self.facility_layer_id)
            )

            # Apply black dot symbology
            symbol = QgsMarkerSymbol.createSimple(
                {
                    "name": "cross_fill",
                    "color": "black",
                    "size": "3",
                }
            )
            self.facility_layer.renderer().setSymbol(symbol)
            self.facility_layer.triggerRepaint()

        iface.mapCanvas().setMapTool(self.pointTool)

        iface.messageBar().pushMessage(
            "Instructions for picking multiple points",
            "Click to add a facility location. Repeat as needed. Delete 'Facilities' layer to start over.",
            level=Qgis.Info,
            duration=15,
        )

    def handle_point_picked(self, point):
        """Handle the event when a point is picked.
        Args:
            point (QgsPointXY): The picked point coordinates.
        """
        geom = QgsGeometry.fromPointXY(point)

        # Add the new point to the same memory layer
        provider = self.facility_layer.dataProvider()
        feat = QgsFeature()
        feat.setGeometry(geom)
        provider.addFeatures([feat])
        self.facility_layer.updateExtents()
        self.facility_layer.triggerRepaint()
        iface.mapCanvas().refresh()

    def log_to_textbox(self, message):
        """Log messages to the plain text edit box.
        Args:
            message (str): The message to log.
        """
        self.plainTextEdit.appendPlainText(str(message))

    def handle_results(self, result_dict):
        """Handle the results from the delivered cost analysis worker.
        Args:
            result_dict (dict): Dictionary containing layer names and their file paths.
        Raises:
            RuntimeError: If any raster layer fails to load or is invalid.
        """
        self.log_to_textbox("Delivered Cost Analysis completed successfully.")
        self.runButton.setEnabled(True)
        i = 2
        try:
            for name, dest_path in result_dict.items():
                layer = QgsRasterLayer(dest_path, name)
                layer.setCustomProperty("delivered_cost_plugin/temp", True)

                # Ensure the raster layer is valid
                if not layer.isValid():
                    raise RuntimeError("Raster layer failed to load.")

                # Trigger stats computation so QGIS knows actual min/max values
                provider = layer.dataProvider()
                stats = provider.bandStatistics(1, QgsRasterBandStats.All)
                min_val = stats.minimumValue
                max_val = stats.maximumValue

                # Now apply symbology AFTER stats are known
                if (
                    "delivered" in name.lower()
                    or "skidder" in name.lower()
                    or "cable" in name.lower()
                ):
                    if max_val > 1000:
                        self.log_to_textbox(
                            f"Applying capped symbology to {name} with max value 1000"
                        )
                        apply_capped_symbology(layer, cap_value=1000)
                    else:
                        self.log_to_textbox(
                            f"Applying uncapped symbology to {name} with max value {max_val}"
                        )
                        apply_capped_symbology(layer, cap_value=max_val)

                QgsProject.instance().addMapLayer(layer, addToLegend=False)
                QgsProject.instance().layerTreeRoot().insertLayer(i, layer)
                i += 1

        except Exception as e:
            self.log_to_textbox(f"Error adding layers to project: {str(e)}")

    def show_error(self, error_message):
        """Show an error message in the log textbox and a message box.
        Args:
            error_message (str): The error message to display.
        """
        self.log_to_textbox(f"Error: {error_message}")
        QMessageBox.critical(self, "Error", error_message)
        self.runButton.setEnabled(True)

    def run_delivered_cost(self):
        """Run the delivered cost analysis with the selected AOI and Facility layers."""
        self.plainTextEdit.clear()  # Clear previous log messages
        aoi_crs = self.get_selected_aoi_geometry()  # Get AOI geometry and CRS
        if aoi_crs is None:  # If AOI layer crs is None, use project CRS
            study_area_coords = qgs_to_coords_list_epsg4326(self.aoi_geometry)
        else:
            study_area_coords = qgs_to_coords_list_epsg4326(
                self.aoi_geometry, source_crs=aoi_crs
            )

        facility_crs = (
            self.get_selected_facility_coords()
        )  # Get Facility coordinates and CRS
        if facility_crs is None:  # If Facility layer crs is None, use project CRS
            saw_coords = [
                qgs_to_coords_list_epsg4326(QgsPointXY(pt[0], pt[1]))
                for pt in self.facility_coords
            ]
        else:
            saw_coords = [
                qgs_to_coords_list_epsg4326(
                    QgsPointXY(pt[0], pt[1]), source_crs=facility_crs
                )
                for pt in self.facility_coords
            ]
        if not self.facility_coords:  # If no facility points are selected
            QMessageBox.warning(
                self,
                "No Facility Point",
                "Please pick a facility point before running the analysis.",
            )
            return
        elif (
            not hasattr(self, "aoi_geometry") or self.aoi_geometry is None
        ):  # If AOI layer is not defined
            QMessageBox.warning(
                self,
                "No AOI Polygon",
                "Please draw an area of interest polygon before running the analysis.",
            )
            return
        if self.roadsComboBox.currentData() is not None:  # If roads layer is selected
            layer = QgsProject.instance().mapLayer(self.roadsComboBox.currentData())
            if layer_correct_fields(layer):
                self.lyr_roads_path = layer.source()
            else:
                QMessageBox.warning(
                    self,
                    "Invalid Roads Layer",
                    "The selected roads layer does not have a 'highway' field. The 'highway' field must be filled with one of the following: motorway, trunk, primary, secondary, tertiary, unclassified, residential. Please select a valid roads layer.",
                )
                return
        else:
            self.lyr_roads_path = None
        if (
            self.barriersComboBox.currentData() is not None
        ):  # If barriers layer is selected
            layer = QgsProject.instance().mapLayer(self.barriersComboBox.currentData())
            self.lyr_barriers_path = layer.source()
        else:
            self.lyr_barriers_path = None

        # Get values from spinboxes
        tr_s = self.rtSpdSpinBox.value()
        cb_s = self.skylineSpdSpinBox.value()

        tr_d = self.rtSkidderMonSpinBox.value()
        cb_d = self.skylineMonSpinBox.value()
        fb_d = self.fellerbunchRateSpinBox.value()
        hf_d = self.handfellingRateSpinBox.value()
        pr_d = self.processingSpinBox.value()
        ha_d = self.haulingSpinBox.value()
        ht_d = self.handTreatmentSpinBox.value()
        pf_d = self.prescribedFireSpinBox.value()

        tr_p = self.rtSkidderPayloadSpinBox.value()
        cb_p = self.skylinePayloadSpinBox.value()
        lt_p = self.logTruckPayloadSpinBox.value()

        cb_o = self.optionalSurfacesCheckBox.isChecked()
        args = {
            "study_area_coords": study_area_coords,
            "saw_coords": saw_coords,
            "lyr_roads_path": self.lyr_roads_path,
            "lyr_barriers_path": self.lyr_barriers_path,
            "sk_r": tr_s,
            "cb_r": cb_s,
            "sk_d": tr_d,
            "cb_d": cb_d,
            "fb_d": fb_d,
            "hf_d": hf_d,
            "pr_d": pr_d,
            "lt_d": ha_d,
            "ht_d": ht_d,
            "pf_d": pf_d,
            "sk_p": tr_p,
            "cb_p": cb_p,
            "lt_p": lt_p,
            "cb_o": cb_o,
        }
        self.runButton.setEnabled(False)  # Disable button to prevent multiple clicks
        self.log_to_textbox("Starting Delivered Cost Analysis...")
        try:
            from .workers import DeliveredCostWorker

            worker = DeliveredCostWorker(args)
            worker.signals.log.connect(self.log_to_textbox)
            worker.signals.progress.connect(self.progressBar.setValue)
            worker.signals.finished.connect(self.handle_results)
            worker.signals.error.connect(self.show_error)
            self.threadpool.start(worker)
            # Reset state after starting the worker
            self.facility_layer_id = None
            self.facility_coords = []
            self.facility_layer = None
            self.aoi_layer_id = None
            self.aoi_geometry = None
        except Exception as e:
            self.log_to_textbox(f"Error initializing worker: {str(e)}")
            self.runButton.setEnabled(True)

    def closeEvent(self, event):
        """Handle the close event of the dock widget.
        Args:
            event (QCloseEvent): The close event.
        """
        self.facility_coords = []  # Reset facility coordinates
        self.aoi_geometry = None  # Reset AOI geometry
        if (
            hasattr(self, "osm_layer_id") and self.osm_layer_id
        ):  # Remove OSM layer if it exists
            layer = QgsProject.instance().mapLayer(self.osm_layer_id)
            if layer:
                QgsProject.instance().removeMapLayer(layer)
            self.osm_layer_id = None
        # if (
        #     hasattr(self, "aoi_layer_id") and self.aoi_layer_id
        # ):  # Remove AOI layer if it exists
        #     layer = QgsProject.instance().mapLayer(self.aoi_layer_id)
        #     if layer:
        #         QgsProject.instance().removeMapLayer(layer)
        #     self.aoi_layer_id = None

        # if (
        #     hasattr(self, "facility_layer_id") and self.facility_layer_id
        # ):  # Remove Facility layer if it exists
        #     layer = QgsProject.instance().mapLayer(self.facility_layer_id)
        #     if layer:
        #         QgsProject.instance().removeMapLayer(layer)
        #     self.facility_layer_id = None

        if (
            hasattr(self, "draw_polygon_tool") and self.draw_polygon_tool
        ):  # Deactivate draw tool if it exists
            self.draw_polygon_tool.deactivate()

        if (
            hasattr(self, "pointTool") and self.pointTool
        ):  # Deactivate point tool if it exists
            self.pointTool.deactivate()

        # Switch back to pan tool explicitly
        pan_tool = QgsMapToolPan(iface.mapCanvas())
        iface.mapCanvas().setMapTool(pan_tool)

        # Remove all background layers
        for config in self.layer_configs.values():
            layer_id = config.get("layer_id")
            if layer_id:
                layer = QgsProject.instance().mapLayer(layer_id)
                if layer:
                    QgsProject.instance().removeMapLayer(layer)
                    iface.mapCanvas().refresh()
                config["layer_id"] = None
        # Uncheck all checkboxes
        for checkbox in self.layer_configs.keys():
            checkbox.setChecked(False)

        self.closingPlugin.emit()
        event.accept()

    def showEvent(self, event):
        """Override showEvent to add OSM layer when the dock widget is shown."""
        if getattr(self, "osm_layer_id", None) is None:
            self.add_osm_basemap()
            QTimer.singleShot(
                100, self.zoom_to_us_extent_3857
            )  # Zoom to U.S. extent after adding OSM layer
        super(DeliveredCostDockWidget, self).showEvent(event)


def qgs_to_coords_list_epsg4326(geom, source_crs=None):
    """Convert a QgsGeometry or QgsPointXY to a list of coordinates in EPSG:4326.
    Args:
        geom (QgsGeometry or QgsPointXY): The geometry to convert.
        source_crs (QgsCoordinateReferenceSystem, optional): The source CRS of the geometry. If None, uses project CRS.
    Returns:
        list: A list of tuples representing coordinates in EPSG:4326.
    Raises:
        ValueError: If the geometry type is unsupported.
    """
    # Get source CRS (usually project CRS)
    if source_crs is None:
        src_crs = QgsProject.instance().crs()
    else:
        src_crs = source_crs
    dest_crs = QgsCoordinateReferenceSystem("EPSG:4326")

    # Setup coordinate transformer
    transform = QgsCoordinateTransform(src_crs, dest_crs, QgsProject.instance())

    # If input is QgsPointXY, convert to QgsGeometry first
    if isinstance(geom, QgsPointXY):
        geom = QgsGeometry.fromPointXY(geom)

    # Transform geometry to EPSG:4326
    if src_crs != dest_crs:
        geom.transform(transform)

    # Convert to shapely geometry via GeoJSON dict
    import json
    from shapely.geometry import shape, Point, Polygon, MultiPolygon

    geojson_dict = json.loads(geom.asJson())
    shapely_geom = shape(geojson_dict)

    if isinstance(shapely_geom, Point):
        return [(shapely_geom.x, shapely_geom.y)]
    elif isinstance(shapely_geom, Polygon):
        return list(shapely_geom.exterior.coords)
    elif isinstance(shapely_geom, MultiPolygon):
        # Take only the first polygon for simplicity
        return list(shapely_geom.geoms[0].exterior.coords)
    else:
        raise ValueError(f"Unsupported geometry type: {type(shapely_geom)}")


def apply_capped_symbology(raster_layer, cap_value=1000):
    """Apply symbology to a raster layer with capped values.
    Args:
        raster_layer (QgsRasterLayer): The raster layer to apply symbology to.
        cap_value (float): The maximum value for the color ramp.
    """
    stats = raster_layer.dataProvider().bandStatistics(1)
    actual_min = stats.minimumValue
    symbology_min = actual_min
    symbology_max = cap_value

    # Define color ramp
    ramp_shader = QgsColorRampShader()
    ramp_shader.setColorRampType(QgsColorRampShader.Interpolated)
    ramp_shader.setColorRampItemList(
        [
            QgsColorRampShader.ColorRampItem(
                symbology_min, QColor("red"), str(symbology_min)
            ),
            QgsColorRampShader.ColorRampItem(
                symbology_max * 0.5, QColor("yellow"), str(symbology_max * 0.5)
            ),
            QgsColorRampShader.ColorRampItem(
                symbology_max, QColor("green"), str(f">{symbology_max}")
            ),
        ]
    )
    ramp_shader.setMinimumValue(symbology_min)
    ramp_shader.setMaximumValue(symbology_max)

    # Build shader and renderer
    shader = QgsRasterShader()
    shader.setRasterShaderFunction(ramp_shader)

    renderer = QgsSingleBandPseudoColorRenderer(raster_layer.dataProvider(), 1, shader)

    # Apply renderer
    raster_layer.setRenderer(renderer)

    # Hint to QGIS to use user-defined contrast (sometimes helps retain settings)
    raster_layer.setCustomProperty("contrastEnhancementMinMax", "User")

    raster_layer.triggerRepaint()


def layer_correct_fields(layer: QgsVectorLayer) -> bool:
    """Check if the given layer has the required 'highway' field.
    Args:
        layer (QgsVectorLayer): The layer to check.
    Returns:
        bool: True if the layer has a 'highway' field, False otherwise.
    """
    fields = [f.name().lower() for f in layer.fields()]
    if "highway" in fields:
        return True
    return False
