import json

from qgis.core import Qgis, QgsApplication
from qgis.PyQt.Qt import QUrl
from qgis.PyQt.QtGui import QFont, QIcon
from qgis.PyQt.QtWidgets import QTreeWidget, QTreeWidgetItem

from qwc2_tools.toolbelt import PlgLogger
from qwc2_tools.toolbelt.network_manager import NetworkRequestsManager
from qwc2_tools.toolbelt.preferences import PlgOptionsManager

FILE = 1001
FOLDER = 1002


class QWC2TreeWidget(QTreeWidget):
    def __init__(self, parent=None):
        """Dialog to load list and load an existing project into QWC2"""
        # init module and ui
        super().__init__(parent)
        self.network_manager = NetworkRequestsManager()
        self.list_project = []
        self.refresh()

    def update_projects_list(self) -> list[str]:
        """Returns the list of projects for the QWC2 instance

        :return: List of projets
        :rtype: list
        """
        url_publish = PlgOptionsManager.get_plg_settings().url_publish
        try:
            req_result = self.network_manager.get_url(
                url=QUrl(f"{url_publish}/listprojectdir")
            )
        except:
            PlgLogger.log(
                message=self.tr(
                    "Impossible to connect QWC2 instance, please check settings"
                ),
                log_level=Qgis.Critical,
                duration=10,
                push=True,
            )
            return
        try:
            self.list_project = json.loads(req_result.data().decode())
        except Exception:
            PlgLogger.log(
                message=self.tr(
                    "Impossible to connect QWC2 instance, please check URL"
                ),
                log_level=Qgis.Critical,
                duration=10,
                push=True,
            )
            self.list_project = []

    def build_folder_recursively(self, folder) -> QTreeWidgetItem:
        """Recursive function that populates the tree from the json returned by the api

        :param folder: folder path
        :type folder: str
        :return: New item
        :rtype: QTreeWidgetItem
        """
        item = QTreeWidgetItem([folder["name"]], type=FOLDER)
        item.setFont(0, QFont("Helvetica", 10, QFont.Bold))
        item.setIcon(0, QIcon(QgsApplication.iconPath("mIconFolder24.svg")))
        for subfolder in folder["folders"]:
            item.addChild(self.build_folder_recursively(subfolder))
        for projet in folder["projects"]:
            child = QTreeWidgetItem([projet, folder["name"]], type=FILE)
            child.setIcon(0, QIcon(QgsApplication.iconPath("mIconQgsProjectFile.svg")))
            item.addChild(child)
        return item

    def refresh(self) -> None:
        """Create or re-create the tree from the list of returned projects."""
        self.update_projects_list()
        self.clear()

        if len(self.list_project) == 0:
            return

        item = QTreeWidgetItem([""], type=FOLDER)

        item.setFont(0, QFont("Helvetica", 10, QFont.Bold))
        item.setIcon(0, QIcon(QgsApplication.iconPath("mIconFolder24.svg")))

        for dossier_racine in self.list_project["folders"]:
            item.addChild(self.build_folder_recursively(dossier_racine))
        self.insertTopLevelItems(0, [item])
        item.setExpanded(True)

        for fichier_racine in self.list_project["projects"]:
            root_item = QTreeWidgetItem([fichier_racine], type=FILE)
            root_item.setIcon(
                0, QIcon(QgsApplication.iconPath("mIconQgsProjectFile.svg"))
            )
            item.addChild(root_item)

    @property
    def selected_item_path(self) -> str:
        """Return to project path

        :return: Project path
        :rtype: str
        """
        selected_item = self.selected_item
        if selected_item:
            path = [selected_item.text(0)]
            parent = selected_item.parent()
            while parent:
                path.insert(0, parent.text(0))
                parent = parent.parent()
            return "/".join(path).lstrip("/")

        return None

    @property
    def selected_item(self) -> str:
        """Return to project path

        :return: Project path
        :rtype: str
        """
        selected_items = self.selectedItems()
        if selected_items:
            selected_item = selected_items[0]

            return selected_item

        return None

    def expand_folder(self, folder: QTreeWidgetItem, recursive: bool = True) -> None:
        """expand all the folders in QTreeWidget

        :param item: Tree Widget target
        :type item: QTreeWidget
        """
        folder.setExpanded(True)
        if recursive:
            for i in range(folder.childCount()):
                child_item = folder.child(i)
                self.expand_folder(child_item)

    def expand_all(self) -> None:
        """Browse all QTreeWidget folders and apply the extend method"""
        for i in range(self.topLevelItemCount()):
            root_item = self.topLevelItem(i)
            self.expand_folder(root_item)
