from datetime import datetime
from pathlib import Path
from tempfile import TemporaryDirectory

from qgis.core import QgsApplication, QgsProject
from qgis.PyQt import uic
from qgis.PyQt.Qt import (
    QHBoxLayout,
    QIcon,
    QLabel,
    QLineEdit,
    QMessageBox,
    QPushButton,
    QUrl,
    QVBoxLayout,
)
from qgis.PyQt.QtCore import QByteArray
from qgis.PyQt.QtWidgets import QDialog

from qwc2_tools.gui.qwc2_treewidget import FILE, FOLDER
from qwc2_tools.toolbelt import PlgLogger
from qwc2_tools.toolbelt.network_manager import NetworkRequestsManager
from qwc2_tools.toolbelt.preferences import PlgOptionsManager


class ProjectsManager(QDialog):
    def __init__(self, parent=None):
        """Dialog to load list and load an existing project into QWC2"""
        # init module and ui
        super().__init__(parent)
        uic.loadUi(Path(__file__).parent / f"{Path(__file__).stem}.ui", self)
        self.network_manager = NetworkRequestsManager()

        self.log = PlgLogger().log

        # Buttons icons
        self.dlg_btn_open_project.setIcon(
            QgsApplication.getThemeIcon("mAddToProject.svg")
        )
        self.dlg_btn_delete_project.setIcon(
            QgsApplication.getThemeIcon("mIconDelete.svg")
        )
        self.dlg_btn_refresh_project.setIcon(
            QgsApplication.getThemeIcon("mActionRefresh.svg")
        )
        self.dlg_btn_create_folder.setIcon(
            QgsApplication.getThemeIcon("mIconFolderOpen.svg")
        )
        self.dlg_btn_expand.setIcon(
            QgsApplication.getThemeIcon("mActionExpandNewTree.svg")
        )

        # Connect buttons
        self.dlg_btn_refresh_project.clicked.connect(self.refresh)
        self.dlg_btn_open_project.clicked.connect(self.open_project)
        self.dlg_btn_delete_project.clicked.connect(self.delete_project)
        self.dlg_btn_expand.clicked.connect(self.the_tree.expand_all)
        self.dlg_btn_create_folder.clicked.connect(self.create_folder)
        self.the_tree.itemSelectionChanged.connect(self.unlock_update_button)

        # initial behaviour
        self.dlg_btn_open_project.setEnabled(False)
        self.dlg_btn_create_folder.setEnabled(False)
        self.dlg_btn_delete_project.setEnabled(False)

    def unlock_update_button(self):
        """Check if a project is selected in the tree,
        if a project is selected it's possible to update this project so
        the button is unlock
        """
        if self.the_tree.selected_item:
            # Root is selected
            if (
                self.the_tree.selected_item.type() == FOLDER
                and self.the_tree.selected_item_path == ""
            ):
                self.dlg_btn_open_project.setEnabled(False)
                self.dlg_btn_create_folder.setEnabled(True)
                self.dlg_btn_delete_project.setEnabled(False)

            else:
                self.dlg_btn_open_project.setEnabled(False)
                self.dlg_btn_create_folder.setEnabled(True)
                self.dlg_btn_delete_project.setEnabled(True)

            if self.the_tree.selected_item.type() == FILE:
                self.dlg_btn_create_folder.setEnabled(False)
                self.dlg_btn_open_project.setEnabled(True)
                self.dlg_btn_delete_project.setEnabled(True)

    def refresh(self) -> None:
        """Refresh the tree"""
        self.the_tree.refresh()

        # initial behaviour
        self.dlg_btn_open_project.setEnabled(False)
        self.dlg_btn_create_folder.setEnabled(False)
        self.dlg_btn_delete_project.setEnabled(False)

    def open_project(self) -> None:
        """Open the selected project in the current canvas"""
        if not self.the_tree.selected_item_path:
            return
        url_publish = PlgOptionsManager.get_plg_settings().url_publish
        url = f"{url_publish}/project/{self.the_tree.selected_item_path}"
        req_result = self.network_manager.get_url(url=QUrl(url))

        # QGS Temporary file
        tmpdir = TemporaryDirectory(prefix="qwc2_tools_")
        qgs_path = Path(tmpdir.name).joinpath(
            f"projet_{datetime.now().strftime('%Y%m%d%H%M%S')}.qgs"
        )

        # Write xml project code to file
        qgs_path.write_text(req_result.data().decode())

        project = QgsProject.instance()
        project.read(str(qgs_path))

    def delete_project(self) -> None:
        """Delete the selected project"""

        url_publish = PlgOptionsManager.get_plg_settings().url_publish

        if not self.the_tree.selected_item_path:
            return

        if self.the_tree.selected_item.type() == FILE:
            url = f"{url_publish}/project/{self.the_tree.selected_item_path}"
            message = self.tr(
                "Are you sure you want to drop the " "project {} ?"
            ).format(self.the_tree.selected_item_path)

        elif self.the_tree.selected_item.type() == FOLDER:
            url = f"{url_publish}/folder/{self.the_tree.selected_item_path}"
            message = self.tr(
                "Are you sure you want to delete the folder {} "
                "and all the projects and subfolders it contains ?"
            ).format(self.the_tree.selected_item_path)

        response = QMessageBox().warning(
            self,
            self.tr("Suppression"),
            message,
            QMessageBox.Ok | QMessageBox.Cancel,
            QMessageBox.Cancel,
        )

        if response == QMessageBox.Ok:
            self.network_manager.delete_url(url=QUrl(url))
            self.the_tree.refresh()

        else:
            pass

    def create_folder(self) -> None:
        """Create a new folder in the tree. Display IHM to pout the new folder name"""

        url_publish = PlgOptionsManager.get_plg_settings().url_publish

        if self.the_tree.selected_item_path:
            if self.the_tree.selected_item.type() == FOLDER:
                base_folder = self.the_tree.selected_item_path + "/"
            else:
                base_folder = "/".join(self.the_tree.selected_item_path.split("/")[:-1])
        else:
            base_folder = ""

        folder_dialog = QDialog()
        folder_dialog.setWindowTitle(self.tr("Create Folder"))
        layout = QVBoxLayout()
        horizontal_layout = QHBoxLayout()
        label = QLabel(base_folder)
        horizontal_layout.addWidget(label)
        folder_name_input = QLineEdit()
        folder_name_input.setMinimumWidth(100)
        horizontal_layout.addWidget(folder_name_input)
        layout.addLayout(horizontal_layout)
        horizontal_layout2 = QHBoxLayout()
        confirm_button = QPushButton(self.tr("Create"))
        confirm_button.setIcon(QgsApplication.getThemeIcon("mActionNewFolder.svg"))
        confirm_button.clicked.connect(folder_dialog.accept)
        horizontal_layout2.addWidget(confirm_button)
        cancel_button = QPushButton(self.tr("Cancel"))
        cancel_button.setIcon(QgsApplication.getThemeIcon("mTaskCancel.svg"))
        cancel_button.clicked.connect(folder_dialog.reject)
        horizontal_layout2.addWidget(cancel_button)
        layout.addLayout(horizontal_layout2)
        folder_dialog.setLayout(layout)
        result = folder_dialog.exec_()

        if result == QDialog.Accepted:
            new_folder = base_folder + folder_name_input.text()
            data = QByteArray()
            data.append(f"name={new_folder}")
            url = f"{url_publish}/folder/"
            self.network_manager.post_url(url=QUrl(url), data=data)
            self.the_tree.refresh()
