import json
import mimetypes
from codecs import encode
from datetime import datetime
from pathlib import Path

from qgis.core import QgsApplication, QgsProject
from qgis.PyQt import uic
from qgis.PyQt.Qt import (
    QFile,
    QHBoxLayout,
    QLabel,
    QLineEdit,
    QMessageBox,
    QPushButton,
    QUrl,
    QVBoxLayout,
)
from qgis.PyQt.QtCore import QFileInfo, QIODevice, QTemporaryDir
from qgis.PyQt.QtWidgets import QDialog

from qwc2_tools.toolbelt.log_handler import PlgLogger
from qwc2_tools.toolbelt.network_manager import NetworkRequestsManager
from qwc2_tools.toolbelt.preferences import PlgOptionsManager

FILE = 1001
FOLDER = 1002


class PublishProject(QDialog):
    def __init__(self, parent=None):
        """Dialog to load list and load an existing project into QWC2"""
        # init module and ui
        super().__init__(parent)
        uic.loadUi(Path(__file__).parent / f"{Path(__file__).stem}.ui", self)
        self.network_manager = NetworkRequestsManager()
        self.log = PlgLogger().log

        # Connect buttons
        self.dlg_btn_refresh.clicked.connect(self.refresh)
        self.dlg_btn_new.clicked.connect(self.publish_new_project)
        self.dlg_btn_update.clicked.connect(self.update_project)
        self.the_tree.itemSelectionChanged.connect(self.unlock_update_button)
        self.dlg_btn_expand.clicked.connect(self.the_tree.expand_all)

        # Buttons icons
        self.dlg_btn_new.setIcon(QgsApplication.getThemeIcon("mIconFolderProject.svg"))
        self.dlg_btn_update.setIcon(QgsApplication.getThemeIcon("mAddToProject.svg"))
        self.dlg_btn_refresh.setIcon(QgsApplication.getThemeIcon("mActionRefresh.svg"))
        self.dlg_btn_expand.setIcon(
            QgsApplication.getThemeIcon("mActionExpandNewTree.svg")
        )

        # Get project extension
        url_publish = PlgOptionsManager.get_plg_settings().url_publish
        response = self.network_manager.get_url(
            url=QUrl(f"{url_publish}/getprojecttype")
        )
        self.extension = "." + json.loads(response.data().decode("utf-8"))["type"]

    def refresh(self) -> None:
        """Refresh the tree"""
        self.the_tree.refresh()

    def publish_new_project(self):
        """Upload the current project as a new project in the QWC2 instance. A popup is
        displayed in which you enter the name you want to give to the project
        """
        tmpdir = QTemporaryDir()
        tmpdir.setAutoRemove(False)
        qgs_path = Path(tmpdir.path()).joinpath(
            f"projet_{datetime.now().strftime('%Y%m%d%H%M%S')}{self.extension}"
        )
        project = QgsProject.instance()
        project.write(str(qgs_path))

        if self.the_tree.selected_item:
            if self.the_tree.selected_item.type() == FOLDER:
                base_folder = self.the_tree.selected_item_path + "/"
            else:
                base_folder = "/".join(self.the_tree.selected_item_path.split("/")[:-1])
        else:
            base_folder = ""

        folder_dialog = QDialog()
        folder_dialog.setWindowTitle(self.tr("Publish new project"))
        layout = QVBoxLayout()
        horizontal_layout = QHBoxLayout()
        label = QLabel(base_folder)
        horizontal_layout.addWidget(label)
        project_name_input = QLineEdit()
        project_name_input.setMinimumWidth(100)
        horizontal_layout.addWidget(project_name_input)
        layout.addLayout(horizontal_layout)
        horizontal_layout2 = QHBoxLayout()
        confirm_button = QPushButton(self.tr("Publish"))
        confirm_button.setIcon(QgsApplication.getThemeIcon("mAddToProject.svg"))
        confirm_button.clicked.connect(folder_dialog.accept)
        horizontal_layout2.addWidget(confirm_button)
        cancel_button = QPushButton(self.tr("Cancel"))
        cancel_button.setIcon(QgsApplication.getThemeIcon("mTaskCancel.svg"))
        cancel_button.clicked.connect(folder_dialog.reject)
        horizontal_layout2.addWidget(cancel_button)
        layout.addLayout(horizontal_layout2)
        folder_dialog.setLayout(layout)
        result = folder_dialog.exec_()

        if result == QDialog.Accepted:
            if project_name_input.text().endswith(self.extension):
                new_project = base_folder + project_name_input.text()
            else:
                new_project = base_folder + project_name_input.text() + self.extension

            fp = QFile(str(qgs_path))
            fp.open(QIODevice.ReadOnly)

            body_list = []
            boundary = "wL36Yn8afVp8Ag7AmP8qZ0SA4n1v9T"
            body_list.append(encode("--" + boundary))
            body_list.append(encode("Content-Disposition: form-data; name=filename;"))
            body_list.append(encode("Content-Type: {}".format("text/plain")))
            body_list.append(encode(""))
            body_list.append(encode(new_project))
            body_list.append(encode("--" + boundary))
            body_list.append(
                encode(
                    "Content-Disposition: form-data; name=file; filename={0}".format(
                        QFileInfo(fp).fileName()
                    )
                )
            )
            fileType = (
                mimetypes.guess_type(str(qgs_path))[0] or "application/octet-stream"
            )
            body_list.append(encode("Content-Type: {}".format(fileType)))
            body_list.append(encode(""))
            body_list.append(fp.readAll())
            body_list.append(encode("--" + boundary + "--"))
            body_list.append(encode(""))
            body = b"\r\n".join(body_list)

            headers = {
                b"Content-type": bytes(
                    "multipart/form-data; boundary={}".format(boundary), "utf8"
                ),
            }
            url_publish = PlgOptionsManager.get_plg_settings().url_publish
            url = f"{url_publish}/project/"
            self.network_manager.post_url(url=QUrl(url), data=body, headers=headers)
            self.the_tree.refresh()

            self.log(
                message=self.tr(
                    "The new project has been successfully published: {}"
                ).format(new_project),
                log_level=3,
                push=True,
                duration=15,
            )
            self.close()

        tmpdir.remove()

    def unlock_update_button(self):
        """Check if a project is selected in the tree,
        if a project is selected it's possible to update this project so
        the button is unlock
        """
        if self.the_tree.selected_item_path:
            if self.the_tree.selected_item_path.endswith(self.extension):
                self.dlg_btn_update.setEnabled(True)

        if self.the_tree.selected_item_path:
            if not self.the_tree.selected_item_path.endswith(self.extension):
                self.dlg_btn_update.setEnabled(False)

    def update_project(self):
        """Update an existing project in QWC2 instance with the current project"""
        tmpdir = QTemporaryDir()
        tmpdir.setAutoRemove(False)
        qgs_path = Path(tmpdir.path()).joinpath(
            f"projet_{datetime.now().strftime('%Y%m%d%H%M%S')}{self.extension}"
        )
        project = QgsProject.instance()
        project.write(str(qgs_path))

        update_project = self.the_tree.selected_item_path

        message = self.tr(
            "Are you sure you're replacing project {} with the current "
            "project content ?"
        ).format(update_project)

        response = QMessageBox().warning(
            self,
            self.tr("Update project"),
            message,
            QMessageBox.Ok | QMessageBox.Cancel,
            QMessageBox.Cancel,
        )

        if response == QMessageBox.Ok:
            fp = QFile(str(qgs_path))
            fp.open(QIODevice.ReadOnly)
            body_list = []
            boundary = "wL36Yn8afVp8Ag7AmP8qZ0SA4n1v9T"
            body_list.append(encode("--" + boundary))
            body_list.append(
                encode(
                    "Content-Disposition: form-data; name=file; filename={0}".format(
                        QFileInfo(fp).fileName()
                    )
                )
            )
            fileType = (
                mimetypes.guess_type(str(qgs_path))[0] or "application/octet-stream"
            )
            body_list.append(encode("Content-Type: {}".format(fileType)))
            body_list.append(encode(""))
            body_list.append(fp.readAll())
            body_list.append(encode("--" + boundary + "--"))
            body_list.append(encode(""))
            body = b"\r\n".join(body_list)
            headers = {
                b"Content-type": bytes(
                    "multipart/form-data; boundary={}".format(boundary), "utf8"
                ),
            }
            url_publish = PlgOptionsManager.get_plg_settings().url_publish
            url = f"{url_publish}/project/{update_project}"
            self.network_manager.put_url(url=QUrl(url), data=body, headers=headers)
            self.the_tree.refresh()

            self.log(
                message=self.tr("The project {} has been successfully updated").format(
                    update_project
                ),
                log_level=3,
                push=True,
                duration=15,
            )
            self.close()

        tmpdir.remove()
