# TabbedConsole.py
import os
from qgis.PyQt.QtWidgets import QDialog, QVBoxLayout, QHBoxLayout, QListWidget, QStackedWidget, QWidget, QListWidgetItem
from qgis.PyQt.QtCore import Qt
from qgis.PyQt.QtGui import QIcon

# Use your existing dialogs (pure Python)
from .ConfigDialog import ConfigDialog
from .CreateDialog import CreateDialog
from .UploadDialog import UploadDialog
from .PublishDialog import PublishDialog

class QWC2Console(QDialog):
    """
    Single tabbed window that embeds your existing dialogs:
      - Configure Access
      - Create Tenant
      - Update Files
      - Publish Map
    """
    def __init__(self, config, parent=None):
        super().__init__(parent)
        self.setWindowTitle("QWC2 Console")
        self.resize(900, 650)
        self.config = config
        self.save_config_callback = None

        # Main horizontal layout
        main_layout = QHBoxLayout(self)
        main_layout.setContentsMargins(0, 0, 0, 0)
        
        # Left side: List widget acting as tabs
        self.tab_list = QListWidget()
        self.tab_list.setMaximumWidth(150)
        self.tab_list.setMinimumWidth(150)
        self.tab_list.setStyleSheet("""
            QListWidget {
                background-color: #5D5D5D;
                border: none;
                outline: none;
		color: white;
            }
            QListWidget::item {
                padding: 12px 8px;
                border-bottom: none;
            }
            QListWidget::item:selected {
                background-color: #36454F;
                color: yellow;
            }
            QListWidget::item:hover {
                background-color: #36454F;
		color: white;
            }
        """)
        
        # Right side: Stacked widget for content
        self.content_stack = QStackedWidget()
        
        # Add to main layout
        main_layout.addWidget(self.tab_list)
        main_layout.addWidget(self.content_stack, 1)  # stretch factor 1
        
        # Create tab items and dialogs
        self.dialogs = {}
        self.setup_tabs(config)
        
        # Connect selection change
        self.tab_list.currentRowChanged.connect(self.content_stack.setCurrentIndex)
    
    def set_save_config_callback(self, callback):
        """Set the callback function to save the config"""
        self.save_config_callback = callback
    
    def save_config(self):
        """Save the config using the callback"""
        if self.save_config_callback:
            self.save_config_callback(self.config)

    def setup_tabs(self, config):
        """Setup the tab list and dialogs"""
        # Get the plugin directory for icons
        plugin_dir = os.path.dirname(__file__)
        icon_dir = os.path.join(plugin_dir, 'icons')
        
        tab_items = [
            ("Configure", ConfigDialog, "configure.png"),
            ("Create Tenant", CreateDialog, "create.png"), 
            ("Upload", UploadDialog, "update.png"),
            ("Publish Map", PublishDialog, "publish.png"),
        ]
        
        for i, (title, dialog_cls, icon_name) in enumerate(tab_items):
            # Create icon
            icon_path = os.path.join(icon_dir, icon_name)
            icon = QIcon(icon_path)
            
            # Add to list widget with icon
            item = QListWidgetItem(icon, title)
            self.tab_list.addItem(item)
            
            # Create dialog
            dlg = dialog_cls(config)
            dlg.setWindowFlags(Qt.Widget)
            # Neutralize accept/reject so inner dialogs don't close the whole console
            dlg.accept = lambda *a, **k: None
            dlg.reject = lambda *a, **k: None
            
            # Pass console reference to all dialogs that need it
            if hasattr(dlg, 'set_console_ref'):
                dlg.set_console_ref(self)
            
            # Pass save callback to ConfigDialog
            if title == "Configure" and hasattr(dlg, 'set_save_callback'):
                dlg.set_save_callback(self.save_config)
                # Also pass callback to refresh server lists in other tabs
                if hasattr(dlg, 'set_refresh_callback'):
                    dlg.set_refresh_callback(self.refresh_server_lists)
            
            # Create container widget
            container = QWidget()
            layout = QVBoxLayout(container)
            layout.setContentsMargins(0, 0, 0, 0)
            layout.setSpacing(0)
            layout.addWidget(dlg)
            layout.addStretch()  # Push content to top
            
            # Add to stacked widget
            self.content_stack.addWidget(container)
            self.dialogs[title] = dlg
        
        # Select first tab
        self.tab_list.setCurrentRow(0)
    
    def refresh_tenant_lists(self):
        """Refresh tenant dropdowns in all dialogs that have them"""
        # Refresh PublishDialog tenant list
        if "Publish Map" in self.dialogs:
            publish_dlg = self.dialogs["Publish Map"]
            if hasattr(publish_dlg, 'updateTenants'):
                publish_dlg.updateTenants()
        
        # Refresh UploadDialog tenant list
        if "Upload" in self.dialogs:
            upload_dlg = self.dialogs["Upload"]
            if hasattr(upload_dlg, 'updateTenants'):
                upload_dlg.updateTenants()
    
    def refresh_server_lists(self):
        """Refresh server dropdowns in all dialogs that have them"""
        # Refresh CreateDialog server list
        if "Create Tenant" in self.dialogs:
            create_dlg = self.dialogs["Create Tenant"]
            if hasattr(create_dlg, 'refresh_server_list'):
                create_dlg.refresh_server_list()
        
        # Refresh UploadDialog server list
        if "Upload" in self.dialogs:
            upload_dlg = self.dialogs["Upload"]
            if hasattr(upload_dlg, 'refresh_server_list'):
                upload_dlg.refresh_server_list()
        
        # Refresh PublishDialog server list
        if "Publish Map" in self.dialogs:
            publish_dlg = self.dialogs["Publish Map"]
            if hasattr(publish_dlg, 'refresh_server_list'):
                publish_dlg.refresh_server_list()
    
    def refresh_theme_lists(self):
        """Refresh theme dropdowns in all dialogs that have them"""
        # Refresh PublishDialog theme list
        if "Publish Map" in self.dialogs:
            publish_dlg = self.dialogs["Publish Map"]
            if hasattr(publish_dlg, 'updateThemes'):
                publish_dlg.updateThemes()
