# -*- coding: utf-8 -*-
"""
/***************************************************************************
 QuickLabeling
                                 A QGIS plugin
 QuickLabeling - Plugin para etiquetado rápido de features
 Generated by Plugin Builder
                              -------------------
        begin                : 2025-06-03
        copyright            : (C) 2025 by Eduardo A
        email                : eduardoaqgis@gmail.com
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QCoreApplication, Qt
from qgis.PyQt.QtWidgets import QAction, QCheckBox
from qgis.PyQt.QtGui import QIcon
from qgis.gui import QgsMapToolIdentifyFeature
from qgis.core import QgsProject, edit, QgsMessageLog, Qgis, QgsFeatureRequest, QgsRectangle, QgsGeometry
from qgis.gui import QgsHighlight
from qgis.PyQt.QtGui import QColor
import os

# Importar nuestro widget personalizado
from .CounterWidget import CounterWidget

class IdentifyAndLabel(QgsMapToolIdentifyFeature):
    """Herramienta para identificar y etiquetar features haciendo clic en el mapa"""
    
    def __init__(self, iface, layer, field, prefix, start_value, counter_widget):
        super().__init__(iface.mapCanvas())
        self.iface = iface
        self.layer = layer
        self.field = field
        self.prefix = prefix
        self.counter = start_value
        self.counter_widget = counter_widget
        
        # Para el resaltado
        self.highlight = None      
        
        # Configurar la herramienta para trabajar solo con la capa seleccionada
        self.setLayer(layer)
    
    def clear_highlight(self):
        """Limpiar resaltado de forma segura"""
        if self.highlight:
            try:
                self.highlight.hide()
                self.highlight = None
            except:
                self.highlight = None

    def canvasMoveEvent(self, event):
        """Resaltar feature al pasar el mouse"""
        try:
            # Verificar si el resaltado está habilitado
            if not self.counter_widget.is_highlight_enabled():
                self.clear_highlight()
                return
            
            # Limpiar resaltado anterior
            self.clear_highlight()
            
            # Identificar features en la posición del mouse
            results = self.identify(event.x(), event.y(), [self.layer], self.TopDownStopAtFirst)
            
            if results:
                # Obtener la primera feature
                feat = results[0].mFeature
                
                # Crear el resaltado (sin reimportar las librerías)
                self.highlight = QgsHighlight(self.iface.mapCanvas(), feat, self.layer)
                self.highlight.setColor(QColor(255, 255, 0, 150))  # Amarillo translúcido
                self.highlight.setWidth(3)
                self.highlight.show()
                
        except Exception as e:
            self.clear_highlight()
            QgsMessageLog.logMessage(
                f"Error en resaltado: {str(e)}", 
                "QuickLabeling", 
                Qgis.Warning
            )

    def canvasReleaseEvent(self, event):
        """Evento que se ejecuta al hacer clic en el mapa"""
        try:
            # Limpiar resaltado inmediatamente al hacer clic
            self.clear_highlight()
                        
            # Identificar features en la posición del clic
            results = self.identify(event.x(), event.y(), [self.layer], self.TopDownAll)
            
            if not results:
                QgsMessageLog.logMessage(
                    "No se encontraron features en esta ubicación", 
                    "QuickLabeling", 
                    Qgis.Warning
                )
                return
            
            # Tomar la primera feature identificada
            feat = results[0].mFeature
            
            # Verificar tipo de campo
            field_index = self.layer.fields().indexFromName(self.field)
            if field_index < 0:
                QgsMessageLog.logMessage(
                    f"Campo '{self.field}' no encontrado en la capa", 
                    "QuickLabeling", 
                    Qgis.Critical
                )
                return
            
            field_type = self.layer.fields().field(field_index).type()
            
            # Generar la etiqueta
            digits = self.counter_widget.get_digits()
            format_str = f"{{:0{digits}d}}"
            
            # Validar si el campo es numérico
            if field_type in [2, 4, 6]:  # Integer, LongLong, Double
                if self.prefix:  # Si hay prefijo, mostrar error
                    QgsMessageLog.logMessage(
                        f"Error: El campo '{self.field}' es numérico. No se puede usar prefijo de texto.", 
                        "QuickLabeling", 
                        Qgis.Critical
                    )
                    return
                label = self.counter  # Solo número
            else:
                label = f"{self.prefix}{format_str.format(self.counter)}"  # Texto completo
            
            # Aplicar la etiqueta
            with edit(self.layer):
                self.layer.changeAttributeValue(feat.id(), field_index, label)
                
                # Incrementar contador
                self.counter += 1
                
                # Actualizar el widget contador
                if hasattr(self.counter_widget, 'update_counter_callback'):
                    self.counter_widget.update_counter_callback(self.counter)
                
                QgsMessageLog.logMessage(
                    f"Feature etiquetada con: {label}", 
                    "QuickLabeling", 
                    Qgis.Info
                )
                        
        except Exception as e:
            # Asegurar que el resaltado se limpia incluso si hay error
            self.clear_highlight()
            
            QgsMessageLog.logMessage(
                f"Error al etiquetar feature: {str(e)}", 
                "QuickLabeling", 
                Qgis.Critical
            )
    def deactivate(self):
        """Limpiar resaltado al desactivar la herramienta"""
        self.clear_highlight()
        super().deactivate()


class QuickLabeling:
    """Clase principal del plugin QuickLabeling"""
    
    def __init__(self, iface):
        self.iface = iface
        self.plugin_dir = os.path.dirname(__file__)
        self.dock_widget = None
        self.canvas_tool = None
        self.action = None
        
        # Nombre del plugin para traducciones
        self.plugin_name = 'QuickLabeling'
    
    def tr(self, message):
        """Función para traducir mensajes"""
        return QCoreApplication.translate(self.plugin_name, message)
    
    def add_action(self, icon_path, text, callback, parent=None):
        """Agregar una acción al menú y toolbar"""
        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        
        # Agregar al toolbar y menú
        self.iface.addToolBarIcon(action)
        self.iface.addPluginToMenu(self.tr(u'&QuickLabeling'), action)
        
        return action
    
    def initGui(self):
        """Inicializar la interfaz gráfica del plugin"""
        # Ruta del icono
        icon_path = os.path.join(self.plugin_dir, 'icon.png')
        
        # Si no existe el icono, usar uno por defecto
        if not os.path.exists(icon_path):
            icon_path = ':/plugins/QuickLabeling/icon.png'
        
        # Crear la acción principal
        self.action = self.add_action(
            icon_path,
            text=self.tr(u'Quick Labeling'),
            callback=self.run,
            parent=self.iface.mainWindow()
        )
        
        # Establecer tooltip
        self.action.setToolTip(self.tr(u'Abrir panel de etiquetado rápido'))
        
        QgsMessageLog.logMessage(
            "Plugin QuickLabeling inicializado correctamente", 
            "QuickLabeling", 
            Qgis.Info
        )
    
    def unload(self):
        """Limpiar recursos cuando se desactiva el plugin"""
        try:
            # Desactivar la herramienta del mapa primero
            if self.canvas_tool is not None:
                if self.iface.mapCanvas().mapTool() == self.canvas_tool:
                    self.iface.mapCanvas().unsetMapTool(self.canvas_tool)
                self.canvas_tool = None
            
            # Manejo seguro del dock widget
            if self.dock_widget is not None:
                try:
                    # Ocultar el widget
                    if self.dock_widget.isVisible():
                        self.dock_widget.hide()
                    
                    # Intentar remover del dock de manera segura
                    dockWidgets = self.iface.mainWindow().findChildren(type(self.dock_widget))
                    for widget in dockWidgets:
                        if widget == self.dock_widget:
                            self.iface.removeDockWidget(widget)
                            break
                    
                except Exception as dock_error:
                    QgsMessageLog.logMessage(
                        f"Advertencia al remover dock widget: {str(dock_error)}", 
                        "QuickLabeling", 
                        Qgis.Warning
                    )
                
                # Cerrar y limpiar el widget
                try:
                    self.dock_widget.close()
                    self.dock_widget.deleteLater()
                except:
                    pass
                finally:
                    self.dock_widget = None
            
            # Remover la acción del menú y toolbar
            if self.action is not None:
                try:
                    self.iface.removePluginMenu(self.tr('&QuickLabeling'), self.action)
                    self.iface.removeToolBarIcon(self.action)
                except:
                    pass
                finally:
                    self.action = None
            
            QgsMessageLog.logMessage(
                "Plugin QuickLabeling desactivado correctamente", 
                "QuickLabeling", 
                Qgis.Info
            )
            
        except Exception as e:
            QgsMessageLog.logMessage(
                f"Error al desactivar plugin: {str(e)}", 
                "QuickLabeling", 
                Qgis.Critical
            )
    
    def run(self):
        """Ejecutar el plugin principal"""
        try:
            # Si ya existe el widget, solo mostrarlo
            if self.dock_widget is not None:
                if not self.dock_widget.isVisible():
                    self.dock_widget.show()
                    self.dock_widget.raise_()
                return
            
            # Función callback para aplicar etiquetas individuales
            def apply_label_tool(layer, field, prefix, counter):
                """Activar la herramienta de etiquetado individual"""
                try:
                    self.canvas_tool = IdentifyAndLabel(
                        self.iface, 
                        layer, 
                        field, 
                        prefix, 
                        counter, 
                        self.dock_widget
                    )
                    self.iface.mapCanvas().setMapTool(self.canvas_tool)
                    return self.canvas_tool
                except Exception as e:
                    QgsMessageLog.logMessage(
                        f"Error al activar herramienta de etiquetado: {str(e)}", 
                        "QuickLabeling", 
                        Qgis.Critical
                    )
                    return None
            
            # Crear el widget contador con el mainWindow como parent
            self.dock_widget = CounterWidget(
                self.iface, 
                apply_label_tool, 
                self.iface.mainWindow()
            )
            
            # Agregar como dock widget al área derecha
            self.iface.addDockWidget(Qt.RightDockWidgetArea, self.dock_widget)
            
            # Hacer que el dock flote por defecto
            self.dock_widget.setFloating(True)

            # Definir posición y tamaño personalizados
            self.dock_widget.move(1200, 200)        # x, y desde la esquina superior izquierda
            # Establecer tamaño inicial óptimo
            optimal_size = self.dock_widget.widget().sizeHint()
            self.dock_widget.resize(350, optimal_size.height() - 100)  # +10 para el título del dock

            # Asegurar que el widget sea visible
            self.dock_widget.show()
            self.dock_widget.raise_()
            
            QgsMessageLog.logMessage(
                "Panel de etiquetado rápido abierto", 
                "QuickLabeling", 
                Qgis.Info
            )
            
        except Exception as e:
            QgsMessageLog.logMessage(
                f"Error al ejecutar plugin: {str(e)}", 
                "QuickLabeling", 
                Qgis.Critical
            )