# -*- coding: utf-8 -*-
from qgis.PyQt.QtCore import Qt
from qgis.PyQt.QtWidgets import (QDockWidget, QWidget, QVBoxLayout, QHBoxLayout, 
                                QLabel, QComboBox, QLineEdit, QPushButton, 
                                QRadioButton, QSpinBox, QGroupBox)
from qgis.core import QgsProject, edit

class CounterWidget(QDockWidget):
    def __init__(self, iface, apply_label_callback, parent=None):
        super().__init__("Etiquetado Rápido", parent)
        self.iface = iface
        self.apply_label_callback = apply_label_callback
        self.counter = 1
        self.canvas_tool = None
        
        # Crear widget contenedor principal
        main_widget = QWidget()
        main_layout = QVBoxLayout(main_widget)
        
        main_layout.setContentsMargins(5, 5, 5, 5)
        main_layout.setSpacing(5)

        # Grupo de configuración de capa y campo
        layer_group = QGroupBox("Configuración")
        layer_layout = QVBoxLayout(layer_group)
        
        layer_layout.addWidget(QLabel("Seleccionar capa:"))
        self.layer_combo = QComboBox()
        layer_layout.addWidget(self.layer_combo)
        
        layer_layout.addWidget(QLabel("Seleccionar campo:"))
        self.field_combo = QComboBox()
        layer_layout.addWidget(self.field_combo)
        
        main_layout.addWidget(layer_group)
        
        # Grupo de etiquetado
        label_group = QGroupBox("Etiquetado")
        label_layout = QVBoxLayout(label_group)
        
        label_layout.addWidget(QLabel("Prefijo:"))
        self.prefix_input = QLineEdit()
        self.prefix_input.setText("A")
        label_layout.addWidget(self.prefix_input)
        
        # Modo de etiquetado
        label_layout.addWidget(QLabel("Modo:"))
        radio_layout = QHBoxLayout()
        self.radio_one = QRadioButton("Etiquetar uno a la vez")
        self.radio_all = QRadioButton("Etiquetar todo")
        self.radio_one.setChecked(True)
        radio_layout.addWidget(self.radio_one)
        radio_layout.addWidget(self.radio_all)
        label_layout.addLayout(radio_layout)
        
        main_layout.addWidget(label_group)
        
        # Grupo de contador
        counter_group = QGroupBox("Contador")
        counter_layout = QVBoxLayout(counter_group)

        # SpinBox para valor inicial
        counter_layout.addWidget(QLabel("Valor inicial:"))
        self.counter_spin = QSpinBox()
        self.counter_spin.setMinimum(1)
        self.counter_spin.setMaximum(9999)
        self.counter_spin.setValue(self.counter)
        counter_layout.addWidget(self.counter_spin)

        # Botones de incremento/decremento
        btn_layout = QHBoxLayout()
        self.btn_down = QPushButton("↓ Decrementar")
        self.btn_down.clicked.connect(self.decrement)
        btn_layout.addWidget(self.btn_down)

        self.btn_up = QPushButton("↑ Incrementar")
        self.btn_up.clicked.connect(self.increment)
        btn_layout.addWidget(self.btn_up)

        counter_layout.addLayout(btn_layout)

        # Etiqueta de vista previa
        self.preview_label = QLabel("Vista previa: ET_01")
        self.preview_label.setStyleSheet("QLabel { background-color: #f0f0f0; padding: 5px; border: 1px solid #ccc; }")
        counter_layout.addWidget(self.preview_label)

        # Botón de parámetros avanzados
        self.advanced_button = QPushButton("▶ Parámetros avanzados")
        self.advanced_button.clicked.connect(self.toggle_advanced_params)
        self.advanced_button.setStyleSheet("QPushButton { text-align: left; padding: 5px; border: none; background: none; }")
        counter_layout.addWidget(self.advanced_button)

        # Widget contenedor para parámetros avanzados
        self.advanced_widget = QWidget()
        self.advanced_widget.setVisible(False)  # Oculto por defecto
        advanced_layout = QVBoxLayout(self.advanced_widget)
        advanced_layout.setContentsMargins(20, 5, 5, 5)  # Indentación

        # Número de dígitos dentro de parámetros avanzados
        advanced_layout.addWidget(QLabel("Número mínimo de dígitos:"))
        self.digits_spin = QSpinBox()
        self.digits_spin.setMinimum(1)
        self.digits_spin.setMaximum(6)
        self.digits_spin.setValue(2)
        advanced_layout.addWidget(self.digits_spin)

        # NUEVO: Checkbox para resaltado
        from qgis.PyQt.QtWidgets import QCheckBox
        self.highlight_checkbox = QCheckBox("Resaltar entidades al navegar")
        self.highlight_checkbox.setChecked(False)  # Activado por defecto
        advanced_layout.addWidget(self.highlight_checkbox)

        counter_layout.addWidget(self.advanced_widget)
        main_layout.addWidget(counter_group)
        
        # Botón de inicio
        self.btn_start = QPushButton("Iniciar Etiquetado")
        self.btn_start.clicked.connect(self.start_labeling)
        self.btn_start.setStyleSheet("""
            QPushButton {
                background-color: #4B9303;
                color: white;
                font-weight: bold;
                padding: 8px;
                border-radius: 4px;
                border: none;
            }
            QPushButton:hover {
                background-color: #407072;
            }
            QPushButton:pressed {
                background-color: #2e5052;
            }
        """)


        main_layout.addWidget(self.btn_start)
        
        # Status label
        self.status_label = QLabel("Listo para etiquetar")
        self.status_label.setStyleSheet("QLabel { color: #666; font-style: italic; }")
        self.status_label.setMaximumHeight(20)  # Altura fija pequeña
        self.status_label.setMinimumHeight(20)  # Altura mínima igual a la máxima
        main_layout.addWidget(self.status_label)

        # Asignar el widget contenedor al dock
        self.setWidget(main_widget)
                
        # Conectar eventos
        self.layer_combo.currentIndexChanged.connect(self.populate_fields)
        self.counter_spin.valueChanged.connect(self.sync_counter_from_spin)
        self.prefix_input.textChanged.connect(self.update_preview)
        self.digits_spin.valueChanged.connect(self.update_preview)

        # Inicializar
        self.populate_layers()
        self.update_preview()

    def populate_layers(self):
        """Poblar el combo de capas con las capas vectoriales del proyecto"""
        self.layer_combo.clear()
        for layer in QgsProject.instance().mapLayers().values():
            if hasattr(layer, 'type') and layer.type() == layer.VectorLayer:
                self.layer_combo.addItem(layer.name(), layer)
    
    def populate_fields(self):
        """Poblar el combo de campos según la capa seleccionada"""
        self.field_combo.clear()
        layer = self.layer_combo.currentData()
        if layer:
            for field in layer.fields():
                self.field_combo.addItem(field.name())
    
    def increment(self):
        """Incrementar el contador"""
        self.counter += 1
        self.counter_spin.setValue(self.counter)
        self.update_preview()
    
    def decrement(self):
        """Decrementar el contador (mínimo 1)"""
        if self.counter > 1:
            self.counter -= 1
            self.counter_spin.setValue(self.counter)
            self.update_preview()
    
    def sync_counter_from_spin(self):
        """Sincronizar el contador interno con el SpinBox"""
        self.counter = self.counter_spin.value()
        self.update_preview()
        
        # Actualizar el contador de la herramienta activa si existe
        if hasattr(self, 'canvas_tool') and self.canvas_tool is not None:
            self.canvas_tool.counter = self.counter
    
    def update_preview(self):
        """Actualizar la vista previa de la etiqueta"""
        prefix = self.prefix_input.text()
        digits = self.digits_spin.value()
        format_str = f"{{:0{digits}d}}"
        self.preview_label.setText(f"Vista previa: {prefix}{format_str.format(self.counter)}")
    
    def start_labeling(self):
        """Iniciar el proceso de etiquetado"""
        layer = self.layer_combo.currentData()
        field = self.field_combo.currentText()
        prefix = self.prefix_input.text()
        start_value = self.counter
        
        if not layer or not field:
            self.status_label.setText("⚠️ Selecciona capa y campo válidos")
            self.status_label.setStyleSheet("QLabel { color: #f44336; font-style: italic; }")
            # DESACTIVAR HERRAMIENTA AL TENER ERROR
            self._deactivate_current_tool()
            return
        
        # Validar tipo de campo
        field_index = layer.fields().indexFromName(field)
        if field_index >= 0:
            field_type = layer.fields().field(field_index).type()
            if field_type in [2, 4, 6] and prefix:  # Campo numérico con prefijo
                self.status_label.setText("⚠️ Campo numérico no puede tener prefijo de texto")
                self.status_label.setStyleSheet("QLabel { color: #f44336; font-style: italic; }")
                # DESACTIVAR HERRAMIENTA AL TENER ERROR
                self._deactivate_current_tool()
                return
        
        if self.radio_all.isChecked():
            # Etiquetar todas las features
            features = list(layer.getFeatures())
            try:    
                digits = self.digits_spin.value()
                format_str = f"{{:0{digits}d}}"
                with edit(layer):
                    for i, feat in enumerate(features):
                        # Validar nuevamente el tipo de campo para cada feature
                        if field_type in [2, 4, 6]:  # Campo numérico
                            label_value = start_value + i
                        else:  # Campo de texto
                            label_value = f"{prefix}{format_str.format(start_value + i)}"
                        
                        layer.changeAttributeValue(
                            feat.id(), 
                            layer.fields().indexFromName(field), 
                            label_value
                        )
                self.status_label.setText(f"✅ Etiquetado completado: {len(features)} features")
                self.status_label.setStyleSheet("QLabel { color: #539091; font-style: italic; }")
                # Actualizar contador para la siguiente serie
                self.counter = start_value + len(features)
                self.counter_spin.setValue(self.counter)
                self.update_preview()
            except Exception as e:
                self.status_label.setText(f"❌ Error: {str(e)}")
                self.status_label.setStyleSheet("QLabel { color: #f44336; font-style: italic; }")
                # DESACTIVAR HERRAMIENTA AL TENER ERROR
                self._deactivate_current_tool()
                
        elif self.radio_one.isChecked():
            # Etiquetar uno a la vez - solo activar herramienta si no hay errores
            self.canvas_tool = self.apply_label_callback(layer, field, prefix, start_value)
            
            if self.canvas_tool is not None:
                self.status_label.setText("🖱️ Haz clic en una entidad para etiquetarla")
                self.status_label.setStyleSheet("QLabel { color: #2196F3; font-style: italic; }")
            else:
                self.status_label.setText("❌ Error al activar herramienta")
                self.status_label.setStyleSheet("QLabel { color: #f44336; font-style: italic; }")

    
    def get_counter(self):
        """Obtener el valor actual del contador"""
        return self.counter
    
    def set_counter(self, val):
        """Establecer el valor del contador"""
        if val >= 1:
            self.counter = val
            self.counter_spin.setValue(val)
            self.update_preview()
    
    def update_counter_callback(self, new_value):
        """Callback para actualizar el contador desde herramientas externas"""
        self.set_counter(new_value)
        digits = self.digits_spin.value()
        format_str = f"{{:0{digits}d}}"
        self.status_label.setText(f"✅ Entidad etiquetada. Siguiente: {self.prefix_input.text()}{format_str.format(new_value)}")
        self.status_label.setStyleSheet("QLabel { color: #4CAF50; font-style: italic; }")
    
    def get_digits(self):
        """Obtener el número de dígitos configurado"""
        return self.digits_spin.value()
    
    def toggle_advanced_params(self):
        """Mostrar/ocultar parámetros avanzados"""
        is_visible = self.advanced_widget.isVisible()
        self.advanced_widget.setVisible(not is_visible)
        
        if is_visible:
            self.advanced_button.setText("▶ Parámetros avanzados")
        else:
            self.advanced_button.setText("▼ Parámetros avanzados")
        
        # Ajustar el tamaño del panel
        from qgis.PyQt.QtWidgets import QApplication
        from qgis.PyQt.QtCore import QTimer
        
        def resize_panel():
            if self.isFloating():
                QApplication.processEvents()
                
                # Obtener el tamaño óptimo del contenido
                optimal_size = self.widget().sizeHint()
                current_width = self.width()
                
                # Calcular nueva altura (contenido + margen para título)
                new_height = optimal_size.height() + 35
                
                # Aplicar el nuevo tamaño
                self.resize(current_width, new_height)
        
        # Ejecutar redimensionamiento con delay
        QTimer.singleShot(10, resize_panel)

    def showEvent(self, event):
        """Recargar capas y campos cada vez que se muestra el widget"""
        super().showEvent(event)
        self.populate_layers()
        self.populate_fields()
        
        # Ajustar el tamaño inicial del panel
        from qgis.PyQt.QtCore import QTimer
        
        def reset_panel_size():
            if self.isFloating():
                from qgis.PyQt.QtWidgets import QApplication
                QApplication.processEvents()
                
                optimal_size = self.widget().sizeHint()
                current_width = self.width()
                new_height = optimal_size.height() + 35
                
                self.resize(current_width, new_height)
        
        QTimer.singleShot(50, reset_panel_size)

    def hideEvent(self, event):
        """Limpiar herramienta cuando se oculta el widget"""
        super().hideEvent(event)
        # Limpiar la herramienta de canvas cuando se cierra el panel
        if self.canvas_tool is not None:
            if self.iface.mapCanvas().mapTool() == self.canvas_tool:
                self.iface.mapCanvas().unsetMapTool(self.canvas_tool)
            self.canvas_tool = None
        
        # Resetear estado
        self.status_label.setText("Listo para etiquetar")
        self.status_label.setStyleSheet("QLabel { color: #666; font-style: italic; }")

    def _deactivate_current_tool(self):
        """Desactivar la herramienta actual del canvas"""
        if self.canvas_tool is not None:
            if self.iface.mapCanvas().mapTool() == self.canvas_tool:
                self.iface.mapCanvas().unsetMapTool(self.canvas_tool)
            self.canvas_tool = None

    def is_highlight_enabled(self):
        """Verificar si el resaltado está habilitado"""
        return self.highlight_checkbox.isChecked()