# -*- coding: utf-8 -*-

"""
/***************************************************************************
 QThermonet
                                 A QGIS plugin
 This plugin links QGIS to pythermonet
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-06-10
        copyright            : (C) 2024 by Jane Lund Andersen/VIA University College
        email                : jana@via.dk
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Jane Lund Andersen/VIA University College'
__date__ = '2025-06-10'
__copyright__ = '(C) 2025 by Jane Lund Andersen/VIA University College'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

import os
import inspect
import pandas as pd
from qgis.PyQt.QtGui import QIcon, QColor

from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import (QgsProcessing,
                       QgsCategorizedSymbolRenderer,
                       QgsCoordinateTransformContext,
                       QgsExpression,
                       QgsExpressionContext,
                       QgsExpressionContextUtils,
                       QgsField,
                       QgsFields,
                       QgsProcessingAlgorithm,
                       QgsProcessingParameterBoolean,
                       QgsProcessingParameterFeatureSource,
                       QgsProcessingParameterFileDestination,
                       QgsProcessingException,
                       QgsProject,
                       QgsRendererCategory,
                       QgsSymbol,
                       QgsVectorFileWriter,
                       QgsVectorLayer,
                       QgsFeature)

from PyQt5.QtCore import QVariant


class LoadCalculationAlgorithm(QgsProcessingAlgorithm):
    
    #Handle input/output
    OUTPUT = 'OUTPUT'
    INPUT = 'INPUT'

    def initAlgorithm(self, config=None):
        
        # 1st input
        param = QgsProcessingParameterFeatureSource(
                self.INPUT,
                self.tr("Input Layer with BBR information:"),
                [QgsProcessing.TypeVectorPolygon]  # Only accept polygon layers
            )
        
        param.setHelp(
            "The input layer must:\n"
            "- Be a polygon layer (e.g., representing building footprints).\n"
            "- Contain the required fields: 'BuildYear', 'BuildCode', "
            " 'BBRarea', and 'BBRUUID'.\n"
            "- Use a compatible CRS (preferably WGS84/EPSG:4326 OR 3857)."
        )

        self.addParameter(param)
        
        # 1st output
        self.addParameter(
            QgsProcessingParameterFileDestination(
                self.OUTPUT,
                self.tr('Output GeoJSON'),
                fileFilter="GeoJSON (*.geojson)"
            )
        )

        # Add a checkbox for opening the output file after running
        self.addParameter(
            QgsProcessingParameterBoolean(
                "OPEN_OUTPUT",
                self.tr("Open output file after running algorithm"),
                defaultValue=True  # Automatically checked
            )
        )
        
        # 2nd output
        self.addParameter(
            QgsProcessingParameterFileDestination(
                "DAT_OUTPUT",
                self.tr("Output DAT file"),
                fileFilter="DAT files (*.dat)"
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        #Handle input layers
        input_layer = self.parameterAsVectorLayer(
            parameters, self.INPUT, context)
        output_path = self.parameterAsFileOutput(
            parameters, self.OUTPUT, context)
        open_output = self.parameterAsBoolean(
            parameters, "OPEN_OUTPUT", context)
        dat_output_path = self.parameterAsFileOutput(
            parameters, "DAT_OUTPUT", context)

        if not input_layer:
            raise QgsProcessingException("Invalid input layer!")

        # Get all features and fields
        input_features = list(input_layer.getFeatures())
        input_fields = input_layer.fields()
        updated_fields = QgsFields()
        geometries = []
             
        # Step 1: Process 'BBRUUID' field
        bbruuid_values = []
        if not any(field.name() == "BBRUUID" for field in input_fields):
            raise QgsProcessingException(
                'The input layer is missing the required "BBRUUID" field!')
        else:
            feedback.pushInfo("Copying 'BBRUUID' features...")
            for feature in input_features:
                bbruuid_values.append(feature.attribute("BBRUUID"))
                
        # Step 2: Process 'Thermonet' field if it doesn't exist already
        thermonet_values = []
        if not any(field.name() == "Thermonet" for field in input_fields):
            feedback.pushInfo("Adding 'Thermonet' features...")
            
            expression = QgsExpression('CASE WHEN "BBRUUID" IS NOT NULL AND "BBRarea" <> 0  AND BuildCode < 200 THEN \'Yes\' ELSE \'No\' END')
            context = QgsExpressionContext()
            context.appendScopes(
                QgsExpressionContextUtils.globalProjectLayerScopes(input_layer))   
            for feature in input_features:
                context.setFeature(feature)
                thermonet_value = expression.evaluate(context)
    
                if expression.hasEvalError():
                    raise QgsProcessingException(
                        f"Expression evaluation error: {expression.evalErrorString()}")
                thermonet_values.append(thermonet_value)
        else:
            feedback.pushInfo("Copying 'Thermonet' features...")
            for feature in input_features:
                thermonet_values.append(feature.attribute("Thermonet"))
             
        # Step 3: Copy or create id.lokalId field for use as Heatpump ID
        lokalId_values = []
        if not any(field.name() == "id.lokalId" for field in input_fields):
            feedback.pushInfo("'id.lokalId' field not found, assigning unique IDs...")
            feature_count = 0
            for feature in input_features:  # Update existing features
                feature_count += 1
                lokalId_values.append(f"{str(feature_count)}")
        else:
            feedback.pushInfo("Copying 'id.lokalId' features...")
            for feature in input_features:
                lokalId_values.append(feature.attribute("id.lokalId"))
        
        # Step 4: Copy BBR information
        BuildYear_values = []
        if not any(field.name() == "BuildYear" for field in input_fields):
            raise QgsProcessingException(
                'The input layer is missing the required "BuildYear" field!')
        else:
            feedback.pushInfo("Copying 'BuildYear' features...")
            for feature in input_features:
                BuildYear_values.append(feature.attribute("BuildYear"))
                
        BuildCode_values = []
        if not any(field.name() == "BuildCode" for field in input_fields):
            raise QgsProcessingException(
                'The input layer is missing the required "BuildCode" field!')
        else:
            feedback.pushInfo("Copying 'BuildCode' features...")
            for feature in input_features:
                BuildCode_values.append(feature.attribute("BuildCode"))
                
        BBRarea_values = []
        if not any(field.name() == "BBRarea" for field in input_fields):
            raise QgsProcessingException(
                'The input layer is missing the required "BBRarea" field!')
        else:
            feedback.pushInfo("Copying 'BBRarea' features...")
            for feature in input_features:
                BBRarea_values.append(feature.attribute("BBRarea"))
                
        # Step 5: Calculate bulding heat loads
        feedback.pushInfo("Calculating building heatload...")
        YHL_values, WHL_values, DHL_values = self.calc_heat_loads(
            BuildYear_values, BuildCode_values, BBRarea_values, thermonet_values, feedback)
        
        # Step 6: Create updated features
        updated_fields.append(QgsField("id.lokalId", QVariant.String))
        updated_fields.append(QgsField("BBRUUID", QVariant.String))
        updated_fields.append(QgsField("Thermonet", QVariant.String))
        updated_fields.append(QgsField("BuildYear", QVariant.Int, "integer", len=4))
        updated_fields.append(QgsField("BuildCode", QVariant.Int, "integer", len=3))
        updated_fields.append(QgsField("BBRarea", QVariant.Int, "integer", len=10))
        updated_fields.append(QgsField("YrHeatLoad", QVariant.Double, "double", len=10, prec=0))
        updated_fields.append(QgsField("WiHeatLoad", QVariant.Double, "double", len=10, prec=0))
        updated_fields.append(QgsField("DyHeatLoad", QVariant.Double, "double", len=10, prec=0))
        
        geometries = [feature.geometry() for feature in input_features] # geometries for all features
        
        updated_features = []
        for idx, geometry in enumerate(geometries):
            new_feature = QgsFeature()
            new_feature.setGeometry(geometry)
            new_feature.setFields(updated_fields)
        
            # Combine attributes
            attributes = [None] * len(updated_fields)
            attributes[updated_fields.indexOf("id.lokalId")] = lokalId_values[idx] if idx < len(lokalId_values) else None
            attributes[updated_fields.indexOf("BBRUUID")] = bbruuid_values[idx] if idx < len(bbruuid_values) else None
            attributes[updated_fields.indexOf("Thermonet")] = thermonet_values[idx] if idx < len(thermonet_values) else None
            attributes[updated_fields.indexOf("BuildYear")] = BuildYear_values[idx] if idx < len(BuildYear_values) else None
            attributes[updated_fields.indexOf("BuildCode")] = BuildCode_values[idx] if idx < len(BuildCode_values) else None
            attributes[updated_fields.indexOf("BBRarea")] = BBRarea_values[idx] if idx < len(BBRarea_values) else None
            attributes[updated_fields.indexOf("YrHeatLoad")] = YHL_values[idx] if idx < len(YHL_values) else None
            attributes[updated_fields.indexOf("WiHeatLoad")] = WHL_values[idx] if idx < len(WHL_values) else None
            attributes[updated_fields.indexOf("DyHeatLoad")] = DHL_values[idx] if idx < len(DHL_values) else None
            new_feature.setAttributes(attributes)
            updated_features.append(new_feature)
                   

        # Step 7: Save updated features to a geojson file (only those in thermonet)
        # Filter updated_features to include only those where "Thermonet" == 'Yes'
        filtered_features = [
            f for f in updated_features
            if f["Thermonet"] and str(f["Thermonet"]).strip().lower() == "yes"
        ]
        
        feedback.pushInfo("Saving to output geojson file...")
        # Use filtered_features for saving the GeoJSON
        self.save_to_geojson(updated_fields, filtered_features, 
                             output_path, input_layer, feedback)

        # self.save_to_geojson(updated_fields, updated_features, 
        #                      output_path, input_layer, feedback)

        # Step 8: Open dat file in writing mode and write header and content
        feedback.pushInfo("Writing dat file ...")
        dat_file = open(dat_output_path, "w", encoding="utf-8")
        dat_file.write(
            "Heat_pump_ID\tYearly_heating_load_(W)\tWinter_heating_load_(W)\tDaily_heating_load_(W)\t"
            "Year_COP\tWinter_COP\tHour_COP\tdT_HP_Heating\tYearly_cooling_load_(W)\tSummer_cooling_load_(W)\t"
            "Daily_cooling_load_(W)\tEER\tdT_HP_Cooling\n"
        )

        # temporary hardcoded values
        Yr_COP = 3.3
        Winter_COP = 3.4
        Hour_COP = 3.0
        dT_HP_Heating = 3
        Year_CoolLoad = 0
        Summer_CoolLoad = 0
        daily_CoolLoad = 0
        EER = 0.0
        dT_HP_Cooling = 0

        for feature in updated_features:
            thermonet_value = feature["Thermonet"] if "Thermonet" in feature.fields(
            ).names() else None
            if thermonet_value and str(thermonet_value).lower() == "yes":
                dat_file.write(
                    f"{feature['id.lokalId']}\t"
                    f"{int(round(feature['YrHeatLoad']))}\t"
                    f"{int(round(feature['WiHeatLoad']))}\t"
                    f"{int(round(feature['DyHeatLoad']))}\t"
                    f"{float(Yr_COP):.1f}\t"
                    f"{float(Winter_COP):.1f}\t"
                    f"{float(Hour_COP):.1f}\t"
                    f"{float(dT_HP_Heating):.1f}\t"
                    f"{int(round(Year_CoolLoad))}\t"
                    f"{int(round(Summer_CoolLoad))}\t"
                    f"{int(round(daily_CoolLoad))}\t"
                    f"{float(EER):.1f}\t"
                    f"{float(dT_HP_Cooling):.1f}\n"
                )

        dat_file.close()

        # Step 9: Open the output file if the checkbox is checked
        if open_output:
            feedback.pushInfo("Opening output file...")
            # Extract the file name without extension to use as the layer name
            import os
            layer_name = os.path.splitext(os.path.basename(output_path))[0]
            layer = QgsVectorLayer(output_path, layer_name, "ogr")
            if not layer.isValid():
                raise QgsProcessingException(
                    "Could not load the output layer!")

            # Add the layer to the QGIS project
            QgsProject.instance().addMapLayer(layer)

            # Set up categorized symbology based on the "Thermonet" field
            feedback.pushInfo("Applying symbology...")
            categories = []

            # Define the categories
            gray_symbol = QgsSymbol.defaultSymbol(layer.geometryType())
            myGray = QColor(200, 200, 200)
            gray_symbol.setColor(myGray)
            categories.append(QgsRendererCategory("No", gray_symbol, "No"))

            red_symbol = QgsSymbol.defaultSymbol(layer.geometryType())
            myRed = QColor(196, 60, 57)
            red_symbol.setColor(myRed)
            categories.append(QgsRendererCategory("Yes", red_symbol, "Yes"))

            # Create and set the categorized renderer
            renderer = QgsCategorizedSymbolRenderer("Thermonet", categories)
            if renderer is not None:
                layer.setRenderer(renderer)
                layer.triggerRepaint()

        feedback.pushInfo("Processing completed successfully.")
        return {
            self.OUTPUT: output_path,
            "DAT_OUTPUT": dat_output_path,
        }


    def save_to_geojson(self, fields, updated_features, output_path, input_layer, feedback):
        """
        Saves updated features with new fields to a GeoJSON file.
    
        :param fields: QgsFields containing the updated field definitions.
        :param updated_features: The list of updated QgsFeature.
        :param output_path: Path to save the output GeoJSON.
        :param input_layer: The input layer for CRS and geometry type.
        :param feedback: Feedback object for progress reporting.
        """
        options = QgsVectorFileWriter.SaveVectorOptions()
        options.driverName = "GeoJSON"  # Specify GeoJSON driver
        options.fileEncoding = "UTF-8"
    
        # Ensure output filename has .geojson extension
        if not output_path.lower().endswith(".geojson"):
            output_path += ".geojson"
    
        writer = QgsVectorFileWriter.create(
            output_path,
            fields,
            input_layer.wkbType(),
            input_layer.sourceCrs(),
            QgsCoordinateTransformContext(),
            options
        )
    
        if writer.hasError() != QgsVectorFileWriter.NoError:
            raise QgsProcessingException("Failed to create output GeoJSON.")
    
        for feature in updated_features:
            writer.addFeature(feature)
    
        del writer
        feedback.pushInfo(f"Output GeoJSON saved to: {output_path}")
        

    def calc_heat_loads(self, BuildYear_values, BuildCode_values, BBRarea_values, Thermonet, feedback):
        """
        Calculates the building heatloads (annual, winter, daily)
        NB: using dummy values to calculate winter and daily loads from 
        annual loads based on fudge factors from Silkeborg case.
        
        """
        # Initiate output variables
        YHLs = []
        WHLs = []
        DHLs = []
        
        for B_year, B_code, B_area, In_Thermonet in zip(BuildYear_values, BuildCode_values, BBRarea_values, Thermonet):  
            if feedback.isCanceled():
                break
            
            # Access field values
            area = int(B_area or 0)
            construction_year = int(B_year or 0)
            build_code = int(B_code or 0)
            inThermonet = str(In_Thermonet).strip().lower()

            ''' The following takes a construction year as input and returns an
            index representing a specific range of years. The index corresponds
            to the ranges in the file
            (heat_atlas_data_long_format_mapped.csv): '''
            # Map construction_year to index for table look-up:
            if construction_year < 1850:
                year_idx = 0
            elif construction_year < 1931:
                year_idx = 1
            elif construction_year < 1951:
                year_idx = 2
            elif construction_year < 1961:
                year_idx = 3
            elif construction_year < 1973:
                year_idx = 4
            elif construction_year < 1979:
                year_idx = 5
            elif construction_year < 1999:
                year_idx = 6
            elif construction_year < 2007:
                year_idx = 7
            elif construction_year > 2006:
                year_idx = 8

            # Calculate HeatLoad
            if construction_year == 0 or area == 0 or inThermonet == "no":  # Handle NULL values
                year_heat_load_value = 0.0
            else:
                # Get heat load from heat atlas table lookup and convert to W
                YHL = self.get_heat_demand(build_code, year_idx, feedback) #kWh m-2 yr-1
                YHL_W = (YHL * 1000)/(8766) #Yearly heat load in W
                # feedback.pushInfo(f"Yearly heat load for building: {round(YHL_W,1)} W m-2")
                year_heat_load_value = YHL_W * area
                
            #Calculate dummy seasonal and daily heat loads for now
            WHLdummy = 1.6  # dummy factor to convert from annual to winter heat load
            DHLdummy = 3.2  # dummy factor to convert from winter to daily heat load
            winter_heat_load_value = round(year_heat_load_value * WHLdummy)
            daily_heat_load_value = round(winter_heat_load_value * DHLdummy)

            YHLs.append(round(year_heat_load_value))
            WHLs.append(winter_heat_load_value)
            DHLs.append(daily_heat_load_value)

        return YHLs, WHLs, DHLs

    def get_heat_demand(self, usage_code, construction_year_bin, feedback):
        """
        Extracts the heat demand from the heat atlas for a specific building 
        code and construction year bin .

        This function filters the heat atlas DataFrame to find the heat
        demand (kWh) per square meter per year for buildings with the specified
        usage code and construction year bin.

        Args:
        :param usage_code: The building code for the building of interest.
        :type usage_code: int
        :param construction_year_bin: The construction year bin for the 
            building of interest.
        :type construction_year_bin: int

        Returns:
        :return: The heat demand (kWh) per square meter per year for the 
            specified building code and construction year bin.
        :rtype: float
        """
        
        # Get the directory of the current Python script
        script_dir = os.path.dirname(os.path.abspath(__file__))
        
        # Build the full path to the CSV file
        heat_atlas_long_file = os.path.join(script_dir, 'heat_atlas_data_long_format_mapped.csv')
        
        # Check if the file exists
        if not os.path.exists(heat_atlas_long_file):
            feedback.pushInfo(f"File not found: {heat_atlas_long_file}")
            return {}
        
        # Try reading the CSV file
        try:
            df_heat_atlas = pd.read_csv(heat_atlas_long_file)
        except Exception as e:
            feedback.pushInfo(f"Error reading the file: {e}")
            return {}
        
        # # Filter the DataFrame
        filtered_data = df_heat_atlas[
            (df_heat_atlas["usage_code"] == usage_code) & 
            (df_heat_atlas["construction_year_bin"] == construction_year_bin)
        ]
        
        # Check if the filtered DataFrame is empty
        if not filtered_data.empty:
            # Extract the heat_demand value
            heat_demand = filtered_data.iloc[0]["heat_demand_kWh_m2_year"]
            feedback.pushInfo(f"Heat demand for usage_code={usage_code} and construction_year_bin={construction_year_bin}: {heat_demand}")
            return heat_demand
        else:
            feedback.pushInfo(f"No data found for usage_code={usage_code} and construction_year_bin={construction_year_bin}")
            return None
    
    
    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'Calculate heatloads'

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr(self.name())

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr(self.groupId())

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return '1. Heatloads'

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def icon(self):
        cmd_folder = os.path.split(inspect.getfile(inspect.currentframe()))[0]
        icon = QIcon(os.path.join(os.path.join(cmd_folder, 'logo2b.png')))
        return icon

    def shortHelpString(self):
        return ("<p><b> This tool: </b><p>"
                "<p> 1. Calculates the building heat load based on the building "
                " code, building year, and building area by lookup in the "
                "Danish Heat Atlas (Mathiesen et al. 2021). <br>"
                "2. Stores the relevant information in new geojson and dat files. <p>"
                "<p> The output dat-file can be used as input for full "
                "dimensioning of the thermonet using pythermonet <p>"
                )
    
    def createInstance(self):
        return LoadCalculationAlgorithm()
