# -*- coding: utf-8 -*-

"""
/***************************************************************************
 QThermonet
                                 A QGIS plugin
 This plugin links QGIS to pythermonet
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2024-12-05
        copyright            : (C) 2024 by Jane Lund Andersen/VIA University College
        email                : jana@via.dk
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Jane Lund Andersen/VIA University College'
__date__ = '2024-12-05'
__copyright__ = '(C) 2024 by Jane Lund Andersen/VIA University College'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

import os
import inspect
from qgis.PyQt.QtGui import QIcon

from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import (
    QgsProcessing,
    QgsProcessingAlgorithm,
    QgsProcessingParameterVectorLayer,
    QgsProcessingParameterFileDestination,
    QgsProcessingParameterNumber,
    QgsProcessingException,
    QgsVectorLayer,
    QgsFeature,
    QgsField)
from qgis.PyQt.QtCore import QVariant
import csv


class AggregatedLoadAlgorithm(QgsProcessingAlgorithm):
    """
    This algorithm calculates the aggregated heat load of buildings

    All Processing algorithms should extend the QgsProcessingAlgorithm
    class.
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    INPUT_LAYER = "INPUT_LAYER"
    YrCOP = "Year_COP"
    WiCOP = "Winter_COP"
    HrCOP = "Hour_COP"
    dT_HP_Heating = "dT_HP_Heating"
    OUTPUT = "OUTPUT"

    def initAlgorithm(self, config=None):
        # Input shapefile
        self.addParameter(
            QgsProcessingParameterVectorLayer(
                self.INPUT_LAYER,
                "Input layer with building heatloads",
                [QgsProcessing.TypeVectorAnyGeometry]
            )
        )

        # Input numerical values for Year_COP
        self.addParameter(
            QgsProcessingParameterNumber(
                self.YrCOP,
                "Year COP Value",
                type=QgsProcessingParameterNumber.Double,
                defaultValue=3.3,
                minValue=0.0
            )
        )
        
        # Input numerical values for Winter_COP
        self.addParameter(
            QgsProcessingParameterNumber(
                self.WiCOP,
                "Winter COP Value",
                type=QgsProcessingParameterNumber.Double,
                defaultValue=3.4,
                minValue=0.0
            )
        )
        
        # Input numerical values for Hour_COP
        self.addParameter(
            QgsProcessingParameterNumber(
                self.HrCOP,
                "Hour COP Value",
                type=QgsProcessingParameterNumber.Double,
                defaultValue=3.0,
                minValue=0.0
            )
        )

        # Input numerical values for dT_HP_Heating
        self.addParameter(
            QgsProcessingParameterNumber(
                self.dT_HP_Heating,
                "dT_HP_Heating Value",
                type=QgsProcessingParameterNumber.Double,
                defaultValue=3.0,
                minValue=0.0
            )
        )

        # Output file
        self.addParameter(
            QgsProcessingParameterFileDestination(
                self.OUTPUT,
                "Output DAT File",
                "DAT files (*.dat)"
            )
        )
        
    def processAlgorithm(self, parameters, context, feedback):
       # Retrieve parameters
        input_layer = self.parameterAsVectorLayer(parameters, self.INPUT_LAYER, context)
        output_path = self.parameterAsFileOutput(parameters, self.OUTPUT, context)
        year_cop_value = self.parameterAsDouble(parameters, self.YrCOP, context)
        winter_cop_value = self.parameterAsDouble(parameters, self.WiCOP, context)
        hour_cop_value = self.parameterAsDouble(parameters, self.HrCOP, context)
        dT_HP_Heating_value = self.parameterAsDouble(parameters, self.dT_HP_Heating, context)

        if not input_layer:
            raise QgsProcessingException("Invalid input layer!")

        # Initialize aggregation
        year_heat_load = 0
        winter_heat_load = 0
        hour_heat_load = 0
        consumers = 0
        feature_count = input_layer.featureCount()

        feedback.pushInfo(f"Processing {feature_count} features...")
        
        # Aggregate HeatLoads
        for current, feature in enumerate(input_layer.getFeatures()):
            if feedback.isCanceled():
                break
            
            if feature["Thermonet"] == "Yes":
            
                year_load = feature["YrHeatLoad"] if "YrHeatLoad" in feature.fields().names() else 0
                year_heat_load += year_load
                
                winter_load = feature["WiHeatLoad"] if "WiHeatLoad" in feature.fields().names() else 0
                winter_heat_load += winter_load
                
                hour_load = feature["DyHeatLoad"] if "DyHeatLoad" in feature.fields().names() else 0
                hour_heat_load += hour_load
                
                ## Only add consumer to total count if Thermonet is set to "Yes"
                #inThermonet = 1 if feature["Thermonet"] == "Yes" else 0
                #consumers += inThermonet
                consumers +=1

            # Update progress
            feedback.setProgress(int((current / feature_count) * 100))

        # Create table content, note that cooling is not handled yet
        table_content = [
            ["No. Consumers","Yearly_heating_load_(W)", "Winter_heating_load_(W)",\
             "Daily_heating_load_(W)", "Year_COP", "Winter_COP", "Hour_COP", "dT_HP_Heating",\
             "Yearly_cooling_load_(W)", "Summer_cooling_load_(W)", \
                 "Daily_cooling_load_(W)", "EER", "dT_HP_Cooling"],
            [consumers, year_heat_load, winter_heat_load, hour_heat_load, \
             year_cop_value, winter_cop_value, hour_cop_value, dT_HP_Heating_value, 0, 0, 0, 0, 0]
        ]
        
        # Write to DAT
        feedback.pushInfo("Writing output to DAT...")
        with open(output_path, "w") as dat_file:
            for row in table_content:
                # Convert each row (list) to a tab-separated string and write to the file
                dat_file.write("\t".join(map(str, row)) + "\n")
        
        # # Write to CSV
        # feedback.pushInfo("Writing output to CSV...")
        # with open(output_path, "w", newline="") as csv_file:
        #     writer = csv.writer(csv_file)
        #     writer.writerows(table_content)

        feedback.pushInfo("Processing complete!")
        
        return {self.OUTPUT: output_path}


    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'Calculate aggregated heat load'

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr(self.name())

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr(self.groupId())

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return ''

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)
    
    def icon(self):
        cmd_folder = os.path.split(inspect.getfile(inspect.currentframe()))[0]
        icon = QIcon(os.path.join(os.path.join(cmd_folder, 'logo3.png')))
        return icon
    
    def shortHelpString(self):
        """
        Return a short help string for the algorithm.
        """
        return ("<p> This algorithm calculates the aggregated building heatload "
                "for an input file of building heatloads. <p> "
                "<p> The output dat-file can be used as input to pythermonet "
                "to dimension sources of a thermonet. <p>"
                "<p> COP values of heatpumps are set from input parameters <p>"
                )
    
    def createInstance(self):
        return AggregatedLoadAlgorithm()
