# -*- coding: utf-8 -*-

"""
/***************************************************************************
 QThermonet
                                 A QGIS plugin
 This plugin links QGIS to pythermonet
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-06-10
        copyright            : (C) 2024 by Jane Lund Andersen/VIA University College
        email                : jana@via.dk
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Jane Lund Andersen/VIA University College'
__date__ = '2025-06-10'
__copyright__ = '(C) 2025 by Jane Lund Andersen/VIA University College'


from qgis.core import (
    QgsProcessing,
    QgsProcessingAlgorithm,
    QgsProcessingException,
    QgsProcessingParameterVectorLayer,
    QgsProcessingParameterFeatureSink,
    QgsProject,
    QgsFeature,
    QgsFeatureSink,
    QgsGeometry,
    QgsFields,
    QgsWkbTypes,
    QgsSpatialIndex,
    QgsDistanceArea,
    QgsFeatureRequest,
    QgsField,
    QgsPointXY,
    QgsPoint,
    QgsLineSymbol, 
    QgsGeometryGeneratorSymbolLayer,
    QgsMarkerSymbol,
    QgsSingleSymbolRenderer,
    QgsVectorLayer
)
from qgis.PyQt.QtCore import QCoreApplication
from qgis.PyQt.QtGui import QIcon
import os
import inspect
from osgeo import ogr

class ServicePipesAlgorithm(QgsProcessingAlgorithm):
    
    #Handle input/output
    BUILDINGS_LAYER = "BUILDINGS_LAYER"
    PIPES_LAYER = "PIPES_LAYER"
    OUTPUT_LAYER = "OUTPUT_LAYER"

    def initAlgorithm(self, config=None):
        
        # 1st input
        param = QgsProcessingParameterVectorLayer(
            self.BUILDINGS_LAYER,
            "Select the Buildings Layer",
            [QgsProcessing.TypeVectorPolygon],
        )

        param.setHelp(
            "The input layer must:\n"
            "- Be a polygon layer (e.g., representing building footprints).\n"
            "- Contain the required fields: 'Thermonet' and 'BBRUUID'.\n"
            "- Use a compatible CRS (preferably WGS84/EPSG:3857)."
        )

        self.addParameter(param)
       
        # 2nd input
        param = QgsProcessingParameterVectorLayer(
            self.PIPES_LAYER,
            "Select the Pipes Layer",
            [QgsProcessing.TypeVectorLine],
        )

        param.setHelp(
            "The input layer must:\n"
            "- Be a Line/polyline layer with thermonet pipes.\n"
            "- Contain the required fields: 'id' and 'Level'.\n"
            "- The 'Level' field should contain the hierarchy of the pipes "
            "(integers 1:1:N, where 1=main pipe, N=lowest-level pipe).\n"
            "- Use a compatible CRS (preferably WGS84/EPSG:3857)."
        )

        self.addParameter(param)
        
        # Output
        param = QgsProcessingParameterFeatureSink(
                self.OUTPUT_LAYER,
                "Output Service Pipes",
            )
        param.setHelp(
            "The output layer will contain the shortest distance service pipes "
            "between each building and the nearest point on the nearest pipe. \n"
            "- It will contain all fields from both input layers.\n"
            "- It is an essential input to the 'Pipe Topology' tool."
            )
        self.addParameter(param)
        
    def processAlgorithm(self, parameters, context, feedback):
        
        feedback.pushInfo("Checking input files ...")
        buildings_layer = self.parameterAsVectorLayer(parameters, self.BUILDINGS_LAYER, context)
        pipes_layer = self.parameterAsVectorLayer(parameters, self.PIPES_LAYER, context)
        
        if not buildings_layer or not pipes_layer:
            raise QgsProcessingException("Invalid input layers!")
            
        # Ensure the "Thermonet" field exists in the buildings layer
        if "Thermonet" not in [field.name() for field in buildings_layer.fields()]:
            raise QgsProcessingException("The source layer does not contain a 'Thermonet' field!")
    
        # Filter buildings with Thermonet = "Yes"
        buildings = [
            feat
            for feat in buildings_layer.getFeatures()
            if feat["Thermonet"] == "Yes"
        ]
    
        if not buildings:
            raise QgsProcessingException("No buildings with 'Thermonet' field set to 'Yes' found!")
        
        # Check if the output file is locked
        feedback.pushInfo("Checking output file destination ...")
        output_path = self.parameterAsOutputLayer(parameters, self.OUTPUT_LAYER, context)
        if self.is_file_locked(output_path, feedback):
            raise QgsProcessingException(
                f"The output file '{output_path}' is locked by QGIS and cannot be overwritten. "
                "Please select another output file name or restart QGIS to remove the lock."
            )
    
        # Prepare output fields
        feedback.pushInfo("Preparing output fields ...")

        output_fields = QgsFields()
        for field in buildings_layer.fields():
            if field.name() == "YrHeatLoad":
                output_fields.append(QgsField(field.name(), field.type(), field.typeName(), 10, 0))
            elif field.name() == "WiHeatLoad":
                output_fields.append(QgsField(field.name(), field.type(), field.typeName(), 10, 0))
            elif field.name() == "DyHeatLoad":
                output_fields.append(QgsField(field.name(), field.type(), field.typeName(), 10, 0))
            else:
                # Use default type/length/precision from input
                output_fields.append(QgsField(field.name(), field.type(), field.typeName(), field.length(), field.precision()))

        for field in pipes_layer.fields():
            if field.name() == "ellip_length":
                output_fields.append(QgsField(field.name(), field.type(), field.typeName(), 10, 1))
            else:
                # Use default type/length/precision from input
                output_fields.append(QgsField(field.name(), field.type(), field.typeName(), field.length(), field.precision()))
                                  
        # Create the sink
        feedback.pushInfo("Creating sink ...")
        (sink, sink_id) = self.parameterAsSink(
            parameters,
            self.OUTPUT_LAYER,
            context,
            output_fields,
            QgsWkbTypes.LineString,
            buildings_layer.sourceCrs()
        )
        
        if not sink:
            raise QgsProcessingException("Invalid output sink.")
                                      
        # Create spatial index for pipes
        feedback.pushInfo("Building spatial index for pipes...")
        spatial_index = QgsSpatialIndex(pipes_layer.getFeatures())
        
        # Set up the distance calculator
        distance_calculator = QgsDistanceArea()
        distance_calculator.setSourceCrs(buildings_layer.sourceCrs(), context.transformContext())
        distance_calculator.setEllipsoid(context.project().ellipsoid())
        
        # Process each building
        feedback.pushInfo("Finding shortest service pipe for each building...")
        for building in buildings:
            if feedback.isCanceled():
                break
        
            building_geom = building.geometry()
        
            # Retrieve n closest candidate pipes
            candidate_pipe_ids = spatial_index.nearestNeighbor(building_geom.boundingBox().center(), 10)  # Adjust '10' as needed
        
            if not candidate_pipe_ids:
                continue
        
            shortest_distance = float("inf")
            closest_pipe_feature = None
            building_nearest_point = None
            pipe_nearest_point = None
        
            for pipe_id in candidate_pipe_ids:
                pipe_request = QgsFeatureRequest().setFilterFid(pipe_id)
                pipe_feature = next(pipes_layer.getFeatures(pipe_request), None)
        
                if pipe_feature is None:
                    continue
        
                pipe_geom = pipe_feature.geometry()
        
                # Calculate the shortest distance between the building and the pipe
                distance, b_nearest_point, p_nearest_point, feedback = self.calculate_distance_between_geometries(
                    building_geom, pipe_geom, distance_calculator, feedback
                )
        
                if distance < shortest_distance:
                    shortest_distance = distance
                    closest_pipe_feature = pipe_feature
                    building_nearest_point = b_nearest_point
                    pipe_nearest_point = p_nearest_point
        
            if closest_pipe_feature is None:
                continue
        
            # Add service pipe feature
            start_point = QgsPointXY(building_nearest_point.x(), building_nearest_point.y())
            end_point = QgsPointXY(pipe_nearest_point.x(), pipe_nearest_point.y())
        
            new_feature = QgsFeature(output_fields)
            new_feature.setGeometry(QgsGeometry.fromPolylineXY([start_point, end_point]))
            new_feature.setAttributes(
                building.attributes() + closest_pipe_feature.attributes()
            )
            sink.addFeature(new_feature, QgsFeatureSink.FastInsert)
           
        
        # Apply symbology if the output layer is loaded
        output_layer = context.getMapLayer(sink_id)
        
        if output_layer is None:
            # Saved output layer: Load it manually
            output_layer_path = self.parameterAsOutputLayer(parameters, self.OUTPUT_LAYER, context)
            if output_layer_path:  # Ensure the path exists
                output_layer = QgsVectorLayer(output_layer_path, "Service pipes", "ogr")
                if output_layer.isValid():
                    feedback.pushInfo("Applying symbology to layer...")
                    QgsProject.instance().addMapLayer(output_layer)
                    self.set_symbology(output_layer, feedback)
                else:
                    feedback.pushInfo("Output layer is not valid, symbology not applied")
            else: 
                feedback.pushInfo("Output layer path is not valid, symbology not applied")
        else:
            # Temporary output layer: Symbology is applied directly
            feedback.pushInfo("Applying symbology to memory layer...")
            self.set_symbology(output_layer, feedback)
        
        if output_layer.isValid():
            feedback.pushInfo("Layer successfully created and symbology applied.")
        else:
            feedback.pushInfo("Layer creation succeeded but was marked as invalid by QGIS.")
           
        feedback.pushInfo("Processing completed.")
        return {self.OUTPUT_LAYER: sink_id}
    
            
    def set_symbology(self, layer, feedback):
        """Set symbology for the service pipes layer."""
        if layer.geometryType() == QgsWkbTypes.LineGeometry:
            
            # Create the main line symbol
            line_symbol = QgsLineSymbol.createSimple({
                'color': 'black',  # Line color
                'width': '0.46',  # Line width
            })
    
            # Create a geometry generator for the start point
            geometry_generator_props = {
                'GeometryType': 'Point',              # Output geometry type
                'OutputType': 'Point'
            }
            
            geometry_generator = QgsGeometryGeneratorSymbolLayer.create(geometry_generator_props)
            geometry_generator.setGeometryExpression('start_point(@geometry)')
            
            # Configure the marker symbol for the geometry generator
            marker_symbol = QgsMarkerSymbol.createSimple({
                'color': 'red',         # Fill color for the circle
                'outline_color': 'black',  # Outline color
                'outline_width': '0.2',  # Outline width
                'size': '2.0'           # Circle size
            })
            geometry_generator.setSubSymbol(marker_symbol)  # Apply marker symbol
    
            # Add the geometry generator to the line symbol
            line_symbol.appendSymbolLayer(geometry_generator)
    
            # Set the layer's renderer to use the composed symbol
            layer.setRenderer(QgsSingleSymbolRenderer(line_symbol))
    
            # Refresh the layer to apply the changes
            layer.triggerRepaint()
                   
    def calculate_distance_between_geometries(self, building_geom, pipe_geom, distance_calculator, feedback):
        """
        Calculate the shortest distance between the nearest point on the building and the nearest point on the pipe.

        Args:
            building_geom (QgsGeometry): The building's geometry.
            pipe_geom (QgsGeometry): The pipe's geometry.
            distance_calculator (QgsDistanceArea): Configured distance calculator.

        Returns:
            float: The shortest distance in meters.
            QgsPoint: The nearest point on the building.
            QgsPoint: The nearest point on the pipe.
        """
        shortest_distance = float("inf")
        building_nearest_point = None
        pipe_nearest_point = None

        # Handle single-part and multi-part geometries
        if pipe_geom.isMultipart():
            segments = pipe_geom.asMultiPolyline()
        else:
            segments = [pipe_geom.asPolyline()]

        # Iterate over each segment in the pipe geometry
        for segment in segments:
            for i in range(len(segment) - 1):
                segment_start = segment[i]
                segment_end = segment[i + 1]
                
                # Convert QgsPointXY to QgsPoint by adding a z-coordinate (e.g., 0)
                segment_start_point = QgsPoint(segment_start.x(), segment_start.y(), 0)
                segment_end_point = QgsPoint(segment_end.x(), segment_end.y(), 0)

                # feedback.pushInfo(f"segment_start: {segment_start}")
                
                # Create a line segment geometry for the current pipe segment
                segment_geom = QgsGeometry.fromPolyline([segment_start_point, segment_end_point])

                # Find the nearest point on the pipe segment to the building geometry
                pipe_point_geom = segment_geom.nearestPoint(building_geom)
                pipe_point = pipe_point_geom.asPoint()

                # Find the nearest point on the building geometry to the pipe point
                building_point_geom = building_geom.nearestPoint(pipe_point_geom)
                building_point = building_point_geom.asPoint()

                # Calculate the distance using the ellipsoid method
                distance = distance_calculator.measureLine(QgsPointXY(building_point), QgsPointXY(pipe_point))

                # Update the shortest distance and nearest points
                if distance < shortest_distance:
                    shortest_distance = distance
                    building_nearest_point = QgsPoint(building_point)
                    pipe_nearest_point = QgsPoint(pipe_point)

        return shortest_distance, building_nearest_point, pipe_nearest_point, feedback

    def is_file_locked(self, file_path, feedback):
        """Check if a file is locked by attempting to access it and provide detailed feedback."""
        if os.path.exists(file_path):
            feedback.pushInfo(f"File exists at path: {file_path}")
            
            # Try opening the file in append mode to check for basic locks
            try:
                with open(file_path, "a"):
                    feedback.pushInfo("File is not locked at the operating system level.")
            except (OSError, IOError) as e:
                feedback.pushInfo(f"File is locked at the OS level. Error encountered: {e}")
                return True
            
            # OGR-specific test for file access
            ogr_ds = ogr.Open(file_path, update=1)  # Open in write mode
            if ogr_ds is None:
                feedback.pushInfo("OGR reports the file is locked or inaccessible.")
                return True
            else:
                feedback.pushInfo("OGR reports the file is accessible.")
                ogr_ds = None  # Close the data source properly
                return False
        else:
            feedback.pushInfo(f"File does not exist at path: {file_path}.")
            return False  # File doesn't exist, so it can't be locked


    def name(self):
        return "Shortest Service Pipes"
    
    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr(self.name())

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr(self.groupId())

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return '2. Thermonet'

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)
    
    def icon(self):
        cmd_folder = os.path.split(inspect.getfile(inspect.currentframe()))[0]
        icon = QIcon(os.path.join(os.path.join(cmd_folder, 'logo-servicep.png')))
        return icon
  
    def shortHelpString(self):
        return (
            "<p><b>Tool Description:</b></p>"
            "<p>This tool creates a new layer of service pipes by connecting "
            "a set of heat pumps/buildings in a buildings input layer with "
            "pipes in a thermonet pipes input layer.</p>"
            "<p><b>How It Works:</b></p>"
            "<ul>"
            "  <li>Calculates the shortest distance between buildings with the "
            "<b>'Thermonet'</b> field set to <b>'Yes'</b> and the nearest pipe.</li>"
            "  <li>Generates a vector layer of service pipes containing all attributes "
            "from both input layers.</li>"
            "</ul>"
            "<p><b>Output Details:</b></p>"
            "<p>After running the tool the output service pipes layer can be "
            "manually modified to place the service pipes in more realistic "
            "locations. <br> " 
            "Note that the output service pipes layer containing ID's of the "
            "buildings/heatloads/HPs is an essential input to the 'Pipe "
            "Topology' tool.</p>"
        )

    def createInstance(self):
        return ServicePipesAlgorithm()
