# -*- coding: utf-8 -*-

"""
/***************************************************************************
 QThermonet
                                 A QGIS plugin
 This plugin links QGIS to pythermonet
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-06-10
        copyright            : (C) 2024 by Jane Lund Andersen/VIA University College
        email                : jana@via.dk
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Jane Lund Andersen/VIA University College'
__date__ = '2025-06-10'
__copyright__ = '(C) 2025 by Jane Lund Andersen/VIA University College'


from qgis.core import (
    QgsCoordinateReferenceSystem,
    QgsExpression,
    QgsExpressionContext,
    QgsExpressionContextUtils,
    QgsFeature,
    QgsFeatureRequest,
    QgsField,
    QgsFields,
    QgsProcessing,
    QgsProcessingAlgorithm,
    QgsProcessingException,
    QgsProcessingParameterFileDestination,
    QgsProcessingParameterVectorLayer,
    QgsProject,
    QgsVectorFileWriter,
    QgsVectorLayer,
    QgsWkbTypes
)
from collections import Counter
from qgis.PyQt.QtCore import QCoreApplication, QVariant
from qgis.PyQt.QtGui import QIcon
from qgis import processing
import os
import inspect

class PipeTopologyAlgorithm(QgsProcessingAlgorithm):
    
    #Handle input/output
    PIPES_LAYER = "PIPES_LAYER"
    SERVICE_PIPES_LAYER = "SERVICE_PIPES_LAYER"
    OUTPUT = "OUTPUT"
    DAT_OUTPUT = "DAT_OUTPUT"

    def initAlgorithm(self, config=None):
        
        #1st input
        param=QgsProcessingParameterVectorLayer(
                self.PIPES_LAYER,
                "Select the Pipes Layer",
                [QgsProcessing.TypeVectorLine],
            )
        param.setHelp(
            "The input layer must:\n"
            "- contain the main pipes network of the thermonet.\n"
            "- Contain the required fields: 'Level' that defines the hierarchy of the pipe-heatpump connections.\n"
            "- Use a compatible CRS (preferably WGS84/EPSG:4326 OR 3857)."
        )

        self.addParameter(param)
        
        #2nd input
        param = QgsProcessingParameterVectorLayer(
                self.SERVICE_PIPES_LAYER,
                "Select the Service Pipes Layer",
                [QgsProcessing.TypeVectorLine],
            )
        param.setHelp(
            "The service pipes layer must:\n"
            "- contain the service pipes connecting the heatpumps to the main thermonet pipes.\n"
            "- Contain the heatpump ID's of the heatpumps in field with name 'id.lokalId' \n"
            "- Use a compatible CRS (preferably WGS84/EPSG:4326 OR 3857)."
        )

        self.addParameter(param)
        
        #output geojson
        self.addParameter(
            QgsProcessingParameterFileDestination(
                self.OUTPUT,
                self.tr('Output GeoJSON'),
                fileFilter="GeoJSON (*.geojson)"  # Filter for file type
            )
        )
        
        #output dat-format
        self.addParameter(
            QgsProcessingParameterFileDestination(
                "DAT_OUTPUT",
                self.tr("Output DAT file"),
                fileFilter="DAT files (*.dat)"
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        pipes_layer = self.parameterAsVectorLayer(parameters, self.PIPES_LAYER, context)
        service_pipes_layer = self.parameterAsVectorLayer(parameters, self.SERVICE_PIPES_LAYER, context)
        output_path = self.parameterAsFileOutput(parameters, self.OUTPUT, context)
        dat_output_path = self.parameterAsFileOutput(parameters, "DAT_OUTPUT", context)

    
        if not pipes_layer or not service_pipes_layer:
            raise QgsProcessingException("Invalid input layers!")
            
        ## Step 0: Re-project layers if necessarty (units should be meters for algorithm to work)
        default_projected_crs = QgsCoordinateReferenceSystem("EPSG:3857")
        
        # Helper: check if CRS is geographic (i.e., degrees)
        def is_geographic(crs):
            return crs.isGeographic()
        
        # Use the projected CRS of the input if it's already projected
        if is_geographic(pipes_layer.crs()):
            feedback.pushInfo(f"Pipes layer is in geographic CRS ({pipes_layer.crs().authid()}), reprojecting to {default_projected_crs.authid()}")
            pipes_layer_proj = processing.run(
                "native:reprojectlayer",
                {
                    'INPUT': pipes_layer,
                    'TARGET_CRS': default_projected_crs,
                    'OUTPUT': 'memory:'
                },
                context=context,
                feedback=feedback
            )['OUTPUT']
        else:
            feedback.pushInfo(f"Pipes layer is already projected: {pipes_layer.crs().authid()}")
            pipes_layer_proj = pipes_layer
        
        # Do the same for source layer
        if is_geographic(service_pipes_layer.crs()):
            feedback.pushInfo(f"Source layer is in geographic CRS ({service_pipes_layer.crs().authid()}), reprojecting to {pipes_layer_proj.crs().authid()}")
            service_pipes_layer_proj = processing.run(
                "native:reprojectlayer",
                {
                    'INPUT': service_pipes_layer,
                    'TARGET_CRS': pipes_layer_proj.crs(),  # ensure it matches pipes
                    'OUTPUT': 'memory:'
                },
                context=context,
                feedback=feedback
            )['OUTPUT']
        else:
            feedback.pushInfo(f"Source layer is already projected: {service_pipes_layer.crs().authid()}")
            service_pipes_layer_proj = service_pipes_layer

            
            
        # Define the fields for the output layer
        feedback.pushInfo("Creating the new fields ...")
        fields = QgsFields()
        
        # List of fields to add: name, type, length, precision (if applicable)
        field_definitions = [
            ("Section", QVariant.String, 255, 0), # String field, length 255
            ("SDR", QVariant.Int, 0, 0),          # Integer field
            ("Trace_(m)", QVariant.Double, 4, 2), # Double field with 2 decimal places
            ("Number_of_traces", QVariant.Int, 0, 0),   # Integer field
            ("Max_pressure_loss_(Pa)", QVariant.Int, 0, 0),   # Integer field
            ("HP_ID_vector", QVariant.String, 0, 0),  # String field
        ]
        
        # Add fields to the QgsFields object
        for field_name, field_type, field_length, field_precision in field_definitions:
            new_field = QgsField(field_name, field_type, len=field_length, prec=field_precision)
            fields.append(new_field)

        # Set CRS from the service_pipes_layer
        crs = service_pipes_layer_proj.crs()

        # Create a writer for the output layer
        writer = QgsVectorFileWriter(
            output_path,
            "UTF-8",
            fields,
            QgsWkbTypes.LineString,
            crs,
            "GeoJSON", #"ESRI Shapefile",
        )

        if writer.hasError() != QgsVectorFileWriter.NoError:
            raise QgsProcessingException(f"Error when creating the output shapefile: {writer.errorMessage()}")
            
        # Prepare the $length expression
        expression = QgsExpression('$length')
        expr_context = QgsExpressionContext()
        expr_context.appendScopes(QgsExpressionContextUtils.globalProjectLayerScopes(service_pipes_layer_proj))
        
        # Open dat file in writing mode and write header        
        dat_file = open(dat_output_path, "w", encoding="utf-8")
        dat_file.write("Section\tSDR\tTrace_(m)\tNumber_of_traces\tMax_pressure_loss_(Pa)\tHP_ID_vector\n")

        # Copy features from the service pipes layer
        feedback.pushInfo("Handling service pipes ...")
        pipeNo = 0
        for feature in service_pipes_layer_proj.getFeatures():
            pipeNo += 1
            new_feature = QgsFeature(fields)
            new_feature.setGeometry(feature.geometry())
            
            # Set the current feature in the expression context
            expr_context.setFeature(feature)
        
            # Evaluate the $length expression
            Trace_length = expression.evaluate(expr_context)
            if expression.hasEvalError():
                raise QgsProcessingException(f"Expression evaluation error: {expression.evalErrorString()}")

            #Update field values
            new_feature["Section"] = "Service_pipe_" + str(pipeNo)
            new_feature["SDR"] = 17
            new_feature["Trace_(m)"] = Trace_length
            new_feature["Number_of_traces"] = 1
            new_feature["Max_pressure_loss_(Pa)"] = 180 * Trace_length
            new_feature["HP_ID_vector"] = feature["id.lokalId"]
            writer.addFeature(new_feature)
            
            # Update dat file
            dat_file.write(
                f"{new_feature['Section']}\t{new_feature['SDR']}\t"
                f"{float(new_feature['Trace_(m)']):.2f}\t"  # Explicitly format to 2 decimal places
                f"{new_feature['Number_of_traces']}\t"
                f"{int(round(new_feature['Max_pressure_loss_(Pa)']))}\t"  # Ensure integer for MaxPLossPa
                f"{new_feature['HP_ID_vector']}\n"
            )
        
        # Copy features from the pipes layer
        feedback.pushInfo("Handling main pipes ...")
        
        # Validate that the 'Level' field exists and all features have valid integer values
        if "Level" not in [field.name() for field in pipes_layer_proj.fields()]:
            raise QgsProcessingException(
                "The 'Level' field is missing in the input pipes layer. "
                "Please ensure a 'Level' field is created and contains integer values "
                "representing the hierarchy of the pipe topology. Each feature in the layer must have a "
                "valid integer value in the 'Level' field before running this algorithm."
            )
            
        # Check that all features have valid integer values in the 'Level' field
        invalid_features = []
        for feature in pipes_layer_proj.getFeatures():
            level_value = feature["Level"]
            if not isinstance(level_value, int):
                invalid_features.append(feature.id())
        
        if invalid_features:
            raise QgsProcessingException(
                f"The following features have invalid or missing values in the 'Level' field: {invalid_features}. "
                "Ensure all features in the 'Level' field contain valid integer values before running this algorithm."
            )

        # Find the lowest level pipes
        max_level = max([feature["Level"] for feature in pipes_layer_proj.getFeatures()])
        
        # Set up sorting of main pipes so topology is handled correctly when assigning Heat-pump ID's
        order_by_clause = QgsFeatureRequest.OrderByClause("Level", ascending=False)
        order_by = QgsFeatureRequest.OrderBy([order_by_clause])
        request = QgsFeatureRequest().setOrderBy(order_by)
        
        # Dictionary to store HP_IDs for processed features
        processed_features = {}

        pipeNo = 0
        # Iterate through the sorted features
        for pipe_feature in pipes_layer_proj.getFeatures(request):
            current_level = pipe_feature["Level"]
            feedback.pushInfo(f"Processing pipe with Level: {current_level}")

            pipeNo += 1
            new_feature = QgsFeature(fields)
            new_feature.setGeometry(pipe_feature.geometry())
            
            # Step 1: calculate the length of the current pipe
            expr_context.setFeature(pipe_feature)
            Trace_length = expression.evaluate(expr_context)
            if expression.hasEvalError():
                raise QgsProcessingException(f"Expression evaluation error: {expression.evalErrorString()}")
            
            
            # Step 2: Retrieve Heat-pump ID's for each pipe
            feedback.pushInfo("") #Empty line for visibility of output
            feedback.pushInfo(f"Retrieving heatpump ID's for pipe {pipeNo} ... ")
            
            # Find nearby service pipes
            Dist = 0.1 # distance in meter, to ensure robustness for near-overlapping features
            nearby_service_pipes = self.find_features_within_distance(pipe_feature, service_pipes_layer_proj, Dist)
            
            # Collect IDs of the nearby service pipes
            connected_ids = []
            for f in nearby_service_pipes:
                if "id.lokalId" in f.fields().names():
                    connected_ids.append(f["id.lokalId"])
                    feedback.pushInfo(f"Service pipe ID found: {f['id.lokalId']}")
                else:
                    feedback.pushInfo("Field 'id.lokalId' not found in feature.")
                    
            # Combine IDs into a comma-separated string
            combined_ids = ", ".join(connected_ids)
                        
            # Print or assign the result
            feedback.pushInfo(f"Pipe ID: {pipe_feature.id()}, Connected Service Pipes: {combined_ids}")
            
            
            # Add HP ID's of lower-level connected pipes
            if current_level < max_level:
                # Find pipes with higher levels within 1 meter of the current feature
                higher_level_ids = []
                for other_feature in pipes_layer_proj.getFeatures():
                    other_level = other_feature["Level"]
                    
                    # Check if the other feature has a higher level and is within 1 meter
                    if other_level > current_level:
                        distance = pipe_feature.geometry().distance(other_feature.geometry())
                        if distance <= 1:  # 1 meter threshold - lower?
                            # Retrieve HP_IDs for the other feature from the processed_features dictionary
                            other_id = other_feature.id()
                            if other_id in processed_features:
                                higher_level_ids.append(processed_features[other_id])
                
                # Combine the higher-level HP_IDs into a single comma-separated string
                if higher_level_ids:
                    combined_ids += ", " + ", ".join(higher_level_ids)
            
            # Check for duplicates and issue a warning
            ids_list = [id.strip() for id in combined_ids.split(",")]  # Split and strip whitespace
            id_counts = Counter(ids_list)  # Count occurrences of each ID
            unique_ids = sorted(set(ids_list))  # Deduplicate and sort
            duplicates = [id for id, count in id_counts.items() if count > 1] # Identify duplicates

            if duplicates:
                feedback.reportError(
                    f"Duplicate HP_IDs found and removed for HP_ID: {', '.join(duplicates)}, make sure this heatpump only connects to one pipe section"
                )
            
            # Reconstruct combined_ids as a comma-separated string
            combined_ids = ", ".join(unique_ids)

            # Update the current feature's combined_ids field
            feedback.pushInfo(f"Feature ID {pipe_feature.id()} updated combined IDs: {combined_ids}")

            
            #Step 3: Update field values
            new_feature["Section"] = "Pipe_branch_" + str(pipeNo)
            new_feature["SDR"] = 17
            new_feature["Trace_(m)"] = Trace_length
            new_feature["Number_of_traces"] = 1
            new_feature["Max_pressure_loss_(Pa)"] = 180 * Trace_length
            new_feature["HP_ID_vector"] = combined_ids
            writer.addFeature(new_feature)
            
            # Store the processed feature's HP_IDs in the dictionary
            current_id = pipe_feature.id()
            processed_features[current_id] = combined_ids

            # Update dat file
            dat_file.write(
                f"{new_feature['Section']}\t{new_feature['SDR']}\t"
                f"{float(new_feature['Trace_(m)']):.2f}\t"  # Explicitly format to 2 decimal places
                f"{new_feature['Number_of_traces']}\t"
                f"{int(round(new_feature['Max_pressure_loss_(Pa)']))}\t"  # Ensure integer for MaxPLossPa
                f"{new_feature['HP_ID_vector']}\n"
            )
           
        
        # Finalize the writer and close dat file
        del writer
        dat_file.close()
        
        # Add the new layer to the QGIS map
        feedback.pushInfo("") # Empty line for visibility of output
        feedback.pushInfo("Adding layer to map ...")
        new_layer = QgsVectorLayer(output_path, "Pipe Topology", "ogr")
        if not new_layer.isValid():
            raise QgsProcessingException(f"Could not load the output file: {output_path}")
        QgsProject.instance().addMapLayer(new_layer)

        feedback.pushInfo("Processing completed successfully.")
        return {
            self.OUTPUT: output_path,
            "DAT_OUTPUT": dat_output_path,
        }
        
    def find_features_within_distance(self, input_feature, reference_layer, distance):
        """
        Find all features in the reference layer that are within a given distance
        of the input feature.
        """
        nearby_features = []
        input_geometry = input_feature.geometry()
        buffer_geometry = input_geometry.buffer(distance, segments=5)  # Create a buffer around the feature
    
        for reference_feature in reference_layer.getFeatures():
            if buffer_geometry.intersects(reference_feature.geometry()):
                nearby_features.append(reference_feature)
        return nearby_features

    def name(self):
        return "Pipe Topology"
    
    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr(self.name())

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr(self.groupId())

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return '2. Thermonet'

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)
    
    def icon(self):
        cmd_folder = os.path.split(inspect.getfile(inspect.currentframe()))[0]
        icon = QIcon(os.path.join(os.path.join(cmd_folder, 'logo6-pipes-alt.png')))
        return icon

    def shortHelpString(self):
        return (
            "<p> This tool creates a pipe topology file for the thermonet that "
            "can be used as input to pythermonet.<p> "
            "<p> The tool requires two input files: <p> "
            "<p> 1. A pipe layer containing the"
            " main pipes of the thermonet with field 'Level' determining the"
            " hierarchy of the pipe network where 0=the highest level and 1,2,3,..."
            " are succesively lower levels. The Level determines which heatpump"
            "ID's are connected to which pipe segment<p> "
            "<p> 2. A service pipes layer containing the service "
            "pipes connecting each building/heatpump to the thermonet. "
            "This layer should hold the ID's of the heatpumps in a field called 'id.lokalId'. <p> "                
            " The tool stores the relevant information in new geojson and dat files. <p>"
            "<p> The output dat-file can be used as input for full "
            "dimensioning of the thermonet using pythermonet <p>"
        )

    def createInstance(self):
        return PipeTopologyAlgorithm()
