# -*- coding: utf-8 -*-

"""
/***************************************************************************
 QThermonet
                                 A QGIS plugin
 This plugin links QGIS to pythermonet
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-06-10
        copyright            : (C) 2024 by Jane Lund Andersen/VIA University College
        email                : jana@via.dk
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Jane Lund Andersen/VIA University College'
__date__ = '2025-06-10'
__copyright__ = '(C) 2025 by Jane Lund Andersen/VIA University College'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

import os
import inspect
from qgis.PyQt.QtGui import QIcon

from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import (QgsProcessingAlgorithm,
                       QgsProcessingException,
                       QgsProcessingParameterEnum,
                       QgsProcessingParameterFile,
                       QgsProcessingParameterFileDestination,
                       QgsProcessingParameterString)

from pythermonet.data.equipment.pipes import load_pipe_catalogue
from pythermonet.io import (
    combine_heatpump_user_and_file,
    read_heat_pump_tsv,
    read_undimensioned_topology_tsv_to_net
)
from pythermonet.core.main import run_full_dimensioning
from pythermonet.domain import BHEConfig, HHEConfig, Brine, HeatPump, Thermonet


class FullDimensioningAlgorithm(QgsProcessingAlgorithm):
    
    #Handle input/output
    PID="PID"
    OUTPUT = 'OUTPUT'
    INPUT_LOAD = 'INPUT_LOAD'
    INPUT_Topology = 'INPUT_Topology'
    HE_MODE = "HE_MODE"
    # INPUT_Brine = 'INPUT_Brine'
    # rhoBrine = "rhoBrine"
    # cBrine = "cBrine"
    # muBrine = "muBrine"
    # lBrine = "lBrine"

    def initAlgorithm(self, config=None):
        """
        Define the parameters for the algorithm.
        """
        # Project name
        param = QgsProcessingParameterString(
                self.PID,
                "Project name"
            )
        param.setHelp(
            "Name of the project (string)\n"
        )
        self.addParameter(param)
        
        
        # Input file for specifying load for heating/cooling
        param = QgsProcessingParameterFile(
                self.INPUT_LOAD,
                self.tr("Input heat/cooling load file:"),
                extension="dat"  # Restrict selection to dat files
            )
        param.setHelp(
            "Heating and/or cooling loads for each heat pump (.dat)\n"
        )
        self.addParameter(param)
        
        
        # Input file containing topology information
        param = QgsProcessingParameterFile(
                self.INPUT_Topology,
                self.tr("Input topology file:"),
                extension="dat"  # Restrict selection to dat files
            )
        param.setHelp(
            "Topology of the thermonet with pipes and service pipes"
            " and hierarchy/level assigned (.dat)\n"
        )
        self.addParameter(param)
        
        # Add a dropdown menu parameter
        param = QgsProcessingParameterEnum(
                self.HE_MODE,
                description="Select heat exchanger source mode",
                options=["BHE", "HHE"],  # Options for the dropdown menu
                defaultValue=0  # Default selection index (0 corresponds to "BHE")
            )
        param.setHelp(
            "Choose whether you would like the source to be configured as a "
            "Borehole Heat Exchanger (BHE) or a Horizontal Heat Exchanger (HHE)"
        )
        self.addParameter(param)
        
        
        # # Input file containing brine information
        # self.addParameter(
        #     QgsProcessingParameterFile(
        #         self.INPUT_Brine,
        #         self.tr("Input brine file:"),
        #         extension="dat"  # Restrict selection to dat files
        #     )
        # )
        
        
        # # Input numerical values for brine density
        # self.addParameter(
        #     QgsProcessingParameterNumber(
        #         self.rhoBrine,
        #         "Brine density (kg/m3), T = 0C",
        #         type=QgsProcessingParameterNumber.Integer,
        #         defaultValue=965,
        #         minValue=0
        #     )
        # )
        
        # # Input numerical values for brine specific heat
        # self.addParameter(
        #     QgsProcessingParameterNumber(
        #         self.cBrine,
        #         "Brine specific heat (J/kg/K)",
        #         type=QgsProcessingParameterNumber.Integer,
        #         defaultValue=4450,
        #         minValue=0
        #     )
        # )
        
        # # Input numerical values for brine dynamic viscosity
        # self.addParameter(
        #     QgsProcessingParameterNumber(
        #         self.muBrine,
        #         "Brine dynamic viscosity (Pa*s)",
        #         type=QgsProcessingParameterNumber.Double,
        #         defaultValue=5e-3,
        #         minValue=0
        #     )
        # )
        
        # # Input numerical values for brine thermal conductivity 
        # self.addParameter(
        #     QgsProcessingParameterNumber(
        #         self.lBrine,
        #         "Brine thermal conductivity (W/m/K)",
        #         type=QgsProcessingParameterNumber.Double,
        #         defaultValue=0.45,
        #         minValue=0.0
        #     )
        # )
        
        # Output file
        self.addParameter(
            QgsProcessingParameterFileDestination(
                self.OUTPUT,
                "Output File",
                "CSV files (*.csv)"
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        output_path = self.parameterAsFileOutput(parameters, self.OUTPUT, context)
        PID = self.parameterAsString(parameters,self.PID,context)
        he_mode_index = self.parameterAsEnum(parameters, self.HE_MODE, context)
        # rho_brine = self.parameterAsInt(parameters,self.rhoBrine, context)
        # c_brine = self.parameterAsInt(parameters,self.cBrine, context)
        # mu_brine = self.parameterAsDouble(parameters,self.muBrine, context)
        # l_brine = self.parameterAsDouble(parameters,self.lBrine, context)
        
        #Handle input layers
        feedback.pushInfo("Checking input layers...")
        HP_file = self.parameterAsFile(parameters, self.INPUT_LOAD, context)
        if not HP_file:
            raise QgsProcessingException("Invalid heat/cooling load input file!")
            
        TOPO_file = self.parameterAsFile(parameters, self.INPUT_Topology, context)
        if not TOPO_file:
            raise QgsProcessingException("Invalid topology input file!")
                        
        # Handle pipe catalogue file
        feedback.pushInfo("Handling pipe file...")
        d_pipes = load_pipe_catalogue().values / 1000 # Convert d_pipes from mm to m
        
        
        # Set brine properties
        feedback.pushInfo("Setting brine properties...")
        # brine_file = self.parameterAsFile(parameters, self.INPUT_Brine, context)
        # if not brine_file:
        #     raise QgsProcessingException("Invalid brine input file!")
        # brine = Brine(rho_brine,c_brine,mu_brine,l_brine)
        brine = Brine(rho=965, c=4450, mu=5e-3, l=0.45)
        
        # Initialise thermonet object
        feedback.pushInfo("Initializing thermonet object...")
        net = Thermonet(
            D_gridpipes=0.3,
            l_p=0.4,
            l_s_H=1.25,
            l_s_C=1.25,
            rhoc_s=2.5e6,
            z_grid=1.2,
            T0=9.03,
            A=7.90
        )

        # Read remaining data from user specified file
        net, pipeGroupNames = read_undimensioned_topology_tsv_to_net(
            TOPO_file, net
        )
        
        # Initialise heat pump object
        feedback.pushInfo("Initializing heat pump object...")
        heat_pump = HeatPump(
            Ti_H=-3,
            Ti_C=20,
            f_peak_H=1,
            t_peak_H=4,
            f_peak_C=1,
            t_peak_C=4
        )
        # Read remaining data from user specified file
        heat_pump_input = read_heat_pump_tsv(HP_file)
        heat_pump = combine_heatpump_user_and_file(heat_pump, heat_pump_input)
        
        # Heat source (either BHE or HHE)
        feedback.pushInfo("Setting heat source configuration...")
        # Map the index to the corresponding option
        he_mode = ["BHE", "HHE"][he_mode_index]
        if he_mode == "BHE":
            feedback.pushInfo("Performing BHE configuration...")
            source_config = BHEConfig(q_geo = 0.0185, r_b=0.152/2, r_p=0.02, SDR=11, l_ss=2.36, rhoc_ss=2.65e6, l_g=1.75, rhoc_g=3e6, D_pipes=0.015, NX=1, D_x=15, NY=6, D_y=15, gFuncMethod='ICS')
        else:
            feedback.pushInfo("Performing HHE configuration...")
            source_config = HHEConfig(N_HHE=6, d=0.04, SDR=17, D=1.5)
            
        # Full dimensioning of pipes and sources - results printed to console
        feedback.pushInfo("Dimensioning pipes and sources...")
        run_full_dimensioning(PID, d_pipes, brine, net, heat_pump, pipeGroupNames, source_config)
        
        # print_project_id(PID)
        # print_source_dimensions(source_config,net)
        
        # # Write to output file
        # feedback.pushInfo("Writing output...")
        # with open(output_path, "w", newline="") as csv_file:
        #     writer = csv.writer(csv_file)
        #     writer.writerows(table_content)

        feedback.pushInfo("Processing complete!")
        
        return {self.OUTPUT: output_path}

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'Full dimensioning'

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr(self.name())

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr(self.groupId())

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return '3. Dimensioning'

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)
    
    def icon(self):
        cmd_folder = os.path.split(inspect.getfile(inspect.currentframe()))[0]
        icon = QIcon(os.path.join(os.path.join(cmd_folder, 'logo.png')))
        return icon
    
    def shortHelpString(self):
        return ("<p> This tool performs full dimensioning of a thermonet. <p>"
                "<p> Input files: <br> "
                "1. Heating/cooling load for each building/heatpump. <p>"
                "2. Pipe network topology: Topology of pipes and service pipes "
                "in thermonet circulating fluid to heatpumps/buildings. "
                "Can be created using the 'Pipe Topology' tool. <p>"
                "3. Heat exchanger source mode: Borehole (BHE) or Horizontal (HHE)"
                "<p> Input parameters and results are stored in an output "
                "report.dat file (not yet implemented). <p>"
                )

    def createInstance(self):
        return FullDimensioningAlgorithm()
