"""
***************************************************************************
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
***************************************************************************
"""

import json
import time
from pathlib import Path
from typing import Any, List, Optional

import processing
from qgis.core import (
    Qgis,
    QgsBlockingNetworkRequest,
    QgsCoordinateReferenceSystem,
    QgsFeature,
    QgsFeatureRequest,
    QgsFeatureSink,
    QgsField,
    QgsFields,
    QgsProcessing,
    QgsProcessingAlgorithm,
    QgsProcessingContext,
    QgsProcessingException,
    QgsProcessingFeatureSource,
    QgsProcessingFeatureSourceDefinition,
    QgsProcessingFeedback,
    QgsProcessingParameterBoolean,
    QgsProcessingParameterDefinition,
    QgsProcessingParameterFeatureSink,
    QgsProcessingParameterFeatureSource,
    QgsProcessingParameterField,
    QgsProcessingParameterNumber,
    QgsProcessingUtils,
    QgsVectorLayer,
    QgsWkbTypes,
)
from qgis.PyQt.QtCore import QCoreApplication, QUrl, QVariant
from qgis.PyQt.QtNetwork import QNetworkRequest

from qsunpotential.toolbelt.pvgis_params import (
    getAngle,
    getAspectFromAzimuth,
    getPeakpower,
)

API_URL = "https://re.jrc.ec.europa.eu/api/PVcalc?"
LOSS = 14
OUTPUT_FORMAT = "json"
USE_HORIZON = 1
RESOURCE_PATH = Path(__file__).parent.parent / "resources"
P50_STYLE_FILE = RESOURCE_PATH / "classification_p50.qml"
EY_STYLE_FILE = RESOURCE_PATH / "classification_ey.qml"


class PVGISRequest(QgsProcessingAlgorithm):
    """
    This is an example algorithm that takes a vector layer and
    creates a new identical one.

    It is meant to be used as an example of how to create your own
    algorithms and explain methods and variables used to do it. An
    algorithm like this will be available in all elements, and there
    is not need for additional work.

    All Processing algorithms should extend the QgsProcessingAlgorithm
    class.
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    LAYER_INPUT = "LAYER_INPUT"
    ID_ROOF = "ID_ROOF"
    OUTPUT_MONTH = "OUTPUT_MONTH"
    OUTPUT_YEAR = "OUTPUT_YEAR"
    OUTPUT_LAYER = "OUTPUT_LAYER"
    OUTPUT_LAYOUT = "OUTPUT_LAYOUT"

    ANGLE_INPUT = "ANGLE_INPUT"
    MIN_ANGLE_INPUT = "MIN_ANGLE_INPUT"
    SURFACE_INPUT = "SURFACE_INPUT"
    COEFF_INPUT = "COEFF_INPUT"
    OCCUPATION_INPUT = "OCCUPATION_INPUT"
    AZIMUTH_INPUT = "AZIMUTH_INPUT"
    FORCED_ANGLE = "FORCED_ANGLE"

    def tr(self, string):
        """
        Returns a translatable string with the self.tr() function.
        """
        return QCoreApplication.translate("Processing", string)

    def name(self) -> str:
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return "myscript"

    def displayName(self) -> str:
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return "Requête API PVGIS"

    def shortHelpString(self) -> str:
        """
        Returns a localised short helper string for the algorithm. This string
        should provide a basic description about what the algorithm does and the
        parameters and outputs associated with it.
        """
        help_text = "Cet algorithme appelle l'API PVGIS sur une couche de toitures."
        return help_text

    def initAlgorithm(self, config: Optional[dict[str, Any]] = None):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """
        layer_input = QgsProcessingParameterFeatureSource(
            self.LAYER_INPUT,
            "Couche des toitures",
            [QgsProcessing.SourceType.VectorPolygon],
        )
        layer_input.setHelp(
            "Couche vecteur de type polygone représentant les toitures."
        )
        self.addParameter(layer_input)
        self.addParameter(
            QgsProcessingParameterField(
                self.ID_ROOF,
                description=self.tr("Identifiant des toitures"),
                parentLayerParameterName=self.LAYER_INPUT,
                type=QgsProcessingParameterField.DataType.String,
            )
        )
        angle_input = QgsProcessingParameterField(
            self.ANGLE_INPUT,
            description=self.tr("Inclinaison des toitures (°)"),
            parentLayerParameterName=self.LAYER_INPUT,
            type=QgsProcessingParameterField.DataType.Numeric,
        )
        self.addParameter(angle_input)
        min_angle_input = QgsProcessingParameterNumber(
            name=self.MIN_ANGLE_INPUT,
            description=self.tr("Angle d'inclinaison minimal (°)"),
            type=QgsProcessingParameterNumber.Double,
            defaultValue=5,
            minValue=0,
        )
        min_angle_input.setHelp(
            "Seuil en déçà duquel l'inclinaison de la toiture est forcée à un angle (de 10° par défaut)",
        )
        min_angle_input.setFlags(
            min_angle_input.flags() | QgsProcessingParameterDefinition.Flag.FlagAdvanced
        )
        self.addParameter(min_angle_input)
        forced_angle_value = QgsProcessingParameterNumber(
            name=self.FORCED_ANGLE,
            description=self.tr("Angle d'inclinaison de forçage (°)"),
            defaultValue=10,
        )
        forced_angle_value.setHelp(
            "Valeur prise par les toitures dont l'inclinaison est "
            "inférieure au seuil minimal fixé"
        )
        forced_angle_value.setFlags(
            forced_angle_value.flags()
            | QgsProcessingParameterDefinition.Flag.FlagAdvanced
        )
        self.addParameter(forced_angle_value)
        surface_input = QgsProcessingParameterField(
            self.SURFACE_INPUT,
            description=self.tr("Surface des toitures (m²)"),
            parentLayerParameterName=self.LAYER_INPUT,
            type=QgsProcessingParameterField.DataType.Numeric,
        )
        surface_input.setHelp("Surface réelle (3D) de la toiture, en mètres carrés")
        self.addParameter(surface_input)
        coeff_peakpower = QgsProcessingParameterNumber(
            name=self.COEFF_INPUT,
            description=self.tr("Coefficient peakpower (kW/m²)"),
            type=QgsProcessingParameterNumber.Double,
            defaultValue=0.215,
            minValue=0,
        )
        coeff_peakpower.setMetadata({"widget_wrapper": {"decimals": 3}})
        coeff_peakpower.setHelp(
            "Ce coefficient est utilisé dans le calcul de la puissance crête (peakpower), "
            "dont la formule est la suivante : "
            "peakpower = surface_toiture_3D * taux_occupation * coeff"
        )
        self.addParameter(coeff_peakpower)
        coeff_peakpower.setFlags(
            coeff_peakpower.flags() | QgsProcessingParameterDefinition.Flag.FlagAdvanced
        )
        occupation_input = QgsProcessingParameterNumber(
            name=self.OCCUPATION_INPUT,
            description=self.tr("Taux d'occupation (%)"),
            type=QgsProcessingParameterNumber.Double,
            defaultValue=70,
            minValue=0,
            maxValue=100,
        )
        occupation_input.setHelp(
            "Taux d'occupation de la toiture par les cellules photovoltaïques"
        )
        occupation_input.setFlags(
            occupation_input.flags()
            | QgsProcessingParameterDefinition.Flag.FlagAdvanced
        )
        self.addParameter(occupation_input)
        azimuth_input = QgsProcessingParameterField(
            self.AZIMUTH_INPUT,
            description=self.tr("Orientation des toitures (°)"),
            parentLayerParameterName=self.LAYER_INPUT,
            type=QgsProcessingParameterField.DataType.Numeric,
        )
        azimuth_input.setHelp(
            "Orientation de la toiture comprise entre 0 et 360 par rapport au Nord."
            "Le plugin convertira cette orientation par rapport au Sud, "
            "selon la norme utilisée par PVGIS."
        )
        self.addParameter(azimuth_input)
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT_MONTH,
                description="PVGIS_PVcalc_monthly",
                type=QgsProcessing.SourceType.Vector,
            )
        )
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT_YEAR,
                description="PVGIS_PVcalc_totals",
                type=QgsProcessing.SourceType.Vector,
            )
        )
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT_LAYER,
                description="Toitures",
                type=QgsProcessing.SourceType.VectorPolygon,
            )
        )
        layout = QgsProcessingParameterBoolean(
            self.OUTPUT_LAYOUT, description="Classification des toitures selon P50"
        )
        layout.setHelp(
            "Par défaut, la couche de toiture en sortie sera classifiée "
            "selon la production d’énergie moyenne par an (champ ey)"
        )
        self.addParameter(layout)

    def processAlgorithm(
        self,
        parameters: dict[str, Any],
        context: QgsProcessingContext,
        feedback: QgsProcessingFeedback,
    ) -> dict[str, Any]:
        """
        Here is where the processing itself takes place.
        """
        source = self.parameterAsSource(parameters, self.LAYER_INPUT, context)
        self.input_for_check = self.get_filtered_layer(parameters, context, feedback)
        source_point = self.prepareRoofPoints(
            feedback,
        )["OUTPUT"]
        roof_id = self.parameterAsString(parameters, self.ID_ROOF, context)
        angle = self.parameterAsString(parameters, self.ANGLE_INPUT, context)
        angle_threshold = self.parameterAsDouble(
            parameters, self.MIN_ANGLE_INPUT, context
        )
        forced_angle = self.parameterAsDouble(parameters, self.FORCED_ANGLE, context)
        azimuth = self.parameterAsString(parameters, self.AZIMUTH_INPUT, context)
        area = self.parameterAsString(parameters, self.SURFACE_INPUT, context)
        coeff = self.parameterAsDouble(parameters, self.COEFF_INPUT, context)
        occupation = self.parameterAsDouble(parameters, self.OCCUPATION_INPUT, context)
        self.layout_p50 = self.parameterAsBoolean(
            parameters, self.OUTPUT_LAYOUT, context
        )

        (sink_year, sink_year_id) = self.parameterAsSink(
            parameters,
            self.OUTPUT_YEAR,
            context,
            self.getYearFields(roof_id_field=roof_id),
        )
        (sink_month, sink_month_id) = self.parameterAsSink(
            parameters,
            self.OUTPUT_MONTH,
            context,
            self.getMonthFields(roof_id_field=roof_id),
        )
        # Couche des toitures contenant les résultats annuels
        (sink_layer, self.sink_layer_id) = self.parameterAsSink(
            parameters,
            self.OUTPUT_LAYER,
            context,
            self.getOutputLayerFields(source, roof_id),
            Qgis.WkbType.Polygon,
            source.sourceCrs(),
        )
        self.sink_year_id = sink_year_id
        if sink_month is None:
            raise QgsProcessingException(
                self.invalidSinkError(parameters, self.OUTPUT_MONTH)
            )
        if sink_year is None:
            raise QgsProcessingException(
                self.invalidSinkError(parameters, self.OUTPUT_YEAR)
            )
        if sink_layer is None:
            raise QgsProcessingException(
                self.invalidSinkError(parameters, self.OUTPUT_LAYER)
            )
        # Compute the number of steps to display within the progress bar and
        # get features from source
        total = 100.0 / source.featureCount() if source.featureCount() else 0
        features = source_point.getFeatures()

        for current, feature in enumerate(features):
            # Stop the algorithm if cancel button has been clicked
            if feedback.isCanceled():
                break
            # Build request URL
            url = API_URL
            url += f"lat={feature.geometry().asPoint().y()}&lon={feature.geometry().asPoint().x()}"
            peakpower = getPeakpower(
                area_3d=feature[area], occupation_rate=occupation, coeff=coeff
            )
            url += f"&peakpower={peakpower}"
            url += f"&loss={LOSS}"
            url += f"&outputformat={OUTPUT_FORMAT}"
            angle_pvgis = getAngle(
                angle_pv=feature[angle],
                min_threshold=angle_threshold,
                forced_angle=forced_angle,
            )
            url += f"&angle={angle_pvgis}"
            azimuth_pv = getAspectFromAzimuth(feature[azimuth])
            url += f"&aspect={azimuth_pv}"  # Azimuth par rapport au Sud
            url += f"&usehorizon={USE_HORIZON}"

            request = QgsBlockingNetworkRequest()
            error_code = request.get(QNetworkRequest(QUrl(url)))
            self.handleErrorResponse(
                error_code=error_code,
                request=request,
                url=url,
                feedback=feedback,
                retry=True,  # Relance une fois la requête dans le cas de surcharge au niveau serveur
            )
            if error_code != QgsBlockingNetworkRequest.ErrorCode.NoError:
                feedback.reportError(f"L'entité {feature[roof_id]} n'est pas traitée.")
                continue
            pvgis_result = request.reply().content().data().decode("utf-8")

            data = json.loads(pvgis_result)

            # Process output features and add in sinks
            # Store year results in table
            year_feature = QgsFeature()
            year_feature.setFields(self.getYearFields(roof_id))
            year_feature = self.getYearFeature(
                year_feature,
                data["outputs"]["totals"],
                peakpower=peakpower,
                roof_id=roof_id,
                roof_id_value=feature[roof_id],
            )
            sink_year.addFeature(year_feature, QgsFeatureSink.Flag.FastInsert)
            # Store year result in toiture
            filter_expr = f"{roof_id}='{feature[roof_id]}'"
            filtered = source.getFeatures(
                request=QgsFeatureRequest().setFilterExpression(filter_expr)
            )
            toiture = QgsFeature()
            toiture.setFields(source.fields())
            filtered.nextFeature(toiture)
            toiture_jointure = QgsFeature()
            toiture_jointure.setFields(self.getOutputLayerFields(source, roof_id))
            toiture_jointure.setGeometry(toiture.geometry())
            for (
                fieldname,
                value,
            ) in toiture.attributeMap().items():
                toiture_jointure[fieldname] = value
            for (
                fieldname,
                value,
            ) in year_feature.attributeMap().items():
                toiture_jointure[fieldname] = value
            sink_layer.addFeature(toiture_jointure)

            # Store month results in table
            month_features = self.getMonthFeatures(
                data["outputs"]["monthly"], roof_id, feature[roof_id]
            )
            for fmonth in month_features:
                sink_month.addFeature(fmonth, QgsFeatureSink.Flag.FastInsert)
            # Update the progress bar
            feedback.setProgress(int(current * total))

        # Sauvegarde les variables utiles dans le post processing
        self.roof_id = roof_id
        self.sink_layer = sink_layer
        return {
            self.OUTPUT_MONTH: sink_month_id,
            self.OUTPUT_YEAR: self.sink_year_id,
            self.OUTPUT_LAYER: self.sink_layer_id,
        }

    def getMonthFeatures(
        self, data_pvgis_monthly: dict, roof_id_field: str, roof_id_value: str
    ) -> List[QgsFeature]:
        """Put PVGIS monthly results on a roof in features.


        :param data_pvgis_monthly: PVGIS month results
        :type data_pvgis_monthly: dict
        :param roof_id: roof id field name
        :type roof_id: str
        :param roof_id_value: feature's roof id value
        :type roof_id_value: str
        :return: list of 12 features (for each month) storing pvgis results
        :rtype: List[QgsFeature]
        """
        features_by_month = []

        for m_data in data_pvgis_monthly["fixed"]:
            fmonth = QgsFeature()
            fmonth.setFields(self.getMonthFields(roof_id_field))
            fmonth.setAttribute(roof_id_field, roof_id_value)
            fmonth.setAttribute("month", m_data["month"])
            fmonth.setAttribute("e_d", m_data["E_d"])
            fmonth.setAttribute("e_m", m_data["E_m"])
            fmonth.setAttribute("h_i_d", m_data["H(i)_d"])
            fmonth.setAttribute("h_i_m", m_data["H(i)_m"])
            fmonth.setAttribute("sd_m", m_data["SD_m"])
            features_by_month.append(fmonth)

        return features_by_month

    def getYearFeature(
        self,
        feature: QgsFeature,
        data_pvgis_totals: dict,
        peakpower: float,
        roof_id: str,
        roof_id_value: str,
    ) -> QgsFeature:
        """Store PVGIS year data in a feature

        :param feature: emmpty feature
        :type feature: QgsFeature
        :param data_pvgis_totals: PVGIS year results
        :type data_pvgis_totals: dict
        :param peakpower: value of peakpower
        :type peakpower: float
        :param roof_id: roof id field name
        :type roof_id: str
        :param roof_id_value: feature's roof id value
        :type roof_id_value: str
        :return: filled feature
        :rtype: QgsFeature
        """
        y_data = data_pvgis_totals["fixed"]
        feature.setAttribute(roof_id, roof_id_value)
        feature.setAttribute("e_d", y_data["E_d"])
        feature.setAttribute("e_m", y_data["E_m"])
        feature.setAttribute("e_y", y_data["E_y"])
        feature.setAttribute("h_i_d", y_data["H(i)_d"])
        feature.setAttribute("h_i_m", y_data["H(i)_m"])
        feature.setAttribute("sd_m", y_data["SD_m"])
        feature.setAttribute("sd_y", y_data["SD_y"])
        feature.setAttribute("l_aoi", y_data["l_aoi"])
        feature.setAttribute("l_spec", y_data["l_spec"])
        feature.setAttribute("l_tg", y_data["l_tg"])
        feature.setAttribute("l_total", y_data["l_total"])
        feature.setAttribute("peakpower", peakpower)
        feature.setAttribute("p50", y_data["E_y"] / peakpower)
        return feature

    def getMonthFields(self, roof_id_field) -> QgsFields:
        """Get PVGIS fields for monthly results.

        :param roof_id_field: roof id field name
        :type roof_id_field: str
        :return: fields for PVGIS monthly output
        :rtype: QgsFields
        """
        fields = QgsFields()
        fields.append(QgsField(roof_id_field, QVariant.String))
        fields.append(QgsField("month", QVariant.Int))
        fields.append(QgsField("e_d", QVariant.Double))
        fields.append(QgsField("e_m", QVariant.Double))
        fields.append(QgsField("h_i_d", QVariant.Double))
        fields.append(QgsField("h_i_m", QVariant.Double))
        fields.append(QgsField("sd_m", QVariant.Double))

        return fields

    def getYearFields(self, roof_id_field) -> QgsFields:
        """Get PVGIS fields for yearly results

        :param roof_id_field: roof id field name
        :type roof_id_field: str
        :return: fields for PVGIS yearly output
        :rtype: QgsFields
        """
        fields = QgsFields()
        fields.append(QgsField(roof_id_field, QVariant.String))
        fields.append(QgsField("e_d", QVariant.Double))
        fields.append(QgsField("e_m", QVariant.Double))
        fields.append(QgsField("e_y", QVariant.Double))
        fields.append(QgsField("h_i_d", QVariant.Double))
        fields.append(QgsField("h_i_m", QVariant.Double))
        fields.append(QgsField("sd_m", QVariant.Double))
        fields.append(QgsField("sd_y", QVariant.Double))
        fields.append(QgsField("l_aoi", QVariant.Double))
        fields.append(QgsField("l_spec", QVariant.Double))
        fields.append(QgsField("l_tg", QVariant.Double))
        fields.append(QgsField("l_total", QVariant.Double))
        fields.append(QgsField("p50", QVariant.Double))
        fields.append(QgsField("peakpower", QVariant.Double))

        return fields

    def get_filtered_layer(self, parameters, context, feedback) -> QgsVectorLayer:
        """Filter layer entities manually.


        :param parameters: _description_
        :type parameters: _type_
        :param context: _description_
        :type context: _type_
        :param feedback: _description_
        :type feedback: _type_
        :raises QgsProcessingException: _description_
        :raises QgsProcessingException: _description_
        :return: _description_
        :rtype: QgsVectorLayer
        """
        layer_param = parameters[self.LAYER_INPUT]

        # Récupérer la source de la couche (feature source)
        feature_source = self.parameterAsSource(parameters, self.LAYER_INPUT, context)

        if feature_source is None:
            raise QgsProcessingException(
                "Impossible d'obtenir la source de la couche d'entrée."
            )

        # On tente de filtrer uniquement si c'est bien une FeatureSourceDefinition
        if (
            isinstance(layer_param, QgsProcessingFeatureSourceDefinition)
            and layer_param.selectedFeaturesOnly
        ):
            feedback.pushInfo("Filtrage manuel des entités sélectionnées...")

            # Récupération des entités filtrées (QGIS filtre déjà selon la sélection ici)
            selected_feats = list(feature_source.getFeatures())

            if not selected_feats:
                raise QgsProcessingException("Aucune entité sélectionnée.")

            # On doit reconstruire une couche mémoire temporaire avec ces entités
            fields = feature_source.fields()
            geom_type = QgsWkbTypes.displayString(feature_source.wkbType())
            crs = feature_source.sourceCrs().authid()
            temp_layer = QgsVectorLayer(
                f"{geom_type}?crs={crs}", "selection_temp", "memory"
            )
            temp_provider = temp_layer.dataProvider()
            temp_provider.addAttributes(fields)
            temp_layer.updateFields()
            temp_provider.addFeatures(selected_feats)

            return temp_layer

        # Si pas de sélection spécifique demandée, retourne la couche complète
        return self.parameterAsVectorLayer(parameters, self.LAYER_INPUT, context)

    def prepareRoofPoints(
        self,
        feedback: QgsProcessingFeedback,
    ) -> dict[str, Any]:
        """Process input polygon layer.

        This function applies the following operations on the polygon layer:
        - check valid geometries among the layer entities
        - retrieve a point feature inside each valid polygon entity
        - transform output layer point into the EPSG:4326 coordinate reference system

        The output layer will then be usable for PVGIS API requests.

        :return: point layer in EPSG:4326
        :rtype: dict[str, Any]
        """
        # Get valid geometries
        valid = processing.run(
            "qgis:checkvalidity",
            {
                "INPUT_LAYER": self.input_for_check,
                "METHOD": 2,  # GEOS
                "IGNORE_RING_SELF_INTERSECTION": False,
                "VALID_OUTPUT": QgsProcessing.TEMPORARY_OUTPUT,
            },
        )["VALID_OUTPUT"]
        feedback.pushInfo("Géométries valides récupérées")
        if feedback.isCanceled():
            return {}
        # Get point on surface from valid geometries
        points = processing.run(
            "native:pointonsurface",
            {
                "INPUT": valid,
                "ALL_PARTS": False,
                "OUTPUT": "TEMPORARY_OUTPUT",
            },
        )["OUTPUT"]
        if feedback.isCanceled():
            return {}
        # Reproject into EPSG:4326
        reprojected = processing.run(
            "native:reprojectlayer",
            {
                "INPUT": points,
                "TARGET_CRS": QgsCoordinateReferenceSystem("EPSG:4326"),
                "CONVERT_CURVED_GEOMETRIES": False,
                "OUTPUT": "TEMPORARY_OUTPUT",
            },
        )

        return reprojected

    def handleErrorResponse(
        self,
        error_code: QgsBlockingNetworkRequest.ErrorCode,
        request: QgsBlockingNetworkRequest,
        url: str,
        feedback: QgsProcessingFeedback,
        retry: bool = False,
    ) -> dict:
        """Handle API response error.

        The following status codes are handled:
        - Status 400: bad request, case when a parameter has a bad value (e.g. angle value is > 90°)
        - Status 429: too many requests, case when API calls exceed the rate limit of
            30 calls/seconds
        - Status 529: Site is overloaded, the request should be repeated after a while

        :param error_code: error code resulting from a previous request
        :type error_code: QgsBlockingNetworkRequest.ErrorCode
        :param request: request used to output error code
        :type request: QgsBlockingNetworkRequest
        :param url: URL to request API
        :type url: str
        :param feedback: feedback used to push info messages or report errors
        :type feedback: QgsProcessingFeedback
        :return: API response content
        :rtype: dict
        """
        # get the API response error to log it
        req_reply = request.reply()
        err_msg = f"{request.errorMessage()}. "
        api_response_error = json.loads(str(req_reply.content(), "UTF8"))
        if error_code != QgsBlockingNetworkRequest.ErrorCode.NoError:
            feedback.reportError(
                self.tr("Erreur lors de la requête PVGIS : {}".format(err_msg))
            )
            status_code = api_response_error["status"]
            feedback.reportError(
                f"API error message (status {status_code}): {api_response_error['message']}"
            )
            if status_code == 429:  # Too Many Requests (> 30 calls/s)
                time.sleep(seconds=0.02)
            elif status_code == 529:  # Site is overloaded
                time.sleep(seconds=5)
                error_code = request.get(QNetworkRequest(url=QUrl(url)))
            else:  # all other possible errors
                return api_response_error
        else:
            return api_response_error
        if retry:  # On relance la fonction
            new_error_code = request.get(QNetworkRequest(url=QUrl(url)))
            new_api_response = self.handleErrorResponse(
                error_code=new_error_code,
                request=request,
                url=url,
                feedback=feedback,
                retry=False,
            )
            return new_api_response
        else:
            return api_response_error

    def getOutputLayerFields(
        self, source: QgsProcessingFeatureSource, roof_id: str
    ) -> QgsFields:
        output_layer_fields = QgsFields()
        for f in source.fields():  # Champs de la couche d'entrée
            output_layer_fields.append(f)
        for f in self.getYearFields(
            roof_id_field=roof_id
        ):  # Champs de la table des résultats annuels
            output_layer_fields.append(f)
        return output_layer_fields

    def postProcessAlgorithm(
        self, context: QgsProcessingContext, feedback: QgsProcessingFeedback
    ) -> dict:
        """Apply style to output layer

        :param context: _description_
        :type context: QgsProcessingContext
        :param feedback: _description_
        :type feedback: QgsProcessingFeedback
        :return: stylished output layer
        :rtype: output layer
        """
        style_file = str(P50_STYLE_FILE) if self.layout_p50 else str(EY_STYLE_FILE)
        processed_layer = QgsProcessingUtils.mapLayerFromString(
            self.sink_layer_id, context
        )
        feedback.pushInfo(f"Classification selon P50 : {self.layout_p50}")
        msg, style_ok = processed_layer.loadNamedStyle(style_file)
        if style_ok is False:
            feedback.reportError(
                f"{msg}. La couche {self.sink_layer_id} sera chargée sans style"
            )
        else:
            feedback.pushInfo(
                f"Application d'une couche de style {style_file} "
                f"sur la couche {self.sink_layer_id}"
            )
            processed_layer.triggerRepaint()
        return {self.OUTPUT_LAYER: self.sink_layer_id}

    def createInstance(self):
        return self.__class__()
