# SPDX-License-Identifier: GPL-2.0-or-later

# QGIS
from qgis.core import QgsProcessing, QgsProcessingFeatureBasedAlgorithm, QgsWkbTypes
from qgis.PyQt.QtCore import QCoreApplication

# Projet
from qsfcgal.core.sfcgal_wrapper import QgsGeometry_to_SFCGAL, SFCGAL_to_QgsGeometry


class Triangulate_2dz(QgsProcessingFeatureBasedAlgorithm):
    """
    This is an example algorithm that takes a vector layer and
    creates a new identical one.

    It is meant to be used as an example of how to create your own
    algorithms and explain methods and variables used to do it. An
    algorithm like this will be available in all elements, and there
    is not need for additional work.

    All Processing algorithms should extend the QgsProcessingAlgorithm
    class.
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    INPUT = "INPUT"

    def tr(self, string):
        """
        Returns a translatable string with the self.tr() function.
        """
        return QCoreApplication.translate("Processing", string)

    def createInstance(self):
        return Triangulate_2dz()

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return "Triangulate_2dz"

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr("Triangulate 2dz")

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr("2D")

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return "geometry"

    def outputName(self):
        return self.tr("Triangulate 2dz")

    def shortHelpString(self):
        """
        Returns a localised short helper string for the algorithm. This string
        should provide a basic description about what the algorithm does and the
        parameters and outputs associated with it..
        """
        return self.tr(
            "This algorithm generates a triangulated irregular network (TIN) from polygon or line"
        )

    def inputLayerTypes(self):
        return [
            QgsProcessing.SourceType.TypeVectorPolygon,
            QgsProcessing.SourceType.TypeVectorLine,
            QgsProcessing.SourceType.TypeVectorPoint,
        ]

    def outputWkbType(self, input_wkb_type):
        return QgsWkbTypes.Type.TIN

    def initParameters(self, config=None):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """
        pass

    def prepareAlgorithm(self, parameters, context, feedback):
        return True

    def processFeature(self, feature, context, feedback):
        if feedback.isCanceled():
            return []

        try:
            geom_input = feature.geometry()
            geom_sfcgal = QgsGeometry_to_SFCGAL(geom_input)
            result_geom = SFCGAL_to_QgsGeometry(geom_sfcgal.triangulate_2dz())
            feature.setGeometry(result_geom)
        except Exception as e:
            feedback.pushInfo(
                f"Error during execution of triangulate_2dz for feature {feature.id()}: {e}"
            )

        return [feature]
