#! python3  # noqa: E265

"""
Processing provider module.
"""

# PyQGIS
from qgis.core import QgsProcessingProvider
from qgis.PyQt.QtCore import QCoreApplication
from qgis.PyQt.QtGui import QIcon

# project
from qsfcgal.__about__ import __icon_path__, __title__, __version__

# Processing
from qsfcgal.processing.alpha_shapes import AlphaShapes
from qsfcgal.processing.approximate_medial_axis import ApproximateMedialAxis
from qsfcgal.processing.convexhull import Convexhull
from qsfcgal.processing.difference import Difference
from qsfcgal.processing.extrude import Extrude
from qsfcgal.processing.extrude_polygon_straight_skeleton import (
    ExtrudePolygonStraightSkeleton,
)
from qsfcgal.processing.force_lhr import Force_LHR
from qsfcgal.processing.force_rhr import Force_RHR
from qsfcgal.processing.intersection import Intersection
from qsfcgal.processing.intersection_3d import Intersection_3D
from qsfcgal.processing.line_sub_string import LineSubString
from qsfcgal.processing.minkowski_sum import Minkowski_sum
from qsfcgal.processing.offset_polygon import OffsetPolygon
from qsfcgal.processing.optimal_alpha_shapes import OptimalAlphaShapes
from qsfcgal.processing.slice_3d import Slice_3D
from qsfcgal.processing.straight_skeleton import StraightSkeleton
from qsfcgal.processing.translate import Translate
from qsfcgal.processing.translate_3d import Translate_3D
from qsfcgal.processing.triangulate_2dz import Triangulate_2dz
from qsfcgal.processing.union import Union

# ############################################################################
# ########## Classes ###############
# ##################################


class QsfcgalProvider(QgsProcessingProvider):
    """
    Processing provider class.
    """

    def loadAlgorithms(self):
        """Loads all algorithms belonging to this provider."""
        self.addAlgorithm(StraightSkeleton())
        self.addAlgorithm(Convexhull())
        self.addAlgorithm(Difference())
        self.addAlgorithm(Intersection())
        self.addAlgorithm(Union())
        self.addAlgorithm(Minkowski_sum())
        self.addAlgorithm(OptimalAlphaShapes())
        self.addAlgorithm(ApproximateMedialAxis())
        self.addAlgorithm(AlphaShapes())
        self.addAlgorithm(OffsetPolygon())
        self.addAlgorithm(Force_LHR())
        self.addAlgorithm(Force_RHR())
        self.addAlgorithm(Translate_3D())
        self.addAlgorithm(Translate())
        self.addAlgorithm(Intersection_3D())
        self.addAlgorithm(Extrude())
        self.addAlgorithm(Slice_3D())
        self.addAlgorithm(LineSubString())
        self.addAlgorithm(ExtrudePolygonStraightSkeleton())
        self.addAlgorithm(Triangulate_2dz())

    def id(self) -> str:  # noqa: A003
        """Unique provider id, used for identifying it. This string should be unique, \
        short, character only string, eg "qgis" or "gdal". \
        This string should not be localised.

        :return: provider ID
        :rtype: str
        """
        return "sfcgal"

    def name(self) -> str:
        """Returns the provider name, which is used to describe the provider
        within the GUI. This string should be short (e.g. "Lastools") and localised.

        :return: provider name
        :rtype: str
        """
        return "SFCGAL"

    def longName(self) -> str:
        """Longer version of the provider name, which can include
        extra details such as version numbers. E.g. "Lastools LIDAR tools". This string should be
        localised. The default implementation returns the same string as name().

        :return: provider long name
        :rtype: str
        """
        return self.tr("{} - Tools".format(__title__))

    def icon(self) -> QIcon:
        """QIcon used for your provider inside the Processing toolbox menu.

        :return: provider icon
        :rtype: QIcon
        """
        return QIcon(str(__icon_path__))

    def tr(self, message: str) -> str:
        """Get the translation for a string using Qt translation API.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: str
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate(self.__class__.__name__, message)

    def versionInfo(self) -> str:
        """Version information for the provider, or an empty string if this is not \
        applicable (e.g. for inbuilt Processing providers). For plugin based providers, \
        this should return the plugin’s version identifier.

        :return: version
        :rtype: str
        """
        return __version__
