# -*- coding: utf-8 -*-
"""
/***************************************************************************
 QRiSDockWidget
                                 A QGIS plugin
 QGIS Riverscapes Studio (QRiS)
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                             -------------------
        begin                : 2021-05-06
        git sha              : $Format:%H$
        copyright            : (C) 2021 by North Arrow Research
        email                : info@northarrowresearch.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

import os
import traceback

from qgis.PyQt import QtCore, QtGui, QtWidgets
from qgis.core import Qgis, QgsMessageLog
from qgis.gui import QgisInterface

from ..model.project import Project
from ..model.analysis import ANALYSIS_MACHINE_CODE, Analysis
from ..model.analysis_metric import AnalysisMetric
from ..model.db_item import DB_MODE_CREATE, DB_MODE_IMPORT, DBItem, DBItemModel
from ..model.event import AS_BUILT_EVENT_TYPE_ID, DCE_EVENT_TYPE_ID, DESIGN_EVENT_TYPE_ID, Event
from ..model.raster import BASEMAP_MACHINE_CODE, Raster
from ..model.sample_frame import get_sample_frame_ids
from ..model.metric import Metric
from ..model.metric_value import MetricValue, load_metric_values

from ..lib.unit_conversion import short_unit_name
from ..gp import analysis_metrics
from ..gp.analysis_metrics import MetricInputMissingError
from ..QRiS.path_utilities import parse_posix_path

from .utilities import add_help_button
from .frm_metric_value import FrmMetricValue
from .frm_calculate_all_metrics import FrmCalculateAllMetrics
from .frm_export_metrics import FrmExportMetrics
from .frm_analysis_properties import FrmAnalysisProperties
from .frm_analysis_units import FrmAnalysisUnits
from .frm_analysis_explorer import FrmAnalysisExplorer
from .widgets.analysis_table import AnalysisTable

class FrmAnalysisDocWidget(QtWidgets.QDockWidget):

    closing = QtCore.pyqtSignal()

    def __init__(self, parent, iface: QgisInterface):

        super(FrmAnalysisDocWidget, self).__init__(parent)
        self.iface = iface
        self.setFeatures(QtWidgets.QDockWidget.DockWidgetClosable | QtWidgets.QDockWidget.DockWidgetMovable | QtWidgets.QDockWidget.DockWidgetFloatable)
        # Store the connections so they can be disconnected when the form is closed
        self.connections = {}
        self.setupUi()
        self.resize(500, 1000)

        self.frmMetricValue = None

        self.destroyed.connect(self.cleanup_connections)

    def configure_analysis(self, project: Project, analysis: Analysis, event: Event):

        self.qris_project = project
        self.analysis = analysis
        self.txtName.setText(analysis.name)

        # Set Sample Frames
        frame_ids = get_sample_frame_ids(self.qris_project.project_file, self.analysis.sample_frame.id)
        self.segments_model = DBItemModel(frame_ids)
        self.cboSampleFrame.setModel(self.segments_model)

        # Events 
        valid_event_types = [DCE_EVENT_TYPE_ID, DESIGN_EVENT_TYPE_ID, AS_BUILT_EVENT_TYPE_ID]
        events = {event_id: event for event_id, event in self.qris_project.events.items() if event.event_type.id in valid_event_types}

        # Filter events if selected_events metadata exists, but only keep those that still exist
        selected_event_ids = self.analysis.metadata.get('selected_events')
        if selected_event_ids:
            valid_event_ids = [eid for eid in selected_event_ids if eid in events]
            events = {eid: evt for eid, evt in events.items() if eid in valid_event_ids}
            # Optionally, update the analysis metadata to remove missing event IDs
            if len(valid_event_ids) != len(selected_event_ids):
                self.analysis.metadata['selected_events'] = valid_event_ids

        self.events_model = DBItemModel(events)
        self.cboEvent.setModel(self.events_model)

        # Configure the table
        self.table.configure(project, analysis)
        self.table.build_table() # Default to all metrics
        self.load_table_values()

    def build_table(self):
        # Delegate to table widget
        self.table.build_table() # Default, internal state managed by table now
        self.load_table_values()

    def load_table_values(self):
        # Delegate to table widget
        event = self.cboEvent.currentData(QtCore.Qt.UserRole)
        mask_feature_id = self.cboSampleFrame.currentData(QtCore.Qt.UserRole).id
        self.table.load_values(event, mask_feature_id)

    # set_status method is removed as it is now in AnalysisTable

    def cmdCalculate_clicked(self):

        frm = FrmCalculateAllMetrics(self)
        result = frm.exec_()

        if result == QtWidgets.QDialog.Accepted:
            sample_frame_features = [self.cboSampleFrame.itemData(i, QtCore.Qt.UserRole) for i in range(self.cboSampleFrame.count())] if frm.rdoAllSF.isChecked() else [self.cboSampleFrame.currentData(QtCore.Qt.UserRole)]
            data_capture_events = [self.cboEvent.itemData(i, QtCore.Qt.UserRole) for i in range(self.cboEvent.count())] if frm.rdoAllDCE.isChecked() else [self.cboEvent.currentData(QtCore.Qt.UserRole)]

            errors = False
            missing_data = False
            analysis_params = {}
            centerline = self.analysis.metadata.get('centerline', None)
            if centerline is not None and centerline in self.qris_project.profiles:
                analysis_params['centerline'] = self.qris_project.profiles[centerline]
            dem = self.analysis.metadata.get('dem', None)
            if dem is not None and dem in self.qris_project.rasters:
                analysis_params['dem'] = self.qris_project.rasters[dem]
            valley_bottom = self.analysis.metadata.get('valley_bottom', None)
            if valley_bottom is not None and valley_bottom in self.qris_project.valley_bottoms: 
                analysis_params['valley_bottom'] = self.qris_project.valley_bottoms[valley_bottom]

            for sample_frame_feature in sample_frame_features:
                for data_capture_event in data_capture_events:
                    metric_values = load_metric_values(self.qris_project.project_file, self.analysis, data_capture_event, sample_frame_feature.id, self.qris_project.metrics)
                    for analysis_metric in self.analysis.analysis_metrics.values():
                        metric: Metric = analysis_metric.metric
                        metric_value: MetricValue = metric_values.get(metric.id, MetricValue(metric, None, None, False, None, None, metric.default_unit_id, None))
                        if metric_value.automated_value is not None and not frm.chkOverwrite.isChecked():
                            continue
                        if metric.metric_function is None:
                            # Metric is not defined in database. continue
                            continue
                        if metric.can_calculate_automated(self.qris_project, data_capture_event.id, self.analysis.id) is False:
                            QgsMessageLog.logMessage(f'Unable to calculate metric {metric.name} for {data_capture_event.name} due to missing required layer in the data capture event.', 'QRiS_Metrics', Qgis.Warning)
                            continue
                        try:
                            metric_calculation = getattr(analysis_metrics, metric.metric_function)
                            result = metric_calculation(self.qris_project.project_file, sample_frame_feature.id, data_capture_event.id, metric.metric_params, analysis_params)
                            metric_value.automated_value = result
                            if frm.chkForceActive.isChecked():
                                metric_value.is_manual = False
                            
                            # Clear any previous errors
                            if metric_value.metadata is None:
                                metric_value.metadata = {}
                            metric_value.metadata.pop('calculation_error', None)
                            
                            metric_value.save(self.qris_project.project_file, self.analysis, data_capture_event, sample_frame_feature.id, metric.default_unit_id)
                            QgsMessageLog.logMessage(f'Successfully calculated metric {metric.name} for {data_capture_event.name} sample frame {sample_frame_feature.id}', 'QRiS_Metrics', Qgis.Info)
                        except MetricInputMissingError as ex:
                            missing_data = True
                            QgsMessageLog.logMessage(f'Error calculating metric {metric.name}: {ex}', 'QRiS_Metrics', Qgis.Warning)
                            
                            # Persist Error
                            if metric_value.metadata is None:
                                metric_value.metadata = {}
                            metric_value.metadata['calculation_error'] = str(ex)
                            metric_value.automated_value = None
                            metric_value.save(self.qris_project.project_file, self.analysis, data_capture_event, sample_frame_feature.id, metric.default_unit_id)
                            
                            continue
                        except Exception as ex:
                            errors = True
                            QgsMessageLog.logMessage(f'Error calculating metric {metric.name}: {ex}', 'QRiS_Metrics', Qgis.Warning)

                            # Persist Error
                            if metric_value.metadata is None:
                                metric_value.metadata = {}
                            metric_value.metadata['calculation_error'] = str(ex)
                            metric_value.automated_value = None
                            metric_value.save(self.qris_project.project_file, self.analysis, data_capture_event, sample_frame_feature.id, metric.default_unit_id)

                            continue
            if errors is False and missing_data is False:
                self.iface.messageBar().pushMessage('Metrics', 'All metrics successfully calculated.', level=Qgis.Success)
            else:
                if missing_data is True:
                    self.iface.messageBar().pushMessage('Metrics', 'One or more metrics were not calculated due to missing data requirements. See log for details.', level=Qgis.Success)
                if errors is True:
                    self.iface.messageBar().pushMessage('Metrics', 'Onr or more metrics were not calculated due to processing error(s). See log for details.', level=Qgis.Warning)
            self.load_table_values()

    def export_table(self):

        # open modal dialog to select export file
        current_sample_frame = self.cboSampleFrame.currentData(QtCore.Qt.UserRole)
        current_data_capture_event = self.cboEvent.currentData(QtCore.Qt.UserRole)
        frm = FrmExportMetrics(self, self.iface, self.qris_project, self.analysis, current_data_capture_event, current_sample_frame)
        result = frm.exec_()

        if result == QtWidgets.QDialog.Accepted:
            self.iface.messageBar().pushMessage('Export Metrics', f'Exported metrics to {frm.txtOutpath.text()}', level=Qgis.Success)

    def cmdProperties_clicked(self):

        frm = FrmAnalysisProperties(self, self.qris_project, self.analysis)
        result = frm.exec_()
        if result is not None and result != 0:
            self.txtName.setText(frm.analysis.name)
            self.configure_analysis(self.qris_project, frm.analysis, None)
            self.build_table()

    def edit_metric_value(self, metric_value):

        event = self.cboEvent.currentData(QtCore.Qt.UserRole)
        mask_feature = self.cboSampleFrame.currentData(QtCore.Qt.UserRole)

        frm = FrmMetricValue(self, self.qris_project, self.analysis, event, mask_feature.id, metric_value)
        result = frm.exec_()
        if result is not None and result != 0:
            self.load_table_values()

    def calculate_metric_value(self, analysis_metric, metric_value):

        event = self.cboEvent.currentData(QtCore.Qt.UserRole)
        mask_feature = self.cboSampleFrame.currentData(QtCore.Qt.UserRole)
        
        if metric_value is None:
             # Create new metric value instance (in memory, will be saved)
             metric = analysis_metric.metric
             metric_value = MetricValue(metric, None, None, False, None, '', None, None)
        
        if analysis_metric.metric.metric_function is None:
             return

        # Gathering params 
        metric_params: dict = analysis_metric.metric.metric_params
        analysis_params = {}
        centerline = self.analysis.metadata.get('centerline', None)
        if centerline is not None and centerline in self.qris_project.profiles:
            analysis_params['centerline'] = self.qris_project.profiles[centerline]
        dem = self.analysis.metadata.get('dem', None)
        if dem is not None and dem in self.qris_project.rasters:
            analysis_params['dem'] = self.qris_project.rasters[dem]
        valley_bottom = self.analysis.metadata.get('valley_bottom', None)
        if valley_bottom is not None and valley_bottom in self.qris_project.valley_bottoms: 
            analysis_params['valley_bottom'] = self.qris_project.valley_bottoms[valley_bottom]
        
        metric_calculation = getattr(analysis_metrics, analysis_metric.metric.metric_function)
        
        try:
            QtWidgets.QApplication.setOverrideCursor(QtCore.Qt.WaitCursor)
            
            result = metric_calculation(self.qris_project.project_file, mask_feature.id, event.id, metric_params, analysis_params)
            metric_value.automated_value = result
            metric_value.is_manual = False 
            
            # Clear any previous errors
            if metric_value.metadata is None:
                metric_value.metadata = {}
            metric_value.metadata.pop('calculation_error', None)
            
            metric_value.save(self.qris_project.project_file, self.analysis, event, mask_feature.id)
            
            QtWidgets.QApplication.restoreOverrideCursor()
            
            self.load_table_values()
            
        except Exception as ex:
            # Persist Error
            if metric_value.metadata is None:
                metric_value.metadata = {}
            metric_value.metadata['calculation_error'] = str(ex)
            metric_value.automated_value = None
            metric_value.save(self.qris_project.project_file, self.analysis, event, mask_feature.id)
            
            QtWidgets.QApplication.restoreOverrideCursor()
            # QtWidgets.QMessageBox.warning(self, f'Error Calculating Metric', f'{ex}\n\nSee log for additional details.')
            QgsMessageLog.logMessage(str(traceback.format_exc()), f'QRiS_Metrics', level=Qgis.Warning)
            
            self.load_table_values()

    def resizeEvent(self, event):
        try:
            if hasattr(self, "table") and isinstance(self.table, QtWidgets.QTableWidget):
                # Dynamically size the Metric column (index 1) to 80% of width
                if self.table.columnWidth(1) != int(self.table.width() * 0.8):
                    self.table.setColumnWidth(1, int(self.table.width() * 0.8))
        except RuntimeError:
            # This can happen if the widget has been deleted during undock
            pass
        super().resizeEvent(event)

    def closeEvent(self, event):
        self.closing.emit()
        for signal in list(self.connections.keys()):
            signal.disconnect(self.connections.pop(signal))
        QtWidgets.QDockWidget.closeEvent(self, event)

    def cleanup_connections(self, *args):
        for signal in list(self.connections.keys()):
            try:
                signal.disconnect(self.connections.pop(signal))
            except Exception:
                pass

    def setupUi(self):

        self.setWindowTitle('QRiS Analysis')
        self.dockWidgetContents = QtWidgets.QWidget(self)

        # Top level layout must include parent. Widgets added to this layout do not need parent.
        self.vert = QtWidgets.QVBoxLayout(self.dockWidgetContents)

        self.grid = QtWidgets.QGridLayout()
        self.vert.addLayout(self.grid)

        self.lblName = QtWidgets.QLabel('Analysis')
        self.grid.addWidget(self.lblName, 0, 0, 1, 1)

        self.horizName = QtWidgets.QHBoxLayout()
        self.grid.addLayout(self.horizName, 0, 1, 1, 1)

        self.txtName = QtWidgets.QLineEdit()
        self.txtName.setMaxLength(255)
        self.txtName.setReadOnly(True)
        self.horizName.addWidget(self.txtName)

        self.cmdProperties = QtWidgets.QPushButton('Properties')
        self.connections[self.cmdProperties.clicked] = self.cmdProperties.clicked.connect(self.cmdProperties_clicked)
        self.cmdProperties.setToolTip('Edit analysis properties, indcluding metrics and indicators')
        self.cmdProperties.setToolTipDuration(2000)
        self.horizName.addWidget(self.cmdProperties)

        self.cmdOpenSummary = QtWidgets.QPushButton()
        icon = QtGui.QIcon(':/plugins/qris_toolbar/analysis_summary')
        self.cmdOpenSummary.setIcon(icon)
        self.cmdOpenSummary.setToolTip('Open Analysis Summary')
        self.cmdOpenSummary.setToolTipDuration(2000)
        self.connections[self.cmdOpenSummary.clicked] = self.cmdOpenSummary.clicked.connect(lambda: FrmAnalysisExplorer(self, self.qris_project, self.analysis.id).exec_())
        self.horizName.addWidget(self.cmdOpenSummary)

        self.lblEvent = QtWidgets.QLabel('Data Capture Event')
        self.grid.addWidget(self.lblEvent, 1, 0, 1, 1)

        self.horizEvent = QtWidgets.QHBoxLayout()
        self.grid.addLayout(self.horizEvent, 1, 1, 1, 1)

        self.cboEvent = QtWidgets.QComboBox()
        self.connections[self.cboEvent.currentIndexChanged] = self.cboEvent.currentIndexChanged.connect(self.load_table_values)
        self.horizEvent.addWidget(self.cboEvent)

        self.cmdCalculate = QtWidgets.QPushButton('Calculate...')
        self.connections[self.cmdCalculate.clicked] = self.cmdCalculate.clicked.connect(self.cmdCalculate_clicked)
        self.cmdCalculate.setSizePolicy(QtWidgets.QSizePolicy.Fixed, QtWidgets.QSizePolicy.Fixed)
        self.cmdCalculate.setToolTip('Options to calculate metrics and indicators for the selected event and sample frame')
        self.cmdCalculate.setToolTipDuration(2000)
        self.horizEvent.addWidget(self.cmdCalculate, 0)

        self.lblSegment = QtWidgets.QLabel('Mask Polygon')
        self.grid.addWidget(self.lblSegment, 2, 0, 1, 1)

        self.cboSampleFrame = QtWidgets.QComboBox()
        self.connections[self.cboSampleFrame.currentIndexChanged] = self.cboSampleFrame.currentIndexChanged.connect(self.load_table_values)
        self.grid.addWidget(self.cboSampleFrame, 2, 1, 1, 1)

        self.horiz = QtWidgets.QHBoxLayout()
        self.vert.addLayout(self.horiz)

        self.spacerItem = QtWidgets.QSpacerItem(40, 20, QtWidgets.QSizePolicy.Expanding, QtWidgets.QSizePolicy.Minimum)
        self.horiz.addItem(self.spacerItem)

        self.table = AnalysisTable()
        self.connections[self.table.metric_edit_requested] = self.table.metric_edit_requested.connect(self.edit_metric_value)
        self.connections[self.table.metric_calculate_requested] = self.table.metric_calculate_requested.connect(self.calculate_metric_value)
        self.vert.addWidget(self.table)

        # add export table button at the bottom right of the dock
        self.horizExport = QtWidgets.QHBoxLayout()
        self.vert.addLayout(self.horizExport)

        self.horizExport.addWidget(add_help_button(self, 'analyses'))

        self.cmdUnits = QtWidgets.QPushButton('Display Units')
        self.cmdUnits.setMenu(self.table.units_menu)
        self.horizExport.addWidget(self.cmdUnits)

        self.spacerItem = QtWidgets.QSpacerItem(40, 20, QtWidgets.QSizePolicy.Expanding, QtWidgets.QSizePolicy.Minimum)
        self.horizExport.addItem(self.spacerItem)

        self.cmdExport = QtWidgets.QPushButton()
        self.cmdExport.setText('Export Metrics Table')
        self.connections[self.cmdExport.clicked] = self.cmdExport.clicked.connect(self.export_table)
        self.cmdExport.setToolTip('Export the table to a csv, json, or Excel file')
        self.horizExport.addWidget(self.cmdExport, 0)

        self.setWidget(self.dockWidgetContents)
