# -*- coding: utf-8 -*-
"""
/***************************************************************************
 QRiSDockWidget
                                 A QGIS plugin
 QGIS Riverscapes Studio (QRiS)
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                             -------------------
        begin                : 2021-05-06
        git sha              : $Format:%H$
        copyright            : (C) 2021 by North Arrow Research
        email                : info@northarrowresearch.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

import os

from PyQt5 import QtCore, QtGui, QtWidgets
from qgis.core import Qgis, QgsMessageLog
from qgis.utils import iface

from ..model.project import Project
from ..model.analysis import ANALYSIS_MACHINE_CODE, Analysis
from ..model.analysis_metric import AnalysisMetric
from ..model.db_item import DB_MODE_CREATE, DB_MODE_IMPORT, DBItem, DBItemModel
from ..model.event import DCE_EVENT_TYPE_ID, Event
from ..model.raster import BASEMAP_MACHINE_CODE, Raster
from ..model.sample_frame import get_sample_frame_ids
from ..model.metric import Metric
from ..model.metric_value import MetricValue, load_metric_values

from ..lib.unit_conversion import short_unit_name
from ..gp import analysis_metrics
from ..gp.analysis_metrics import MetricInputMissingError
from ..QRiS.path_utilities import parse_posix_path

from .utilities import add_help_button
from .frm_metric_value import FrmMetricValue
from .frm_calculate_all_metrics import FrmCalculateAllMetrics
from .frm_export_metrics import FrmExportMetrics
from .frm_analysis_properties import FrmAnalysisProperties
from .frm_analysis_units import FrmAnalysisUnits
from .frm_analysis_explorer import FrmAnalysisExplorer

column = {
    'edit': 0,
    'metric': 1,
    'units': 2,
    'value': 3,
    'uncertainty': 4,
    'status': 5,
    'warnings': 6
}

class FrmAnalysisDocWidget(QtWidgets.QDockWidget):

    def __init__(self, parent):

        super(FrmAnalysisDocWidget, self).__init__(parent)
        self.setAttribute(QtCore.Qt.WA_QuitOnClose)
        self.setFeatures(QtWidgets.QDockWidget.DockWidgetClosable)  # <--- Add this line
        # Store the connections so they can be disconnected when the form is closed
        self.connections = {}
        self.setupUi()

        self.frmMetricValue = None

        self.destroyed.connect(self.cleanup_connections)

    def configure_analysis(self, project: Project, analysis: Analysis, event: Event):

        self.qris_project = project
        self.analysis = analysis
        self.txtName.setText(analysis.name)

        # Set Sample Frames
        frame_ids = get_sample_frame_ids(self.qris_project.project_file, self.analysis.sample_frame.id)
        self.segments_model = DBItemModel(frame_ids)
        self.cboSampleFrame.setModel(self.segments_model)

        # Events 
        self.events_model = DBItemModel({event_id: event for event_id, event in self.qris_project.events.items() if event.event_type.id == DCE_EVENT_TYPE_ID})
        self.cboEvent.setModel(self.events_model)

        # Build the metric table (this will also load the metric values)
        self.build_table()

    def build_table(self):

        # Disconnect the signals if they're already connected
        for signal in [self.table.itemClicked, self.table.doubleClicked]:
            if signal in self.connections:
                signal.disconnect(self.connections.pop(signal))

        self.table.clearContents()

        self.table.setRowCount(0)
        analysis_metrics = list(metric for metric in self.analysis.analysis_metrics.values() if metric.level_id == 2) if self.rdoIndicators.isChecked() else list(self.analysis.analysis_metrics.values())
        self.table.setRowCount(len(analysis_metrics))
        for row in range(len(analysis_metrics)):
            analysis_metric: AnalysisMetric = analysis_metrics[row]
            metric: Metric = analysis_metric.metric
            
            edit_icon = QtGui.QIcon(':/plugins/qris_toolbar/options')
            item = QtWidgets.QTableWidgetItem()
            item.setIcon(edit_icon)
            self.table.setItem(row, column['edit'], item)
        
            label_metric = QtWidgets.QTableWidgetItem()
            label_metric.setText(metric.name)
            self.table.setItem(row, column['metric'], label_metric)
            label_metric.setData(QtCore.Qt.UserRole, analysis_metric)

            label_units = QtWidgets.QTableWidgetItem()
            display_unit = short_unit_name(self.analysis.units.get(metric.unit_type, None))
            if metric.normalized:
                if display_unit != 'ratio':
                    normalization_unit = self.analysis.units['distance'] # TODO only normaization by distance is supported, need to add support for other Normalization types
                    display_unit = f'{display_unit}/{short_unit_name(normalization_unit)}'
            label_units.setText(display_unit)
            self.table.setItem(row, column['units'], label_units)           

            label_value = QtWidgets.QTableWidgetItem()
            self.table.setItem(row, column['value'], label_value)

            label_uncertainty = QtWidgets.QTableWidgetItem()
            self.table.setItem(row, column['uncertainty'], label_uncertainty)

            status_item = QtWidgets.QTableWidgetItem()
            self.table.setItem(row, column['status'], status_item)

        # Connect the signals and store the connections
        self.connections[self.table.itemClicked] = self.table.itemClicked.connect(lambda item: self.edit_metric_value(item) if item.column() == column['edit'] else None)
        self.connections[self.table.doubleClicked] = self.table.doubleClicked.connect(self.edit_metric_value)
        self.table.resizeColumnToContents(1)
        
        self.load_table_values()

    def load_table_values(self):

        event = self.cboEvent.currentData(QtCore.Qt.UserRole)
        mask_feature_id = self.cboSampleFrame.currentData(QtCore.Qt.UserRole).id

        if event is not None and mask_feature_id is not None:
            # Load latest metric values from DB
            metric_values = load_metric_values(self.qris_project.project_file, self.analysis, event, mask_feature_id, self.qris_project.metrics)

            # Loop over active metrics and load values into grid
            for row in range(self.table.rowCount()):
                analysis_metric: AnalysisMetric = self.table.item(row, column['metric']).data(QtCore.Qt.UserRole)
                metric: Metric = analysis_metric.metric
                metric_value = None
                metric_value_text = 'null'
                uncertainty_text = ''
                if metric.id in metric_values:
                    metric_value = metric_values[metric.id]
                    display_unit = self.analysis.units.get(metric.unit_type, None)
                    if metric.normalized:
                        if display_unit != 'ratio':
                            display_unit = self.analysis.units['distance']
                    display_unit = None if display_unit in ['count', 'ratio'] else display_unit
                    metric_value_text = metric_value.current_value_as_string(display_unit)
                    uncertainty_text = metric_value.uncertainty_as_string()
                self.table.item(row, column['value']).setData(QtCore.Qt.UserRole, metric_value)
                self.table.item(row, column['value']).setText(metric_value_text)
                self.table.item(row, column['uncertainty']).setText(uncertainty_text)
                self.set_status(row)
        self.table.resizeColumnToContents(2)

    def set_status(self, row):

        # Get the metric, value and status item for the row
        status_item = self.table.item(row, column['status'])
        metric: Metric = self.table.item(row, 1).data(QtCore.Qt.UserRole).metric
        metric_value: MetricValue = self.table.item(row, column['value']).data(QtCore.Qt.UserRole)

        # Default Status none exists or selected
        status_manual_icon = QtGui.QPixmap(':/plugins/qris_toolbar/manual_none')
        status_auto_icon = QtGui.QPixmap(':/plugins/qris_toolbar/auto_none')
        status_text = None

        if metric_value is not None:
            # set icons for value existence
            if metric_value.manual_value is not None:
                status_manual_icon = QtGui.QPixmap(':/plugins/qris_toolbar/manual_exists')
                status_text = 'Manual Value'
            if metric_value.automated_value is not None:
                status_auto_icon = QtGui.QPixmap(':/plugins/qris_toolbar/auto_exists')
                status_text = 'Automated Value' if status_text is None else 'Manual and Automated Values'
            if status_text is not None:
                status_text = f'{status_text} exists for this metric.'
            # set icons for value selection
            if metric_value.is_manual == 1 and metric_value.manual_value is not None:
                status_manual_icon = QtGui.QPixmap(':/plugins/qris_toolbar/manual_selected')
                status_text = f'{status_text} (Manual Value Selected)'
                if metric_value.automated_value is not None:
                    # set warining icon if manual value is more than 10% different from automated value
                    if metric.tolerance is not None and abs(metric_value.manual_value - metric_value.automated_value) > metric.tolerance * metric_value.automated_value:
                        status_manual_icon = QtGui.QPixmap(':/plugins/qris_toolbar/manual_selected_warning')
                        status_text = f'{status_text} (Manual Value Selected - Warning: Manual value differs from Automated value by more than {metric.tolerance * 100}%.)'
            if metric_value.is_manual != 1 and metric_value.automated_value is not None:
                status_auto_icon = QtGui.QPixmap(':/plugins/qris_toolbar/auto_selected')
                status_text = f'{status_text} (Automated Value Selected)'

        # set the tool tip
        if status_text is None:
            status_text = 'No Values currently exist for this metric.'
        status_item.setToolTip(status_text)
        # set the Icons
        icon = QtGui.QIcon()
        icon.actualSize(QtCore.QSize(32, 16))
        pixmap = QtGui.QPixmap(32, 16)
        pixmap.fill(QtCore.Qt.transparent)
        painter = QtGui.QPainter()
        painter.begin(pixmap)
        painter.setBackgroundMode(QtCore.Qt.TransparentMode)
        painter.drawPixmap(0, 0, status_manual_icon)
        painter.drawPixmap(16, 0, status_auto_icon)
        icon.addPixmap(pixmap)
        status_item.setIcon(icon)
        painter = None

    def cmdCalculate_clicked(self):

        frm = FrmCalculateAllMetrics(self)
        result = frm.exec_()

        if result == QtWidgets.QDialog.Accepted:
            sample_frame_features = [self.cboSampleFrame.itemData(i, QtCore.Qt.UserRole) for i in range(self.cboSampleFrame.count())] if frm.rdoAllSF.isChecked() else [self.cboSampleFrame.currentData(QtCore.Qt.UserRole)]
            data_capture_events = [self.cboEvent.itemData(i, QtCore.Qt.UserRole) for i in range(self.cboEvent.count())] if frm.rdoAllDCE.isChecked() else [self.cboEvent.currentData(QtCore.Qt.UserRole)]

            errors = False
            missing_data = False
            analysis_params = {}
            centerline = self.analysis.metadata.get('centerline', None)
            if centerline is not None and centerline in self.qris_project.profiles:
                analysis_params['centerline'] = self.qris_project.profiles[centerline]
            dem = self.analysis.metadata.get('dem', None)
            if dem is not None and dem in self.qris_project.rasters:
                analysis_params['dem'] = self.qris_project.rasters[dem]
            valley_bottom = self.analysis.metadata.get('valley_bottom', None)
            if valley_bottom is not None and valley_bottom in self.qris_project.valley_bottoms: 
                analysis_params['valley_bottom'] = self.qris_project.valley_bottoms[valley_bottom]

            for sample_frame_feature in sample_frame_features:
                for data_capture_event in data_capture_events:
                    metric_values = load_metric_values(self.qris_project.project_file, self.analysis, data_capture_event, sample_frame_feature.id, self.qris_project.metrics)
                    for analysis_metric in self.analysis.analysis_metrics.values():
                        metric: Metric = analysis_metric.metric
                        metric_value: MetricValue = metric_values.get(metric.id, MetricValue(metric, None, None, False, None, None, metric.default_unit_id, None))
                        if metric_value.automated_value is not None and not frm.chkOverwrite.isChecked():
                            continue
                        if metric.metric_function is None:
                            # Metric is not defined in database. continue
                            continue
                        if metric.can_calculate_automated(self.qris_project, data_capture_event.id, self.analysis.id) is False:
                            QgsMessageLog.logMessage(f'Unable to calculate metric {metric.name} for {data_capture_event.name} due to missing required layer in the data capture event.', 'QRiS_Metrics', Qgis.Warning)
                            continue
                        try:
                            metric_calculation = getattr(analysis_metrics, metric.metric_function)
                            result = metric_calculation(self.qris_project.project_file, sample_frame_feature.id, data_capture_event.id, metric.metric_params, analysis_params)
                            metric_value.automated_value = result
                            if frm.chkForceActive.isChecked():
                                metric_value.is_manual = False
                            metric_value.save(self.qris_project.project_file, self.analysis, data_capture_event, sample_frame_feature.id, metric.default_unit_id)
                            QgsMessageLog.logMessage(f'Successfully calculated metric {metric.name} for {data_capture_event.name} sample frame {sample_frame_feature.id}', 'QRiS_Metrics', Qgis.Info)
                        except MetricInputMissingError as ex:
                            missing_data = True
                            QgsMessageLog.logMessage(f'Error calculating metric {metric.name}: {ex}', 'QRiS_Metrics', Qgis.Warning)
                            continue
                        except Exception as ex:
                            errors = True
                            QgsMessageLog.logMessage(f'Error calculating metric {metric.name}: {ex}', 'QRiS_Metrics', Qgis.Warning)
                            continue
            if errors is False and missing_data is False:
                iface.messageBar().pushMessage('Metrics', 'All metrics successfully calculated.', level=Qgis.Success)
            else:
                if missing_data is True:
                    iface.messageBar().pushMessage('Metrics', 'One or more metrics were not calculated due to missing data requirements. See log for details.', level=Qgis.Success)
                if errors is True:
                    iface.messageBar().pushMessage('Metrics', 'Onr or more metrics were not calculated due to processing error(s). See log for details.', level=Qgis.Warning)
            self.load_table_values()

    def export_table(self):

        # open modal dialog to select export file
        current_sample_frame = self.cboSampleFrame.currentData(QtCore.Qt.UserRole)
        current_data_capture_event = self.cboEvent.currentData(QtCore.Qt.UserRole)
        frm = FrmExportMetrics(self, self.qris_project, self.analysis, current_data_capture_event, current_sample_frame)
        result = frm.exec_()

        if result == QtWidgets.QDialog.Accepted:
            iface.messageBar().pushMessage('Export Metrics', f'Exported metrics to {frm.txtOutpath.text()}', level=Qgis.Success)

    def cmdProperties_clicked(self):

        frm = FrmAnalysisProperties(self, self.qris_project, self.analysis)
        result = frm.exec_()
        if result is not None and result != 0:
            self.txtName.setText(frm.analysis.name)
            self.configure_analysis(self.qris_project, frm.analysis, None)
            self.build_table()

    def edit_metric_value(self, mi):

        metric_value = self.table.item(mi.row(), column['value']).data(QtCore.Qt.UserRole)
        if metric_value is None:
            metric: Metric = self.table.item(mi.row(), column['metric']).data(QtCore.Qt.UserRole).metric
            metric_value = MetricValue(metric, None, None, True, None, None, metric.default_unit_id, {})
        event = self.cboEvent.currentData(QtCore.Qt.UserRole)
        mask_feature = self.cboSampleFrame.currentData(QtCore.Qt.UserRole)

        frm = FrmMetricValue(self, self.qris_project, self.qris_project.metrics, self.analysis, event, mask_feature.id, metric_value)
        result = frm.exec_()
        if result is not None and result != 0:
            self.build_table()

    def open_units_dialog(self):
        dialog = FrmAnalysisUnits(self, self.analysis)
        result = dialog.exec_()
        if result == QtWidgets.QDialog.Accepted:
            self.build_table()

    def resizeEvent(self, event):
        try:
            if hasattr(self, "table") and isinstance(self.table, QtWidgets.QTableWidget):
                if self.table.columnWidth(0) > self.table.width():
                    self.table.setColumnWidth(0, int(self.table.width() * 0.8))
        except RuntimeError:
            # This can happen if the widget has been deleted during undock
            pass
        super().resizeEvent(event)

    def closeEvent(self, event):
        for signal in list(self.connections.keys()):
            signal.disconnect(self.connections.pop(signal))
        QtWidgets.QDockWidget.closeEvent(self, event)

    def cleanup_connections(self, *args):
        for signal in list(self.connections.keys()):
            try:
                signal.disconnect(self.connections.pop(signal))
            except Exception:
                pass

    def setupUi(self):

        self.setWindowTitle('QRiS Analysis')
        self.dockWidgetContents = QtWidgets.QWidget(self)

        # Top level layout must include parent. Widgets added to this layout do not need parent.
        self.vert = QtWidgets.QVBoxLayout(self.dockWidgetContents)

        self.grid = QtWidgets.QGridLayout()
        self.vert.addLayout(self.grid)

        self.lblName = QtWidgets.QLabel('Analysis')
        self.grid.addWidget(self.lblName, 0, 0, 1, 1)

        self.horizName = QtWidgets.QHBoxLayout()
        self.grid.addLayout(self.horizName, 0, 1, 1, 1)

        self.txtName = QtWidgets.QLineEdit()
        self.txtName.setMaxLength(255)
        self.txtName.setReadOnly(True)
        self.horizName.addWidget(self.txtName)

        self.cmdProperties = QtWidgets.QPushButton('Properties')
        self.connections[self.cmdProperties.clicked] = self.cmdProperties.clicked.connect(self.cmdProperties_clicked)
        self.cmdProperties.setToolTip('Edit analysis properties, indcluding metrics and indicators')
        self.cmdProperties.setToolTipDuration(2000)
        self.horizName.addWidget(self.cmdProperties)

        self.cmdOpenSummary = QtWidgets.QPushButton()
        icon = QtGui.QIcon(':/plugins/qris_toolbar/analysis_summary')
        self.cmdOpenSummary.setIcon(icon)
        self.cmdOpenSummary.setToolTip('Open Analysis Summary')
        self.cmdOpenSummary.setToolTipDuration(2000)
        self.connections[self.cmdOpenSummary.clicked] = self.cmdOpenSummary.clicked.connect(lambda: FrmAnalysisExplorer(self, self.qris_project, self.analysis.id).exec_())
        self.horizName.addWidget(self.cmdOpenSummary)

        self.lblEvent = QtWidgets.QLabel('Data Capture Event')
        self.grid.addWidget(self.lblEvent, 1, 0, 1, 1)

        self.horizEvent = QtWidgets.QHBoxLayout()
        self.grid.addLayout(self.horizEvent, 1, 1, 1, 1)

        self.cboEvent = QtWidgets.QComboBox()
        self.connections[self.cboEvent.currentIndexChanged] = self.cboEvent.currentIndexChanged.connect(self.load_table_values)
        self.horizEvent.addWidget(self.cboEvent)

        self.cmdCalculate = QtWidgets.QPushButton('Calculate...')
        self.connections[self.cmdCalculate.clicked] = self.cmdCalculate.clicked.connect(self.cmdCalculate_clicked)
        self.cmdCalculate.setSizePolicy(QtWidgets.QSizePolicy.Fixed, QtWidgets.QSizePolicy.Fixed)
        self.cmdCalculate.setToolTip('Options to calculate metrics and indicators for the selected event and sample frame')
        self.cmdCalculate.setToolTipDuration(2000)
        self.horizEvent.addWidget(self.cmdCalculate, 0)

        self.lblSegment = QtWidgets.QLabel('Mask Polygon')
        self.grid.addWidget(self.lblSegment, 2, 0, 1, 1)

        self.cboSampleFrame = QtWidgets.QComboBox()
        self.connections[self.cboSampleFrame.currentIndexChanged] = self.cboSampleFrame.currentIndexChanged.connect(self.load_table_values)
        self.grid.addWidget(self.cboSampleFrame, 2, 1, 1, 1)

        self.lblDisplay = QtWidgets.QLabel('Display Values')
        self.grid.addWidget(self.lblDisplay, 3, 0, 1, 1)

        self.layoutDisplay = QtWidgets.QHBoxLayout()

        self.rdoAll = QtWidgets.QRadioButton('Metrics and Indicators')
        self.rdoAll.setChecked(True)
        self.connections[self.rdoAll.toggled] = self.rdoAll.toggled.connect(self.build_table)
        self.layoutDisplay.addWidget(self.rdoAll)

        self.rdoIndicators = QtWidgets.QRadioButton('Indicators Only')
        self.rdoIndicators.setChecked(False)
        self.layoutDisplay.addWidget(self.rdoIndicators)

        self.layoutDisplay.addStretch()
        self.btnUnits = QtWidgets.QPushButton('Units')
        self.btnUnits.setToolTip('Change metric display units')
        self.connections[self.btnUnits.clicked] = self.btnUnits.clicked.connect(self.open_units_dialog)
        self.layoutDisplay.addWidget(self.btnUnits)

        self.grid.addLayout(self.layoutDisplay, 3, 1, 1, 1)

        self.horiz = QtWidgets.QHBoxLayout()
        self.vert.addLayout(self.horiz)

        self.spacerItem = QtWidgets.QSpacerItem(40, 20, QtWidgets.QSizePolicy.Expanding, QtWidgets.QSizePolicy.Minimum)
        self.horiz.addItem(self.spacerItem)

        self.table = QtWidgets.QTableWidget(0, 7)
        self.table.setEditTriggers(QtWidgets.QTableWidget.NoEditTriggers)
        self.table.verticalHeader().setVisible(False)
        self.table.setSelectionBehavior(QtWidgets.QAbstractItemView.SelectRows)
        self.table.setSelectionMode(QtWidgets.QAbstractItemView.SingleSelection)
        self.table.setHorizontalHeaderLabels(['', 'Metric', 'Units', 'Value', 'Uncertainty', 'Status', ''])
        self.table.horizontalHeader().setSectionResizeMode(0, QtWidgets.QHeaderView.Fixed) 
        self.table.horizontalHeader().setSectionResizeMode(4, QtWidgets.QHeaderView.Fixed)
        self.table.setColumnWidth(column['edit'], 16)
        self.table.setColumnWidth(column['metric'], int(self.table.width() * 0.8))
        self.table.setColumnWidth(column['units'], 75)
        self.table.setColumnWidth(column['value'], 100)
        self.table.setColumnWidth(column['uncertainty'], 100)
        self.table.setColumnWidth(column['status'], 50)
        self.table.setColumnWidth(column['warnings'], 16)

        self.table.setIconSize(QtCore.QSize(32, 16))
        self.vert.addWidget(self.table)

        # add export table button at the bottom right of the dock
        self.horizExport = QtWidgets.QHBoxLayout()
        self.vert.addLayout(self.horizExport)

        self.horizExport.addWidget(add_help_button(self, 'analyses'))
        self.horizExport.addWidget(add_help_button(self, 'analyses/#metric-table', 'Icon Legend'))

        self.spacerItem = QtWidgets.QSpacerItem(40, 20, QtWidgets.QSizePolicy.Expanding, QtWidgets.QSizePolicy.Minimum)
        self.horizExport.addItem(self.spacerItem)

        self.cmdExport = QtWidgets.QPushButton()
        self.cmdExport.setText('Export Metrics Table')
        self.connections[self.cmdExport.clicked] = self.cmdExport.clicked.connect(self.export_table)
        self.cmdExport.setToolTip('Export the table to a csv, json, or Excel file')
        self.horizExport.addWidget(self.cmdExport, 0)

        self.setWidget(self.dockWidgetContents)
