# -*- coding: utf-8 -*-
"""
/***************************************************************************
 qosm
                                 A QGIS plugin
 Open Street Map tiles in QGIS
                              -------------------
        begin                : 2015-12-08
        git sha              : $Format:%H$
        copyright            : (C) 2015 by Dewey Dunnington / Fish & Whistle
        email                : dewey@fishandwhistle.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
import os.path

from PyQt4.QtCore import QSettings, QTranslator, qVersion, QCoreApplication, pyqtSlot
from PyQt4.QtGui import QAction, QIcon, QMenu
from qgis.core import QgsMapLayerRegistry, QgsMapLayer, QgsPluginLayerRegistry

import qosmpy.resources_rc
import qosmpy.qosmlogging as qosmlogging
from qosmpy.qosmtilelayer import QOSMTileLayer, QOSMTileLayerType
from qosmpy.dialog_settings import DialogSettings



class qosm:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'qosm_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)

            if qVersion() > '4.3.3':
                QCoreApplication.installTranslator(self.translator)

        # Create the dialog (after translation) and keep reference
        self.dlg_settings = None #TODO: add settings dialog

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&QOSM')
        
        self.toolbar = self.iface.addToolBar(u'QOSM')
        self.toolbar.setObjectName(u'QOSM')
        
        self.layers = []

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('qosm', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            self.toolbar.addAction(action)

        if add_to_menu:
            self.menuObject.addAction(action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        #start logging
        qosmlogging.initialize_logging()
        qosmlogging.log("Initizlizing GUI")
        
        self.settingsdialog = DialogSettings(None)
        
        #add menu to web
        self.menuObject = QMenu(self.menu)
        self.menuObject.setIcon(QIcon(":/plugins/qosm/icon.png"))
        
        tmpAction = QAction("_tmp", self.iface.mainWindow())
        self.iface.addPluginToWebMenu("_tmp", tmpAction)
        self._menu = self.iface.webMenu()
        self._menu.addMenu(self.menuObject)
        self.iface.removePluginWebMenu("_tmp", tmpAction)        
        
        self.add_action(
            ':/plugins/qosm/icon_newlayer.png',
            text=self.tr(u'Add OSM tile layer'),
            callback=self.addOSMLayer,
            parent=self.iface.mainWindow())
        
        self.add_action(
            ':/plugins/qosm/icon_refresh.png',
            text=self.tr(u'Refresh OSM tile layers'),
            callback=self.refreshOSMLayers,
            parent=self.iface.mainWindow())
        
        self.add_action(
            ':/plugins/qosm/icon_settings.png',
            text=self.tr(u'QOSM Settings'),
            callback=self.qosmSettings,
            parent=self.iface.mainWindow())
        
        self.pluginLayerType = QOSMTileLayerType(self, self.onAddOSMLayer)
        QgsPluginLayerRegistry.instance().addPluginLayerType(self.pluginLayerType)
        QgsMapLayerRegistry.instance().layersWillBeRemoved["QStringList"].connect(self.cleanLayerResources)


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginWebMenu(
                self.menu,
                action)
            self.iface.removeToolBarIcon(action)
            
        self.iface.webMenu().removeAction(self.menuObject.menuAction())
            
        # remove the toolbar
        del self.toolbar
        del self.menuObject
        
        QgsPluginLayerRegistry.instance().removePluginLayerType(QOSMTileLayer.LAYER_TYPE)
        QgsMapLayerRegistry.instance().layersWillBeRemoved["QStringList"].disconnect(self.cleanLayerResources)

    @pyqtSlot()
    def cleanLayerResources(self, layers):
        reg = QgsMapLayerRegistry.instance()
        for layerid in layers:
            layer = reg.mapLayer(layerid)
            if isinstance(layer, QOSMTileLayer):
                layer.cleantiles()
                #remove from internal cache of layers
                for i in range(len(self.layers)):
                    if self.layers[i].id() == layerid:
                        self.layers.pop(i)
                        break
        
    
    def addOSMLayer(self):
        """Adds a new OSM layer and opens the properties dialog to default settings"""
        # add a new qosmtilelayer
        layer = self.pluginLayerType.createLayer()
        QgsMapLayerRegistry.instance().addMapLayer(layer)
        self.pluginLayerType.showLayerProperties(layer, newlayer=True)
    
    def refreshOSMLayers(self):
        #manual refresh OSM Layers
        for layer in self.layers:
            layer.refreshtiles()
    
    def qosmSettings(self):
        self.settingsdialog.sync_dialog()
        self.settingsdialog.show()
    
    def onAddOSMLayer(self, layer):
        self.layers.append(layer)

        
        
